"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PloneDeployment = void 0;
// import { log } from 'console';
const cdk8s_1 = require("cdk8s");
const kplus = require("cdk8s-plus-30");
const constructs_1 = require("constructs");
const k8s = require("./imports/k8s");
const pdb_1 = require("./pdb");
/**
 * PloneDeployment creates a Kubernetes Deployment for Plone containers.
 *
 * This is an internal construct used by the Plone class.
 * It creates a Deployment with configurable replicas, resources, probes,
 * and an optional PodDisruptionBudget.
 */
class PloneDeployment extends constructs_1.Construct {
    constructor(scope, id, options) {
        super(scope, id);
        const image = options.image ?? {};
        const replicas = options.replicas ?? 2;
        const label = { app: cdk8s_1.Names.toLabelValue(this) };
        const optionLabels = options.labels ?? {};
        const deploymentLabels = {
            'app.kubernetes.io/name': optionLabels['app.kubernetes.io/name'] + '-deployment',
            'app.kubernetes.io/component': optionLabels['app.kubernetes.io/component'] ?? '' + '-deployment',
        };
        const template_labels = {
            ...optionLabels,
            ...label,
            'app.kubernetes.io/part-of': 'plone',
            'app.kubernetes.io/managed-by': 'cdk8s-plone',
        };
        const kpEnv = options?.environment ?? new kplus.Env([], {});
        var env = [];
        for (const name in kpEnv.variables) {
            env.push({ name: name, value: kpEnv.variables[name].value, valueFrom: kpEnv.variables[name].valueFrom });
        }
        var envFrom = [];
        for (const idx in kpEnv.sources) {
            const source = kpEnv.sources[idx];
            envFrom.push(source._toKube());
        }
        var ploneContainerSpec = {
            name: id + '-container', // here the namespaced name shold be used, but how?
            image: image.image,
            imagePullPolicy: image.imagePullPolicy,
            env: env,
            envFrom: envFrom,
            resources: {
                limits: {
                    cpu: k8s.Quantity.fromString(options.limitCpu ?? '1000m'),
                    memory: k8s.Quantity.fromString(options.limitMemory ?? '1Gi'),
                },
                requests: {
                    cpu: k8s.Quantity.fromString(options.requestCpu ?? '200m'),
                    memory: k8s.Quantity.fromString(options.requestMemory ?? '300Mi'),
                },
            },
            livenessProbe: options.livenessProbe ?? undefined,
            readinessProbe: options.readinessProbe ?? undefined,
        };
        const deploymentOptions = {
            metadata: {
                labels: deploymentLabels,
                annotations: options.annotations,
            },
            spec: {
                replicas,
                selector: {
                    matchLabels: label,
                },
                template: {
                    metadata: {
                        labels: template_labels,
                        annotations: options.podAnnotations,
                    },
                    spec: {
                        imagePullSecrets: (image.imagePullSecrets ?? []).map((name) => ({ name: name })),
                        containers: [
                            ploneContainerSpec,
                            ...options.sidecars ?? [],
                        ],
                    },
                },
            },
        };
        new k8s.KubeDeployment(this, 'deployment', deploymentOptions);
        if (options.pdb ?? false) {
            const pdbOptions = options.pdb ?? {};
            new pdb_1.PlonePDB(this, 'pdb', label, pdbOptions);
        }
    }
}
exports.PloneDeployment = PloneDeployment;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95bWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9kZXBsb3ltZW50LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLGlDQUFpQztBQUNqQyxpQ0FBOEI7QUFDOUIsdUNBQXVDO0FBQ3ZDLDJDQUF1QztBQUN2QyxxQ0FBcUM7QUFDckMsK0JBQWtEO0FBdUlsRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLGVBQWdCLFNBQVEsc0JBQVM7SUFFNUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxPQUErQjtRQUN2RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxLQUFLLElBQUksRUFBRSxDQUFDO1FBQ2xDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLElBQUksQ0FBQyxDQUFDO1FBQ3ZDLE1BQU0sS0FBSyxHQUFHLEVBQUUsR0FBRyxFQUFFLGFBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztRQUNoRCxNQUFNLFlBQVksR0FBRyxPQUFPLENBQUMsTUFBTSxJQUFJLEVBQUUsQ0FBQztRQUMxQyxNQUFNLGdCQUFnQixHQUFHO1lBQ3ZCLHdCQUF3QixFQUFFLFlBQVksQ0FBQyx3QkFBd0IsQ0FBQyxHQUFHLGFBQWE7WUFDaEYsNkJBQTZCLEVBQUUsWUFBWSxDQUFDLDZCQUE2QixDQUFDLElBQUksRUFBRSxHQUFHLGFBQWE7U0FDakcsQ0FBQztRQUNGLE1BQU0sZUFBZSxHQUFHO1lBQ3RCLEdBQUcsWUFBWTtZQUNmLEdBQUcsS0FBSztZQUNSLDJCQUEyQixFQUFFLE9BQU87WUFDcEMsOEJBQThCLEVBQUUsYUFBYTtTQUM5QyxDQUFDO1FBQ0YsTUFBTSxLQUFLLEdBQWMsT0FBTyxFQUFFLFdBQVcsSUFBSSxJQUFJLEtBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksR0FBRyxHQUFpQixFQUFFLENBQUM7UUFDM0IsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7WUFDbkMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDM0csQ0FBQztRQUNELElBQUksT0FBTyxHQUF3QixFQUFFLENBQUM7UUFDdEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDaEMsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUNsQyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBQ2pDLENBQUM7UUFDRCxJQUFJLGtCQUFrQixHQUFrQjtZQUN0QyxJQUFJLEVBQUUsRUFBRSxHQUFHLFlBQVksRUFBRSxtREFBbUQ7WUFDNUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO1lBQ2xCLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtZQUN0QyxHQUFHLEVBQUUsR0FBRztZQUNSLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLFNBQVMsRUFBRTtnQkFDVCxNQUFNLEVBQUU7b0JBQ04sR0FBRyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxRQUFRLElBQUksT0FBTyxDQUFDO29CQUN6RCxNQUFNLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFdBQVcsSUFBSSxLQUFLLENBQUM7aUJBQzlEO2dCQUNELFFBQVEsRUFBRTtvQkFDUixHQUFHLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFVBQVUsSUFBSSxNQUFNLENBQUM7b0JBQzFELE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsYUFBYSxJQUFJLE9BQU8sQ0FBQztpQkFDbEU7YUFDRjtZQUNELGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYSxJQUFJLFNBQVM7WUFDakQsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjLElBQUksU0FBUztTQUNwRCxDQUFDO1FBQ0YsTUFBTSxpQkFBaUIsR0FBNEI7WUFDakQsUUFBUSxFQUFFO2dCQUNSLE1BQU0sRUFBRSxnQkFBZ0I7Z0JBQ3hCLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVzthQUNqQztZQUNELElBQUksRUFBRTtnQkFDSixRQUFRO2dCQUNSLFFBQVEsRUFBRTtvQkFDUixXQUFXLEVBQUUsS0FBSztpQkFDbkI7Z0JBQ0QsUUFBUSxFQUFFO29CQUNSLFFBQVEsRUFBRTt3QkFDUixNQUFNLEVBQUUsZUFBZTt3QkFDdkIsV0FBVyxFQUFFLE9BQU8sQ0FBQyxjQUFjO3FCQUNwQztvQkFDRCxJQUFJLEVBQUU7d0JBQ0osZ0JBQWdCLEVBQUUsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLElBQUksRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7d0JBQ2hGLFVBQVUsRUFBRTs0QkFDVixrQkFBa0I7NEJBQ2xCLEdBQUcsT0FBTyxDQUFDLFFBQVEsSUFBSSxFQUFFO3lCQUMxQjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztRQUVGLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFFOUQsSUFBSSxPQUFPLENBQUMsR0FBRyxJQUFJLEtBQUssRUFBRSxDQUFDO1lBQ3pCLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxHQUFHLElBQUksRUFBRSxDQUFDO1lBQ3JDLElBQUksY0FBUSxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLENBQUM7SUFDSCxDQUFDO0NBQ0Y7QUFoRkQsMENBZ0ZDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gaW1wb3J0IHsgbG9nIH0gZnJvbSAnY29uc29sZSc7XG5pbXBvcnQgeyBOYW1lcyB9IGZyb20gJ2NkazhzJztcbmltcG9ydCAqIGFzIGtwbHVzIGZyb20gJ2NkazhzLXBsdXMtMzAnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBrOHMgZnJvbSAnLi9pbXBvcnRzL2s4cyc7XG5pbXBvcnQgeyBQbG9uZVBEQiwgUGxvbmVQREJPcHRpb25zIH0gZnJvbSAnLi9wZGInO1xuXG4vKipcbiAqIENvbnRhaW5lciBpbWFnZSBjb25maWd1cmF0aW9uIG9wdGlvbnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUGxvbmVJbWFnZU9wdGlvbnMge1xuICAvKipcbiAgICogQ29udGFpbmVyIGltYWdlIG5hbWUgYW5kIHRhZy5cbiAgICogQGV4YW1wbGUgJ3Bsb25lL3Bsb25lLWJhY2tlbmQ6Ni4wLjEwJ1xuICAgKiBAZGVmYXVsdCAncGxvbmUvcGxvbmUtYmFja2VuZDpsYXRlc3QnXG4gICAqL1xuICByZWFkb25seSBpbWFnZT86IHN0cmluZztcblxuICAvKipcbiAgICogTmFtZXMgb2YgS3ViZXJuZXRlcyBzZWNyZXRzIGZvciBwdWxsaW5nIHByaXZhdGUgaW1hZ2VzLlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2VQdWxsU2VjcmV0cz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBJbWFnZSBwdWxsIHBvbGljeSAoQWx3YXlzLCBJZk5vdFByZXNlbnQsIE5ldmVyKS5cbiAgICogQGRlZmF1bHQgJ0lmTm90UHJlc2VudCdcbiAgICovXG4gIHJlYWRvbmx5IGltYWdlUHVsbFBvbGljeT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIFBsb25lRGVwbG95bWVudC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQbG9uZURlcGxveW1lbnRPcHRpb25zIHtcbiAgLyoqXG4gICAqIENvbnRhaW5lciBpbWFnZSBjb25maWd1cmF0aW9uLlxuICAgKiBAZGVmYXVsdCAncGxvbmUvcGxvbmUtYmFja2VuZDpsYXRlc3QnXG4gICAqL1xuICByZWFkb25seSBpbWFnZT86IFBsb25lSW1hZ2VPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBFbnZpcm9ubWVudCB2YXJpYWJsZXMgZm9yIHRoZSBjb250YWluZXIuXG4gICAqIFVzZSBjZGs4cy1wbHVzLTMwIEVudiB0byBkZWZpbmUgdmFyaWFibGVzIGFuZCBzb3VyY2VzLlxuICAgKiBAZGVmYXVsdCAtIG5vIGFkZGl0aW9uYWwgZW52aXJvbm1lbnQgdmFyaWFibGVzXG4gICAqL1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IGtwbHVzLkVudjtcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHBvZCByZXBsaWNhcyB0byBydW4uXG4gICAqIEBkZWZhdWx0IDJcbiAgICovXG4gIHJlYWRvbmx5IHJlcGxpY2FzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBDUFUgbGltaXQgZm9yIHRoZSBjb250YWluZXIuXG4gICAqIEBkZWZhdWx0ICcxMDAwbSdcbiAgICovXG4gIHJlYWRvbmx5IGxpbWl0Q3B1Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBNZW1vcnkgbGltaXQgZm9yIHRoZSBjb250YWluZXIuXG4gICAqIEBkZWZhdWx0ICcxR2knXG4gICAqL1xuICByZWFkb25seSBsaW1pdE1lbW9yeT86IHN0cmluZztcblxuICAvKipcbiAgICogQ1BVIHJlcXVlc3QgZm9yIHRoZSBjb250YWluZXIuXG4gICAqIEBkZWZhdWx0ICcyMDBtJ1xuICAgKi9cbiAgcmVhZG9ubHkgcmVxdWVzdENwdT86IHN0cmluZztcblxuICAvKipcbiAgICogTWVtb3J5IHJlcXVlc3QgZm9yIHRoZSBjb250YWluZXIuXG4gICAqIEBkZWZhdWx0ICczMDBNaSdcbiAgICovXG4gIHJlYWRvbmx5IHJlcXVlc3RNZW1vcnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENvbnRhaW5lciBwb3J0IG51bWJlciB0byBleHBvc2UuXG4gICAqL1xuICByZWFkb25seSBwb3J0OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgS3ViZXJuZXRlcyBsYWJlbHMgZm9yIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIHN0YW5kYXJkIFBsb25lIGxhYmVscyBvbmx5XG4gICAqL1xuICByZWFkb25seSBsYWJlbHM/OiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogQW5ub3RhdGlvbnMgdG8gYWRkIHRvIHRoZSBEZXBsb3ltZW50IG1ldGFkYXRhLlxuICAgKiBDb21tb24gYW5ub3RhdGlvbnMgaW5jbHVkZTogZGVwbG95bWVudCB0aW1lc3RhbXBzLCBjaGFuZ2UgdHJhY2tpbmcsIGV0Yy5cbiAgICogQGV4YW1wbGUgeyAnZGVwbG95bWVudC5rdWJlcm5ldGVzLmlvL3JldmlzaW9uJzogJzEnIH1cbiAgICogQGRlZmF1bHQgLSBubyBhZGRpdGlvbmFsIGFubm90YXRpb25zXG4gICAqL1xuICByZWFkb25seSBhbm5vdGF0aW9ucz86IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBBbm5vdGF0aW9ucyB0byBhZGQgdG8gdGhlIFBvZCB0ZW1wbGF0ZSBtZXRhZGF0YS5cbiAgICogQ29tbW9uIGFubm90YXRpb25zIGluY2x1ZGU6IFByb21ldGhldXMgc2NyYXBpbmcgY29uZmlnLCBJc3RpbyBjb25maWcsXG4gICAqIGJhY2t1cCBwb2xpY2llcywgbG9nZ2luZyBjb25maWd1cmF0aW9ucywgZXRjLlxuICAgKiBAZXhhbXBsZSB7ICdwcm9tZXRoZXVzLmlvL3NjcmFwZSc6ICd0cnVlJywgJ3Byb21ldGhldXMuaW8vcG9ydCc6ICc4MDgwJyB9XG4gICAqIEBkZWZhdWx0IC0gbm8gYWRkaXRpb25hbCBhbm5vdGF0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgcG9kQW5ub3RhdGlvbnM/OiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBjb250YWluZXIgc3BlY2lmaWNhdGlvbiBvdmVycmlkZXMuXG4gICAqIEFkdmFuY2VkIHVzZSBvbmx5IC0gbWVyZ2VzIHdpdGggZ2VuZXJhdGVkIGNvbnRhaW5lciBzcGVjLlxuICAgKiBAZGVmYXVsdCAtIHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgcGxvbmVDb250YWluZXI/OiBrOHMuQ29udGFpbmVyO1xuXG4gIC8qKlxuICAgKiBTaWRlY2FyIGNvbnRhaW5lcnMgdG8gcnVuIGFsb25nc2lkZSB0aGUgbWFpbiBjb250YWluZXIuXG4gICAqIEBleGFtcGxlIFt7IG5hbWU6ICdsb2ctZm9yd2FyZGVyJywgaW1hZ2U6ICdmbHVlbnRkOmxhdGVzdCcgfV1cbiAgICogQGRlZmF1bHQgW11cbiAgICovXG4gIHJlYWRvbmx5IHNpZGVjYXJzPzogazhzLkNvbnRhaW5lcltdO1xuXG4gIC8qKlxuICAgKiBQb2REaXNydXB0aW9uQnVkZ2V0IGNvbmZpZ3VyYXRpb24gZm9yIGhpZ2ggYXZhaWxhYmlsaXR5LlxuICAgKiBJZiBwcm92aWRlZCwgY3JlYXRlcyBhIFBEQiB3aXRoIHRoZSBzcGVjaWZpZWQgY29uc3RyYWludHMuXG4gICAqIEBkZWZhdWx0IC0gbm8gUERCIGNyZWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHBkYj86IFBsb25lUERCT3B0aW9ucztcblxuICAvKipcbiAgICogTGl2ZW5lc3MgcHJvYmUgY29uZmlndXJhdGlvbiBmb3IgdGhlIGNvbnRhaW5lci5cbiAgICogQGRlZmF1bHQgLSB1bmRlZmluZWQgKG5vIGxpdmVuZXNzIHByb2JlKVxuICAgKi9cbiAgbGl2ZW5lc3NQcm9iZT86IGs4cy5Qcm9iZTtcblxuICAvKipcbiAgICogUmVhZGluZXNzIHByb2JlIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBjb250YWluZXIuXG4gICAqIEBkZWZhdWx0IC0gdW5kZWZpbmVkIChubyByZWFkaW5lc3MgcHJvYmUpXG4gICAqL1xuICByZWFkaW5lc3NQcm9iZT86IGs4cy5Qcm9iZTtcbn1cblxuLyoqXG4gKiBQbG9uZURlcGxveW1lbnQgY3JlYXRlcyBhIEt1YmVybmV0ZXMgRGVwbG95bWVudCBmb3IgUGxvbmUgY29udGFpbmVycy5cbiAqXG4gKiBUaGlzIGlzIGFuIGludGVybmFsIGNvbnN0cnVjdCB1c2VkIGJ5IHRoZSBQbG9uZSBjbGFzcy5cbiAqIEl0IGNyZWF0ZXMgYSBEZXBsb3ltZW50IHdpdGggY29uZmlndXJhYmxlIHJlcGxpY2FzLCByZXNvdXJjZXMsIHByb2JlcyxcbiAqIGFuZCBhbiBvcHRpb25hbCBQb2REaXNydXB0aW9uQnVkZ2V0LlxuICovXG5leHBvcnQgY2xhc3MgUGxvbmVEZXBsb3ltZW50IGV4dGVuZHMgQ29uc3RydWN0IHtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBvcHRpb25zOiBQbG9uZURlcGxveW1lbnRPcHRpb25zKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICBjb25zdCBpbWFnZSA9IG9wdGlvbnMuaW1hZ2UgPz8ge307XG4gICAgY29uc3QgcmVwbGljYXMgPSBvcHRpb25zLnJlcGxpY2FzID8/IDI7XG4gICAgY29uc3QgbGFiZWwgPSB7IGFwcDogTmFtZXMudG9MYWJlbFZhbHVlKHRoaXMpIH07XG4gICAgY29uc3Qgb3B0aW9uTGFiZWxzID0gb3B0aW9ucy5sYWJlbHMgPz8ge307XG4gICAgY29uc3QgZGVwbG95bWVudExhYmVscyA9IHtcbiAgICAgICdhcHAua3ViZXJuZXRlcy5pby9uYW1lJzogb3B0aW9uTGFiZWxzWydhcHAua3ViZXJuZXRlcy5pby9uYW1lJ10gKyAnLWRlcGxveW1lbnQnLFxuICAgICAgJ2FwcC5rdWJlcm5ldGVzLmlvL2NvbXBvbmVudCc6IG9wdGlvbkxhYmVsc1snYXBwLmt1YmVybmV0ZXMuaW8vY29tcG9uZW50J10gPz8gJycgKyAnLWRlcGxveW1lbnQnLFxuICAgIH07XG4gICAgY29uc3QgdGVtcGxhdGVfbGFiZWxzID0ge1xuICAgICAgLi4ub3B0aW9uTGFiZWxzLFxuICAgICAgLi4ubGFiZWwsXG4gICAgICAnYXBwLmt1YmVybmV0ZXMuaW8vcGFydC1vZic6ICdwbG9uZScsXG4gICAgICAnYXBwLmt1YmVybmV0ZXMuaW8vbWFuYWdlZC1ieSc6ICdjZGs4cy1wbG9uZScsXG4gICAgfTtcbiAgICBjb25zdCBrcEVudjoga3BsdXMuRW52ID0gb3B0aW9ucz8uZW52aXJvbm1lbnQgPz8gbmV3IGtwbHVzLkVudihbXSwge30pO1xuICAgIHZhciBlbnY6IGs4cy5FbnZWYXJbXSA9IFtdO1xuICAgIGZvciAoY29uc3QgbmFtZSBpbiBrcEVudi52YXJpYWJsZXMpIHtcbiAgICAgIGVudi5wdXNoKHsgbmFtZTogbmFtZSwgdmFsdWU6IGtwRW52LnZhcmlhYmxlc1tuYW1lXS52YWx1ZSwgdmFsdWVGcm9tOiBrcEVudi52YXJpYWJsZXNbbmFtZV0udmFsdWVGcm9tIH0pO1xuICAgIH1cbiAgICB2YXIgZW52RnJvbTogazhzLkVudkZyb21Tb3VyY2VbXSA9IFtdO1xuICAgIGZvciAoY29uc3QgaWR4IGluIGtwRW52LnNvdXJjZXMpIHtcbiAgICAgIGNvbnN0IHNvdXJjZSA9IGtwRW52LnNvdXJjZXNbaWR4XTtcbiAgICAgIGVudkZyb20ucHVzaChzb3VyY2UuX3RvS3ViZSgpKTtcbiAgICB9XG4gICAgdmFyIHBsb25lQ29udGFpbmVyU3BlYzogazhzLkNvbnRhaW5lciA9IHtcbiAgICAgIG5hbWU6IGlkICsgJy1jb250YWluZXInLCAvLyBoZXJlIHRoZSBuYW1lc3BhY2VkIG5hbWUgc2hvbGQgYmUgdXNlZCwgYnV0IGhvdz9cbiAgICAgIGltYWdlOiBpbWFnZS5pbWFnZSxcbiAgICAgIGltYWdlUHVsbFBvbGljeTogaW1hZ2UuaW1hZ2VQdWxsUG9saWN5LFxuICAgICAgZW52OiBlbnYsXG4gICAgICBlbnZGcm9tOiBlbnZGcm9tLFxuICAgICAgcmVzb3VyY2VzOiB7XG4gICAgICAgIGxpbWl0czoge1xuICAgICAgICAgIGNwdTogazhzLlF1YW50aXR5LmZyb21TdHJpbmcob3B0aW9ucy5saW1pdENwdSA/PyAnMTAwMG0nKSxcbiAgICAgICAgICBtZW1vcnk6IGs4cy5RdWFudGl0eS5mcm9tU3RyaW5nKG9wdGlvbnMubGltaXRNZW1vcnkgPz8gJzFHaScpLFxuICAgICAgICB9LFxuICAgICAgICByZXF1ZXN0czoge1xuICAgICAgICAgIGNwdTogazhzLlF1YW50aXR5LmZyb21TdHJpbmcob3B0aW9ucy5yZXF1ZXN0Q3B1ID8/ICcyMDBtJyksXG4gICAgICAgICAgbWVtb3J5OiBrOHMuUXVhbnRpdHkuZnJvbVN0cmluZyhvcHRpb25zLnJlcXVlc3RNZW1vcnkgPz8gJzMwME1pJyksXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgbGl2ZW5lc3NQcm9iZTogb3B0aW9ucy5saXZlbmVzc1Byb2JlID8/IHVuZGVmaW5lZCxcbiAgICAgIHJlYWRpbmVzc1Byb2JlOiBvcHRpb25zLnJlYWRpbmVzc1Byb2JlID8/IHVuZGVmaW5lZCxcbiAgICB9O1xuICAgIGNvbnN0IGRlcGxveW1lbnRPcHRpb25zOiBrOHMuS3ViZURlcGxveW1lbnRQcm9wcyA9IHtcbiAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgIGxhYmVsczogZGVwbG95bWVudExhYmVscyxcbiAgICAgICAgYW5ub3RhdGlvbnM6IG9wdGlvbnMuYW5ub3RhdGlvbnMsXG4gICAgICB9LFxuICAgICAgc3BlYzoge1xuICAgICAgICByZXBsaWNhcyxcbiAgICAgICAgc2VsZWN0b3I6IHtcbiAgICAgICAgICBtYXRjaExhYmVsczogbGFiZWwsXG4gICAgICAgIH0sXG4gICAgICAgIHRlbXBsYXRlOiB7XG4gICAgICAgICAgbWV0YWRhdGE6IHtcbiAgICAgICAgICAgIGxhYmVsczogdGVtcGxhdGVfbGFiZWxzLFxuICAgICAgICAgICAgYW5ub3RhdGlvbnM6IG9wdGlvbnMucG9kQW5ub3RhdGlvbnMsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzcGVjOiB7XG4gICAgICAgICAgICBpbWFnZVB1bGxTZWNyZXRzOiAoaW1hZ2UuaW1hZ2VQdWxsU2VjcmV0cyA/PyBbXSkubWFwKChuYW1lKSA9PiAoeyBuYW1lOiBuYW1lIH0pKSxcbiAgICAgICAgICAgIGNvbnRhaW5lcnM6IFtcbiAgICAgICAgICAgICAgcGxvbmVDb250YWluZXJTcGVjLFxuICAgICAgICAgICAgICAuLi5vcHRpb25zLnNpZGVjYXJzID8/IFtdLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgbmV3IGs4cy5LdWJlRGVwbG95bWVudCh0aGlzLCAnZGVwbG95bWVudCcsIGRlcGxveW1lbnRPcHRpb25zKTtcblxuICAgIGlmIChvcHRpb25zLnBkYiA/PyBmYWxzZSkge1xuICAgICAgY29uc3QgcGRiT3B0aW9ucyA9IG9wdGlvbnMucGRiID8/IHt9O1xuICAgICAgbmV3IFBsb25lUERCKHRoaXMsICdwZGInLCBsYWJlbCwgcGRiT3B0aW9ucyk7XG4gICAgfVxuICB9XG59XG4iXX0=