"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PloneHttpcache = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
/**
 * PloneHttpcache construct for deploying Varnish HTTP caching layer.
 *
 * Uses the mittwald/kube-httpcache Helm chart to deploy Varnish as a
 * caching proxy in front of Plone backend and/or frontend services.
 *
 * The cache automatically connects to the Plone services and provides
 * HTTP cache invalidation capabilities.
 *
 * @example
 * const plone = new Plone(chart, 'plone');
 * const cache = new PloneHttpcache(chart, 'cache', {
 *   plone: plone,
 *   existingSecret: 'varnish-secret',
 * });
 */
class PloneHttpcache extends constructs_1.Construct {
    constructor(scope, id, options) {
        super(scope, id);
        let varnishVcl;
        if (!options.varnishVcl) {
            let vclPath;
            if (!options.varnishVclFile) {
                vclPath = path.join(__dirname, 'config', 'varnish.tpl.vcl');
            }
            else {
                vclPath = options.varnishVclFile;
            }
            varnishVcl = fs.readFileSync(vclPath, 'utf8');
        }
        else {
            varnishVcl = options.varnishVcl;
        }
        const httpcache = new cdk8s_1.Helm(this, 'httpcache', {
            // see https://github.com/mittwald/kube-httpcache/chart
            repo: 'https://helm.mittwald.de',
            chart: 'kube-httpcache',
            version: options.chartVersion,
            values: {
                replicaCount: options.replicas ?? 2,
                cache: {
                    // need to looks at the frontendWatch, do we need it?
                    frontendWatch: false,
                    backendWatch: false,
                    existingSecret: options.existingSecret ?? undefined,
                },
                vclTemplate: varnishVcl,
                extraEnvVars: [
                    { name: 'BACKEND_SERVICE_NAME', value: options.plone.backendServiceName },
                    { name: 'BACKEND_SERVICE_PORT', value: '8080' },
                    { name: 'BACKEND_SITE_ID', value: options.plone.siteId },
                    { name: 'FRONTEND_SERVICE_NAME', value: options.plone.frontendServiceName },
                    { name: 'FRONTEND_SERVICE_PORT', value: '3000' },
                ],
                // see https://github.com/mittwald/kube-httpcache/issues/253
                nodeSelector: {
                    'kubernetes.io/arch': 'amd64',
                },
                resources: {
                    limits: {
                        cpu: options.limitCpu || '500m',
                        memory: options.limitMemory || '500Mi',
                    },
                    requests: {
                        cpu: options.requestCpu || '100m',
                        memory: options.requestMemory || '100Mi',
                    },
                },
                rbac: {
                    enabled: false,
                },
                exporter: {
                    enabled: options.exporterEnabled ?? true,
                    resources: {
                        limits: {
                            cpu: '100m',
                            memory: '100Mi',
                        },
                        requests: {
                            cpu: '10m',
                            memory: '50Mi',
                        },
                    },
                },
                serviceMonitor: {
                    enabled: options.servicemonitor || false,
                    scrapeSignaller: options.servicemonitor || false,
                },
            },
        });
        const httpcacheService = httpcache.apiObjects.find((construct) => {
            if ((construct.kind === 'Service') && (construct.metadata.name?.endsWith('kube-httpcache'))) {
                return construct.name;
            }
            return undefined;
        });
        if (httpcacheService === undefined) {
            throw new Error('Could not find httpcache service');
        }
        this.httpcacheServiceName = httpcacheService.name;
    }
}
exports.PloneHttpcache = PloneHttpcache;
_a = JSII_RTTI_SYMBOL_1;
PloneHttpcache[_a] = { fqn: "@bluedynamics/cdk8s-plone.PloneHttpcache", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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