"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Plone = exports.PloneVariant = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const kplus = require("cdk8s-plus-30");
const constructs_1 = require("constructs");
const deployment_1 = require("./deployment");
const k8s_1 = require("./imports/k8s");
const service_1 = require("./service");
/**
 * Plone deployment variants.
 */
var PloneVariant;
(function (PloneVariant) {
    /**
     * Volto variant: ReactJS frontend (Volto) with REST API backend.
     * Deploys both frontend and backend services.
     */
    PloneVariant["VOLTO"] = "volto";
    /**
     * Classic UI variant: Traditional Plone with server-side rendering.
     * Deploys only the backend service.
     */
    PloneVariant["CLASSICUI"] = "classicui";
})(PloneVariant || (exports.PloneVariant = PloneVariant = {}));
/**
 * Plone construct for deploying Plone CMS to Kubernetes.
 *
 * This construct creates all necessary Kubernetes resources for running Plone:
 * - Deployment(s) for backend (and optionally frontend)
 * - Service(s) for network access
 * - Optional PodDisruptionBudget for high availability
 *
 * Supports two deployment variants:
 * - VOLTO: Modern React frontend with REST API backend (default)
 * - CLASSICUI: Traditional server-side rendered Plone
 *
 * @example
 * new Plone(chart, 'my-plone', {
 *   variant: PloneVariant.VOLTO,
 *   backend: {
 *     image: 'plone/plone-backend:6.0.10',
 *     replicas: 3,
 *   },
 *   frontend: {
 *     image: 'plone/plone-frontend:16.0.0',
 *   },
 * });
 */
class Plone extends constructs_1.Construct {
    constructor(scope, id, options = {}) {
        super(scope, id);
        this.frontendServiceName = undefined;
        this.siteId = options.siteId ?? 'Plone';
        this.variant = options.variant ?? PloneVariant.VOLTO;
        // ------------------------------------------------------------------------
        // Backend
        const backend = options.backend ?? {};
        const backendLabels = {
            'app.kubernetes.io/name': 'plone-backend',
            'app.kubernetes.io/component': 'backend',
            'app.kubernetes.io/version': options.version ?? 'undefined',
        };
        const backendPort = 8080;
        // Options
        var backendOptions = {
            labels: backendLabels,
            image: {
                image: backend.image ?? 'plone/plone-backend:latest',
                imagePullSecrets: options.imagePullSecrets ?? [],
                imagePullPolicy: backend.imagePullPolicy ?? 'IfNotPresent',
            },
            replicas: backend.replicas,
            limitCpu: backend.limitCpu ?? '500m',
            limitMemory: backend.limitMemory ?? '512Mi',
            requestCpu: backend.requestCpu ?? '200m',
            requestMemory: backend.requestMemory ?? '256Mi',
            pdb: {
                maxUnavailable: backend.maxUnavailable ?? undefined,
                minAvailable: backend.minAvailable ?? undefined,
            },
            port: backendPort,
            environment: backend.environment,
            annotations: backend.annotations,
            podAnnotations: backend.podAnnotations,
        };
        // Probing
        const backendActionHttpGet = {
            path: '/',
            port: k8s_1.IntOrString.fromNumber(backendPort),
        };
        if (backend.livenessEnabled ?? false) {
            backendOptions.livenessProbe = {
                httpGet: backendActionHttpGet,
                initialDelaySeconds: backend.livenessInitialDelaySeconds ?? 30,
                timeoutSeconds: backend.livenessTimeoutSeconds ?? 5,
                periodSeconds: backend.livenessPeriodSeconds ?? 10,
                successThreshold: backend.livenessSuccessThreshold ?? 1,
                failureThreshold: backend.livenessFailureThreshold ?? 3,
            };
        }
        if (backend.readinessEnabled ?? true) {
            backendOptions.readinessProbe = {
                httpGet: backendActionHttpGet,
                initialDelaySeconds: backend.readinessInitialDelaySeconds ?? 10,
                timeoutSeconds: backend.readinessTimeoutSeconds ?? 15,
                periodSeconds: backend.readinessPeriodSeconds ?? 10,
                successThreshold: backend.readinessSuccessThreshold ?? 1,
                failureThreshold: backend.readinessFailureThreshold ?? 3,
            };
        }
        // Deployment
        var backendDeployment = new deployment_1.PloneDeployment(this, 'backend', backendOptions);
        // Service
        const backendService = new service_1.PloneService(backendDeployment, 'service', {
            labels: {
                'app.kubernetes.io/name': 'plone-backend-service',
                'app.kubernetes.io/component': 'service',
            },
            targetPort: backendPort,
            selectorLabel: { app: cdk8s_1.Names.toLabelValue(backendDeployment) },
            portName: 'backend-http',
            annotations: backend.serviceAnnotations,
        });
        this.backendServiceName = backendService.name;
        // ------------------------------------------------------------------------
        // Frontend
        if (this.variant == PloneVariant.VOLTO) {
            const frontend = options.frontend ?? {};
            const frontendPort = 3000;
            const frontendLabels = {
                'app.kubernetes.io/name': 'plone-frontend',
                'app.kubernetes.io/component': 'frontend',
                'app.kubernetes.io/version': options.version ?? 'undefined',
            };
            // Environment for RAZZLE
            var frontendEnvironment = frontend.environment ?? new kplus.Env([], {});
            if (frontendEnvironment.variables.RAZZLE_INTERNAL_API_PATH === undefined) {
                // connect with backend service
                frontendEnvironment?.addVariable('RAZZLE_INTERNAL_API_PATH', kplus.EnvValue.fromValue(`http://${backendService.name}:${backendPort}/${this.siteId}`));
            }
            // Options
            var frontendOptions = {
                labels: frontendLabels,
                image: {
                    image: frontend.image ?? 'plone/plone-frontend:latest',
                    imagePullSecrets: options.imagePullSecrets ?? [],
                    imagePullPolicy: frontend.imagePullPolicy ?? 'IfNotPresent',
                },
                replicas: frontend.replicas,
                limitCpu: frontend.limitCpu ?? '500m',
                limitMemory: frontend.limitMemory ?? '1Gi',
                requestCpu: frontend.requestCpu ?? '200m',
                requestMemory: frontend.requestMemory ?? '256Mi',
                pdb: {
                    maxUnavailable: frontend.maxUnavailable ?? undefined,
                    minAvailable: frontend.minAvailable ?? undefined,
                },
                port: frontendPort,
                environment: frontendEnvironment,
                annotations: frontend.annotations,
                podAnnotations: frontend.podAnnotations,
            };
            // Probing
            const frontendActionHttpGet = {
                path: '/',
                port: k8s_1.IntOrString.fromNumber(frontendPort),
            };
            if (frontend.livenessEnabled ?? false) {
                frontendOptions.livenessProbe = {
                    httpGet: frontendActionHttpGet,
                    initialDelaySeconds: frontend.livenessInitialDelaySeconds ?? 30,
                    timeoutSeconds: frontend.livenessTimeoutSeconds ?? 5,
                    periodSeconds: frontend.livenessPeriodSeconds ?? 10,
                    successThreshold: frontend.livenessSuccessThreshold ?? 1,
                    failureThreshold: frontend.livenessFailureThreshold ?? 3,
                };
            }
            if (frontend.readinessEnabled ?? true) {
                frontendOptions.readinessProbe = {
                    httpGet: frontendActionHttpGet,
                    initialDelaySeconds: frontend.readinessInitialDelaySeconds ?? 10,
                    timeoutSeconds: frontend.readinessTimeoutSeconds ?? 15,
                    periodSeconds: frontend.readinessPeriodSeconds ?? 10,
                    successThreshold: frontend.readinessSuccessThreshold ?? 1,
                    failureThreshold: frontend.readinessFailureThreshold ?? 3,
                };
            }
            // Deployment
            const frontendDeployment = new deployment_1.PloneDeployment(this, 'frontend', frontendOptions);
            // Service
            const frontendService = new service_1.PloneService(frontendDeployment, 'service', {
                labels: {
                    'app.kubernetes.io/name': 'plone-frontend-service',
                    'app.kubernetes.io/component': 'service',
                },
                targetPort: frontendPort,
                selectorLabel: { app: cdk8s_1.Names.toLabelValue(frontendDeployment) },
                portName: 'frontend-http',
                annotations: frontend.serviceAnnotations,
            });
            this.frontendServiceName = frontendService.name;
        }
    }
}
exports.Plone = Plone;
_a = JSII_RTTI_SYMBOL_1;
Plone[_a] = { fqn: "@bluedynamics/cdk8s-plone.Plone", version: "0.1.2" };
//# sourceMappingURL=data:application/json;base64,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