"""
This module contains elements for interacting with the Groww API.
"""
import random
import time
import uuid
from typing import Optional, Tuple, Final

import requests
import pandas as pd
from pandas import DataFrame
from colorama import Fore, Style, init
import hashlib

from growwapi.common.files import get_cwd
from growwapi.groww.exceptions import (
    GrowwAPIException,
    GrowwAPITimeoutException,
    GrowwAPIAuthenticationException,
    GrowwAPIAuthorisationException,
    GrowwAPIBadRequestException,
    GrowwAPINotFoundException,
    GrowwAPIRateLimitException, InstrumentNotFoundException,
)

class GrowwAPI:
    _ERROR_MAP = {
        400: GrowwAPIBadRequestException,
        401: GrowwAPIAuthenticationException,
        403: GrowwAPIAuthorisationException,
        404: GrowwAPINotFoundException,
        429: GrowwAPIRateLimitException,
        504: GrowwAPITimeoutException
    }

    """
    A client for interacting with the Groww API.
    """

    # Validity constants
    VALIDITY_DAY = "DAY"
    VALIDITY_EOS = "EOS"
    VALIDITY_IOC = "IOC"
    VALIDITY_GTC = "GTC"
    VALIDITY_GTD = "GTD"

    # Exchange constants
    EXCHANGE_BSE = "BSE"
    EXCHANGE_MCX = "MCX"
    EXCHANGE_MCXSX = "MCXSX"
    EXCHANGE_NCDEX = "NCDEX"
    EXCHANGE_NSE = "NSE"
    EXCHANGE_US = "US"

    # OrderType constants
    ORDER_TYPE_LIMIT = "LIMIT"
    ORDER_TYPE_MARKET = "MARKET"
    ORDER_TYPE_STOP_LOSS = "SL"
    ORDER_TYPE_STOP_LOSS_MARKET = "SL_M"

    # Product constants
    PRODUCT_ARBITRAGE = "ARB"
    PRODUCT_BO = "BO"
    PRODUCT_CNC = "CNC"
    PRODUCT_CO = "CO"
    PRODUCT_NRML = "NRML"
    PRODUCT_MIS = "MIS"
    PRODUCT_MTF = "MTF"

    # Segment constants
    SEGMENT_CASH = "CASH"
    SEGMENT_CURRENCY = "CURRENCY"
    SEGMENT_COMMODITY = "COMMODITY"
    SEGMENT_FNO = "FNO"

    # TransactionType constants
    TRANSACTION_TYPE_BUY = "BUY"
    TRANSACTION_TYPE_SELL = "SELL"

    INSTRUMENT_CSV_URL = "https://growwapi-assets.groww.in/instruments/instrument.csv"
    _GROWW_GENERATE_SOCKET_TOKEN_URL: Final[str] = "https://api.groww.in/v1/api/apex/v1/socket/token/create/"


    def __init__(self, token: str) -> None:
        """
        Initialize the GrowwAPI with the given token and key.

        Args:
            token (str): API token for authentication.
        """
        self.domain = "https://api.groww.in/v1"
        self.token = token
        self.instruments = None
        self._display_changelog()
        print("Ready to Groww!")
    
    def _display_changelog(self):
        """
        Display the changelog for the Groww API by printing messages to the console.
        """

        init(autoreset=True)
        changelog = self._get_changelog()
        info_messages = changelog.get("info", [])
        warning_messages = changelog.get("warning", [])
        
        if info_messages:
            print(Fore.YELLOW + "INFO:")
            for message in info_messages:
                print(Fore.YELLOW + "- " + message)
        
        if warning_messages:
            print(Fore.RED + "WARNING:")
            for message in warning_messages:
                print(Fore.RED + "- " + message)
            

    def _get_changelog(self):
        """
        Get the changelog for the Groww API.
        :return: dict: A dictionary containing the JSON response from the API, or an empty dictionary in case of failure.
        The dictionary may contain the following keys:
            - "info" (list): Informational messages.
            - "warning" (list): Warning messages.
        """
        url = self.domain + "/changelog"
        headers = self._build_headers()
        try:
            response = self._request_get(
                url=url,
                headers=headers,
            )
            return response.json()
        except Exception as e:
            return {}
        

    
    def _download_and_load_instruments(self) -> DataFrame:
        """
        Download the instruments CSV file and load it into a DataFrame.

        Returns:
            DataFrame: The instruments data.
        """
        response = requests.get(self.INSTRUMENT_CSV_URL)
        response.raise_for_status()  # Raise an exception for HTTP errors
        csv_path = get_cwd() + "/../instruments.csv" # instrument.csv stored in common directory.
        with open(csv_path, 'wb') as f:
            f.write(response.content)
        return pd.read_csv(csv_path, dtype='str')

    def _load_instruments(self) -> DataFrame:
        """
        Load the instruments data into a DataFrame.

        Returns:
            DataFrame: The instruments data.
        """
        if self.instruments is None:
            self.instruments = self._download_and_load_instruments()
        return self.instruments

    def place_order(
        self,
        validity: str,
        exchange: str,
        order_type: str,
        product: str,
        quantity: int,
        segment: str,
        trading_symbol: str,
        transaction_type: str,
        order_reference_id: Optional[str] = None,
        price: Optional[float] = 0.0,
        trigger_price: Optional[float] = None,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Place a new order.

        Args:
            validity (str): The validity of the order.
            exchange (str): The exchange to place the order on.
            order_type (str): The type of order.
            price (float): The price of the order in Rupee.
            product (str): The product type.
            quantity (int): The quantity of the order.
            segment (str): The segment of the order.
            trading_symbol (str): The trading symbol to place the order for.
            transaction_type (str): The transaction type.
            order_reference_id (Optional[str]): The reference ID to track the order with. Defaults to a random 8-digit number.
            trigger_price (float): The trigger price of the order in Rupee.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The placed order response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/order/create"
        
        order_reference_id = order_reference_id if order_reference_id is not None else str(random.randint(10000000, 99999999))
        headers = self._build_headers()
        request_body = {
            "trading_symbol": trading_symbol,
            "quantity": quantity,
            "price": price,
            "trigger_price": trigger_price,
            "validity": validity,
            "exchange": exchange,
            "segment": segment,
            "product": product,
            "order_type": order_type,
            "transaction_type": transaction_type,
            "order_reference_id": order_reference_id,
        }

        response = self._request_post(
            url=url,
            json=request_body,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)

    def modify_order(
        self,
        order_type: str,
        segment: str,
        groww_order_id: str,
        quantity: int,
        price: Optional[float] = None,
        trigger_price: Optional[float] = None,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Modify an existing order.

        Args:
            order_type (str): The type of order.
            price (float): The price of the order in Rupee.
            quantity (int): The quantity of the order.
            segment (str): The segment of the order.
            groww_order_id (Optional[str]): The Groww order ID.
            trigger_price (float): The trigger price of the order in Rupee.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The modified order response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/order/modify"
        headers = self._build_headers()
        request_body = {
            "quantity": quantity,
            "price": price,
            "trigger_price": trigger_price,
            "groww_order_id": groww_order_id,
            "order_type": order_type,
            "segment": segment,
        }

        response = self._request_post(
            url=url,
            json=request_body,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)

    def cancel_order(
        self,
        groww_order_id: str,
        segment: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Cancel an existing order.

        Args:
            groww_order_id (str): The Groww order ID.
            segment (str): The segment of the order.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The cancelled order response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/order/cancel"
        headers = self._build_headers()
        request_body = {
            "segment": segment,
            "groww_order_id": groww_order_id,
        }

        response = self._request_post(
            url=url,
            json=request_body,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_holdings_for_user(self, timeout: Optional[int] = None) -> dict:
        """
        Get the holdings for the user.

        Args:
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The user's holdings response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/holdings/user"
        response = self._request_get(
            url=url, headers=self._build_headers(), timeout=timeout
        )
        return self._parse_response(response)

    def get_quote(
        self,
        trading_symbol: str,
        exchange: str,
        segment: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Fetch the latest quote data for an instrument.

        Args:
            symbol (str): The symbol to fetch the data for.
            exchange (str): The exchange to fetch the data from.
            segment (str): The segment to fetch the data from.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The latest quote data.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = f"{self.domain}/live-data/quote"
        params = {"exchange": exchange, "segment": segment, "trading_symbol": trading_symbol}
        response = self._request_get(
            url=url,
            headers=self._build_headers(),
            params=params,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_ltp(
        self,
        exchange_trading_symbols: Tuple[str],
        segment: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Fetch the LTP data for a list of instruments.

        Args:
            
            exchange_trading_symbol (str): A list of exchange_trading_symbols to fetch the ltp for. Example: "NSE_RELIANCE, NSE_INFY" or  ("NSE_RELIANCE", "NSE_INFY")
            segment (Segment): The segment to fetch the data from.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The LTP data.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = f"{self.domain}/live-data/ltp"
        params = {"segment": segment, "exchange_symbols": exchange_trading_symbols}
        response = self._request_get(
            url=url,
            headers=self._build_headers(),
            params=params,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_ohlc(
        self,
        exchange_trading_symbols: Tuple[str],
        segment: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Fetch the OHLC data for a list of instruments.

        Args:
            exchange_trading_symbol (str): A list of exchange_trading_symbols to fetch the ohlc for. Example: "NSE:RELIANCE, NSE:INFY" or  ("NSE:RELIANCE", "NSE:INFY")
            segment (str): The segment to fetch the data from.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The OHLC data.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = f"{self.domain}/live-data/ohlc"
        params = {"segment": segment, "exchange_symbols": exchange_trading_symbols}
        response = self._request_get(
            url=url,
            headers=self._build_headers(),
            params=params,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_order_detail(
        self,
        segment: str,
        groww_order_id: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the details of an order.

        Args:
            segment (str): The segment of the order.
            groww_order_id (str): The Groww order ID.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The order details response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url: str = self.domain + "/order/detail/" + groww_order_id
        headers = self._build_headers()
        params = {"segment": segment}

        response = self._request_get(
            url=url,
            params=params,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_order_list(
        self,
        page: Optional[int] = 0,
        page_size: Optional[int] = 25,
        segment: Optional[str] = None,
        timeout: Optional[int] = None,
    ) -> dict:
        """
          Get a list of orders.

          Args:
              page (Optonal[int]): The page number for the orders. Defaults to 0.
              page_size (Optional[int]): The number of orders per page. Defaults to 25.
              segment (Optional[str]): The segment of the orders.
              timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

          Returns:
              dict: The list of orders response.

          Raises:
              GrowwAPIException: If the request fails.
        """
        url = self.domain + "/order/list"

        headers = self._build_headers()
        params = {"segment": segment, "page": page} if segment else {}

        response = self._request_get(
            url=url,
            params=params,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)
    
    def get_order_status(
        self,
        segment: str,
        groww_order_id: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the status of an order.

        Args:
            segment (str): The segment of the order.
            groww_order_id (str): The Groww order ID.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The order status response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/order/status/" + groww_order_id
        headers = self._build_headers()
        params = {"segment": segment}

        response = self._request_get(
            url=url,
            params=params,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)
    
    def get_order_status_by_reference(
        self,
        segment: str,
        order_reference_id: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the status of an order by reference ID.

        Args:
            segment (str): The segment of the order.
            order_reference_id (str): The reference ID of the order.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The order status response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = f"{self.domain}/order/status/reference/{order_reference_id}"
        headers = self._build_headers()
        params = {"segment": segment}

        response = self._request_get(
            url=url,
            params=params,
            headers=headers,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_position_for_trading_symbol(
        self,
        trading_symbol: str,
        segment: str,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the positions for a symbol.

        Args:
            trading_symbol (str): The trading symbol to get the positions for.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).
            segment (str): The segment of the trading_symbol.

        Returns:
            dict: The positions response for the symbol.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/positions/trading-symbol"
        response = self._request_get(
            url=url,
            headers=self._build_headers(),
            params={"trading_symbol": trading_symbol, "segment": segment},
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_positions_for_user(
        self,
        segment: Optional[str] = None,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the positions for the user for all the symbols they have positions in.

        Args:
            segment (str): The segment of the positions.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The user's positions response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/positions/user"
        response = self._request_get(
            url=url,
            params={"segment": segment},
            headers=self._build_headers(),
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_trade_list_for_order(
        self,
        groww_order_id: str,
        segment: str,
        page: Optional[int] = 0,
        page_size: Optional[int] = 25,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the list of trades for a specific order.

        Args:
            groww_order_id (str): The Groww order ID.
            segment (str): The segment of the order.
            page (Optional[int]): The page number for the trades. Defaults to 0.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The list of trades response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/order/trades/" + groww_order_id
        headers = self._build_headers()
        params = {"segment": segment, "page": page, "page_size": page_size}

        response = self._request_get(
            url=url,
            params=params,
            headers=headers,
            timeout=timeout,
        )

        parsed_response = self._parse_response(response)
        return parsed_response

    def get_available_margin_details(self, timeout: Optional[int] = None) -> dict:
        """
        Get the available margin details for the user.

        Args:
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The user's margin details response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = self.domain + "/margins/detail/user"
        response = self._request_get(
            url=url, headers=self._build_headers(), timeout=timeout
        )
        return self._parse_response(response)

    def get_all_instruments(self) -> DataFrame:
        """
        Get a dataframe containing all the instruments.
        :return: DataFrame
        """
        return self._load_instruments()

    def get_instrument_by_exchange_and_trading_symbol(self, exchange: str, trading_symbol: str) -> dict:
        """
        Get the instrument details for a trading symbol on an exchange.
        :param exchange:
        :param trading_symbol:
        :return: dict
        """
        self._load_instruments()
        df = self.instruments.loc[(self.instruments['exchange'] == exchange)
                                  & (self.instruments[
                                         'trading_symbol'] == trading_symbol)]
        if df.empty: raise InstrumentNotFoundException()
        return df.iloc[0].to_dict()

    def get_instrument_by_groww_symbol(self, groww_symbol: str) -> dict:
        """
        Get the instrument details for the groww_symbol.
        :param groww_symbol:
        :return: dict
        """
        self._load_instruments()
        df = self.instruments.loc[
            (self.instruments['groww_symbol'] == groww_symbol)]
        if df.empty: raise InstrumentNotFoundException()
        return df.iloc[0].to_dict()

    def get_instrument_by_exchange_token(self, exchange_token: str) -> dict:
        """
        Get the instrument details for the exchange_token.
        :param exchange_token:
        :return:
        """
        self._load_instruments()
        df = self.instruments.loc[
            (self.instruments['exchange_token'] == exchange_token)]
        if df.empty: raise InstrumentNotFoundException()
        return df.iloc[0].to_dict()


    def get_historical_candle_data(
        self,
        trading_symbol: str,
        exchange: str,
        segment: str,
        start_time: str,
        end_time: str,
        interval_in_minutes: Optional[int] = None,
        timeout: Optional[int] = None,
    ) -> dict:
        """
        Get the historical data for an instrument.

        Args:
            trading_symbol (str): The symbol to fetch the data for.
            exchange (str): The exchange to fetch the data from.
            segment (str): The segment to fetch the data from.
            start_time (str): The start time in epoch milliseconds or yyyy-MM-dd HH:mm:ss format.
            end_time (str): The end time in epoch milliseconds or yyyy-MM-dd HH:mm:ss format.
            interval_in_minutes (Optional[int]): The interval in minutes.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).

        Returns:
            dict: The historical data.

        Raises:
            GrowwAPIException: If the request fails.
        """
        url = f"{self.domain}/historical/candle/range"
        params = {
            "exchange": exchange,
            "segment": segment,
            "trading_symbol": trading_symbol,
            "start_time": start_time,
            "end_time": end_time,
            "interval_in_minutes": interval_in_minutes,
        }
        response = self._request_get(
            url=url,
            headers=self._build_headers(),
            params=params,
            timeout=timeout,
        )
        return self._parse_response(response)

    def get_order_margin_details(
        self,
        segment: str,
        orders: list[dict],
        timeout: Optional[int] = None,
    ) -> dict:

        url = f"{self.domain}/margins/detail/orders"
        params = {"segment": segment}
        request_body = [
            {
                "trading_symbol": order["trading_symbol"],
                "transaction_type": order["transaction_type"],
                "quantity": order["quantity"],
                "price": order["price"],
                "order_type": order["order_type"],
                "product": order["product"],
                "exchange": order["exchange"],
            }
            for order in orders
        ]

        response = self._request_post(
            url=url,
            headers=self._build_headers(),
            json=request_body,
            params=params,
        )

        return self._parse_response(response)

    def generate_socket_token(self, key_pair) -> dict:
        headers = self._build_headers()
        request_body = {
            "socketKey" : key_pair.public_key.decode("utf-8"),
        }
        response = self._request_post(
            url=self._GROWW_GENERATE_SOCKET_TOKEN_URL,
            json=request_body,
            headers=headers,
        )
        return self._parse_response(response)

    def _build_headers(self) -> dict:
        """
        Build the headers for the API request.

        Returns:
            dict: The headers for the API request.
        """
        return {
            "x-request-id": str(uuid.uuid4()),
            "Authorization": "Bearer " + self.token,
            "Content-Type": "application/json",
            "x-client-id": "growwapi",
            "x-client-platform" : "growwapi-python-client",
            "x-client-platform-version": "0.0.9",
            "x-api-version" : "1.0",
        }

    def _request_get(
        self,
        url: str,
        params: dict = None,
        headers: dict = None,
        timeout: Optional[int] = None,
        **kwargs: any,
    ) -> requests.Response:
        """
        Send a GET request to the API.

        Args:
            url (str): The URL to send the request to.
            params (dict): The parameters to send with the request.
            headers (dict): The headers to send with the request.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).
            **kwargs: Optional arguments that ``request`` takes.

        Returns:
            requests.Response: The response from the API.

        Raises:
            GrowwAPIException: If the request fails.
        """
        try:
            return requests.get(
                url,
                params=params,
                headers=headers,
                timeout=timeout,
                **kwargs,
            )
        except requests.Timeout as e:
            raise GrowwAPITimeoutException() from e

    def _request_post(
        self,
        url: str,
        json: dict = None,
        headers: dict = None,
        timeout: Optional[int] = None,
        **kwargs: any,
    ) -> requests.Response:
        """
        Send a POST request to the API.

        Args:
            url (str): The URL to send the request to.
            json (dict): The JSON data to send with the request.
            headers (dict): The headers to send with the request.
            timeout (Optional[int]): The timeout for the request in seconds. Defaults to None (infinite).
            **kwargs: Optional arguments that ``request`` takes.

        Returns:
            requests.Response: The response from the API.

        Raises:
            GrowwAPIException: If the request fails.
        """
        try:
            return requests.post(
                url=url,
                json=json,
                headers=headers,
                timeout=timeout,
                **kwargs,
            )
        except requests.Timeout as e:
            raise GrowwAPITimeoutException() from e

    def _parse_response(self, response: requests.Response) -> dict:
        """
        Parse the response from the API.

        Args:
            response (requests.Response): The response from the API.

        Returns:
            BaseGrowwResponse: The parsed response.

        Raises:
            GrowwAPIException: If the request fails.
        """
        response_map = response.json()
        if response_map.get("status") == "FAILURE":
            error = response_map["error"]
            raise GrowwAPIException(code=error["code"], msg=error["message"])
        if response.status_code in self._ERROR_MAP:
            raise self._ERROR_MAP[response.status_code]()
        if not response.ok:
            raise GrowwAPIException(
                code=str(response.status_code),
                msg="The request to the Groww API failed.",
            )
        return dict(response_map["payload"] if "payload" in response_map else response_map)


    @staticmethod
    def _build_request_data(totp: Optional[str] = None,
        secret: Optional[str] = None) -> dict:
        """
        Builds the request data payload based on authentication method.
        """
        if totp is not None and secret is not None:
            raise ValueError(
                "Either totp or secret should be provided, not both.")
        if totp is None and secret is None:
            raise ValueError("Either totp or secret should be provided.")

        if totp is not None:
            if not totp.strip():
                raise ValueError("TOTP cannot be empty")

            return {
                'key_type': 'totp',
                'totp': totp.strip()
            }

        # secret is not None
        if not secret.strip():
            raise ValueError("Secret cannot be empty")

        timestamp = int(time.time())
        checksum = GrowwAPI._generate_checksum(secret, str(timestamp))
        return {
            'key_type': 'approval',
            'checksum': checksum,
            'timestamp': timestamp
        }

    @staticmethod
    def get_access_token(api_key: str, totp: Optional[str] = None,
        secret: Optional[str] = None) -> dict:
        """
        Args:
            api_key (str): Bearer token or API key for the Authorization header.
            totp (str): If TOTP api key is provided. The TOTP code as a string.
            secret (str): If approval api key is provided. The secret value as a string.
        Returns:
            dict: The JSON response from the API.
        Raises:
            requests.HTTPError: If the request fails.
        """
        import requests
        url = 'https://api.groww.in/v1/token/api/access'
        headers = {
            'Content-Type': 'application/json',
            'Authorization': 'Bearer' + api_key,
            "x-client-id": "growwapi",
            "x-client-platform": "growwapi-python-client",
            "x-client-platform-version": "0.0.9",
            "x-api-version": "1.0",
        }
        data = GrowwAPI._build_request_data(totp=totp, secret=secret)
        response = requests.post(url, headers=headers, json=data, timeout=15)
        if response.status_code == 400:
            try:
                msg = response.json().get('error', {}).get('displayMessage',
                                                           'Bad Request')
            except Exception:
                msg = 'Bad Request'
            raise GrowwAPIException(code=str(response.status_code),
                                    msg=f"Groww API Error 400: {msg}")
        if response.status_code in GrowwAPI._ERROR_MAP:
            raise GrowwAPI._ERROR_MAP[response.status_code]()
        if not response.ok:
            raise GrowwAPIException(code=str(response.status_code),
                                    msg="The request to the Groww API failed.")
        return response.json()['token']


    @staticmethod
    def _generate_checksum(data: str, salt: str) -> str:
        """
        Generates a SHA-256 checksum for the given data and salt.
        :param secret: The api secret value
        :return: Hexadecimal SHA-256 checksum
        """
        input_str = data + salt
        sha256 = hashlib.sha256()
        sha256.update(input_str.encode('utf-8'))
        return sha256.hexdigest()



