"""LangGraph workflow tailored for ChatKit-triggered conversations.

This script demonstrates how to build a minimal LangGraph workflow that can
be ingested into the Orcheo backend and triggered from the OpenAI ChatKit
interface. The graph expects the backend to forward an input payload containing
at least a ``message`` field (as produced by the ChatKit client tool trigger).

Nodes in the workflow:
- ``normalize_message``: sanitize and normalize the inbound text
- ``classify_intent``: derive a coarse intent label from the message
- ``compose_reply``: craft the final textual reply surfaced back to ChatKit

To ingest this workflow into the Orcheo backend, run the
``examples/ingest_langgraph.py`` helper or register it via the SDK/CLI by
providing ``build_graph`` as the entrypoint.
"""

from typing import Any
from langgraph.graph import END, START, StateGraph


State = dict[str, Any]


def echo_message(state: State) -> State:
    """Craft a conversational reply that ChatKit can surface."""
    message = str(state.get("message", ""))

    reply = f"This is a reply generated by the Orcheo workflow. You said: {message}"
    return {
        "reply": reply,
        "echo": message,
    }


def build_graph() -> StateGraph:
    """Construct and return the LangGraph workflow."""
    graph = StateGraph(dict)
    graph.add_node("echo_message", echo_message)
    graph.add_edge(START, "echo_message")
    graph.add_edge("echo_message", END)
    graph.set_entry_point("echo_message")
    graph.set_finish_point("echo_message")
    return graph


if __name__ == "__main__":
    compiled = build_graph().compile()
    payload = {"message": "Hello! Can you help me run the pipeline?"}
    result = compiled.invoke(payload)
    print(result)
    print(f"Chat reply: {result['reply']}")
