from atmoswing_api.app.utils.logger import create_logger
logger = create_logger()

from fastapi import FastAPI, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from starlette.exceptions import HTTPException as StarletteHTTPException
from atmoswing_api.__version__ import __version__
from atmoswing_api.app.routes import meta, forecasts, aggregations, docs
from slowapi import Limiter
from slowapi.util import get_remote_address
from slowapi.errors import RateLimitExceeded
from slowapi.middleware import SlowAPIMiddleware
import traceback
import os

app = FastAPI(
    title="AtmoSwing Web Forecast API",
    description="API to provide forecasts generated by AtmoSwing.",
    version=__version__,
)

# Set up rate limiter
limiter = Limiter(key_func=get_remote_address, default_limits=["120/minute"])
app.state.limiter = limiter

# Add SlowAPI middleware for rate limiting
app.add_middleware(SlowAPIMiddleware)

# Configure CORS middleware -- disabled as it is not working as expected.
#app.add_middleware(
#    CORSMiddleware,
#    allow_origins=["*"],
#    allow_methods=["GET", "OPTIONS"],
#    allow_headers=["*"],
#)

# Include the routes
app.include_router(docs.router, tags=["Documentation"])
app.include_router(meta.router, prefix="/meta", tags=["Metadata"])
app.include_router(forecasts.router, prefix="/forecasts", tags=["Data from a single forecast"])
app.include_router(aggregations.router, prefix="/aggregations", tags=["Aggregated forecast data"])

@app.exception_handler(StarletteHTTPException)
async def custom_http_exception_handler(request: Request, exc: StarletteHTTPException):
    if exc.status_code == 404:
        return JSONResponse(
            status_code=404,
            content={"detail": "This route does not exist. "
                               "Please check the URL or refer to the documentation."},
        )
    return JSONResponse(
        status_code=exc.status_code,
        content={"detail": exc.detail},
    )

@app.exception_handler(Exception)
async def global_exception_handler(request: Request, exc: Exception):
    tb = traceback.format_exc()
    logger.error(f"Unhandled exception: {exc}\nTraceback:\n{tb}")
    if isinstance(exc, MemoryError):
        logger.error("MemoryError: Possible resource exhaustion!")
    return JSONResponse(status_code=500, content={"detail": "Internal server error."})

@app.exception_handler(RateLimitExceeded)
async def rate_limit_handler(request: Request, exc: RateLimitExceeded):
    logger.warning(f"Rate limit exceeded: {exc}")
    return JSONResponse(status_code=429, content={"detail": "Rate limit exceeded. Please try again later."})

# FastAPI startup: check Redis availability and log status so you can see cache usage info in logs
@app.on_event("startup")
async def check_redis_on_startup():
    try:
        from atmoswing_api import cache
        await cache.redis_client.ping()
        logger.info("Redis reachable at %s:%s; caching enabled", os.getenv("REDIS_HOST", "localhost"), os.getenv("REDIS_PORT", 6379))
    except Exception as e:
        logger.warning("Redis unreachable; caching will be bypassed until Redis is available: %s", e)
