# SPDX-FileCopyrightText: 2021-2024 MTS PJSC
# SPDX-License-Identifier: Apache-2.0
from __future__ import annotations

from typing import Optional

from typing_extensions import deprecated

try:
    from pydantic.v1 import Field
except (ImportError, AttributeError):
    from pydantic import Field  # type: ignore[no-redef, assignment]

from onetl.impl import GenericOptions

# options generated by JDBCMixin methods
PROHIBITED_OPTIONS = frozenset(
    (
        "user",
        "password",
        "driver",
        "url",
    ),
)


@deprecated("Deprecated in 0.11.0 and will be removed in 1.0.0. Use FetchOptions or ExecuteOptions instead")
class JDBCOptions(GenericOptions):
    """Generic options, related to specific JDBC driver.

    .. deprecated:: 0.11.0
        Use ``SomeDB.FetchOptions`` or ``SomeDB.ExecuteOptions`` instead

    Examples
    --------

    .. note ::

        You can pass any value supported by underlying JDBC driver class,
        even if it is not mentioned in this documentation.

    .. code:: python

        from onetl.connection import SomeDB

        options = SomeDB.Options(
            queryTimeout=60_000,
            fetchsize=100_000,
            customSparkOption="value",
        )
    """

    class Config:
        prohibited_options = PROHIBITED_OPTIONS
        extra = "allow"

    query_timeout: Optional[int] = Field(default=None, alias="queryTimeout")
    """The number of seconds the driver will wait for a statement to execute.
    Zero means there is no limit.

    This option depends on driver implementation,
    some drivers can check the timeout of each query instead of an entire JDBC batch.
    """

    fetchsize: Optional[int] = None
    """How many rows to fetch per round trip.

    Tuning this option can influence performance of reading.

    .. warning::

        Default value depends on driver. For example, Oracle has
        default ``fetchsize=10``.
    """


class JDBCFetchOptions(GenericOptions):
    """Options related to fetching data from databases via JDBC.

    .. versionadded:: 0.11.0
        Replace ``SomeDB.JDBCOptions`` → ``SomeDB.FetchOptions``

    Examples
    --------

    .. note ::

        You can pass any value supported by underlying JDBC driver class,
        even if it is not mentioned in this documentation.

    .. code:: python

        from onetl.connection import SomeDB

        options = SomeDB.FetchOptions(
            queryTimeout=60_000,
            fetchsize=100_000,
            customSparkOption="value",
        )
    """

    class Config:
        prohibited_options = PROHIBITED_OPTIONS
        extra = "allow"

    query_timeout: Optional[int] = Field(default=None, alias="queryTimeout")
    """The number of seconds the driver will wait for a statement to execute.
    Zero means there is no limit.

    This option depends on driver implementation,
    some drivers can check the timeout of each query instead of an entire JDBC batch.
    """

    fetchsize: Optional[int] = None
    """How many rows to fetch per round trip.

    Tuning this option can influence performance of reading.

    .. warning::
        Default value depends on driver. For example, Oracle has
        default ``fetchsize=10``.
    """


class JDBCExecuteOptions(GenericOptions):
    """Options related to executing statements in databases via JDBC.

    .. versionadded:: 0.11.0
        Replace ``SomeDB.JDBCOptions`` → ``SomeDB.ExecuteOptions``

    Examples
    --------

    .. note ::

        You can pass any value supported by underlying JDBC driver class,
        even if it is not mentioned in this documentation.

    .. code:: python

        from onetl.connection import SomeDB

        options = SomeDB.ExecuteOptions(
            queryTimeout=60_000,
            customSparkOption="value",
        )
    """

    class Config:
        prohibited_options = PROHIBITED_OPTIONS
        extra = "allow"

    query_timeout: Optional[int] = Field(default=None, alias="queryTimeout")
    """The number of seconds the driver will wait for a statement to execute.
    Zero means there is no limit.

    This option depends on driver implementation,
    some drivers can check the timeout of each query instead of an entire JDBC batch.
    """

    fetchsize: Optional[int] = None
    """How many rows to fetch per round trip.

    Tuning this option can influence performance of reading.

    .. warning::
        Default value depends on driver. For example, Oracle has
        default ``fetchsize=10``.
    """
