from ast import Dict
import os
import pathlib
from scrapy.crawler import Crawler
from scrapy.statscollectors import StatsCollector
from typing import Self
import logging
from scrapy import Spider, signals
from scrapy.exceptions import NotConfigured
from scrapy.http import Request, Response
from scrapy.core.downloader import Slot
from twisted.internet import task
from scrapy.logformatter import LogFormatter

import scrapy_zen.signals
from scrapy_zen.logging import ScrapyDictFormatter, ZenBufferedLogHandler


logger = logging.getLogger(__name__)



class ZenStats:
    """
    Allows to calculate average latency across requests and shows logstats
    """

    def __init__(self, stats: StatsCollector) -> None:
        self.stats = stats
        self.interval: float = 60.0
        self.multiplier: float = 60.0 / self.interval
        self.task: task.LoopingCall | None = None

    @classmethod
    def from_crawler(cls, crawler: Crawler) -> Self:
        ext = cls(crawler.stats)
        crawler.signals.connect(ext.spider_opened, signal=signals.spider_opened)
        crawler.signals.connect(ext.spider_closed, signal=signals.spider_closed)
        return ext

    def spider_opened(self, spider: Spider) -> None:
        self.pagesprev: int = 0
        self.itemsprev: int = 0
        self.task = task.LoopingCall(self.log, spider)
        self.task.start(self.interval)

    def log(self, spider: Spider) -> None:
        self.calculate_stats()
        msg = (
            "Crawled %(pages)d pages (at %(pagerate)d pages/min), "
            "scraped %(items)d items (at %(itemrate)d items/min) [%(spider_name)s]"
        )
        log_args = {
            "pages": self.pages,
            "pagerate": self.prate,
            "items": self.items,
            "itemrate": self.irate,
            "spider_name": spider.name,
        }
        logger.info(msg, log_args, extra={"spider": spider})

    def calculate_stats(self) -> None:
        self.items: int = self.stats.get_value("item_scraped_count", 0)
        self.pages: int = self.stats.get_value("response_received_count", 0)
        self.irate: float = (self.items - self.itemsprev) * self.multiplier
        self.prate: float = (self.pages - self.pagesprev) * self.multiplier
        self.pagesprev, self.itemsprev = self.pages, self.items

    def spider_closed(self, spider: Spider, reason: str) -> None:
        if self.task and self.task.running:
            self.task.stop()

        rpm_final, ipm_final = self.calculate_final_stats(spider)
        self.stats.set_value("responses_per_minute", rpm_final)
        self.stats.set_value("items_per_minute", ipm_final)

    def calculate_final_stats(
        self, spider: Spider
    ) -> tuple[None, None] | tuple[float, float]:
        start_time = self.stats.get_value("start_time")
        finish_time = self.stats.get_value("finish_time")

        if not start_time or not finish_time:
            return None, None

        mins_elapsed = (finish_time - start_time).seconds / 60

        if mins_elapsed == 0:
            return None, None

        items = self.stats.get_value("item_scraped_count", 0)
        pages = self.stats.get_value("response_received_count", 0)

        return (pages / mins_elapsed), (items / mins_elapsed)



class ZenBufferedLogs:

    def __init__(self, handler: ZenBufferedLogHandler):
        logging.getLogger().addHandler(handler)
        self.handler = handler

    @classmethod
    def from_crawler(cls, crawler: Crawler) -> Self:
        if not crawler.settings.getbool("ZEN_BUFFEREDLOGS_ENABLED"):
            raise NotConfigured
        log_file = crawler.settings.get("LOG_FILE")
        if not log_file:
            raise NotConfigured

        buffer_length = crawler.settings.getint("ZEN_BUFFEREDLOGS_LENGTH", 30)
        log_format = crawler.settings.get('LOG_FORMAT', '%(asctime)s [%(name)s] %(levelname)s: %(message)s')
        log_dateformat = crawler.settings.get('LOG_DATEFORMAT', '%Y-%m-%d %H:%M:%S')
        formatter = ScrapyDictFormatter(fmt=log_format, datefmt=log_dateformat)
        filename = pathlib.Path(log_file).stem + "_buffered.log"
        handler = ZenBufferedLogHandler(filename, buffer_length)
        handler.setFormatter(formatter)

        ext = cls(handler)

        crawler.signals.connect(ext.new_item, signal=scrapy_zen.signals.new_item)
        return ext

    def new_item(self, item: Dict, spider: Spider):
        self.handler.flush()


class ZenAutoThrottle:
    def __init__(self, crawler: Crawler):
        self.crawler: Crawler = crawler
        if not crawler.settings.getbool("ZEN_AUTOTHROTTLE_ENABLED"):
            raise NotConfigured

        self.inc_factor: float = crawler.settings.getfloat(
            "ZEN_AUTOTHROTTLE_BACKOFF", 1.5
        )
        self.debug: bool = crawler.settings.getbool("ZEN_AUTOTHROTTLE_DEBUG")
        self.target_concurrency: float = crawler.settings.getfloat(
            "ZEN_AUTOTHROTTLE_TARGET_CONCURRENCY"
        )
        if self.target_concurrency <= 0.0:
            raise NotConfigured(
                f"ZEN_AUTOTHROTTLE_TARGET_CONCURRENCY "
                f"({self.target_concurrency!r}) must be higher than 0."
            )
        crawler.signals.connect(self._spider_opened, signal=signals.spider_opened)
        crawler.signals.connect(
            self._response_downloaded, signal=signals.response_downloaded
        )

    @classmethod
    def from_crawler(cls, crawler: Crawler) -> Self:
        return cls(crawler)

    def _spider_opened(self, spider: Spider) -> None:
        self.mindelay = self._min_delay(spider)
        self.maxdelay = self._max_delay(spider)
        spider.download_delay = self._start_delay(spider)  # type: ignore[attr-defined]

    def _min_delay(self, spider: Spider) -> float:
        s = self.crawler.settings
        return getattr(spider, "download_delay", s.getfloat("DOWNLOAD_DELAY"))

    def _max_delay(self, spider: Spider) -> float:
        return self.crawler.settings.getfloat("ZEN_AUTOTHROTTLE_MAX_DELAY")

    def _start_delay(self, spider: Spider) -> float:
        return max(
            self.mindelay, self.crawler.settings.getfloat("ZEN_AUTOTHROTTLE_START_DELAY")
        )

    def _response_downloaded(
        self, response: Response, request: Request, spider: Spider
    ) -> None:
        key, slot = self._get_slot(request, spider)
        latency = request.meta.get("download_latency")
        if (
            latency is None
            or slot is None
            or request.meta.get("autothrottle_dont_adjust_delay", False) is True
        ):
            return

        olddelay = slot.delay
        self._adjust_delay(slot, latency, response)
        if self.debug:
            diff = slot.delay - olddelay
            size = len(response.body)
            conc = len(slot.transferring)
            logger.info(
                "slot: %(slot)s | conc:%(concurrency)2d | "
                "delay:%(delay)5d ms (%(delaydiff)+d) | "
                "latency:%(latency)5d ms | size:%(size)6d bytes",
                {
                    "slot": key,
                    "concurrency": conc,
                    "delay": slot.delay * 1000,
                    "delaydiff": diff * 1000,
                    "latency": latency * 1000,
                    "size": size,
                },
                extra={"spider": spider},
            )

    def _get_slot(
        self, request: Request, spider: Spider
    ) -> tuple[str | None, Slot | None]:
        key: str | None = request.meta.get("download_slot")
        if key is None:
            return None, None
        assert self.crawler.engine
        return key, self.crawler.engine.downloader.slots.get(key)

    def _adjust_delay(self, slot: Slot, latency: float, response: Response) -> None:
        """Define delay adjustment policy"""

        # If a server needs `latency` seconds to respond then
        # we should send a request each `latency/N` seconds
        # to have N requests processed in parallel
        target_delay = latency / self.target_concurrency
        current_delay = slot.delay

        if response.status == 429:
            new_delay = current_delay * self.inc_factor
        else:
            # Adjust the delay to make it closer to target_delay
            new_delay = (slot.delay + target_delay) / 2.0

            # If target delay is bigger than old delay, then use it instead of mean.
            # It works better with problematic sites.
            new_delay = max(target_delay, new_delay)

            # Make sure self.mindelay <= new_delay <= self.max_delay
            new_delay = min(max(self.mindelay, new_delay), self.maxdelay)

            # Dont adjust delay if response status != 200 and new delay is smaller
            # than old one, as error pages (and redirections) are usually small and
            # so tend to reduce latency, thus provoking a positive feedback by
            # reducing delay instead of increase.
            if response.status != 200 and new_delay <= slot.delay:
                return

        slot.delay = new_delay
