# Python Library for Europeana Data Model

A Python library providing utilities for working with the **Europeana Data Model (EDM)**. This library maps all EDM classes and properties to Pydantic models, enabling type-safe validation, parsing, and serialization of EDM records.

## Features

- **Validation**: Built-in validation for EDM record structure and property constraints
- **Type-safe EDM**: All EDM classes implemented as Pydantic models
- **XML/RDF parsing**: Parse EDM records from XML or RDF formats
- **RDF serialization**: Export EDM records back to RDF formats

## Quick Start

EDM specifications are encapsulated in `EDM_Record`. All data is validated at instantiation, ensuring compliance with EDM.

### Parse EDM Records

```python
from edmlib import EDM_Parser

# Parse from string
record = EDM_Parser.from_string("""<?xml version="1.0" encoding="UTF-8"?>
<rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
    xmlns:ore="http://www.openarchives.org/ore/terms/"
    xmlns:edm="http://www.europeana.eu/schemas/edm/"
    xmlns:dc="http://purl.org/dc/elements/1.1/">
    <edm:ProvidedCHO
        rdf:about="http://uri.test/edm123#CHO">
        <dc:type xml:lang="en">Text</dc:type>
        <dc:title xml:lang="de">Titel</dc:title>
        <dc:identifier>123</dc:identifier>
        <dc:language>de</dc:language>
        <edm:type>TEXT</edm:type>
    </edm:ProvidedCHO>
    <ore:Aggregation
        rdf:about="http://uri.test/edm123#Aggregation">
        <edm:aggregatedCHO
            rdf:resource="http://uri.test/edm123#CHO" />
        <edm:dataProvider>Test</edm:dataProvider>
        <edm:isShownAt
            rdf:resource="http://uri.test/edm123.jpg" />
        <edm:isShownBy
            rdf:resource="http://uri.test/edm123.jpg" />
        <edm:provider>Kulturpool</edm:provider>
        <edm:rights rdf:resource="http://creativecommons.org/licenses/by-nc-sa/4.0/" />
    </ore:Aggregation>
</rdf:RDF>
""").parse()

# Parse from file
record = EDM_Parser.from_file("edm_record.xml").parse()

# Parse other formats
record = EDM_Parser.from_file("edm_record.ttl", format="ttl").parse()
```

### Create EDM Records programmatically

```python
from edmlib import EDM_Record, EDM_ProvidedCHO, ORE_Aggregation, Ref, Lit

record = EDM_Record(
    provided_cho=EDM_ProvidedCHO(
        id=Ref(value="http://uri.test/edm123#CHO"),
        dc_type=[Lit(value="Text", lang="en")],
        dc_title=[Lit(value="Titel", lang="de")],
        dc_identifier=[Lit(value="123")],
        dc_language=[Lit(value="de")],
        edm_type=Lit(value="TEXT")
    ),
    aggregation=ORE_Aggregation(
        id=Ref(value="http://uri.test/edm123#Aggregation"),
        edm_aggregatedCHO=Ref(value="http://uri.test/edm123#CHO"),
        edm_dataProvider=Lit(value="Test"),
        edm_isShownAt=Ref(value="http://uri.test/edm123.jpg"),
        edm_isShownBy=Ref(value="http://uri.test/edm123.jpg"),
        edm_provider=Lit(value="Kulturpool"),
        edm_rights=Ref(value="http://creativecommons.org/licenses/by-nc-sa/4.0/")
    )
)
```

### Work with EDM Records

```python
from edmlib import EDM_ProvidedCHO, ORE_Aggregation, EDM_WebResource, EDM_Agent, EDM_Place, EDM_TimeSpan, SKOS_Concept, CC_License, SVCS_Service

# Access record components
assert record.provided_cho.id == Ref(value="http://uri.test/edm123#CHO")
assert record.aggregation.id == Ref(value="http://uri.test/edm123#Aggregation")
```

### Serialize

Multiple serialization formats for exporting EDM records are supported:

```python
# Pretty-XML format
xml_str = record.serialize()

# JSON-LD format
jsonld_str = record.serialize(format="json-ld")
```

All rdflib graph serialization formats are supported, including XML, Turtle (TTL), and others.


## Component Classes

- `EDM_ProvidedCHO` - Cultural Heritage Object (CHO)
- `ORE_Aggregation` - Resource aggregation
- `EDM_WebResource` - Web resources
- `EDM_Agent` - Agents (persons, organizations)
- `EDM_Place` - Places and locations
- `EDM_TimeSpan` - Time periods
- `SKOS_Concept` - Concepts and classifications
- `CC_License` - Creative Commons licenses
- `SVCS_Service` - SIOC Services

### Composition
```python
assert isinstance(          record.provided_cho,    EDM_ProvidedCHO)
assert isinstance(          record.aggregation,     ORE_Aggregation)

assert isinstance(next(iter(record.web_resource)),  EDM_WebResource) # optional
assert isinstance(next(iter(record.skos_concept)),  SKOS_Concept)    # optional
assert isinstance(next(iter(record.edm_agent)),     EDM_Agent)       # optional
assert isinstance(next(iter(record.edm_time_span)), EDM_TimeSpan)    # optional
assert isinstance(next(iter(record.edm_place)),     EDM_Place)       # optional
assert isinstance(next(iter(record.cc_license)),    CC_License)      # optional
assert isinstance(next(iter(record.svcs_service)),  SVCS_Service)    # optional
```

## Development

### Setup

```bash
# Clone repository
git clone <repository-url>

cd edmlib

# Install dependencies
poetry install

# Run tests
pytest .

# Generate API Documentation
poetry run update-api-docs
```

### Requirements

- Python >=3.10,<4.0.0
- Dependencies: 
  - rdflib ^7.0.0
  - lxml ^5.1.0
  - pydantic ^2.10.3
  - pyld ^2.0.3
  - requests ^2.32.3

<!--pdoc-start-->
## API documentation

*  [EDM_Record](#edm_record)             
    *  [provided_cho](#edm_recordprovided_cho)  
    *  [aggregation](#edm_recordaggregation)  
    *  [web_resource](#edm_recordweb_resource)  
    *  [skos_concept](#edm_recordskos_concept)  
    *  [edm_agent](#edm_recordedm_agent)  
    *  [edm_time_span](#edm_recordedm_time_span)  
    *  [edm_place](#edm_recordedm_place)  
    *  [cc_license](#edm_recordcc_license)  
    *  [svcs_service](#edm_recordsvcs_service)  
    *  [get_rdf_graph](#edm_recordget_rdf_graph)  
    *  [serialize](#edm_recordserialize)  
    *  [get_framed_json_ld](#edm_recordget_framed_json_ld)  
    *  [validate_provided_cho_identity](#edm_recordvalidate_provided_cho_identity)  
    *  [fetch_edm_isShownBy_head](#edm_recordfetch_edm_isshownby_head)  
    *  [has_edm_object](#edm_recordhas_edm_object)  
    *  [fetch_edm_object_head](#edm_recordfetch_edm_object_head)  
    *  [has_edm_hasView](#edm_recordhas_edm_hasview)  
    *  [fetch_edm_hasView_heads](#edm_recordfetch_edm_hasview_heads)  
    *  [fetch_edm_isShownAt_head](#edm_recordfetch_edm_isshownat_head)  
    *  [model_config](#edm_recordmodel_config)

*  [EDM_ProvidedCHO](#edm_providedcho)             
    *  [edm_type](#edm_providedchoedm_type)  
    *  [dc_contributor](#edm_providedchodc_contributor)  
    *  [dc_coverage](#edm_providedchodc_coverage)  
    *  [dc_creator](#edm_providedchodc_creator)  
    *  [dc_date](#edm_providedchodc_date)  
    *  [dc_description](#edm_providedchodc_description)  
    *  [dc_format](#edm_providedchodc_format)  
    *  [dc_identifier](#edm_providedchodc_identifier)  
    *  [dc_language](#edm_providedchodc_language)  
    *  [dc_publisher](#edm_providedchodc_publisher)  
    *  [dc_relation](#edm_providedchodc_relation)  
    *  [dc_rights](#edm_providedchodc_rights)  
    *  [dc_source](#edm_providedchodc_source)  
    *  [dc_subject](#edm_providedchodc_subject)  
    *  [dc_title](#edm_providedchodc_title)  
    *  [dc_type](#edm_providedchodc_type)  
    *  [dcterms_alternative](#edm_providedchodcterms_alternative)  
    *  [dcterms_conformsTo](#edm_providedchodcterms_conformsto)  
    *  [dcterms_created](#edm_providedchodcterms_created)  
    *  [dcterms_extent](#edm_providedchodcterms_extent)  
    *  [dcterms_hasFormat](#edm_providedchodcterms_hasformat)  
    *  [dcterms_hasPart](#edm_providedchodcterms_haspart)  
    *  [dcterms_hasVersion](#edm_providedchodcterms_hasversion)  
    *  [dcterms_isFormatOf](#edm_providedchodcterms_isformatof)  
    *  [dcterms_isPartOf](#edm_providedchodcterms_ispartof)  
    *  [dcterms_isReferencedBy](#edm_providedchodcterms_isreferencedby)  
    *  [dcterms_isReplacedBy](#edm_providedchodcterms_isreplacedby)  
    *  [dcterms_isRequiredBy](#edm_providedchodcterms_isrequiredby)  
    *  [dcterms_issued](#edm_providedchodcterms_issued)  
    *  [dcterms_isVersionOf](#edm_providedchodcterms_isversionof)  
    *  [dcterms_medium](#edm_providedchodcterms_medium)  
    *  [dcterms_provenance](#edm_providedchodcterms_provenance)  
    *  [dcterms_references](#edm_providedchodcterms_references)  
    *  [dcterms_replaces](#edm_providedchodcterms_replaces)  
    *  [dcterms_requires](#edm_providedchodcterms_requires)  
    *  [dcterms_spatial](#edm_providedchodcterms_spatial)  
    *  [dcterms_tableOfContents](#edm_providedchodcterms_tableofcontents)  
    *  [dcterms_temporal](#edm_providedchodcterms_temporal)  
    *  [edm_currentLocation](#edm_providedchoedm_currentlocation)  
    *  [edm_hasMet](#edm_providedchoedm_hasmet)  
    *  [edm_hasType](#edm_providedchoedm_hastype)  
    *  [edm_incorporates](#edm_providedchoedm_incorporates)  
    *  [edm_isDerivativeOf](#edm_providedchoedm_isderivativeof)  
    *  [edm_isNextInSequence](#edm_providedchoedm_isnextinsequence)  
    *  [edm_isRelatedTo](#edm_providedchoedm_isrelatedto)  
    *  [edm_isRepresentationOf](#edm_providedchoedm_isrepresentationof)  
    *  [edm_isSimilarTo](#edm_providedchoedm_issimilarto)  
    *  [edm_isSuccessorOf](#edm_providedchoedm_issuccessorof)  
    *  [edm_realizes](#edm_providedchoedm_realizes)  
    *  [owl_sameAs](#edm_providedchoowl_sameas)  
    *  [validate_dependent_edm](#edm_providedchovalidate_dependent_edm)  
    *  [model_config](#edm_providedchomodel_config)

*  [ORE_Aggregation](#ore_aggregation)             
    *  [edm_aggregatedCHO](#ore_aggregationedm_aggregatedcho)  
    *  [edm_dataProvider](#ore_aggregationedm_dataprovider)  
    *  [edm_provider](#ore_aggregationedm_provider)  
    *  [edm_rights](#ore_aggregationedm_rights)  
    *  [edm_hasView](#ore_aggregationedm_hasview)  
    *  [edm_isShownAt](#ore_aggregationedm_isshownat)  
    *  [edm_isShownBy](#ore_aggregationedm_isshownby)  
    *  [edm_object](#ore_aggregationedm_object)  
    *  [dc_rights](#ore_aggregationdc_rights)  
    *  [edm_ugc](#ore_aggregationedm_ugc)  
    *  [edm_intermediateProvider](#ore_aggregationedm_intermediateprovider)  
    *  [validate_conditional_attributes](#ore_aggregationvalidate_conditional_attributes)  
    *  [model_config](#ore_aggregationmodel_config)

*  [EDM_WebResource](#edm_webresource)             
    *  [dc_creator](#edm_webresourcedc_creator)  
    *  [dc_description](#edm_webresourcedc_description)  
    *  [dc_format](#edm_webresourcedc_format)  
    *  [dc_rights](#edm_webresourcedc_rights)  
    *  [dc_source](#edm_webresourcedc_source)  
    *  [dc_type](#edm_webresourcedc_type)  
    *  [dcterms_conformsTo](#edm_webresourcedcterms_conformsto)  
    *  [dcterms_created](#edm_webresourcedcterms_created)  
    *  [dcterms_extent](#edm_webresourcedcterms_extent)  
    *  [dcterms_hasPart](#edm_webresourcedcterms_haspart)  
    *  [dcterms_isFormatOf](#edm_webresourcedcterms_isformatof)  
    *  [dcterms_isPartOf](#edm_webresourcedcterms_ispartof)  
    *  [dcterms_isReferencedBy](#edm_webresourcedcterms_isreferencedby)  
    *  [dcterms_issued](#edm_webresourcedcterms_issued)  
    *  [edm_isNextInSequence](#edm_webresourceedm_isnextinsequence)  
    *  [edm_rights](#edm_webresourceedm_rights)  
    *  [owl_sameAs](#edm_webresourceowl_sameas)  
    *  [svcs_has_service](#edm_webresourcesvcs_has_service)  
    *  [validate_web_resource](#edm_webresourcevalidate_web_resource)  
    *  [model_config](#edm_webresourcemodel_config)

*  [CC_License](#cc_license)             
    *  [odrl_inheritFrom](#cc_licenseodrl_inheritfrom)  
    *  [cc_deprecatedOn](#cc_licensecc_deprecatedon)  
    *  [model_config](#cc_licensemodel_config)

*  [SKOS_Concept](#skos_concept)             
    *  [skos_prefLabel](#skos_conceptskos_preflabel)  
    *  [skos_altLabel](#skos_conceptskos_altlabel)  
    *  [skos_broader](#skos_conceptskos_broader)  
    *  [skos_narrower](#skos_conceptskos_narrower)  
    *  [skos_related](#skos_conceptskos_related)  
    *  [skos_broadMatch](#skos_conceptskos_broadmatch)  
    *  [skos_narrowMatch](#skos_conceptskos_narrowmatch)  
    *  [skos_relatedMatch](#skos_conceptskos_relatedmatch)  
    *  [skos_exactMatch](#skos_conceptskos_exactmatch)  
    *  [skos_closeMatch](#skos_conceptskos_closematch)  
    *  [skos_note](#skos_conceptskos_note)  
    *  [skos_notation](#skos_conceptskos_notation)  
    *  [skos_inScheme](#skos_conceptskos_inscheme)  
    *  [validate_skos_pref_label](#skos_conceptvalidate_skos_pref_label)  
    *  [model_config](#skos_conceptmodel_config)

*  [EDM_Agent](#edm_agent)             
    *  [skos_prefLabel](#edm_agentskos_preflabel)  
    *  [skos_altLabel](#edm_agentskos_altlabel)  
    *  [skos_note](#edm_agentskos_note)  
    *  [dc_date](#edm_agentdc_date)  
    *  [dc_identifier](#edm_agentdc_identifier)  
    *  [dcterms_hasPart](#edm_agentdcterms_haspart)  
    *  [dcterms_isPartOf](#edm_agentdcterms_ispartof)  
    *  [edm_begin](#edm_agentedm_begin)  
    *  [edm_end](#edm_agentedm_end)  
    *  [edm_hasMet](#edm_agentedm_hasmet)  
    *  [edm_isRelatedTo](#edm_agentedm_isrelatedto)  
    *  [foaf_name](#edm_agentfoaf_name)  
    *  [rdagr2_biographicalInformation](#edm_agentrdagr2_biographicalinformation)  
    *  [rdagr2_dateOfBirth](#edm_agentrdagr2_dateofbirth)  
    *  [rdagr2_dateOfDeath](#edm_agentrdagr2_dateofdeath)  
    *  [rdagr2_dateOfEstablishment](#edm_agentrdagr2_dateofestablishment)  
    *  [rdagr2_dateOfTermination](#edm_agentrdagr2_dateoftermination)  
    *  [rdagr2_gender](#edm_agentrdagr2_gender)  
    *  [rdagr2_placeOfBirth](#edm_agentrdagr2_placeofbirth)  
    *  [rdagr2_placeOfDeath](#edm_agentrdagr2_placeofdeath)  
    *  [rdagr2_professionOrOccupation](#edm_agentrdagr2_professionoroccupation)  
    *  [owl_sameAs](#edm_agentowl_sameas)  
    *  [validate_skos_pref_label](#edm_agentvalidate_skos_pref_label)  
    *  [model_config](#edm_agentmodel_config)

*  [EDM_TimeSpan](#edm_timespan)             
    *  [skos_prefLabel](#edm_timespanskos_preflabel)  
    *  [skos_altLabel](#edm_timespanskos_altlabel)  
    *  [skos_note](#edm_timespanskos_note)  
    *  [dcterms_hasPart](#edm_timespandcterms_haspart)  
    *  [dcterms_isPartOf](#edm_timespandcterms_ispartof)  
    *  [edm_begin](#edm_timespanedm_begin)  
    *  [edm_end](#edm_timespanedm_end)  
    *  [edm_isNextInSequence](#edm_timespanedm_isnextinsequence)  
    *  [owl_sameAs](#edm_timespanowl_sameas)  
    *  [validate_skos_pref_label](#edm_timespanvalidate_skos_pref_label)  
    *  [model_config](#edm_timespanmodel_config)

*  [EDM_Place](#edm_place)             
    *  [wgs84_pos_lat](#edm_placewgs84_pos_lat)  
    *  [wgs84_pos_long](#edm_placewgs84_pos_long)  
    *  [wgs84_pos_alt](#edm_placewgs84_pos_alt)  
    *  [skos_prefLabel](#edm_placeskos_preflabel)  
    *  [skos_altLabel](#edm_placeskos_altlabel)  
    *  [skos_note](#edm_placeskos_note)  
    *  [dcterms_hasPart](#edm_placedcterms_haspart)  
    *  [dcterms_isPartOf](#edm_placedcterms_ispartof)  
    *  [edm_isNextInSequence](#edm_placeedm_isnextinsequence)  
    *  [owl_sameAs](#edm_placeowl_sameas)  
    *  [validate_skos_pref_label](#edm_placevalidate_skos_pref_label)  
    *  [model_config](#edm_placemodel_config)

*  [SVCS_Service](#svcs_service)             
    *  [dcterms_conformsTo](#svcs_servicedcterms_conformsto)  
    *  [doap_implements](#svcs_servicedoap_implements)  
    *  [model_config](#svcs_servicemodel_config)

*  [MixedValuesList](#mixedvalueslist)  
*  [EDM_Parser](#edm_parser)             
    *  [EDM_Parser](#edm_parser__init__)  
    *  [from_file](#edm_parserfrom_file)  
    *  [from_string](#edm_parserfrom_string)  
    *  [graph](#edm_parsergraph)  
    *  [get_single_ref](#edm_parserget_single_ref)  
    *  [get_many_ref](#edm_parserget_many_ref)  
    *  [get_triples](#edm_parserget_triples)  
    *  [get_aggregation](#edm_parserget_aggregation)  
    *  [get_webresources](#edm_parserget_webresources)  
    *  [get_instance_triples](#edm_parserget_instance_triples)  
    *  [parse_single_class](#edm_parserparse_single_class)  
    *  [parse_many_class](#edm_parserparse_many_class)  
    *  [parse](#edm_parserparse)

*  [Ref](#ref)             
    *  [value](#refvalue)  
    *  [is_ref](#refis_ref)  
    *  [validate_value_as_uri](#refvalidate_value_as_uri)  
    *  [to_rdflib](#refto_rdflib)  
    *  [model_config](#refmodel_config)

*  [Lit](#lit)             
    *  [value](#litvalue)  
    *  [lang](#litlang)  
    *  [datatype](#litdatatype)  
    *  [normalize](#litnormalize)  
    *  [validate_consistency](#litvalidate_consistency)  
    *  [to_rdflib](#litto_rdflib)  
    *  [model_config](#litmodel_config)





---
### EDM_Record
```python
class EDM_Record(pydantic.main.BaseModel):
``` 



<div class="docstring"><p>Pydantic model representing an edm record, as a fully typed structure.
All contained non-standard types are themselves BaseModels, and the fields are always also either BaseModels or
standard-types. This ensures that without further conversion, an instance of this class can be
dumped as a dict (or json) and restored from such a dict (or json).</p>

<p>Validation:
This model is responsible for validating the overall structure, order and completeness
of the record.
The individual models for each of its properties are responsible for validating their own attributes –
their completeness, cardinality and structure.
Finally, the special type models - Ref and Lit - within those container types are responsible for validating
the indiviudal values.</p>
</div> 

#### EDM_Record.provided_cho
```python
provided_cho: edmlib.edm.classes.core.EDM_ProvidedCHO
```





#### EDM_Record.aggregation
```python
aggregation: edmlib.edm.classes.core.ORE_Aggregation
```





#### EDM_Record.web_resource
```python
web_resource: Optional[List[edmlib.edm.classes.core.EDM_WebResource]]
```





#### EDM_Record.skos_concept
```python
skos_concept: Optional[List[edmlib.edm.classes.context.SKOS_Concept]]
```





#### EDM_Record.edm_agent
```python
edm_agent: Optional[List[edmlib.edm.classes.context.EDM_Agent]]
```





#### EDM_Record.edm_time_span
```python
edm_time_span: Optional[List[edmlib.edm.classes.context.EDM_TimeSpan]]
```





#### EDM_Record.edm_place
```python
edm_place: Optional[List[edmlib.edm.classes.context.EDM_Place]]
```





#### EDM_Record.cc_license
```python
cc_license: Optional[List[edmlib.edm.classes.context.CC_License]]
```





#### EDM_Record.svcs_service
```python
svcs_service: Optional[List[edmlib.edm.classes.service.SVCS_Service]]
```





#### EDM_Record.get_rdf_graph
```python
def get_rdf_graph(self)
```



<div class="docstring"><p>Return whole record as as an RDF - rdflib.Graph object.</p>
</div> 

#### EDM_Record.serialize
```python
def serialize(self, format: str = 'pretty-xml', max_depth: int = 1) -> str
```



<div class="docstring"><p>Serialize graph to rdf/xml with pretty-formatting.</p>
</div> 

#### EDM_Record.get_framed_json_ld
```python
def get_framed_json_ld(self)
```





#### EDM_Record.validate_provided_cho_identity
```python
@model_validator(mode='after') def validate_provided_cho_identity(self) -> Self
```





#### EDM_Record.fetch_edm_isShownBy_head
```python
def fetch_edm_isShownBy_head(self, **kwargs) -> requests.models.Response
```





#### EDM_Record.has_edm_object
```python
def has_edm_object(self) -> bool
```





#### EDM_Record.fetch_edm_object_head
```python
def fetch_edm_object_head(self, **kwargs) -> requests.models.Response
```





#### EDM_Record.has_edm_hasView
```python
def has_edm_hasView(self) -> bool
```





#### EDM_Record.fetch_edm_hasView_heads
```python
def fetch_edm_hasView_heads(self, **kwargs) -> list[requests.models.Response]
```





#### EDM_Record.fetch_edm_isShownAt_head
```python
def fetch_edm_isShownAt_head(self, **kwargs) -> requests.models.Response
```





#### EDM_Record.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### EDM_ProvidedCHO
```python
class EDM_ProvidedCHO(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>mandatory-properties: DC_description, DC_language, DC_subject, DC_title, DC_type, DCTERMS_spatial, DCTERMS_temporal, EDM_type</p>

<p>optional-properties: DC_coverage, DC_format, DC_relation, DC_rights, DCTERMS_conformsTo, DCTERMS_extent, DCTERMS_hasFormat, DCTERMS_hasPart, DCTERMS_hasVersion, DCTERMS_isFormatOf, DCTERMS_isReferencedBy, DCTERMS_isReplacedBy, DCTERMS_isRequiredBy, DCTERMS_isVersionOf, DCTERMS_medium, DCTERMS_provenance, DCTERMS_references, DCTERMS_replaces, DCTERMS_requires, DCTERMS_tableOfContents , EDM_currentLocation, EDM_hasMet, EDM_hasType, EDM_incorporates, EDM_isDerivativeOf, EDM_isRelatedTo, EDM_isRepresentationOf, EDM_isSimilarTo, EDM_isSuccessorOf, EDM_realizes, OWL_sameAs</p>

<p>recommended-properties: DC_contributor, DC_creator, DC_date, DC_identifier, DC_publisher, DC_source, DCTERMS_alternative, DCTERMS_created, DCTERMS_isPartOf, DCTERMS_issued, EDM_IsNextInSequence</p>
</div> 

#### EDM_ProvidedCHO.edm_type
```python
edm_type: edmlib.edm.value_types.Lit
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
exactly_one</p>

<p>Value-Type:
Lit</p>

<p>Description: </p>

<p>The value must be one of the types accepted by Europeana as it will support portal fun
ctionality: TEXT, VIDEO, SOUND, IMAGE, 3D. (For 3D, when applicable, use the value “3D
‐PDF” in dc:format ) <code>&lt;edm:type&gt;IMAGE&lt;/edm:type&gt;</code> (upper-­case &amp; case sensitive) <code>&lt;edm:ty
pe&gt;3D&lt;/edm:type&gt;</code> (upper-­case &amp; case sensitive)</p>
</div> 

#### EDM_ProvidedCHO.dc_contributor
```python
dc_contributor: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use for contributors to the CHO. If possible supply the identifier of the contributor 
from an authority source. Providers with richer role terms can elect to map a subset t
o dc:contributor and others to dc:creator. Repeat for multiple contributors. <code>&lt;dc:contr
ibutor&gt;Maria Callas&lt;/dc:contributor&gt;</code> or create a reference to an instance of the Agent
class <code>&lt;dc:contributor rdf:resource=“http://www.example.com/MariaCallas”&gt;</code>For recommend
ations on medata quality see Tier A-C requirements ,</p>
</div> 

#### EDM_ProvidedCHO.dc_coverage
```python
dc_coverage: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The spatial or temporal topic of the CHO. Use the more precise dcterms:spatial or dcte
rms:temporal properties if the data will support it. <code>&lt;dc:coverage&gt;1995-­1996&lt;/dc:cover
age&gt;</code> or <code>&lt;dc:coverage&gt;Berlin&lt;/dc:coverage&gt;</code> or create a reference to an instance of a co
ntextual class, for example, a Place class <code>&lt;dc:coverage rdf:resource=“https://sws.geon
ames.org/2950159/ ”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_creator
```python
dc_creator: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>For the creator of the CHO. If possible supply the identifier of the creator from an a
uthority source. Repeat for multiple creators.  <code>&lt;dc:creator&gt;Shakespeare, William&lt;/dc:c
reator&gt;</code> or create a reference to an instance of the Agent class <code>&lt;dc:creator rdf:resour
ce=“http://viaf.org/viaf/96994048”/&gt;</code>For recommendations on medata quality see Tier A-C
requirements .</p>
</div> 

#### EDM_ProvidedCHO.dc_date
```python
dc_date: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use for a significant date in the life of the CHO.  Europeana recommends date conformi
ng to ISO 8601 starting with the year and with hyphens (YYYY-­MM-DD). NB: other EDM el
ements are relevant for expressing dates of different events in the life of the CHO: d
cterms:temporal, dcterms:created and dcterms:issued. Be careful and choose the most ap
propriate one! <code>&lt;dc:date&gt;Early 20th century&lt;/dc:date&gt;</code> or <code>&lt;dc:date&gt;1919&lt;/dc:date&gt;</code> or cre
ate a reference to an instance of the TimeSpan class <code>&lt;dc:date rdf:resource=“http://sem
ium.org/time/19xx_1_third”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_description
```python
dc_description: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A description of the CHO. If there is no dc:description for an object, there must be a
dc:title. If both are  available, provide both. <code>&lt;dc:description&gt;Illustrated guide to 
airport markings and lighting signals, with particular reference to SMGCS  (Surface Mo
vement Guidance and Control System) for airports with low visibility conditions.&lt;/dc:d
escription&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_format
```python
dc_format: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use for the terms generally applied to indicate the format of the cultural heritage ob
ject or the file format of a born digital object. Use the value “3D-­PDF” if appropria
te. <code>&lt;dc:format&gt;paper&lt;/dc:format&gt;</code>For recommendations on medata quality see Tier A-C req
uirements .</p>
</div> 

#### EDM_ProvidedCHO.dc_identifier
```python
dc_identifier: List[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>An identifier of the original CHO. <code>&lt;dc:identifier&gt;RP-­T-­1952-­380&lt;/dc:identifier&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_language
```python
dc_language: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The language of text CHOs and also for other types of CHO if there is a language aspec
t. Mandatory for TEXT objects, strongly recommended for other object types with a lang
uage element. Best practice is to use ISO 639 two- or three-letter primary language ta
gs.Repeat for multiple languages. We also recommend the use of the ISO 639-­2 code for
no linguistic content (ZXX).</p>
</div> 

#### EDM_ProvidedCHO.dc_publisher
```python
dc_publisher: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The name of the publisher of the CHO. If possible supply the identifier of the publish
er from an authority source. <code>&lt;dc:publisher&gt;Oxford University Press&lt;/dc:publisher&gt;</code> or c
reate a reference to an instance of the Agent class <code>&lt;dc:publisher rdf:resource=“http:/
/www.oup.com/”/&gt;</code>For recommendations on medata quality see Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.dc_relation
```python
dc_relation: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The name or identifier of a related resource, generally used for other related CHOs. C
f edm:isRelatedTo. <code>&lt;dc:relation&gt;maps.crace.1/33&lt;/dc:relation&gt;</code> (Shelf mark) Or to provi
de a link to another object: <code>&lt;dc:relation rdf:resource=“http://www.identifier/relatedO
bject”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_rights
```python
dc_rights: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use to give the name of the rights holder of the CHO if possible or for more general r
ights information. (Note that the controlled edm:rights property relates to the digita
l objects and applies to the edm:WebResource and/or edm:Aggregation). <code>&lt;dc:rights&gt;Copyr
ight © British Library Board&lt;/dc:rights&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_source
```python
dc_source: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A related resource from which the described resource is derived in whole or in part i.
e. the source of the original CHO.  (Not the name of the content holder: for this see 
edm:dataProvider.) <code>&lt;dc:source&gt;Security Magazine pp 3-12&lt;/dc:source&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_subject
```python
dc_subject: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The subject of the CHO.One of dc:subject or dc:type or dcterms:spatial or dcterms:temp
oral must be  provided; if more than one of these properties is available, please prov
ide them all. High-­level dc:subject  values like 'archaeology' are allowed, especiall
y when there is no other subject that can be easily filled in. <code>&lt;dc:subject&gt;archeology&lt;
/dc:subject&gt;</code>or create a reference to an instance of the Concept class <code>&lt;skos:Concept rd
f:about="http://semantics.gr/authorities/ekt-unesco/560215094"&gt;   &lt;skos:prefLabel xml:
lang="el"&gt;Αρχαιολογία&lt;/skos:prefLabel&gt;   &lt;skos:prefLabel xml:lang="en"&gt;Archaeology&lt;/sk
os:prefLabel&gt;&lt;/skos:Concept&gt;</code>For recommendations on medata quality see Tier A-C require
ments .</p>
</div> 

#### EDM_ProvidedCHO.dc_title
```python
dc_title: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>A name given to the CHO. dc:title should be present; but if there is no dc:title avail
able, it is acceptable to have dc:description instead. dc:title and dc:description sho
uld be distinct. Exact translations of the title can be  provided using appropriate xm
l language attributes. <code>&lt;dc:title xml:lang=“en”&gt;Eight Weeks&lt;/dc:title&gt; &lt;dc:title xml:la
ng=“it”&gt;Ocho semanas&lt;/ dc:title&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dc_type
```python
dc_type: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The nature or genre of the CHO. Ideally the term(s) will be taken from a controlled vo
cabulary. One of dc:type or dc:subject or dcterms:spatial or dcterms:temporal must be 
provided; if more than one of these properties is available, please provide them all. 
dc:type should not be (strictly) identical to edm:type. <code>&lt;dc:type&gt;Book&lt;/dc:type&gt;</code> or <code>&lt;dc
:type&gt;trombone&lt;/dc:type&gt;</code> or create a reference to an instance of the Concept class <code>&lt;dc
:type rdf:resource=“http://www.mimo-­db.eu/HornbostelAndSachs/356/”&gt;</code>For recommendation
s on medata quality see Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.dcterms_alternative
```python
dcterms_alternative: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Any alternative title of the CHO including abbreviations or translations that may not 
be exact. <code>&lt;dcterms:alternativexml:lang=“en”&gt;Eight weeks: a novel&lt;/dcterms:alternative&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_conformsTo
```python
dcterms_conformsTo: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>An established standard to which the CHO conforms. <code>&lt;dcterms:conformsTo&gt;W3C WCAG 2.0&lt;/d
cterms:conformsTo&gt;</code> (conforms to web content accessibility guidelines). Or link to the 
resource <code>&lt;dcterms:conformsTo rdf:resource=“http://www.w3.org/TR/WCAG/”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_created
```python
dcterms_created: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The date of creation of the CHO. Europeana recommends date conforming to ISO 8601 star
ting with the year and with hyphens (YYYY-­MM-DD). NB: other EDM elements are relevant
for expressing dates of different events in the life of the CHO: dc:date, dcterms:tem
poral and dcterms:issued. Be careful and choose the most appropriate one! <code>&lt;dcterms:cre
ated&gt;Mid 16th century&lt;/dcterms:created&gt;</code> or <code>&lt;dcterms:created&gt;1584&lt;/dcterms:created&gt;</code> or 
create a reference to an instance of the TimeSpan class<code>&lt;dcterms:created rdf:resource=“
http://semium.org/time/15xx_3_third”/&gt;</code>For recommendations on medata quality see Tier A
-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.dcterms_extent
```python
dcterms_extent: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The size or duration of the CHO. <code>&lt;dcterms:extent&gt;13 cm&lt;/dcterms:extent&gt;</code> (the width of 
an original object). <code>&lt;dcterms:extent&gt;34 minutes&lt;/dcterms:extent&gt;</code> (the duration of an a
udio file)</p>
</div> 

#### EDM_ProvidedCHO.dcterms_hasFormat
```python
dcterms_hasFormat: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A resource related to the CHO that is substantially the same as the CHO but in another
format. <code>&lt;dcterms:hasFormat&gt;http://upload.wikimedia.org/wikipedia/en/f/f3/Europeana_lo
go.png&lt;/dcterms:hasFormat&gt;</code> for a png image file of the described tiff resource Or as a
link to a resource <code>&lt;dcterms:hasFormat rdf:resource=“http://upload.wikimedia.org/wikip
edia/en/f/f3/Europeana_logo.png’’/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_hasPart
```python
dcterms_hasPart: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A resource that is included either physically or logically in the CHO. It is possible 
to use either dcterms:isPartOf or dcterms:hasPart to express relation between objects 
in a hierarchy. However in many cases (especially when a parent object has many childr
en) it is preferable to use dcterms:isPartOf. <code>&lt;dcterms:hasPart&gt;Vol.2. Issue 1&lt;/dcterms
:hasPart&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_hasVersion
```python
dcterms_hasVersion: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another, later resource that is a version, edition or adaptation of the CHO demonstrat
ing substantive changes in content rather than format. <code>&lt;dcterms:hasVersion&gt;The Sorcere
r’s Apprentice (translation by Edwin Zeydel, 1955)&lt;/dcterms:hasVersion&gt;</code> In this exampl
e the 1955 translation is a version of the described resource.</p>
</div> 

#### EDM_ProvidedCHO.dcterms_isFormatOf
```python
dcterms_isFormatOf: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another resource that is substantially the same as the CHO but in another format. <code>&lt;dct
erms:isFormatOf&gt;Europeana_logo.tiff&lt;/dcterms:isFormatOf&gt;</code> where the resource being desc
ribed is a png image file</p>
</div> 

#### EDM_ProvidedCHO.dcterms_isPartOf
```python
dcterms_isPartOf: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A resource in which the CHO is physically or logically included. This property can be 
used for objects that are part of a hierarchy and will be used to support an appropria
te display in the portal. For that purpose it will be necessary to supply a reference 
as the value. See the Task Force report on representing hierarchical entities.  It is 
possible to use either dcterms:isPartOf or dcterms:hasPart to express relation between
objects in a hierarchy. However in many cases (especially when a parent object has ma
ny children) it is preferable to use dcterms:isPartOf. <code>&lt;dcterms:isPartOf&gt;Crace Collect
ion of Maps of London&lt;/dcterms:isPartOf&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_isReferencedBy
```python
dcterms_isReferencedBy: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another resource that references, cites or otherwise points to the CHO. <code>&lt;dcterms:isRef
erencedBy&gt;Till, Nicholas (1994) Mozart and the Enlightenment: Truth, Virtue and Beauty
in Mozart’s Operas, W. W. Norton &amp; Company&lt;/dcterms:isReferencedBy&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_isReplacedBy
```python
dcterms_isReplacedBy: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another resource that supplants , displaces, or supersedes the CHO. <code>&lt;dcterms:isReplace
dBy&gt;http://dublincore.org/about/2009/01/05/bylaws/&lt;/dcterms:isReplacedBy&gt;</code> where the re
source described is an older version (<a href="http://dublincore.org/about/2006/01/01/bylaws/">http://dublincore.org/about/2006/01/01/bylaws/</a>) 
or link <code>&lt;dcterms:isReplacedBy rdf:resource=“http://dublincore.org/about/2009/01/05/byl
aws/”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_isRequiredBy
```python
dcterms_isRequiredBy: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another related resource that requires the CHO to support its function, delivery or co
herence <code>&lt;isRequiredBy&gt;http://www.myslides.com/myslideshow.ppt&lt;/isRequiredBy&gt;</code> where the
image being described is required for an online slideshow.</p>
</div> 

#### EDM_ProvidedCHO.dcterms_issued
```python
dcterms_issued: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Date of formal issuance or publication of the CHO. Europeana recommends date conformin
g to ISO 8601  starting with the year and with hyphens (YYYY-­MM-DD). NB: other EDM el
ements are relevant for expressing dates of different events in the life of the CHO: d
c:date, dcterms:temporal and dcterms:created. Be careful and choose the most appropria
te one! <code>&lt;dcterms:issued&gt;1993&lt;/dcterms:issued&gt;</code> or create a reference to an instance of 
the TimeSpan class `<dcterms:issued rdf:resource=“<a href="http://semium.org/time/17xx_3_third”/">http://semium.org/time/17xx_3_third”/</a></p>

<blockquote>
<p>` (late 18th century)For recommendations on medata quality see Tier A-C requirements .</p>
</blockquote>
</div> 

#### EDM_ProvidedCHO.dcterms_isVersionOf
```python
dcterms_isVersionOf: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another, earlier resource of which the CHO is a version, edition or adaptation, demons
trating substantive changes in content rather than format. <code>&lt;dcterms:isVersionOf&gt;The So
rcerer’s Apprentice&lt;dcterms:isVersionOf&gt;</code>In this example The Sorcerer’s Apprentice (tra
nslation by Edwin Zeydel, 1955) is the resource being described.</p>
</div> 

#### EDM_ProvidedCHO.dcterms_medium
```python
dcterms_medium: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The material or physical carrier of the CHO.  <code>&lt;dcterms:medium&gt;metal&lt;/dcterms:medium&gt;</code>Fo
r recommendations on medata quality see Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.dcterms_provenance
```python
dcterms_provenance: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A statement of changes in ownership and custody of the CHO since its creation. Signifi
cant for authenticity, integrity and interpretation. <code>&lt;dcterms:provenance&gt;Donated to Th
e National Library in 1965&lt;/dcterms:provenance&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_references
```python
dcterms_references: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Other resources referenced, cited or otherwise pointed to by the CHO. <code>&lt;dcterms:referen
ces&gt;Honderd jaar Noorse schilderkunst &lt;/dcterms:references&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_replaces
```python
dcterms_replaces: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A related resource that is supplanted, displaced, or superseded by the CHO. <code>&lt;dcterms:r
eplaces&gt;http://dublincore.org/about/2006/01/01/bylaws/&lt;/dcterms:replaces&gt;</code> where the re
source described is a newer version (<a href="http://dublincore.org/about/2009/01/05/bylaws/">http://dublincore.org/about/2009/01/05/bylaws/</a>) o
r link to resource <code>&lt;dcterms:replaces rdf:resource=“http://dublincore.org/about/2006/01
/01/bylaws/”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_requires
```python
dcterms_requires: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another resource that is required by the described resource to support its function, d
elivery or coherence. <code>&lt;dcterms:requires&gt;http://ads.ahds.ac.uk/project/userinfo/css/old
browsers.css &lt;/dcterms:requires&gt;</code> where the resource described is an HTML file at http:
//ads.ahds.ac.uk/project/userinfo/digitalTextArchiving.html</p>
</div> 

#### EDM_ProvidedCHO.dcterms_spatial
```python
dcterms_spatial: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Spatial characteristics of the CHO. i.e. what the CHO represents or depicts in terms o
f space (e.g. a location, coordinate or place). Either dcterms:spatial or dc:type or d
c:subject or dcterms:temporal must be provided; if more than one of these properties i
s available, please provide them all. dcterms:spatial is used to record the place depi
cted in the CHO and other locations associated with it as opposed to edm:currentLocati
on which is used only to record the place where the CHO is currently held (e.g. a muse
um or gallery). Be careful to choose the most appropriate one! <code>&lt;dcterms:spatial&gt;Portug
al&lt;/dcterms:spatial&gt;</code> or create a reference to an instance of the Place class <code>&lt;dcterms:
spatial rdf:resource=“https://sws.geonames.org/2264397/ ”/&gt;</code>For recommendations on meda
ta quality see Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.dcterms_tableOfContents
```python
dcterms_tableOfContents: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>A list of sub‐units of the CHO.<code>&lt;dcterms:tableOfContents&gt;Chapter 1. Introduction, Chapt
er 2. History &lt;/dcterms:tableOfContents&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.dcterms_temporal
```python
dcterms_temporal: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Temporal characteristics of the CHO. i.e. what the CHO is about or depicts in terms of
time (e.g. a period, date or date range.) Either dcterms:temporal or dc:type or dc:su
bject or dcterms:spatial must be provided; if more than one of these properties is ava
ilable, please provide them all. Europeana recommends date conforming to ISO 8601 star
ting with the year and with hyphens (YYYY-MM-DD). NB: other EDM elements are relevant 
for expressing dates of different events in the life of the CHO: dc:date, dcterms:crea
ted and dcterms:issued. Be careful and choose the most appropriate one! <code>&lt;dcterms:tempo
ral&gt;Roman Empire&lt;/dcterms:temporal&gt;</code> or create a reference to an instance of the TimeSp
an class <code>&lt;dcterms:temporal rdf:resource=“http://semium.org/time/roman_empire”/&gt;</code>For rec
ommendations on medata quality see Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.edm_currentLocation
```python
edm_currentLocation: Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref, NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Union[Lit, Ref]]</p>

<p>Description: </p>

<p>The geographic location whose boundaries presently include the CHO. This location must
have a position within an established positioning system: a location with coordinates
or address or inside another location that has a position, such as a room within a (m
useum) building. Ideally this position should be provided with the value of the proper
ty, either by using a reference (to a Place entity) that has coordinates or an address
attribute, or as a simple Lit. edm:currentLocation is used only to record the pla
ce where the CHO is currently held (e.g. a museum or gallery)dcterms:spatial is used t
o record the place depicted in the CHO and other locations associated with itBe carefu
l to choose the most appropriate one!<code>&lt;edm:currentLocation rdf:resource=“https://sws.ge
onames.org/2950159/”&gt;</code> (Identifier for Berlin)For recommendations on medata quality see
Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.edm_hasMet
```python
edm_hasMet: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of an agent, a place, a time period or any other identifiable entity th
at the CHO may have “met” in its life. <code>&lt;edm:hasMet rdf:resource=“http://viaf.org/viaf/
96994048/”&gt; (Identifier for William Shakespeare) &lt;edm:hasMet rdf:resource=“https://sws
.geonames.org/6620265/ ”&gt;</code> (location identifier for Shakespeare’s Globe theatre.)For re
commendations on medata quality see Tier A-C requirements .</p>
</div> 

#### EDM_ProvidedCHO.edm_hasType
```python
edm_hasType: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The identifier of a concept, or a word or phrase from a controlled vocabulary (thesaur
us etc) giving the type of the CHO. E.g. Painting from the AAT thesaurus. This propert
y can be seen as a super-­property of e.g. dc:format or dc:type to support “What” ques
tions. <code>&lt;edm:hasType&gt;Painting&lt;/edm:hasType&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_incorporates
```python
edm_incorporates: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of another resource that is incorporated in the described CHO. E.g. the
movie “A Clockwork Orange” incorporates Rossini’s La Gazza Ladra” in its soundtrack. 
<code>&lt;edm:incorporates rdf:resource=“http://www.identifier/IncorporatedResource/“&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_isDerivativeOf
```python
edm_isDerivativeOf: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of another resource from which the described CHO has been derived. E.g.
the identifier of Moby Dick when the Italian translation is the described CHO. <code>&lt;edm:i
sDerivativeOf rdf:resource=“http://www.identifier/SourceResource/”&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_isNextInSequence
```python
edm_isNextInSequence: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of the preceding object where both objects are part of the same overall
resource. Use this for objects that are part of a hierarchy or sequence to ensure cor
rect display in the portal. <code>&lt;edm:isNextInSequence rdf:resource=“http://www.identifier/
PrecedingResource”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_isRelatedTo
```python
edm_isRelatedTo: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The identifier or name of a concept or other resource to which the described CHO is re
lated. E.g. Moby Dick is related to XIX Century literature. Cf dc:relation. <code>&lt;edm:isRel
atedTo&gt;Literature&lt;/edm:isRelatedTo&gt;</code> Or link to resource <code>&lt;edm:isRelatedTo rdf:resource=
“http://www.eionet.europa.eu/gemet/concept?cp=4850/”&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_isRepresentationOf
```python
edm_isRepresentationOf: Optional[edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Ref]</p>

<p>Description: </p>

<p>The identifier of another object of which the described CHO is a representation. E.g. 
the identifier of the statue when the CHO being described is a painting of that statue
. <code>&lt;edm:isRepresentativeOf rdf:resource=“http://www.identifier/RepresentedResource/”&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_isSimilarTo
```python
edm_isSimilarTo: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of another resource to which the described CHO is similar. <code>&lt;edm:isSimil
arTo rdf:resource=“http://www.identifier/SimilarResource”/&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_isSuccessorOf
```python
edm_isSuccessorOf: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a resource to which the described CHO is a successor. E.g. “The Two
Towers” is a successor of “Fellowship of the Ring”. <code>&lt;edm:isSuccessorOf rdf:resource=“
http://dbpedia.org/resource/The_Fellowship_of_the_Ring/”&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.edm_realizes
```python
edm_realizes: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>If the CHO described is of type edm:PhysicalThing it may realize an information object
. E.g. a copy of the Gutenberg publication realizes the Bible.</p>
</div> 

#### EDM_ProvidedCHO.owl_sameAs
```python
owl_sameAs: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Use to point to your own (linked data) representation of the object, if you have already minted a URI identifier for it. It is also possible to provide URIs minted by third-parties for the object. <code>&lt;owl:sameAs rdf:resource=“http://www.identifier/SameResourceElsewhere/”&gt;</code></p>
</div> 

#### EDM_ProvidedCHO.validate_dependent_edm
```python
@model_validator(mode='after') def validate_dependent_edm(self) -> Self
```





#### EDM_ProvidedCHO.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### ORE_Aggregation
```python
class ORE_Aggregation(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>ORE Aggregation</p>

<p>mandatory-properties: EDM_aggregatedCHO, EDM_dataProvider, EDM_isShownAt, EDM_isShownBy, EDM_provider, EDM_rights</p>

<p>optional-properties: EDM_hasView, DC_rights, EDM_ugc</p>

<p>recommended-properties: EDM_object, EDM_intermediateProvider</p>
</div> 

#### ORE_Aggregation.edm_aggregatedCHO
```python
edm_aggregatedCHO: edmlib.edm.value_types.Ref
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
exactly_one</p>

<p>Value-Type:
Ref</p>

<p>Description: </p>

<p>The identifier of the source object e.g. the Mona Lisa itself. This could be a full li
nked open data URI or an internal identifier. <code>&lt;edm:aggregatedCHO rdf:resource=“#UEDIN:
214”/&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_dataProvider
```python
edm_dataProvider: Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
exactly_one</p>

<p>Value-Type:
Union[Lit, Ref]</p>

<p>Description: </p>

<p>The name or identifier of the data provider of the object (i.e. the organisation provi
ding data to an aggregator). Identifiers will not be available until Europeana has imp
lemented its Organization profile.  In the case of the data provider Zuidwestbrabants 
Museum, which delivers data through Erfgoedplus.be to LoCloud, the properties would lo
ok like this: <code>&lt;edm:dataProvider&gt;Zuidwestbrabants  Museum&lt;/edm:dataProvider&gt; &lt;edm:inter
mediateProvider&gt;Erfgoedplus.be&lt;/edm:intermediateProvider&gt; &lt;edm:provider&gt;LoCloud&lt;/edm:p
rovider&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_provider
```python
edm_provider: Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
exactly_one</p>

<p>Value-Type:
Union[Lit, Ref]</p>

<p>Description: </p>

<p>The name or identifier of the provider of the object (i.e. the organisation providing 
data directly to Europeana). Identifiers will not be available until Europeana has imp
lemented its Organization profile.  In the case of the provider LoCloud, which collect
s data from the data provider Zuidwestbrabants Museum through Erfgoedplus.be, the prop
erties would look like this: <code>&lt;edm:dataProvider&gt;Zuidwestbrabants Museum&lt;/edm:dataProvid
er&gt; &lt;edm:intermediateProvider&gt;Erfgoedplus.be&lt;/edm:intermediateProvider&gt;&lt;edm:provider&gt;L
oCloud&lt;/edm:provider&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_rights
```python
edm_rights: edmlib.edm.value_types.Ref
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
exactly_one</p>

<p>Value-Type:
Ref</p>

<p>Description: </p>

<p>This is a mandatory property and the value given here should be the rights statement t
hat applies to the digital representation as given (for example) in edm:object or edm:
isShownAt/By, when these resources are not provided with their own edm:rights (see edm
:rights documentation). The value for the rights statement in this element must be a U
RI from the list of available values. Note: rights statements must be exactly as speci
fied there, which means they must start with http and not https. (For assesing rights 
imformation check <a href="https://pro.europeana.eu/page/available-rights-statements">https://pro.europeana.eu/page/available-rights-statements</a> ) The righ
ts statement given in this property will also by default apply to the previews used in
the portal and will support portal search and display functionality.  Where there are
several web resources attached to one edm:ProvidedCHO the rights statement given here
will be regarded as the “reference” value for all the web resources. Therefore a suit
able value should be chosen with care if the rights statements vary between different 
resources. In fact in such cases Europeana encourages the provision of separate rights
statements for each individual web resource. Please note that the object page on http
://europeana.eu   displays the rights of the digital representation selected in the vi
ewer, which is found in the edm:rights of the WebResource that corresponds to the sele
cted edm:isShownBy or edm:hasView. If there is no such edm:isShownBy or edm:hasView re
presentation available, or if there is one but there is no specific edm:rights attache
d to it, then by default the page displays the edm:rights attached to the ore:Aggregat
ion.For example, a low­‐resolution of a JPEG file could be CC‐BY, while the high resol
ution version or a video showing the object would be CC-­BY-­NC. In such cases the rig
hts statements given for the individual web resources would ‘override’ the one specifi
ed at the ore:Aggregation level. Any other associated web resources would still be gov
erned by the edm:rights of the ore:Aggregation.   <code>&lt;edm:rights rdf:resource=“http://cre
ativecommons.org/publicdomain/mark/1.0/”/&gt; &lt;edm:rights rdf:resource=“http://rightsstat
ements.org/vocab/InC/1.0/”/&gt;</code>  Or create a reference to an instance of the cc:License c
lass where additional details of the rights can be provided (such as an expiry date fo
r the restrictions): <a href="http://rightsstatements.org/vocab/NoC-­NC/1.0/">http://rightsstatements.org/vocab/NoC-­NC/1.0/</a> or <code>&lt;edm:rights rdf
:resource="#statement_3000095353971"/&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_hasView
```python
edm_hasView: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The URL of a web resource which is a digital representation of the CHO. This may be th
e source object itself in the case of a born digital cultural heritage object. edm:has
View should only be used where there are several views of the CHO and one (or both) of
the mandatory edm:isShownAt or edm:isShownBy properties have already been used. It is
for cases where one CHO has several views of the same object. (e.g. a shoe and a deta
il of the label of the shoe)  <code>&lt;edm:hasView rdf:resource="http://www.mimo‐db.eu/media/U
EDIN/VIDEO/0032195v.mpg"/&gt; &lt;edm:hasView rdf:resource="http://www.mimo-­db.eu/media/UED
IN/AUDIO/0032195s.mp3"/&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_isShownAt
```python
edm_isShownAt: Optional[edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Ref]</p>

<p>Description: </p>

<p>The URL of a web view of the object in full information context. An edm:isShownAt must
be provided. If there is no edm:isShownAt for an object, there must be a edm:isShownB
y. If both are available, provide both. The use of edm:isShownBy is preferred. Providi
ng an edm:isShownAt is strongly recommended in all cases.<code>&lt;edm:isShownAt rdf:resource="
http://www.mimo-­‐db.eu/UEDIN/214"/&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_isShownBy
```python
edm_isShownBy: Optional[edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Ref]</p>

<p>Description: </p>

<p>The URL of a web view of the object. An edm:isShownBy must be provided. If there is no
edm:isShownBy for an object, there must be a edm:isShownAt. The use of edm:isShownBy 
is preferred. Europeana generates previews for any direct link to an image file. See E
uropeana Media Policy or information regarding the specifications of previews. <code>&lt;edm:is
ShownBy rdf:resource="http://www.mimo‐db.eu/media/UEDIN/IMAGE/0032195c.jpg"/&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_object
```python
edm_object: Optional[edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Ref]</p>

<p>Description: </p>

<p>The URL of a representation of the CHO which will be used for generating previews for 
use in the Europeana portal. This may be the same URL as edm:isShownBy.See Europeana M
edia Policy for information regarding the specifications of previews. This must be an 
image, even if it is for a sound object. <code>&lt;edm:object rdf:resource="http://www.mimo-‐db
.eu/media/UEDIN/IMAGE/0032195c.jpg"/&gt;</code>In accordance with Europeana's 2023 data publicat
ion approach, objects with edm:type=IMAGE that have no edm:isShownBy nor edm:object wi
ll not be published in Europeana. (See also ContentTier 1: Image type )</p>
</div> 

#### ORE_Aggregation.dc_rights
```python
dc_rights: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Ideally this should be applied to the edm:WebResource or the edm:ProvidedCHO. It is in
cluded here for the conversion of data from ESE where it is not known which object the
rights apply to.</p>
</div> 

#### ORE_Aggregation.edm_ugc
```python
edm_ugc: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>This is a mandatory property for objects that are user generated or user created that 
have been collected by crowdsourcing or project activity. The property is used to iden
tify such content and can only take the value “true” (lower case). <code>&lt;edm:ugc&gt;true&lt;/edm:
ugc&gt;</code></p>
</div> 

#### ORE_Aggregation.edm_intermediateProvider
```python
edm_intermediateProvider: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The name or identifier of the intermediate organization that selects, collates, or cur
ates data from a Data Provider that is then aggregated by a Provider from which Europe
ana harvests. The Intermediate Provider must be distinct from both the Data Provider a
nd the Provider in the data supply chain. Identifiers will not be available until Euro
peana has implemented its Organization profile. In the case of the Erfgoedplus.be, whi
ch collects data from Zuidwestbrabants Museum and provides it to LoCloud, the properti
es would look like this: <code>&lt;edm:dataProvider&gt;Zuidwestbrabants Museum&lt;/edm:dataProvider&gt; 
&lt;edm:provider&gt;LoCloud&lt;/edm:provider&gt; &lt;edm:intermediateProvider&gt;Erfgoedplus.be&lt;/edm:int
ermediateProvider&gt;</code></p>
</div> 

#### ORE_Aggregation.validate_conditional_attributes
```python
@model_validator(mode='after') def validate_conditional_attributes(self) -> Self
```





#### ORE_Aggregation.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### EDM_WebResource
```python
class EDM_WebResource(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>optional-properties: DC_creator, DC_description, DC_format, DC_rights, DC_source, DC_type, DCTERMS_conformsTo, DCTERMS_created, DCTERMS_extent, DCTERMS_hasPart, DCTERMS_isFormatOf, DCTERMS_isPartOf, DCTERMS_isReferencedBy, DCTERMS_issued, EDM_isNextInSequence, OWL_sameAs, SVCS_has_service, DCTERMS_IsReferencedBy</p>

<p>recommended-properties: EDM_rights</p>
</div> 

#### EDM_WebResource.dc_creator
```python
dc_creator: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>For the creator of the web resource. If possible supply the identifier of the creator 
from an authority source. Repeat for multiple creators. <code>&lt;dc:creator xml:lang=“es”&gt;Bibl
icoteca Nacional de España&lt;/dc:creator&gt;</code> or create a reference to an instance of the Ag
ent class <code>&lt;dc:creator rdf:resource=“http://viaf.org/viaf/147143794/”/&gt;</code></p>
</div> 

#### EDM_WebResource.dc_description
```python
dc_description: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use for an account or description of this digital representation <code>&lt;dc:description&gt;Perfo
rmance with Buccin trombone&lt;/dc:description&gt;</code></p>
</div> 

#### EDM_WebResource.dc_format
```python
dc_format: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use for the format of this digital representation. (Use the value “3D‐PDF” if appropri
ate.)<code>&lt;dc:format&gt;image/jpeg&lt;/dc:format&gt;</code></p>
</div> 

#### EDM_WebResource.dc_rights
```python
dc_rights: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Use for the name of the rights holder of this digital representation if possible or fo
r more general rights information. Note the difference between this property and the m
andatory, controlled edm:rights property below. <code>&lt;dc:rights&gt; Copyright © British Librar
y Board&lt;/dc:rights&gt;</code></p>
</div> 

#### EDM_WebResource.dc_source
```python
dc_source: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A related resource from which the Web resource is derived in whole or in part. <code>&lt;dc:sou
rce&gt;The name of the source video tape &lt;dc:source&gt;</code></p>
</div> 

#### EDM_WebResource.dc_type
```python
dc_type: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The nature or genre of the digital representation. Ideally the term(s) will be taken f
rom a controlled vocabulary.dc:type should not be (strictly) identical to edm:type. <code>&lt;d
c:type&gt;video&lt;/dc:type&gt;</code> or create a reference to an instance of the Concept class <code>&lt;dc:t
ype rdf:about= “http://schema.org/VideoObject” &gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_conformsTo
```python
dcterms_conformsTo: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>An established standard to which the web resource conforms. <code>&lt;dcterms:conformsTo&gt;W3C WC
AG 2.0&lt;/dcterms:conformsTo&gt;</code> (web content accessibility guidelines).</p>
</div> 

#### EDM_WebResource.dcterms_created
```python
dcterms_created: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Date of creation of the Web resource. Europeana recommends date conforming to ISO 8601
starting with the year and with hyphens (YYYY-MM-DD). <code>&lt;dcterms:created&gt;2010&lt;/dcterms:
created&gt;</code> or create a reference to an instance of the TimeSpan class <code>&lt;dc:date rdf:resou
rce=“http://semium.org/time/2010”/&gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_extent
```python
dcterms_extent: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The size or duration of the digital resource. <code>&lt;dcterms:extent&gt;1h 26 min 41 sec&lt;/dcterm
s:extent&gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_hasPart
```python
dcterms_hasPart: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>A resource that is included either physically or logically in the web resource. <code>&lt;dcter
ms:hasPart rdf:resource=“http://www.identifier/Part”/&gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_isFormatOf
```python
dcterms_isFormatOf: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Another resource that is substantially the same as the web resource but in another for
mat.  <code>&lt;dcterms:isFormatOf&gt;http://upload.wikimedia.org/wikipedia/en/f/f3/Europeana_logo
.png&lt;/dcterms:isFormatOf&gt;</code> for a png image file of the described tiff web resource. Or 
as a link to a resource <code>&lt;dcterms:isFormatOf rdf:resource=“http://upload.wikimedia.org/
wikipedia/en/f/f3/Europeana_logo.png”/&gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_isPartOf
```python
dcterms_isPartOf: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>A resource in which the WebResource is physically or logically included. This property
can be used for web resources that are part of a hierarchy. Hierarchies can be repres
ented as hierarchies of ProvidedCHOs or hierarchies of web resources but not both at t
he same time. See the Task Force report on representing hierarchical entities. <code>&lt;dcterm
s:isPartOf rdf:resource=“http://data.europeana.eu/item/08701/1B0BACAA44D5A807E43D9B411
C9781AAD2F96E65”/&gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_isReferencedBy
```python
dcterms_isReferencedBy: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A related resource that references, cites, or otherwise points to the described resour
ce. In a IIIF implementation, dcterms:isReferencedBy can be used to connect a edm:WebR
esource to a IIIF manifest URI. <code>&lt;dcterms:isReferencedBy rdf:resource="https://gallica.
bnf.fr/iiif/ark:/12148/btv1b55001425m/manifest.json"/&gt;</code></p>
</div> 

#### EDM_WebResource.dcterms_issued
```python
dcterms_issued: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>Date of formal issuance or publication of the web resource. Europeana recommends date 
conforming to ISO 8601 starting with the year and with hyphens (YYYY‐MM-DD). <code>&lt;dcterms:
issued&gt;1999&lt;/dcterms:issued&gt;</code> or create a reference to an instance of the TimeSpan clas
s<code>&lt;dcterms:issued rdf:resource=“http://semium.org/time/2010”/&gt;</code></p>
</div> 

#### EDM_WebResource.edm_isNextInSequence
```python
edm_isNextInSequence: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Where one CHO has several web resources, shown by multiple instances of the edm:hasVie
w property on the ore:Aggregation this property can be used to show the sequence of th
e objects. Each web resource (apart from the first in the sequence) should use this pr
operty to give the URI of the preceding resource in the sequence.</p>
</div> 

#### EDM_WebResource.edm_rights
```python
edm_rights: Optional[edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Ref]</p>

<p>Description: </p>

<p>The value in this element will indicate the copyright, usage and access rights that ap
ply to this digital representation. It is strongly recommended that a value is supplie
d for this property for each instance of a web resource.The value for the rights state
ment in this element must be a URI from the list of available values. Note: rights sta
tements must be exactly as specified there, which means they must start with http and 
not https. The rights statement specified at the level of the web resource will ‘overr
ide’ the statement specified at the level of the aggregation. <code>&lt;edm:rights rdf:resource
=“http://creativecommons.org/publicdomain/mark/1.0/”/&gt; &lt;edm:rights rdf:resource=“http:
//rightsstatements.org/vocab/InC/1.0/”/&gt;</code>  Or create a reference to an instance of the 
cc:License class where additional details of the rights can be provided (such as an ex
piry date for the restrictions): <a href="http://rightsstatements.org/vocab/NoC-NC/1.0/or">http://rightsstatements.org/vocab/NoC-NC/1.0/or</a> <code>&lt;edm:
rights rdf:resource="#statement_3000095353971"/&gt;</code>This is a recommended property.</p>
</div> 

#### EDM_WebResource.owl_sameAs
```python
owl_sameAs: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Provide the URI of another web representation of the same resource. <code>&lt;owl:sameAs rdf:re
source=”urn:soundcloud:150424305&gt;</code></p>
</div> 

#### EDM_WebResource.svcs_has_service
```python
svcs_has_service: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<pre><code>Cardinality: 
zero_to_many

Value-Type:
Optional[List[Ref]]

Description: 

The identifier of the svcs:Service required to consume the edm:WebResource. Example:
</code></pre>

<p><code>&lt;svcs:has_service rdf:resource="http://www.example.org/Service/IIIF"&gt;</code></p>
</div> 

#### EDM_WebResource.validate_web_resource
```python
@model_validator(mode='after') def validate_web_resource(self) -> Self
```





#### EDM_WebResource.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### CC_License
```python
class CC_License(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>mandatory-properties: ODRL_inheritFrom</p>

<p>optional-properties: CC_deprecatedOn</p>
</div> 

#### CC_License.odrl_inheritFrom
```python
odrl_inheritFrom: edmlib.edm.value_types.Ref
```



<div class="docstring"><p>Mandate: 
mandatory</p>

<p>Cardinality: 
exactly_one</p>

<p>Value-Type:
Ref</p>

<p>Description: </p>

<p>ID of a base rights statement from which the described License is derived. This value 
must come for alist of statements controlled by Europeana.<code>&lt;odrl:inheritFrom rdf:resour
ce=“http://rightsstatements.org/vocab/NoC-­NC/1.0/”/&gt;</code></p>
</div> 

#### CC_License.cc_deprecatedOn
```python
cc_deprecatedOn: Any
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Any</p>

<p>Description: </p>

<p>The date that the license expires, as it has been described, which implies among other
things the expiration of the restrictions specified by the license.<code>&lt;cc:deprecatedOn r
df:datatype=”http://www.w3.org/2001/XMLSchema#date”&gt;2029‐06-­01&lt;/cc:deprecatedOn&gt;</code> Note
this datatype is mandatory for cc:deprecatedOn.</p>
</div> 

#### CC_License.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### SKOS_Concept
```python
class SKOS_Concept(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>optional-properties: SKOS_broader, SKOS_narrower, SKOS_related, SKOS_broadMatch, SKOS_narrowMatch, SKOS_relatedMatch, SKOS_exactMatch, SKOS_closeMatch, SKOS_note, SKOS_notation, SKOS_inScheme</p>

<p>recommended-properties: SKOS_prefLabel, SKOS_altLabel</p>
</div> 

#### SKOS_Concept.skos_prefLabel
```python
skos_prefLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The preferred form of the name of the concept. Although the maximum number of occurren
ces is set at 1, it can be interpreted as 1 per language tag. At least one skos:prefLa
bel SHOULD be provided. Several prefLabels with languages tags are strongly recommende
d for language variants and translations.This is a recommended property for this class
.<code>&lt;skos:prefLabel xml:lang="fr"&gt;Buccin&lt;/skos:prefLabel&gt;&lt;skos:prefLabel xml:lang="de"&gt;Bu
ccin&lt;/skos:prefLabel&gt;&lt;skos:prefLabel xml:lang="nl"&gt;Buccin&lt;/skos:prefLabel&gt;</code>For recommen
dations on medata quality see Tier A-C requirements , more specifically Metadata Tier 
B and Metadata Tier C</p>
</div> 

#### SKOS_Concept.skos_altLabel
```python
skos_altLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Alternative forms of the name of the concept. Recommended unless several prefLabel are
already given with different language tags (altLabel is not suitable for translations
of prefLabel).<code>&lt;skos:altLabel xml:lang="en"&gt;Buccin&lt;/skos:altLabel&gt;</code>This is a recommende
d property for this class.</p>
</div> 

#### SKOS_Concept.skos_broader
```python
skos_broader: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a broader concept in the same thesaurus or controlled vocabulary.<code>&lt;sk
os:broader rdf:resource=“http://www.mimo-db.eu/InstrumentsKeywords/4369_1 ”/&gt;</code>For recom
mendations on medata quality see Tier A-C requirements , more specifically Metadata Ti
er B and Metadata Tier C</p>
</div> 

#### SKOS_Concept.skos_narrower
```python
skos_narrower: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a narrower concept.<code>&lt;skos:narrower rdf:resource=“http://narrower.term
/”/&gt;</code>For recommendations on medata quality see Tier A-C requirements , more specificall
y Metadata Tier B and Metadata Tier C</p>
</div> 

#### SKOS_Concept.skos_related
```python
skos_related: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a related concept<code>&lt;skos:related rdf:resource=“http://related.term/”/&gt;</code>
For recommendations on medata quality see Tier A-C requirements , more specifically Me
tadata Tier B and Metadata Tier C</p>
</div> 

#### SKOS_Concept.skos_broadMatch
```python
skos_broadMatch: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a broader, narrower or related matching concepts from other concept 
schemes.<code>&lt;skos:broadMatch rdf:resource=“http://broadMatch.term/”/&gt;&lt;skos:narrowMatch rdf
:resource=“http://narrowMatch.term/”/&gt;&lt;skos:relatedMatch rdf:resource=“http://relatedM
atch.term/”/&gt;</code></p>
</div> 

#### SKOS_Concept.skos_narrowMatch
```python
skos_narrowMatch: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a broader, narrower or related matching concepts from other concept 
schemes.<code>&lt;skos:broadMatch rdf:resource=“http://broadMatch.term/”/&gt;&lt;skos:narrowMatch rdf
:resource=“http://narrowMatch.term/”/&gt;&lt;skos:relatedMatch rdf:resource=“http://relatedM
atch.term/”/&gt;</code></p>
</div> 

#### SKOS_Concept.skos_relatedMatch
```python
skos_relatedMatch: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of a broader, narrower or related matching concepts from other concept 
schemes.<code>&lt;skos:broadMatch rdf:resource=“http://broadMatch.term/”/&gt;&lt;skos:narrowMatch rdf
:resource=“http://narrowMatch.term/”/&gt;&lt;skos:relatedMatch rdf:resource=“http://relatedM
atch.term/”/&gt;</code></p>
</div> 

#### SKOS_Concept.skos_exactMatch
```python
skos_exactMatch: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of close or exactly matching concepts from other concept schemes.<code>&lt;skos:
exactMatch rdf:resource=“http://exactMatch.term/”/&gt;&lt;skos:closeMatch rdf:resource=“http
://closeMatch.term/”/&gt;</code>For recommendations on medata quality see Tier A-C requirements 
, more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### SKOS_Concept.skos_closeMatch
```python
skos_closeMatch: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The identifier of close or exactly matching concepts from other concept schemes.<code>&lt;skos:
exactMatch rdf:resource=“http://exactMatch.term/”/&gt;&lt;skos:closeMatch rdf:resource=“http
://closeMatch.term/”/&gt;</code>For recommendations on medata quality see Tier A-C requirements 
, more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### SKOS_Concept.skos_note
```python
skos_note: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Information relating to the concept.<code>&lt;skos:note&gt;The buccin is a visually distinctive tr
ombone popularized in military bands in France between 1810–1845 which subsequently fa
ded into obscurity.&lt;/skos:note&gt;</code>For recommendations on medata quality see Tier A-C requ
irements, more specifically Metadata Tier B and Metadata Tier C.</p>
</div> 

#### SKOS_Concept.skos_notation
```python
skos_notation: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The notation in which the concept is represented. This may not be words in natural lan
guage for someknowledge organisation systems e.g. algebra<code>&lt;skos:notation rdf:datatype=“
http://www.w3.org/2001/XMLSchema#int”&gt;123&lt;/skos:notation&gt;</code></p>
</div> 

#### SKOS_Concept.skos_inScheme
```python
skos_inScheme: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The URI of a concept scheme</p>
</div> 

#### SKOS_Concept.validate_skos_pref_label
```python
@model_validator(mode='after') def validate_skos_pref_label(self) -> Self
```





#### SKOS_Concept.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### EDM_Agent
```python
class EDM_Agent(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>optional-properties: SKOS_note, DC_date, DC_identifier, DCTERMS_hasPart, DCTERMS_isPartOf, EDM_begin, EDM_end, EDM_hasMet, EDM_isRelatedTo, FOAF_name, RDAGR2_biographicalInformation, RDAGR2_dateOfEstablishment, RDAGR2_dateOfTermination, RDAGR2_gender, RDAGR2_placeOfBirth, RDAGR2_placeOfDeath, RDAGR2_professionOrOccupation, OWL_sameAs</p>

<p>recommended-properties: SKOS_prefLabel, SKOS_altLabel, RDAGR2_dateOfBirth, RDAGR2_dateOfDeath</p>
</div> 

#### EDM_Agent.skos_prefLabel
```python
skos_prefLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The preferred form of the name of the agent. Although the maximum number of occurrence
s is set at 1, it can be interpreted as 1 per language tag. At least one skos:prefLabe
l SHOULD be provided. Several prefLabels with languages tags are strongly recommended 
for language variants and translations. This is a recommended property for this class.
<code>&lt;skos:prefLabel xml:lang=''fr''&gt;Courtois neveu aîné&lt;/skos:prefLabel&gt;&lt;skos:prefLabel xm
l:lang=''en''&gt;Courtois’eldest nephew&lt;/skos:prefLabel&gt;</code> For recommendations on medata qu
ality see Tier A-C requirements , more specifically Metadata Tier B and Metadata Tier 
C</p>
</div> 

#### EDM_Agent.skos_altLabel
```python
skos_altLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Alternative forms of the name of the agent. This is a recommended property for this cl
ass.<code>&lt;skos:altLabel xml:lang="en"&gt;Courtois&lt;/skos:altLabel&gt;&lt;skos:altLabel xml:lang="fr"&gt;
Augte. Courtois aîné&lt;/skos:altLabel&gt;</code></p>
</div> 

#### EDM_Agent.skos_note
```python
skos_note: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>A note about the agent e.g. biographical notes.<code>&lt;skos:note&gt; Courtois neveu aîné started
a company of the same name manufacturing brass instruments in Paris in 1803&lt;/skos:not
e&gt;</code></p>
</div> 

#### EDM_Agent.dc_date
```python
dc_date: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>A significant date associated with the Agent. Europeana recommends date conforming to 
ISO 8601 starting with the year and with hyphens (YYYY-MM-DD).<code>&lt;dc:date&gt;1803&lt;/dc:date/&gt;</code></p>
</div> 

#### EDM_Agent.dc_identifier
```python
dc_identifier: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>An identifier of the agent.<code>&lt;dc:identifier&gt;http://viaf.org/viaf/96994048  &lt;/dc:identifi
er&gt;</code></p>
</div> 

#### EDM_Agent.dcterms_hasPart
```python
dcterms_hasPart: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to an Agent that is part of the Agent being described (e.g. a part of a corp
oration).<code>&lt;dcterms:hasPart rdf:resource=“http://identifier/partOfCorporation/”&gt;</code></p>
</div> 

#### EDM_Agent.dcterms_isPartOf
```python
dcterms_isPartOf: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to an agent that the described agent is part of.<code>&lt;dcterms:isPartOf rdf:resour
ce=“http://identifier/parentCorporation/”&gt;</code></p>
</div> 

#### EDM_Agent.edm_begin
```python
edm_begin: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date the agent was born/established. Europeana recommends date conforming to ISO 8
601 starting with the year and with hyphens (YYYY-MM-DD).<code>&lt;edm:begin&gt;1795&lt;/edm:begin&gt;</code>Ge
neric "begin" and "end" properties are being used to indicate start date and end date 
generically for edm:Agent and edm:TimeSpan. For edm:Agent this can be interpreted andb
irth and death dates.For recommendations on medata quality see Tier A-C requirements ,
more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Agent.edm_end
```python
edm_end: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>Generic "begin" and "end" properties are being used to indicate start date and end dat
e generically for edm:Agent and edm:TimeSpan. For edm:Agent this can be interpreted an
dbirth and death dates.For recommendations on medata quality see Tier A-C requirements
, more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Agent.edm_hasMet
```python
edm_hasMet: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to another entity which the agent has “met” in a broad sense. For example a 
reference to a Place class<code>&lt;edm:hasMet rdf:resource=“http://sws.geonames.org/6620265/”&gt;</code></p>
</div> 

#### EDM_Agent.edm_isRelatedTo
```python
edm_isRelatedTo: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to other entities, particularly other agents, with whom the agent is related
in a generic sense.<code>&lt;edm:isRelatedTo rdf:resource=“http://identifier/relatedAgent/”&gt;</code></p>
</div> 

#### EDM_Agent.foaf_name
```python
foaf_name: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The name of the agent as a simple textual string.<code>&lt;foaf:name&gt;Auguste Courtois&lt;/foaf:nam
e&gt;</code></p>
</div> 

#### EDM_Agent.rdagr2_biographicalInformation
```python
rdagr2_biographicalInformation: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Information pertaining to the life or history of the agent.<code>&lt;rdaGr2:biographicalInforma
tion&gt;Courtois neveu aîné started a company of the same name manufacturing brass instru
ments in Paris in 1803&lt;/rdaGr2:biographicalInformation&gt;</code></p>
</div> 

#### EDM_Agent.rdagr2_dateOfBirth
```python
rdagr2_dateOfBirth: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date the agent (person) was born. Europeana recommends date conforming to ISO 8601
starting with the year and with hyphens (YYYY-MM-DD). This is a recommended property 
for this class.<code>&lt;rdaGr2:dateOfBirth&gt;1795&lt;/rdaGr2:dateOfBirth&gt;</code>dates.For recommendations 
on medata quality see Tier A-C requirements , more specifically Metadata Tier B and Me
tadata Tier C</p>
</div> 

#### EDM_Agent.rdagr2_dateOfDeath
```python
rdagr2_dateOfDeath: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date the agent (person) died. Europeana recommends date conforming to ISO 8601 sta
rting with the year and with hyphens (YYYY‐MM-DD). This is a recommended property for 
this class.<code>&lt;rdaGr2:dateOfDeath&gt;1895&lt;/rdaGr2:dateOfDeath&gt;</code>For recommendations on medata 
quality see Tier A-C requirements , more specifically Metadata Tier B and Metadata Tie
r C</p>
</div> 

#### EDM_Agent.rdagr2_dateOfEstablishment
```python
rdagr2_dateOfEstablishment: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date on which the agent (corporate body) was established or founded.<code>&lt;rdaGr2:dateOf
Establishment&gt;1795&lt;/rdaGr2:dateOfEstablishment&gt;</code></p>
</div> 

#### EDM_Agent.rdagr2_dateOfTermination
```python
rdagr2_dateOfTermination: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date on which the agent (corporate body) was terminated or dissolved.<code>&lt;rdaGr2:dateO
fTermination&gt;1895&lt;/rdaGr2:dateOfTermination&gt;</code></p>
</div> 

#### EDM_Agent.rdagr2_gender
```python
rdagr2_gender: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The gender with which the agent identifies.<code>&lt; rdaGr2:gender&gt;Female&lt;/rdaGr2:gender&gt;</code></p>
</div> 

#### EDM_Agent.rdagr2_placeOfBirth
```python
rdagr2_placeOfBirth: Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref, NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Union[Lit, Ref]]</p>

<p>Description: </p>

<p>The town, city, province, state, and/or country in which a person was born.<code>&lt;rdaGr2:pla
ceOfBirth&gt;Lusaka, Northern Rhodesia&lt;/rdaGr2:placeOfBirth&gt;&lt;rdaGr2:placeOfBirth rdf:reso
urce=”http://sws.geonames.org/909137/”/&gt;</code>For recommendations on medata quality see Tier
A-C requirements , more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Agent.rdagr2_placeOfDeath
```python
rdagr2_placeOfDeath: Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref, NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Union[Lit, Ref]]</p>

<p>Description: </p>

<p>The town, city, province, state, and/or country in which a person died.<code>&lt;rdaGr2:placeOf
Death&gt;London, United Kingdom&lt;/rdaGr2:placeOfDeath&gt;&lt;rdaGr2:placeOfDeath rdf:resource=“h
ttp://sws.geonames.org/2635167/”/&gt;</code>For recommendations on medata quality see Tier A-C r
equirements , more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Agent.rdagr2_professionOrOccupation
```python
rdagr2_professionOrOccupation: Union[List[Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], List[edmlib.edm.value_types.Ref], List[edmlib.edm.value_types.Lit], NoneType]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[MixedValuesList]</p>

<p>Description: </p>

<p>The profession or occupation in which the agent works or has worked.<code>&lt;rdaGr2:profession
OrOccupation&gt;Instrument Maker&lt;/rdaGr2:professionOrOccupation&gt;</code></p>
</div> 

#### EDM_Agent.owl_sameAs
```python
owl_sameAs: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Another URI of the same agent.<code>&lt;owl:sameAs rdf:resource=“http://www.identifier/sameReso
urceElsewhere”/&gt;</code></p>
</div> 

#### EDM_Agent.validate_skos_pref_label
```python
@model_validator(mode='after') def validate_skos_pref_label(self) -> Self
```





#### EDM_Agent.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### EDM_TimeSpan
```python
class EDM_TimeSpan(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>optional-properties: SKOS_altLabel, SKOS_note, DCTERMS_hasPart, DCTERMS_isPartOf, EDM_isNextInSequence, OWL_sameAs</p>

<p>recommended-properties: SKOS_prefLabel, EDM_begin, EDM_end</p>
</div> 

#### EDM_TimeSpan.skos_prefLabel
```python
skos_prefLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The preferred form of the name of the timespan or period. Although the maximum number 
of occurrences is set at 1, it can be interpreted as 1 per language tag. At least one 
skos:prefLabel SHOULD be provided. Several prefLabels with languages tags are strongly
recommended for language variants andtranslations.<code>&lt;skos:prefLabel xml:lang=“en”&gt;Roman
Empire&lt;/skos:prefLabel&gt;</code>This is a recommended property for this class.</p>
</div> 

#### EDM_TimeSpan.skos_altLabel
```python
skos_altLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Alternative forms of the name of the timespan or period. <code>&lt;skos:altLabel xml:lang=''fr'
'&gt;Empire romain (27 avant J.-­‐C.-­‐476 après J.-­C.)&lt;/skos:altLabel &gt;</code></p>
</div> 

#### EDM_TimeSpan.skos_note
```python
skos_note: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Information relating to the timespan or period.<code>&lt;skos:note&gt;The Roman Empire (Latin: Imp
erium Romanum) was the post-­Republican period of the ancient Roman civilization, char
acterised by an autocratic form of government and large territorial holdings around th
e Mediterranean in Europe, Africa, and Asia.&lt;/skos:note&gt;</code></p>
</div> 

#### EDM_TimeSpan.dcterms_hasPart
```python
dcterms_hasPart: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to a timespan which is part of the described timespan.</p>
</div> 

#### EDM_TimeSpan.dcterms_isPartOf
```python
dcterms_isPartOf: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to a timespan of which the described timespan is a part.</p>
</div> 

#### EDM_TimeSpan.edm_begin
```python
edm_begin: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date the timespan started. Europeana recommends date conforming to ISO 8601 starti
ng with the year and with hyphens (YYYY-­MM-DD). Providing edm:begin in combination wi
th edm:end is recommended for this class.Example 1: <code>&lt;edm:begin&gt;-0026&lt;/edm:begin&gt;</code>Exampl
e:2: <edm:begin>27 BC</edm:begin>Note: '27 BC', while allowed, does not follow the abo
ve recommendation.For recommendations on medata quality see Tier A-C requirements , mo
re specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_TimeSpan.edm_end
```python
edm_end: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
recommended</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The date the timespan finished. Europeana recommends date conforming to ISO 8601 start
ing with the year and with hyphens (YYYY‐MM-DD). Providing edm:end in combination with
edm:begin is recommended for this class.<code>&lt;edm:end&gt;1770&lt;/edm:end&gt;</code>For recommendations on
medata quality see Tier A-C requirements , more specifically Metadata Tier B and Meta
data Tier C</p>
</div> 

#### EDM_TimeSpan.edm_isNextInSequence
```python
edm_isNextInSequence: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Can be used to represent a sequence of Time periods. Use this for objects that are par
t of a hierarchy or sequence to ensure correct display in the portal.<code>&lt;edm:isNextInSequ
ence rdf:resource=“http://semium.org/time/roman_republic”/&gt;</code> (The Roman Empire was prec
eded by the Roman Republic)</p>
</div> 

#### EDM_TimeSpan.owl_sameAs
```python
owl_sameAs: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>The URI of a timespan<code>&lt;owl:sameAs rdf:resource=“http://semium.org/time/roman_empire”/&gt;</code></p>
</div> 

#### EDM_TimeSpan.validate_skos_pref_label
```python
@model_validator(mode='after') def validate_skos_pref_label(self) -> Self
```





#### EDM_TimeSpan.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### EDM_Place
```python
class EDM_Place(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>optional-properties: WGS84_POS_lat, WGS84_POS_long, WGS84_POS_alt, SKOS_prefLabel, SKOS_altLabel, SKOS_note, DCTERMS_hasPart, DCTERMS_isPartOf, EDM_isNextInSequence, OWL_sameAs</p>
</div> 

#### EDM_Place.wgs84_pos_lat
```python
wgs84_pos_lat: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The latitude of a spatial thing (decimal degrees). This is a recommended property for 
this class.<code>&lt;wgs84_pos:lat&gt;51.5075&lt;/wgs84_pos:lat&gt;</code>For recommendations on medata quality
see Tier A-C requirements , more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Place.wgs84_pos_long
```python
wgs84_pos_long: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The longitude of a spatial thing (decimal degrees). This is a recommended property for
this class.<code>&lt;wgs84_pos:long&gt;-­‐0.1231&lt;/wgs84_pos:long&gt;</code>For recommendations on medata qu
ality see Tier A-C requirements, more specifically Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Place.wgs84_pos_alt
```python
wgs84_pos_alt: Optional[edmlib.edm.value_types.Lit]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[Lit]</p>

<p>Description: </p>

<p>The altitude of a spatial thing (decimal metres above the reference)<code>&lt;wgs84_pos:alt&gt;21&lt;
/wgs84_pos:alt&gt;</code></p>
</div> 

#### EDM_Place.skos_prefLabel
```python
skos_prefLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_one</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>The preferred form of the name of the place. Although the maximum number of occurrence
s is set at 1, it can be interpreted as 1 per language tag. At least one skos:prefLabe
l SHOULD be provided. Several prefLabels with languages tags are strongly recommended 
for language variants and translations.<code>&lt;skos:prefLabel xml:lang="en"&gt;London&lt;/skos:pref
Label&gt;</code>For recommendations on medata quality see Tier A-C requirements , more specifica
lly Metadata Tier B and Metadata Tier C</p>
</div> 

#### EDM_Place.skos_altLabel
```python
skos_altLabel: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Alternative forms of the name of the place.<code>&lt;skos:altLabel xml:lang="en"&gt;Greater London
&lt;/skos:altLabel&gt;</code></p>
</div> 

#### EDM_Place.skos_note
```python
skos_note: Optional[List[edmlib.edm.value_types.Lit]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Lit]]</p>

<p>Description: </p>

<p>Information relating to the place.<code>&lt;skos:note xml:lang="en"&gt;Pop. 21m&lt;/skos:note&gt;</code></p>
</div> 

#### EDM_Place.dcterms_hasPart
```python
dcterms_hasPart: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to a place that is part of the place being described.<code>&lt;dcterms:hasPart rdf:re
source=“http://sws.geonames.org/2643741/”/&gt;</code> (City of London)</p>
</div> 

#### EDM_Place.dcterms_isPartOf
```python
dcterms_isPartOf: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Reference to a place that the described place is part of.<code>&lt;dcterms:isPartOf rdf:resourc
e=“http://sws.geonames.org/2635167/”/&gt;</code> (United Kingdom)</p>
</div> 

#### EDM_Place.edm_isNextInSequence
```python
edm_isNextInSequence: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>Can be used to represent a sequence of Place entities over time e.g. the historical la
yers of the city of Troy. Use this for objects that are part of a hierarchy or sequenc
e to ensure correct display in the portal.</p>
</div> 

#### EDM_Place.owl_sameAs
```python
owl_sameAs: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
optional</p>

<p>Cardinality: 
zero_to_many</p>

<p>Value-Type:
Optional[List[Ref]]</p>

<p>Description: </p>

<p>URI of a Place<code>&lt;owl:sameAs rdf:resource=“http://sws.geonames.org/2635167/”/&gt;</code>(London)</p>
</div> 

#### EDM_Place.validate_skos_pref_label
```python
@model_validator(mode='after') def validate_skos_pref_label(self) -> Self
```





#### EDM_Place.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### SVCS_Service
```python
class SVCS_Service(edmlib.edm.base.EDM_BaseClass):
``` 



<div class="docstring"><p>(Manually copied)</p>

<p>Optional-Properties:
dcterms_conformsTo, doap_implements</p>

<p>Mandatory-Properties: None
Recommended-Proeprties: None</p>

<p>Definition:
An established standard to which the web resource or service conforms.
W3C WCAG 2.0 (web content accessibility guidelines).
If the Service describes a IIIF resource, dcterms:conformsTo must be used
to describe the IIIF protocol the resource is conforming to.</p>

<p>Example:
<code>&lt;dcterms:conformsTo rdf:resource="http://iiif.io/api/image"/&gt;</code></p>
</div> 

#### SVCS_Service.dcterms_conformsTo
```python
dcterms_conformsTo: Optional[List[edmlib.edm.value_types.Ref]]
```



<div class="docstring"><p>Mandate: 
Optional</p>

<p>Definition: 
An established standard to which the web resource or service conforms. 
W3C WCAG 2.0 (web content accessibility guidelines). If the Service describes 
a IIIF resource, dcterms:conformsTo must be used to describe the IIIF protocol 
the resource is conforming to.</p>

<p>Example: 
<code>&lt;dcterms:conformsTo rdf:resource="http://iiif.io/api/image"/&gt;</code></p>
</div> 

#### SVCS_Service.doap_implements
```python
doap_implements: Optional[edmlib.edm.value_types.Ref]
```



<div class="docstring"><p>Mandate: 
Optional</p>

<p>Definition: 
A specification that a project implements. Could be a standard, API or legally defined level of conformance. 
In IIIF doap:implements refers to the the protocol implemented in IIIF.</p>

<p>Example: 
<code>&lt;doap:implements rdf:resource="http://iiif.io/api/image/2/level1.json"/&gt;</code></p>
</div> 

#### SVCS_Service.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

### MixedValuesList
```python
MixedValuesList = typing.Union[typing.List[typing.Union[edmlib.edm.value_types.Lit, edmlib.edm.value_types.Ref]], typing.List[edmlib.edm.value_types.Ref], typing.List[edmlib.edm.value_types.Lit]]
```





---
### EDM_Parser
```python
class EDM_Parser:
``` 



<div class="docstring"><p>Parser for edm-xml records. Returns an edm_python.edm.EDM_Record object.</p>
</div> 

#### EDM_Parser.__init__
```python
EDM_Parser(graph: rdflib.graph.Graph)
```





#### EDM_Parser.from_file
```python
@classmethod def from_file(cls, path: str, format: str = 'xml') -> Self
```





#### EDM_Parser.from_string
```python
@classmethod def from_string(cls, content: str, format: str = 'xml') -> Self
```





#### EDM_Parser.graph
```python
graph: rdflib.graph.Graph
```





#### EDM_Parser.get_single_ref
```python
def get_single_ref(self, obj_cls: object) -> rdflib.term.URIRef
```



<div class="docstring"><p>Loooks up instances of a given obj_cls (a edm_python edm-class) and returns
a single IRI.
This method expects that the cardinality of the obj_cls is one per record.</p>
</div> 

#### EDM_Parser.get_many_ref
```python
def get_many_ref(self, obj_cls: object) -> List[rdflib.term.URIRef]
```



<div class="docstring"><p>Loooks up instances of a given obj_cls (a edm_python edm-class) and returns
a list of instance-IRIs.
This method expects that the cardinality of the obj_cls is one or more.</p>
</div> 

#### EDM_Parser.get_triples
```python
def get_triples(self, ref: rdflib.term.URIRef)
```



<div class="docstring"><p>Return all predicate-object triples for a given URIRef within the instance`s-graph.</p>
</div> 

#### EDM_Parser.get_aggregation
```python
def get_aggregation(self)
```





#### EDM_Parser.get_webresources
```python
def get_webresources(self) -> list[typing.Any]
```





#### EDM_Parser.get_instance_triples
```python
def get_instance_triples(self, instance: rdflib.term.URIRef, cls_obj: object) -> Dict[str, Any]
```





#### EDM_Parser.parse_single_class
```python
def parse_single_class(self, cls_obj: object) -> Any
```





#### EDM_Parser.parse_many_class
```python
def parse_many_class(self, cls_obj: Any) -> List[Any]
```





#### EDM_Parser.parse
```python
def parse(self) -> edmlib.edm.record.EDM_Record
```






---

---
### Ref
```python
class Ref(pydantic.main.BaseModel):
``` 



<div class="docstring"><p>About IRIs (from the rdflib.URIRef docstring):</p>

<p>RDF 1.1's IRI Section <a href="https://www.w3.org/TR/rdf11-concepts/#section-IRIs">https://www.w3.org/TR/rdf11-concepts/#section-IRIs</a>
An IRI (Internationalized Resource Identifier) within an RDF graph is a Unicode string that conforms to the syntax defined in RFC 3987.
IRIs in the RDF abstract syntax MUST be absolute, and MAY contain a fragment identifier.
IRIs are a generalization of URIs [RFC3986] that permits a wider range of Unicode characters.</p>
</div> 

#### Ref.value
```python
value: Annotated[str, StringConstraints(strip_whitespace=True, to_upper=None, to_lower=None, strict=None, min_length=1, max_length=None, pattern=None)]
```





#### Ref.is_ref
```python
is_ref: bool
```





#### Ref.validate_value_as_uri
```python
@field_validator('value') @classmethod def validate_value_as_uri(cls, value: str)
```





#### Ref.to_rdflib
```python
def to_rdflib(self)
```



<div class="docstring"><p>Helper to convert this custom type to the rdflib equivalent
Used in the graph serialization of the EDM_Base-Class</p>
</div> 

#### Ref.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

---
### Lit
```python
class Lit(pydantic.main.BaseModel):
``` 



<div class="docstring"><p>Overrides the RDFLib Literal with a custom class, so that it is serializable in pydantic model.
For the same reason, it uses the same attribute names.
Ignore the normalize attribute, it is just added for completeness.</p>
</div> 

#### Lit.value
```python
value: Annotated[str, StringConstraints(strip_whitespace=True, to_upper=None, to_lower=None, strict=None, min_length=1, max_length=None, pattern=None)]
```





#### Lit.lang
```python
lang: Optional[str]
```





#### Lit.datatype
```python
datatype: Optional[str]
```





#### Lit.normalize
```python
normalize: Optional[bool]
```





#### Lit.validate_consistency
```python
@model_validator(mode='after') def validate_consistency(self) -> Self
```



<div class="docstring"><p>Checks that literal either has a lang_tag or a datatype, not both.</p>
</div> 

#### Lit.to_rdflib
```python
def to_rdflib(self)
```



<div class="docstring"><p>Helper to convert this custom type to the rdflib equivalent
Used in the graph serialization of the EDM_Base-Class</p>
</div> 

#### Lit.model_config
```python
model_config: ClassVar[pydantic.config.ConfigDict] = {}
```



<div class="docstring"><p>Configuration for the model, should be a dictionary conforming to [<code>ConfigDict</code>][pydantic.config.ConfigDict].</p>
</div> 


---

<!--pdoc-end-->