"""These tests are for the dev.py CLI used to create the bump and changelog entry"""

from pathlib import Path
from unittest.mock import MagicMock

import pytest

import dev


def get_release_process_test_cases():
    yield pytest.param(
        """# Description

**REVIEWER** Without the generated code this is a ~200 lines PR. Everything inside `/examples` is generated.

## Bump

- [x] Patch
- [ ] Minor
- [ ] Skip

## Changelog
### Added

- [alpha] New method in generated SDK, `.iterate(...)`. This supports looping over items to keep memory footprint small
when retrieving large amount of items.
""",
        "1.2.1",
        """
### Added

- [alpha] New method in generated SDK, `.iterate(...)`. This supports looping over items to keep memory footprint small
when retrieving large amount of items.""",
        "1.2.2",
        id="Patch bump",
    )

    yield pytest.param(
        """# Description

Support for filtering on enum properites.

**Reviewer Notes**
* Tried to have a clean history as most of the code lines are generated.
* All code inside the `examples/` directory is generated.
* This PR is ~40 lines code, ~10 lines test and the rest is generated code.

## Bump

- [ ] Patch
- [x] Minor
- [ ] Skip

## Changelog
### Added

- The generated SDK now will create filter options for enums, for example, if you create an SDK for the `CogniteCore`
model, you can now filter on `.list(type_="numeric")` to only get numeric timeseries.
---------

Co-authored-by: Member <member@users.noreply.github.com>
Co-authored-by: Member <member@cognite.com>""",
        "1.1.1",
        """### Added

- The generated SDK now will create filter options for enums, for example, if you create an SDK for the `CogniteCore`
model, you can now filter on `.list(type_="numeric")` to only get numeric timeseries.""",
        "1.2.0",
        id="Minor bump with co-authors",
    )

    yield pytest.param(
        """This PR contains the following updates:

| Package | Type | Update | Change |
|---|---|---|---|
| [actions/checkout](https://redirect.github.com/actions/checkout) | action | major | `v3` -> `v4` |

---

### Release Notes

<details>
<summary>actions/checkout (actions/checkout)</summary>

### [`v4`](https://redirect.github.com/actions/checkout/blob/HEAD/CHANGELOG.md#v422)

[Compare Source](https://redirect.github.com/actions/checkout/compare/v3...v4)

-   `url-helper.ts` now leverages well-known environment variables by [@&#8203;jww3](https://redirect.github.com/jww3)
in [https://github.com/actions/checkout/pull/1941](https://redirect.github.com/actions/checkout/pull/1941)
-   Expand unit test coverage for `isGhes` by [@&#8203;jww3](https://redirect.github.com/jww3) in
[https://github.com/actions/checkout/pull/1946](https://redirect.github.com/actions/checkout/pull/1946)

</details>

---

### Configuration

📅 **Schedule**: Branch creation - At any time (no schedule defined), Automerge - At any time (no schedule defined).

🚦 **Automerge**: Disabled by config. Please merge this manually once you are satisfied.

♻ **Rebasing**: Whenever PR is behind base branch, or you tick the rebase/retry checkbox.

🔕 **Ignore**: Close this PR and you won't be reminded about this update again.

---

 - [ ] <!-- rebase-check -->If you want to rebase/retry this PR, check this box

---

This PR was generated by [Mend Renovate](https://mend.io/renovate/). View the [repository job log](https://developer.mend.io/github/cognitedata/pygen).
<!--renovate-debug:eyJjcmVhdGVkSW5WZXIiOiIzOS4yMDAuMCIsInVwZGF0ZWRJblZlciI6IjM5LjIyNy4zIiwidGFyZ2V0QnJhbmNoIjoibWFpbiIsImxhYmVscyI6W119-->
""",
        "1.2.4",
        "",
        "1.2.4",
        id="Skip bump with renovate",
    )


class TestReleaseProcess:
    @pytest.mark.parametrize(
        "last_git_message, last_version, expected_changelog, expected_version", list(get_release_process_test_cases())
    )
    def test_bump_and_create_changelog_entry(
        self, last_git_message: str, last_version: str, expected_changelog: str, expected_version: str, monkeypatch
    ) -> None:
        actual_changelog_entry: str | None = None
        actual_version: str | None = None
        last_git_message_file = MagicMock(spec=Path)
        last_git_message_file.read_text.return_value = last_git_message
        last_version_file = MagicMock(spec=Path)
        last_version_file.read_text.return_value = last_version

        def mock_write_changelog(content, encoding=None):
            nonlocal actual_changelog_entry
            actual_changelog_entry = content

        changelog_file = MagicMock(spec=Path)
        changelog_file.write_text = mock_write_changelog

        version_file = MagicMock(spec=Path)
        version_file.read_text.return_value = dev.VERSION_PLACEHOLDER

        def mock_write_version(content, **_):
            nonlocal actual_version
            actual_version = content

        version_file.write_text = mock_write_version

        monkeypatch.setattr(dev, "LAST_GIT_MESSAGE_FILE", last_git_message_file)
        monkeypatch.setattr(dev, "LAST_VERSION", last_version_file)
        monkeypatch.setattr(dev, "CHANGELOG_ENTRY_FILE", changelog_file)
        monkeypatch.setattr(dev, "VERSION_FILES", [version_file])

        dev.bump()
        dev.create_changelog_entry()

        is_skip = last_version == expected_version
        if is_skip:
            assert actual_version is None, "Version was updated when it should not have been"
            assert actual_changelog_entry is None, "Changelog entry was created when it should not have been"
            return

        assert actual_version is not None, "Version was not updated"
        assert actual_version == expected_version

        assert actual_changelog_entry is not None, "Changelog entry was not created"
        removed_trailing_space = "\n".join([line.rstrip() for line in actual_changelog_entry.splitlines()])
        assert removed_trailing_space == expected_changelog
