from uuid import uuid4
from typing import List

import dateutil.parser
from textual.app import App, ComposeResult
from textual.widgets import Button, Header
from textual import on
from textual.app import App, ComposeResult
from textual.validation import Number
from textual.widgets import Input, Label, Pretty, Checkbox, Footer
from textual.containers import Horizontal
from textual.validation import Length
from rich import print
from exploitfarm.utils.config import ClientConfig
from exploitfarm.utils.reqs import get_url
import dateutil, traceback
from datetime import datetime as dt, timedelta 
import datetime
from exploitfarm.utils import DEV_MODE

TOLLERANCE_TIME = timedelta(seconds=5)

class InitialConfiguration(App):
    
    def __init__(self, config: ClientConfig, initial_error: str|None = None):
        super().__init__()
        self.title = "xFarm - Initial Configuration"
        self.config = config
        self.initial_error = "[bold red]"+initial_error+"[/]" if initial_error else ""
        self.ignore_connection_failed = False
    
    BINDINGS = [
        ("ctrl+t", "test_connection()", "Test connection to server"),
        ("ctrl+s", "save()", "Save configuration"),
        ("ctrl+c", "cancel()", "Cancel configuration")
    ]
    
    CSS = """
    Input.-valid {
        border: tall $success 60%;
    }
    Input.-valid:focus {
        border: tall $success;
    }
    Input {
        margin: 1 1;
    }
    Pretty {
        margin-left: 2;
    }
    #error{
        margin: 2 0 0 2;
    }
    .input-label {
        margin: 1 2;
    }
    .error-box {
        margin: 1 2;
        height: 1.8;
    }
    Button {
        margin: 1 2;
    }
    .float-right {
        content-align: right middle;
    }
    .button-box {
        height: 4;
        margin: 2 0;
        margin-top: 1;
    }
    .max-width {
        width: 100%;
    }
    .hidden {
        display: none;
    }
    """

    def compose(self) -> ComposeResult:
        yield Header("xFarm - Initial Configuration")
        yield Label("[bold]Welcome to xFarm, please fill the following fields to configure the client", classes="error-box")
        yield Label("[bold]Server address:[/]", classes="input-label")
        yield Input(
            placeholder="127.0.0.1",
            value=self.config.server.address,
            validators=[Length(minimum=1)],
            id="address",
            classes="form-input"
        )
        yield Horizontal(
            Label("[bold]Errors:[/]"),
            Pretty([], id="address_errors"),
            classes="error-box hidden",
            id="address_errors_box"
        )
        yield Label("[bold]Server port:[/]", classes="input-label")
        yield Input(
            placeholder="5050",
            value=str(self.config.server.port),
            validators=[Number(minimum=1, maximum=65535)],
            id="port",
            classes="form-input"
        )
        yield Horizontal(
            Label("[bold]Errors:[/]"),
            Pretty([], id="port_errors"),
            classes="error-box hidden",
            id="port_errors_box"
        )
        yield Label("[bold]Nickname:[/]", classes="input-label")
        yield Input(
            placeholder="John Doe",
            value=self.config.client_name,
            validators=[Length(minimum=1)],
            id="nickname",
            classes="form-input"
        )
        yield Horizontal(
            Label("[bold]Errors:[/]"),
            Pretty([], id="nickname_errors"),
            classes="error-box hidden",
            id="nickname_errors_box"
        )
        yield Checkbox("Use HTTPS", id="https", value=self.config.server.https, classes="input-label")
        yield Checkbox("Ignore connection failed", id="ign_conn_failed", value=self.ignore_connection_failed, classes="input-label")
        yield Label(self.initial_error, classes=f"{'' if self.initial_error else 'hidden'}", id="error")
        yield Horizontal(
            Button("Save", id="save", variant="success"),
            Button("Cancel", id="cancel", variant="error"),
            Button("Test Connection", id="test", variant="primary", classes="float-right"),
            classes="max-width button-box"
        )
        yield Footer()
    
    def action_save(self):
        self.save()
    
    def action_cancel(self):
        self.cancel()
    
    def action_test_connection(self):
        self.test()
        
    @on(Button.Pressed, "#save")
    def save(self):
        input_forms: List[Input] = self.query(".form-input")
        errors = []
        for form in input_forms:
            errors.extend([ele.failure_description for ele in form.validators if ele.failure_description])
        if self.config.server.address.startswith("http://"):
            self.config.server.https = False
            self.config.server.address = self.config.server.address[len("http://"):]
        if self.config.server.address.startswith("https://"):
            self.config.server.https = True
            self.config.server.address = self.config.server.address[len("https://"):]
            
        if len(errors) == 0 and (self.ignore_connection_failed or self.test()):
            self.config.write()
            self.exit(99 if self.ignore_connection_failed else 0)
    
    @on(Button.Pressed, "#cancel")
    def cancel(self):
        self.exit(1)
    
    @on(Button.Pressed, "#test")
    def test(self):
        error_label = self.query_one("#error", Label)
        error_label.remove_class("hidden")
        try:
            self.config.fetch_status()
            error_label.update(f"[bold green]Connection done successfully to server[/]: {get_url('//', self.config)}")
            return True
        except Exception as e:
            message = f" {traceback.format_exc()}" if DEV_MODE else ""
            error_label.update(f"[bold red]Failed to connect to server at {get_url('//', self.config)}[/]: [orange]{e}{message}[/]")
            return False
    
    @on(Checkbox.Changed, "#https")
    def https_change(self, event: Checkbox.Changed):
        self.config.server.https = event.value
    
    @on(Checkbox.Changed, "#ign_conn_failed")
    def ignore_connection_failed_change(self, event: Checkbox.Changed):
        self.ignore_connection_failed = event.value
    
    @on(Input.Changed, "#address")
    def address_check(self, event: Input.Changed) -> None:
        # Updating the UI to show the reasons why validation failed
        error_label = self.query_one("#address_errors", Pretty)
        error_box = self.query_one("#address_errors_box", Horizontal)
        if not event.validation_result.is_valid:
            error_box.remove_class("hidden")
            error_label.update(event.validation_result.failure_descriptions)
        else:
            error_box.add_class("hidden")
            self.config.server.address = event.input.value
    
    @on(Input.Changed, "#port")
    def port_check(self, event: Input.Changed) -> None:
        # Updating the UI to show the reasons why validation failed
        error_label = self.query_one("#port_errors",Pretty)
        error_box = self.query_one("#port_errors_box", Horizontal)
        if not event.validation_result.is_valid:
            error_box.remove_class("hidden")
            error_label.update(event.validation_result.failure_descriptions)
        else:
            error_box.add_class("hidden")
            self.config.server.port = int(event.input.value)
            
    
    @on(Input.Changed, "#nickname")
    def nickname_check(self, event: Input.Changed) -> None:
        # Updating the UI to show the reasons why validation failed
        error_label = self.query_one("#nickname_errors",Pretty)
        error_box = self.query_one("#nickname_errors_box", Horizontal)
        if not event.validation_result.is_valid:
            error_box.remove_class("hidden")
            error_label.update(event.validation_result.failure_descriptions)
        else:        
            error_box.add_class("hidden")
            self.config.client_name = event.input.value

def _run_setup_config(config: ClientConfig, initial_error: str|None = None):
    init_conf = InitialConfiguration(config, initial_error)
    status = init_conf.run()
    if status == 99:
        return False
    elif status != 0:
        print(f"[bold red]Failed to configure the client[/]")
        exit(1)
    return True

def inital_config_setup(config: ClientConfig, interactive: bool = True) -> bool:
    
    if config.client_id is None:
        config.client_id = uuid4()
        config.write()
    
    if (
        config.server.address is None   or
        config.client_name is None      or
        config.client_name == ""
    ):
        if not interactive:
            print(f"[bold red]Missing configuration, set it with 'xfarm config' command[/]")
            exit(1)
        if not _run_setup_config(config):
            return False
    try:
        delta_request = config.delta_fetch_status()
    except Exception:
        msg = f"Connection to {get_url('//', config)} failed, the url is wrong or the server is down"
        if not interactive:
            print(f"[bold red]{msg}[/]")
            exit(1)
        if not _run_setup_config(config, msg):
            return False
        try:
            delta_request = config.delta_fetch_status()
        except Exception:
            print(f"[bold red]{msg}[/]")
            exit(1)

    if config.status:
        server_time = dateutil.parser.parse(config.status["server_time"], tzinfos={"UTC": datetime.timezone.utc})
        delta = server_time - dt.now(datetime.timezone.utc)
        delta = abs(delta.total_seconds())
        tollerance = TOLLERANCE_TIME+delta_request
        tollerance = abs(tollerance.total_seconds())
        if delta > tollerance:
            print(f"[bold yellow]⚠️ WARNING! The server time is not synchronized with the client, please check the server/client time is correct (delta of {delta} > tol: {tollerance})[/]")
    return True
