from exploitfarm.utils import *
from exploitfarm.utils.reqs import *
from typing import List
from rich import print

from textual.app import App, ComposeResult
from textual.widgets import Button, Header
from textual import on
from textual.app import App, ComposeResult
from textual.widgets import Input, Label, Pretty, Checkbox, Footer
from textual.containers import Horizontal
from textual.validation import Length
from typing import Tuple

def login_required(config: ClientConfig, interactive:bool = True, subscribe_client:bool = True):
    if config.status["loggined"]:
        pass
    else:
        if not interactive:
            print(f"[bold red]Login failed, you can't connect to {get_url('//', config)}, to access this functionality login is required![/]")
            exit(1)
        login = LoginScreen(config)
        status = login.run()
        if status == 0:
            print("[bold green]Loggined successful[/]")
        else:
            print(f"[bold red]Login failed, you can't connect to {get_url('//', config)}, to access this functionality login is required![/]")
            exit(1)
    if subscribe_client:
        config.reqs.self_subscribe_client()

def try_authenticate(password:str, config: ClientConfig) -> Tuple[bool, str|None]:
    if not password:
        return False, "Password is required"
    msg = "Unexpected error"
    try:
        auth = config.reqs.authenicate(password)
        if auth:
            return True, None
        msg = "Failed to login: probably wrong password"
    except ReqsError as e:
        msg = f"Wrong Password: {e}"
    except Exception as e:
        msg = f"Connection error, probably server unreachable: {str(e)[:100]}"
    return False, msg

class LoginScreen(App):
    
    BINDINGS = [
        ("ctrl+s", "login()", "Login to xFarm server"),
        ("ctrl+c", "cancel()", "Cancel operation")
    ]
    
    def __init__(self, config: ClientConfig):
        super().__init__()
        self.title = "xFarm - Server Authentication Required"
        self.config = config
        self.password_value = ""
    
    CSS = """
    Input.-valid {
        border: tall $success 60%;
    }
    Input.-valid:focus {
        border: tall $success;
    }
    Input {
        margin: 1 1;
    }
    Pretty {
        margin-left: 2;
    }
    .input-label {
        margin: 1 2;
    }
    .error-box {
        margin: 1 2;
        height: 1.8;
    }
    Button {
        margin: 1 2;
    }
    .button-box {
        height: 4;
        margin: 2 0;
        margin-top: 1;
    }
    .max-width {
        width: 100%;
    }
    .hidden {
        display: none;
    }
    """
    
    def action_login(self):
        self.save()
    
    def action_cancel(self):
        self.cancel()

    def compose(self) -> ComposeResult:
        yield Header("xFarm - Login")
        yield Label(f"[bold]xFarm login to {get_url('//', self.config)}", classes="error-box")
        yield Label("[bold]Password:[/]", classes="input-label")
        yield Input(
            placeholder="*******",
            value=self.password_value,
            validators=[Length(minimum=1)],
            password=True,
            id="password",
            classes="form-input"
        )
        yield Horizontal(
            Label("[bold]Errors:[/]"),
            Pretty([], id="password_errors"),
            classes="error-box hidden",
            id="password_errors_box"
        )
        yield Checkbox("Show Password", id="show-psw", value=self.config.server.https, classes="input-label")
        yield Horizontal(
            Button("Login", id="save", variant="success"),
            Button("Cancel", id="cancel", variant="error"),
            classes="max-width button-box"
        )
        yield Footer()
        
    @on(Checkbox.Changed, "#show-psw")
    def https_change(self, event: Checkbox.Changed):
        self.query_one("#password", Input).password = not event.value
    
    @on(Button.Pressed, "#save")
    def save(self):
        input_forms: List[Input] = self.query(".form-input")
        errors = []
        for form in input_forms:
            errors.extend([ele.failure_description for ele in form.validators if ele.failure_description])
        if len(errors) == 0:
            done, error = try_authenticate(self.password_value, self.config)
            if done:
                self.config.fetch_status()
                self.exit(0)
            else:
                error_label = self.query_one("#password_errors",Pretty)
                error_box = self.query_one("#password_errors_box", Horizontal)
                error_box.remove_class("hidden")
                error_label.update([error])
    
    @on(Button.Pressed, "#cancel")
    def cancel(self):
        self.exit(1)
    
    @on(Input.Changed, "#password")
    def password_change(self, event: Input.Changed) -> None:
        # Updating the UI to show the reasons why validation failed
        error_label = self.query_one("#password_errors",Pretty)
        error_box = self.query_one("#password_errors_box", Horizontal)
        if not event.validation_result.is_valid:
            error_box.remove_class("hidden")
            error_label.update(event.validation_result.failure_descriptions)
        else:        
            error_box.add_class("hidden")
            self.password_value = event.input.value
