import requests, os
from uuid import UUID
from typing import Dict, Any
from exploitfarm.utils.config import ClientConfig
import orjson
from datetime import datetime as dt
from exploitfarm.model import AttackMode, SetupStatus
from posixpath import join as urljoin

HTTP_TIMEOUT = float(os.getenv("XFARM_HTTP_TIMEOUT", 5))
PATH_OFFSET = os.getenv("XFARM_PATH_OFFSET", "/api")

def get_url(path:str, config: ClientConfig|None = None) -> str:
    protocol = "https" if config.server.https else "http"
    result = f"{protocol}://{config.server.address}:{config.server.port}"+urljoin("/", PATH_OFFSET, path[1:] if path.startswith('/') else path)
    if result.endswith("/"): result = result[:-1]
    return result

def jsonify(data: Any) -> str:
    return orjson.loads(orjson.dumps(data))

def get_session(auth:str|None|ClientConfig = None) -> requests.Session:
    res = requests.Session()
    if auth is not None:
        if isinstance(auth, str):
            res.headers.update({"Authorization": "Bearer "+auth})
        elif isinstance(auth, ClientConfig) and auth.server.auth_key is not None:
            res.headers.update({"Authorization": "Bearer "+auth.server.auth_key})
    return res

class ReqsError(Exception):pass

class EMPTY:pass

def requests_check(res: Dict) -> Any:
    if res["status"] != "ok":
        msg = res.get("message", None)
        if isinstance(msg, str):
            raise ReqsError(msg)
        else:
            raise ReqsError("Unknown error")
    else:
        return res["response"]

class Reqs:
    def __init__(self, config: ClientConfig):
        self.config = config
        self.session = get_session(config)
    
    def refresh_session(self, auth:str|ClientConfig|None = None):
        self.session = get_session(auth if auth else self.config)
    
    def status(self) -> Dict[str, Any]:
        res = self.session.get(get_url("/status", self.config), timeout=HTTP_TIMEOUT).json()
        try:
            if res["whoami"] != "exploitfarm":
                raise Exception()
        except Exception:
            raise ReqsError("Invalid server")
        return res
    
    def submitters(self) -> Any:
        return self.session.get(get_url("/submitters", self.config), timeout=HTTP_TIMEOUT).json()

    def teams(self) -> Any:
        return self.session.get(get_url("/teams", self.config), timeout=HTTP_TIMEOUT).json()
    
    def clients(self) -> Any:
        return self.session.get(get_url("/clients", self.config), timeout=HTTP_TIMEOUT).json()
    
    def flags(self) -> Any:
        return self.session.get(get_url("/flags", self.config), timeout=HTTP_TIMEOUT).json()

    def exploits(self) -> Any:
        return self.session.get(get_url("/exploits", self.config), timeout=HTTP_TIMEOUT).json()
    
    def services(self) -> Any:
        return self.session.get(get_url("/services", self.config), timeout=HTTP_TIMEOUT).json()
    
    def new_submitter(self, data: dict) -> Any:
        return requests_check(self.session.post(get_url("/submitters", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
    
    def delete_submitter(self, submitter_id:int) -> Any:
        return requests_check(self.session.delete(get_url(f"/submitters/{submitter_id}", self.config), timeout=HTTP_TIMEOUT).json())
    
    def test_submitter(self, submitter_id:int, flags:list[str]) -> Any:
        return requests_check(self.session.post(get_url(f"/submitters/{submitter_id}/test", self.config), json=flags, timeout=HTTP_TIMEOUT).json())
    
    def setup(self, data: dict) -> Any:
        return requests_check(self.session.post(get_url("/setup", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
    
    def configure_server(self,
        flag_regex:str|EMPTY = EMPTY,
        tick_duration:int|EMPTY = EMPTY,
        submitter:int|EMPTY = EMPTY,
        start_time:dt|None|EMPTY = EMPTY,
        end_time:dt|None|EMPTY = EMPTY,
        attack_mode:AttackMode|EMPTY = EMPTY,
        loop_attack_delay:int|EMPTY = EMPTY,
        attack_time_tick_delay:int|EMPTY = EMPTY,
        flag_timeout:int|EMPTY = EMPTY,
        flag_submit_limit:int|None|EMPTY = EMPTY,
        submit_delay:int|EMPTY = EMPTY,
        submitter_timeout:int|EMPTY = EMPTY,
        authentication_required:bool|EMPTY = EMPTY,
        password_hash:str|None|EMPTY = EMPTY,
        setup_status:SetupStatus|EMPTY = EMPTY,
        set_running:bool = False
    ) -> Any:
        request = {
            "FLAG_REGEX": flag_regex,
            "START_TIME": start_time,
            "END_TIME": end_time,
            "TICK_DURATION": tick_duration,
            "ATTACK_MODE": attack_mode,
            "LOOP_ATTACK_DELAY": loop_attack_delay,
            "ATTACK_TIME_TICK_DELAY": attack_time_tick_delay,
            "FLAG_TIMEOUT": flag_timeout,
            "FLAG_SUBMIT_LIMIT": flag_submit_limit,
            "SUBMIT_DELAY": submit_delay,
            "SUBMITTER": submitter,
            "SUBMITTER_TIMEOUT": submitter_timeout,
            "AUTHENTICATION_REQUIRED": authentication_required,
            "PASSWORD_HASH": password_hash,
            "SETUP_STATUS": setup_status if not set_running else SetupStatus.RUNNING.value
        }
        request = {key: value for key, value in request.items() if value is not EMPTY}
        return self.setup(request)
    
    def login(self,password:str) -> str:
        try:
            return self.session.post(get_url("/login", self.config), data={"grant_type":"password", "username": "user", "password": password}, timeout=HTTP_TIMEOUT).json()["access_token"]
        except KeyError:
            raise ReqsError("Failed to login")
    
    def authenicate(self, password:str, save:bool = True) -> str|None:
        auth_key = self.login(password)
        self.refresh_session(auth_key)
        self.config.server.auth_key = auth_key
        if save: self.config.write()
        return auth_key
    
    def new_client(self, data: dict) -> Any:
        return requests_check(self.session.post(get_url("/clients", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
    
    def self_subscribe_client(self) -> Any:
        return self.new_client({"name": self.config.client_name, "id": self.config.client_id})
            
    def new_teams(self, data: list) -> Any:
        return requests_check(self.session.post(get_url("/teams", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
    
    def new_service(self, data: dict) -> Any:
        return requests_check(self.session.post(get_url("/services", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
    
    def new_exploit(self, data: dict) -> Any:
        return requests_check(self.session.post(get_url("/exploits", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
    
    def submit_flags(self, data: list[dict], exploit:UUID|None = None) -> Any:
        if not exploit:
            return requests_check(self.session.post(get_url("/exploits/submit", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())
        else:
            return requests_check(self.session.post(get_url(f"/exploits/{exploit}/submit", self.config), json=jsonify(data), timeout=HTTP_TIMEOUT).json())

