# AssertLang CLI Guide

Complete guide to using the `assertlang` command-line interface.

## Installation

### From Source

```bash
# Clone repository
git clone https://github.com/assertlang/assertlang.git
cd assertlang

# Install in development mode
pip install -e .
```

### From PyPI (when published)

```bash
pip install assertlang
```

### Verify Installation

```bash
assertlang --version
# Output: AssertLang 0.1.0
```

## Quick Start

### 1. Create New Agent

```bash
# Create from basic template
assertlang init my-agent

# Create API agent
assertlang init api-service --template api --port 8080

# Create AI agent
assertlang init chatbot --template ai
```

### 2. Validate Agent

```bash
# Basic validation
assertlang validate my-agent.al

# Detailed validation with verbose output
assertlang validate my-agent.al --verbose
```

### 3. Generate Server

```bash
# Generate Python server
assertlang generate my-agent.al --lang python

# Generate Go server
assertlang generate my-agent.al --lang go

# Generate Node.js server with custom output
assertlang generate my-agent.al --lang typescript --output ./servers/nodejs
```

### 4. Build & Run

**Python:**
```bash
cd generated/my-agent
pip install -r requirements.txt
python my-agent_server.py
```

**Node.js:**
```bash
cd generated/my-agent
npm install
node my-agent_server.js
```

**Go:**
```bash
python3 scripts/build_server.py my-agent.al go
./examples/demo/go/my-agent
```

**C#:**
```bash
python3 scripts/build_server.py my-agent.al dotnet
cd examples/demo/dotnet && dotnet run
```

**Rust:**
```bash
python3 scripts/build_server.py my-agent.al rust
./examples/demo/rust/target/release/my-agent
```

### 5. Test Your Agent

Once your agent is running, test it with auto-generated tests:

```bash
# Health check and verb discovery
asl test http://localhost:3000

# Run full integration test suite
asl test http://localhost:3000 --auto

# Load test critical verbs
asl test http://localhost:3000 --load --verb user.create@v1 --requests 500

# Generate coverage report
asl test http://localhost:3000 --auto --coverage
```

## Commands Reference

### `asl test`

Test a running MCP agent with auto-generated integration tests and load testing.

**Usage:**
```bash
asl test <agent-url> [OPTIONS]
```

**Options:**
- `--auto` - Auto-generate and run integration tests from verb schemas
- `--load` - Run load tests (requires `--verb`)
- `--verb VERB` - Specific verb to load test
- `--requests NUM` - Number of load test requests (default: 100)
- `--concurrency NUM` - Concurrent requests for load testing (default: 10)
- `--coverage` - Export coverage report to coverage.json
- `--timeout SEC` - Request timeout in seconds (default: 30)

**Examples:**
```bash
# Basic health check and verb discovery
asl test http://localhost:3000

# Run auto-generated integration tests
asl test http://localhost:3000 --auto

# Load test a specific verb
asl test http://localhost:3000 --load --verb user.create@v1 --requests 1000 --concurrency 50

# Full test suite with coverage report
asl test http://localhost:3000 --auto --coverage
```

**What it does:**

1. **Health Check** - Verifies agent is running and responding
2. **Verb Discovery** - Lists all available verbs and their schemas
3. **Integration Tests** (with `--auto`):
   - Auto-generates test cases from verb schemas
   - Happy path tests with valid parameters
   - Error tests with missing/invalid parameters
   - Validates expected response fields
4. **Load Tests** (with `--load`):
   - Concurrent requests with configurable parallelism
   - Latency statistics (avg, min, max, P95, P99)
   - Throughput measurement (requests per second)
   - Error tracking and reporting
5. **Coverage Report** (with `--coverage`):
   - Exports JSON report with tested verbs
   - Percentage of verbs tested
   - Timestamp and agent URL

**Output:**
```
🧪 Testing agent at http://localhost:3000

✓ Health check passed: {'status': 'alive', 'uptime_seconds': 3600}

✓ Discovered 5 verbs

🧪 Running 10 integration tests...

[1/10] test_user_create_v1_happy_path... ✓ PASS (45ms)
[2/10] test_user_create_v1_missing_param... ✓ PASS (12ms)
...

============================================================
📊 Test Summary
============================================================
Total:    10
Passed:   10 ✓
Failed:   0 ✗
Coverage: 100.0%
============================================================

✨ Testing complete!
```

### `assertlang generate`

Generate MCP server from .al file.

**Usage:**
```bash
assertlang generate <file.al> [OPTIONS]
```

**Options:**
- `--lang LANGUAGE` - Target language (python, typescript, go, csharp, rust). Default: python
- `--output DIR` - Output directory. Default: `./generated/<agent-name>`
- `--build` - Build server after generation (for compiled languages)
- `--yes, -y` - Skip confirmation prompts (for CI/scripts)
- `--dry-run` - Show what would be generated without writing files
- `--quiet, -q` - Minimal output (for CI/scripts)

**Examples:**
```bash
# Python server (default)
assertlang generate agent.al

# Preview without writing files
assertlang generate agent.al --dry-run

# Go server with build (skip confirmation)
assertlang generate agent.al --lang go --build --yes

# Quiet mode for CI/CD
assertlang generate agent.al --quiet --yes

# Rust server with custom output
assertlang generate agent.al --lang rust --output ./production/rust-server
```

**Safety Features:**

- **Confirmation Prompt** - By default, shows what will be created and asks for confirmation
- **Dry-Run Mode** - Use `--dry-run` to preview without writing any files
- **Skip Confirmation** - Use `--yes/-y` to bypass prompts (ideal for CI/CD)
- **Quiet Mode** - Use `--quiet/-q` for minimal output (only errors)
- **NO_COLOR Support** - Respects `NO_COLOR` environment variable for plain text output

### `assertlang validate`

Validate .al file syntax and structure.

**Usage:**
```bash
assertlang validate <file.al> [OPTIONS]
```

**Options:**
- `--verbose` - Show detailed validation output including agent details

**Examples:**
```bash
# Basic validation
assertlang validate agent.al

# Detailed output
assertlang validate agent.al --verbose
```

**Output (verbose):**
```
🔍 Validating agent.pw...
✓ Syntax valid

📋 Agent Details:
  Name: user-service
  Port: 3000
  Verbs: 2
    - user.create@v1 (2 params, 1 returns)
    - user.get@v1 (1 params, 1 returns)
  Tools: http, auth, logger
```

### `assertlang list-tools`

List all available tools that can be used in .al agents.

**Usage:**
```bash
assertlang list-tools [OPTIONS]
```

**Options:**
- `--lang LANGUAGE` - Show tools for specific language (python, typescript, go, csharp, rust, all). Default: all
- `--category CATEGORY` - Filter by category

**Categories:**
- HTTP & APIs
- Authentication
- Storage & Data
- Flow Control
- Logging & Monitoring
- Scheduling
- Media
- System

**Examples:**
```bash
# List all tools
assertlang list-tools

# List Python tools only
assertlang list-tools --lang python

# List HTTP & API tools
assertlang list-tools --category "HTTP & APIs"
```

**Output:**
```
🛠️  Available AssertLang Tools

📦 HTTP & APIs
  • http                     [python, nodejs, go, csharp, rust]
  • rest                     [python, nodejs, go, csharp, rust]
  • api-auth                 [python, nodejs, go, csharp, rust]

📦 Authentication
  • auth                     [python, nodejs, go, csharp, rust]
  • encryption               [python, nodejs, go, csharp, rust]

Total: 38 tools
```

### `assertlang init`

Create new .al agent from template.

**Usage:**
```bash
assertlang init <name> [OPTIONS]
```

**Options:**
- `--template TYPE` - Agent template (basic, api, workflow, ai). Default: basic
- `--port PORT` - Server port. Default: 3000

**Templates:**

1. **basic** - Simple agent with single verb
2. **api** - API agent with HTTP tools
3. **workflow** - Workflow agent with scheduler
4. **ai** - AI-powered agent with LLM

**Examples:**
```bash
# Basic agent
assertlang init my-agent

# API service on port 8080
assertlang init api-service --template api --port 8080

# AI chatbot
assertlang init chatbot --template ai
```

### `assertlang config`

Manage global and per-project configuration.

**Usage:**
```bash
assertlang config <subcommand> [OPTIONS]
```

**Subcommands:**

#### `config set`

Set configuration value.

```bash
assertlang config set <key> <value> [--project]
```

**Examples:**
```bash
# Set default language globally
assertlang config set defaults.language rust

# Set default language for current project only
assertlang config set defaults.language nodejs --project

# Set auto-confirm for generate command
assertlang config set generate.auto_confirm true

# Set default port
assertlang config set init.port 8080
```

#### `config get`

Get configuration value.

```bash
assertlang config get <key>
```

**Examples:**
```bash
# Get default language
assertlang config get defaults.language

# Get init port
assertlang config get init.port
```

#### `config unset`

Remove configuration value.

```bash
assertlang config unset <key> [--project]
```

**Examples:**
```bash
# Remove global language default
assertlang config unset defaults.language

# Remove project-level setting
assertlang config unset defaults.language --project
```

#### `config list`

List all configuration values.

```bash
assertlang config list
```

**Output:**
```json
{
  "defaults": {
    "language": "rust",
    "template": "basic",
    "output_dir": null
  },
  "generate": {
    "auto_confirm": false
  },
  "init": {
    "port": 3000
  }
}
```

#### `config path`

Show path to config file.

```bash
assertlang config path [--project]
```

**Examples:**
```bash
# Show global config file path
assertlang config path

# Show project config file path
assertlang config path --project
```

#### `config edit`

Open config file in editor.

```bash
assertlang config edit [--project]
```

**Configuration Files:**

- **Global**: `~/.config/assertlang/config.toml` (XDG-compliant)
- **Project**: `.assertlang/config.toml` (in current directory)

**Precedence:**

CLI arguments > Project config > Global config > Defaults

**Available Keys:**

| Key | Type | Default | Description |
|-----|------|---------|-------------|
| `defaults.language` | string | `python` | Default target language |
| `defaults.template` | string | `basic` | Default init template |
| `defaults.output_dir` | string | `null` | Default output directory |
| `generate.auto_confirm` | boolean | `false` | Skip confirmations by default |
| `init.port` | integer | `3000` | Default server port |

**Format:**

Supports both TOML and JSON formats. TOML is preferred:

```toml
[defaults]
language = "rust"
template = "api"

[generate]
auto_confirm = false

[init]
port = 8080
```

### `assertlang help`

Show help for commands.

**Usage:**
```bash
assertlang help [COMMAND]
```

**Examples:**
```bash
# General help
assertlang help

# Command-specific help
assertlang help generate
assertlang help validate
assertlang help config
```

### `assertlang version`

Show version information.

**Usage:**
```bash
assertlang --version
```

## Environment Variables

| Variable | Description | Default |
|----------|-------------|---------|
| `ASSERTLANG_HOME` | AssertLang installation directory | Auto-detected |
| `XDG_CONFIG_HOME` | Base directory for config files | `~/.config` |
| `NO_COLOR` | Disable colored output | Not set |
| `ANTHROPIC_API_KEY` | API key for AI-powered agents | None |
| `ALLOWED_ORIGINS` | CORS allowed origins | `*` |
| `DEBUG` | Enable debug mode | `false` |
| `RATE_LIMIT` | Requests per minute per IP | `100` |

## Common Workflows

### Development Workflow

```bash
# 1. Set up preferences (optional)
assertlang config set defaults.language python

# 2. Create agent
assertlang init my-service --template api

# 3. Edit my-service.al (customize verbs, tools)

# 4. Validate
assertlang validate my-service.al --verbose

# 5. Preview generation (dry-run)
assertlang generate my-service.al --dry-run

# 6. Generate development server
assertlang generate my-service.al

# 7. Run and test
cd generated/my-service
pip install -r requirements.txt
python my-service_server.py
```

### Production Workflow

```bash
# 1. Validate agent
assertlang validate my-service.al

# 2. Generate production server (skip confirmation for CI)
assertlang generate my-service.al --lang go --output ./production --yes --quiet

# 3. Build
python3 scripts/build_server.py my-service.al go

# 4. Deploy binary
./examples/demo/go/my-service
```

### CI/CD Workflow

```bash
# Use --yes and --quiet for automated pipelines
assertlang generate agent.al --yes --quiet

# Or set config for entire project
assertlang config set generate.auto_confirm true --project

# Disable colored output
export NO_COLOR=1
assertlang generate agent.al --yes
```

### Multi-Language Workflow

```bash
# Generate servers in all languages
assertlang generate agent.al --lang python --output ./servers/python
assertlang generate agent.al --lang typescript --output ./servers/nodejs
assertlang generate agent.al --lang go --output ./servers/go
assertlang generate agent.al --lang csharp --output ./servers/csharp
assertlang generate agent.al --lang rust --output ./servers/rust
```

## Troubleshooting

### Command Not Found

If `assertlang` command is not found after installation:

```bash
# Ensure pip bin directory is in PATH
export PATH="$HOME/.local/bin:$PATH"

# Or use python -m
python3 -m assertlang.cli --help
```

### Import Errors

If you get import errors:

```bash
# Reinstall in development mode
pip install -e .

# Or add to PYTHONPATH
export PYTHONPATH="/path/to/assertlang:$PYTHONPATH"
```

### Build Failures (Compiled Languages)

For Go/C#/Rust build issues:

```bash
# Verify runtime installed
go version
dotnet --version
cargo --version

# Use build script directly
python3 scripts/build_server.py agent.al go
```

## Shell Completion

### Bash

```bash
# Add to ~/.bashrc
eval "$(_ASSERTLANG_COMPLETE=bash_source assertlang)"
```

### Zsh

```bash
# Add to ~/.zshrc
eval "$(_ASSERTLANG_COMPLETE=zsh_source assertlang)"
```

### Fish

```bash
# Add to ~/.config/fish/completions/assertlang.fish
_ASSERTLANG_COMPLETE=fish_source assertlang | source
```

## Next Steps

- [Writing .al Agents](./assertlang-dsl-spec.md)
- [Production Deployment](./production-hardening.md)
- [Tool Development](./tool-development.md)
- [API Reference](./api-reference.md)
