import os
import shlex
from pathlib import Path

from harbor.agents.installed.base import BaseInstalledAgent, ExecInput
from harbor.models.agent.context import AgentContext
from harbor.models.agent.name import AgentName


class CursorCli(BaseInstalledAgent):
    """
    The Cursor CLI agent uses the Cursor AI command-line tool to solve tasks.
    """

    @staticmethod
    def name() -> str:
        return AgentName.CURSOR_CLI.value

    @property
    def _install_agent_template_path(self) -> Path:
        return Path(__file__).parent / "install-cursor-cli.sh.j2"

    def populate_context_post_run(self, context: AgentContext) -> None:
        pass

    def create_run_agent_commands(self, instruction: str) -> list[ExecInput]:
        escaped_instruction = shlex.quote(instruction)

        if not self.model_name or "/" not in self.model_name:
            raise ValueError("Model name must be in the format provider/model_name")

        model = self.model_name.split("/")[-1]

        # Handle API key
        env = {}
        if "CURSOR_API_KEY" in os.environ:
            env["CURSOR_API_KEY"] = os.environ["CURSOR_API_KEY"]
        else:
            raise ValueError(
                "CURSOR_API_KEY environment variable is required. "
                "Please set your Cursor API key."
            )

        return [
            ExecInput(
                command=(
                    f"cursor-agent -p {escaped_instruction} --model {model} "
                    f"2>&1 | tee /logs/agent/cursor-cli.txt"
                ),
                env=env,
            )
        ]
