"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * IAM Groups have at least one IAM User
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnGroup) {
        const groupLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const groupName = monocdk_1.Stack.of(node).resolve(node.groupName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_iam_1.CfnUser) {
                if (isMatchingUser(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
            else if (child instanceof aws_iam_1.CfnUserToGroupAddition) {
                if (isMatchingGroupAddition(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the IAM User belongs to the IAM Group
 * @param node the CfnUser to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUser is in the given group
 */
function isMatchingUser(node, groupLogicalId, groupName) {
    const groups = monocdk_1.Stack.of(node).resolve(node.groups);
    if (Array.isArray(groups)) {
        for (const group of groups) {
            const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(group));
            if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
                (groupName != undefined &&
                    new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Helper function to check whether the User to Group Addition mentions the specified Group
 * @param node the CfnUserToGroupAddition to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUserToGroupAddition references the given group
 */
function isMatchingGroupAddition(node, groupLogicalId, groupName) {
    const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.groupName));
    if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
        (groupName != undefined &&
            new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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