"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("monocdk/aws-s3");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets prohibit public read access through their Block Public Access configurations and bucket ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        const publicAccessBlockConfiguration = monocdk_1.Stack.of(node).resolve(node.publicAccessBlockConfiguration);
        if (publicAccessBlockConfiguration === undefined ||
            nag_rules_1.NagRules.resolveIfPrimitive(node, publicAccessBlockConfiguration.blockPublicPolicy) !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const accessControl = nag_rules_1.NagRules.resolveIfPrimitive(node, node.accessControl);
        const blockPublicAcls = nag_rules_1.NagRules.resolveIfPrimitive(node, publicAccessBlockConfiguration.blockPublicAcls);
        if ((accessControl === 'PublicRead' ||
            accessControl === 'PublicReadWrite') &&
            blockPublicAcls !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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