"""Functions to translate entire Ladybug core objects to Rhino geometries.

The methods here are intended to help translate groups of geometry that are commonly
generated by several objects in Ladybug core (ie. legends, compasses, etc.)
"""

from .fromgeometry import from_mesh3d
from .text import text_objects


def legend_objects(legend):
    """Translate a Ladybug Legend object into Grasshopper geometry.

    Args:
        legend: A Ladybug Legend object to be converted to Rhino geometry.

    Returns:
        A list of Rhino geometries in the following order.

        -   legend_mesh -- A colored mesh for the legend.

        -   legend_title -- A bake-able text object for the legend title.

        -   legend_text -- Bake-able text objects for the rest of the legend text.
    """
    _height = legend.legend_parameters.text_height
    _font = legend.legend_parameters.font
    legend_mesh = from_mesh3d(legend.segment_mesh)
    legend_title = text_objects(legend.title, legend.title_location, _height, _font)
    if legend.legend_parameters.continuous_legend is False:
        legend_text = [text_objects(txt, loc, _height, _font, 0, 5) for txt, loc in
                       zip(legend.segment_text, legend.segment_text_location)]
    elif legend.legend_parameters.vertical is True:
        legend_text = [text_objects(txt, loc, _height, _font, 0, 3) for txt, loc in
                       zip(legend.segment_text, legend.segment_text_location)]
    else:
        legend_text = [text_objects(txt, loc, _height, _font, 1, 5) for txt, loc in
                       zip(legend.segment_text, legend.segment_text_location)]
    return [legend_mesh] + [legend_title] + legend_text
