# Copyright © 2025 Cognizant Technology Solutions Corp, www.cognizant.com.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# END COPYRIGHT
import logging
import os
import subprocess
import tempfile
from typing import Annotated, Optional

from fastapi import APIRouter, File, Form, HTTPException, UploadFile
from fastapi.responses import JSONResponse
from pydantic import StringConstraints
from werkzeug.utils import secure_filename

# Adjust these to your repo/paths
working_directory = os.getcwd()
REPO_DIR = os.path.join(working_directory, "..", "ml-fastvlm")
PREDICT = os.path.join(REPO_DIR, "predict.py")
PREDICT_VIDEO = os.path.join(REPO_DIR, "predict_video.py")
MODEL_PATH = os.path.join(REPO_DIR, "checkpoints")
DEFAULT_MODEL_NAME = "llava-fastvithd_0.5b_stage3"
PYTHON_CMD = os.path.join(REPO_DIR, "venv/bin/python")
ACCEPTABLE_MODEL_NAMES = [
    "llava-fastvithd_1.5b_stage2",
    "llava-fastvithd_7b_stage3",
    "llava-fastvithd_0.5b_stage2",
    "llava-fastvithd_1.5b_stage3",
    "llava-fastvithd_0.5b_stage3",
    "llava-fastvithd_7b_stage2",
]

router = APIRouter(prefix="/api/v1")

QuestionField = Annotated[str, StringConstraints(min_length=3, max_length=256, strip_whitespace=True)]


@router.post("/vqa")
async def vqa(
    question: QuestionField = Form(...),
    image: UploadFile = File(...),
    model_name: Optional[str] = Form(None),
    timeout_sec: int = Form(120),
):
    """
    Answer a query about the image using the model

    Args:
        question: Query about the image. Response generated by the model
        image: Image we have a query about
        model_path: Path to the model to be used to answer the query
        timeout_sec: timeout in seconds


    Returns:
        JSON response containing the answer in text

    To test the endpoint with curl

    curl -X POST http://127.0.0.1:8005/api/v1/vqa \
        -F 'question=How many people are in the photo?' \
        -F 'image=@/Users/joe/vacation.jpg' \
        -F 'model_name=llava-fastvithd_7b_stage3' \
        -F 'timeout_sec=240'
    """
    if model_name is None:
        model_name = DEFAULT_MODEL_NAME
    if model_name not in ACCEPTABLE_MODEL_NAMES:
        raise HTTPException(500, f"Invalid model name {model_name}!")
    model_name = secure_filename(model_name)
    model = os.path.join(MODEL_PATH, model_name)
    print(f"model: {model}")

    if not os.path.exists(PREDICT):
        raise HTTPException(500, f"predict.py not found at {PREDICT}")
    if not os.path.exists(model):
        raise HTTPException(500, f"model_path not found: {model}")

    # Save uploaded image to a temp file
    try:
        with tempfile.NamedTemporaryFile(delete=False, suffix=os.path.splitext(image.filename or "")[-1]) as tmp:
            img_path = tmp.name
            content = await image.read()
            tmp.write(content)
    except Exception as e:
        raise HTTPException(400, f"Failed to save uploaded image: {e}") from e

    # Call predict.py
    cmd = [
        PYTHON_CMD,
        PREDICT,
        "--model-path",
        model,
        "--image-file",
        img_path,
        "--prompt",
        question,
    ]

    try:
        proc = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=timeout_sec,
            check=False,  # don’t raise automatically; we’ll return stderr if needed
        )
    except subprocess.TimeoutExpired as e:
        os.unlink(img_path)
        raise HTTPException(504, f"Prediction timed out after {timeout_sec}s") from e

    # Clean up the temp file
    try:
        os.unlink(img_path)
    except Exception:
        pass

    # Parse/return
    payload = {
        "exit_code": proc.returncode,
        "stdout": proc.stdout.strip(),
        "stderr": proc.stderr.strip(),
    }
    if proc.returncode != 0:
        # Bubble up a 500 with stderr for easier debugging
        raise HTTPException(500, detail=payload)

    # If predict.py prints extra text, you can post-process `proc.stdout` here.
    return JSONResponse(
        {
            "answer": proc.stdout.strip(),
            "model_path": model,
        }
    )


@router.post("/vqa_video")
async def vqa_video(
    question: QuestionField = Form(...),
    video: UploadFile = File(...),
    model_name: Optional[str] = Form(None),
    expected_num_of_frames: int = Form(8),
    max_new_tokens: int = Form(128),
    temperature: float = Form(0.2),
    timeout_sec: int = Form(120),
):
    """
    Answer a query about the video using the model

    Args:
        question: Query about the video. Response generated by the model
        video: Video we have a query about
        model_path: Path to the model to be used to answer the query
        timeout_sec: timeout in seconds


    Returns:
        JSON response containing the answer in text

    To test the endpoint with curl

    curl -X POST http://127.0.0.1:8005/api/v1/vqa_video \
        -F 'question=What is this video about?' \
        -F 'image=@/Users/joe/vacation.mp4' \
        -F 'model_name=llava-fastvithd_7b_stage3' \
        -F 'expected_num_of_frames=8' \
        -F 'max_new_tokens=128' \
        -F 'temperature=0.2' \
        -F 'timeout_sec=240'
    """
    if model_name is None:
        model_name = DEFAULT_MODEL_NAME
    if model_name not in ACCEPTABLE_MODEL_NAMES:
        raise HTTPException(500, f"Invalid model name {model_name}!")
    model_name = secure_filename(model_name)
    model = os.path.join(MODEL_PATH, model_name)
    print(f"model: {model}")

    if not os.path.exists(PREDICT_VIDEO):
        raise HTTPException(500, f"predict.py not found at {PREDICT_VIDEO}")
    if not os.path.exists(model):
        raise HTTPException(500, f"model_path not found: {model}")

    # Save uploaded video to a temp file
    try:
        with tempfile.NamedTemporaryFile(delete=False, suffix=os.path.splitext(video.filename or "")[-1]) as tmp:
            video_path = tmp.name
            content = await video.read()
            tmp.write(content)
    except Exception as e:
        raise HTTPException(400, f"Failed to save uploaded video: {e}")

    # Call predict.py
    cmd = [
        PYTHON_CMD,
        PREDICT_VIDEO,
        "--model-path",
        model,
        "--video-path",
        video_path,
        "--prompt",
        question,
        "--expected_num_of_frames",
        str(expected_num_of_frames),
        "--max_new_tokens",
        str(max_new_tokens),
        "--temperature",
        str(temperature),
    ]

    try:
        proc = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=timeout_sec,
            check=False,  # don’t raise automatically; we’ll return stderr if needed
        )
    except subprocess.TimeoutExpired:
        os.unlink(video_path)
        raise HTTPException(504, f"Prediction timed out after {timeout_sec}s")

    # Clean up the temp file
    try:
        os.unlink(video_path)
    except Exception:
        pass

    # Parse/return
    payload = {
        "exit_code": proc.returncode,
        "stdout": proc.stdout.strip(),
        "stderr": proc.stderr.strip(),
    }
    if proc.returncode != 0:
        # Bubble up a 500 with stderr for easier debugging
        raise HTTPException(500, detail=payload)

    # If predict.py prints extra text, you can post-process `proc.stdout` here.
    return JSONResponse(
        {
            "answer": proc.stdout.strip(),
            "model_path": model,
        }
    )
