"""
Classifies: CHEBI:25029 leukotriene
"""
"""
Classifies: leukotriene
"""
from rdkit import Chem
from rdkit.Chem import AllChem

def is_leukotriene(smiles: str):
    """
    Determines if a molecule is a leukotriene based on its SMILES string.
    A leukotriene is an icosanoid derived from C20 polyunsaturated fatty acids,
    with at least four double bonds (three of which are conjugated),
    possibly containing an epoxide ring, generated by leukocytes from arachidonic acid.
    
    Args:
        smiles (str): SMILES string of the molecule
    
    Returns:
        bool: True if molecule is a leukotriene, False otherwise
        str: Reason for classification
    """
    # Parse SMILES
    mol = Chem.MolFromSmiles(smiles)
    if mol is None:
        return False, "Invalid SMILES string"

    # Remove stereochemistry to simplify matching
    mol2 = Chem.Mol(mol)
    Chem.RemoveStereochemistry(mol2)

    # Count the number of carbon atoms
    c_count = sum(1 for atom in mol2.GetAtoms() if atom.GetAtomicNum() == 6)
    if c_count < 20:
        return False, f"Molecule has {c_count} carbon atoms, expected at least 20 for leukotriene"

    # Identify conjugated triene system in a linear chain (exclude rings)
    conjugated_triene = Chem.MolFromSmarts("[#6]-[CH]=[CH]-[CH]=[CH]-[CH]=[CH]-[#6]")
    matches_triene = mol2.HasSubstructMatch(conjugated_triene)
    if not matches_triene:
        return False, "No conjugated triene system found"

    # Check for at least four double bonds (total)
    double_bonds = [bond for bond in mol2.GetBonds() if bond.GetBondType() == Chem.rdchem.BondType.DOUBLE]
    if len(double_bonds) < 4:
        return False, f"Molecule has {len(double_bonds)} double bonds, expected at least 4 for leukotriene"

    # Check for terminal carboxylic acid group (-COOH)
    carboxylic_acid = Chem.MolFromSmarts("C(=O)[O;H1]")
    if not mol2.HasSubstructMatch(carboxylic_acid):
        return False, "No terminal carboxylic acid group found"

    # Check for linear long chain fatty acid (exclude rings except epoxides)
    ring_info = mol2.GetRingInfo()
    non_epoxide_rings = 0
    for ring in ring_info.AtomRings():
        ring_atoms = [mol2.GetAtomWithIdx(idx) for idx in ring]
        # Check if ring is epoxide (3-membered ring with one oxygen)
        if len(ring) == 3:
            o_count = sum(1 for atom in ring_atoms if atom.GetAtomicNum() == 8)
            if o_count == 1:
                continue  # Epoxide ring is acceptable
        non_epoxide_rings += 1
    if non_epoxide_rings > 0:
        return False, "Molecule contains ring structures other than epoxides"

    # Check for leukotriene-specific functional groups
    # Optional hydroxyl groups (-OH) at C5, C12 positions
    hydroxyl_groups = Chem.MolFromSmarts("[#6][CH](O)[#6]")
    hydroxyl_matches = mol2.GetSubstructMatches(hydroxyl_groups)
    if len(hydroxyl_matches) == 0:
        return False, "No hydroxyl groups found, expected at least one"

    # Exclude molecules known to be other eicosanoids (e.g., resolvins, lipoxins)
    # For simplicity, exclude if molecule has more than one hydroxyl group beyond C5 and C12
    if len(hydroxyl_matches) > 2:
        return False, "Molecule has multiple hydroxyl groups, may not be a leukotriene"

    # If molecule meets all criteria, classify as leukotriene
    return True, "Molecule matches criteria for a leukotriene"

__metadata__ = {   'chemical_class': {   'id': 'LEUKOTRIENE',
                              'name': 'leukotriene',
                              'definition': 'Any icosanoid from that family of C20 polyunsaturated fatty acids and their derivatives generated by leukocytes from arachidonic acid, each member having four double bonds of which three are conjugated.',
                              'parents': ['ICOSANOID', 'FATTY_ACID_DERIVATIVE']},
        'config': {   'llm_model_name': 'lbl/claude-sonnet',
                      'f1_threshold': 0.8,
                      'max_attempts': 5,
                      'max_positive_instances': None,
                      'max_positive_to_test': None,
                      'max_negative_to_test': None,
                      'max_positive_in_prompt': 50,
                      'max_negative_in_prompt': 20,
                      'max_instances_in_prompt': 100,
                      'test_proportion': 0.1},
        'message': None,
        'attempt': 3,
        'success': True,
        'best': True,
        'error': '',
        'stdout': None,
        'num_true_positives': None,
        'num_false_positives': None,
        'num_true_negatives': None,
        'num_false_negatives': None,
        'num_negatives': None,
        'precision': None,
        'recall': None,
        'f1': None,
        'accuracy': None}


__metadata__ = {   'chemical_class': {   'id': 'CHEBI:25029',
                          'name': 'leukotriene',
                          'definition': 'Any icosanoid from that family of C20 '
                                        'polyunsaturated fatty acids and their '
                                        'derivatives generated by leukocytes '
                                        'from arachidonic acid, each member '
                                        'having four double bonds of which '
                                        'three are conjugated.',
                          'parents': ['CHEBI:23899'],
                          'xrefs': [   'LIPID_MAPS_class:LMFA0302',
                                       'PMID:10922142',
                                       'PMID:15794321',
                                       'PMID:3001504',
                                       'PMID:6129158',
                                       'PMID:6293196',
                                       'PMID:6311078',
                                       'PMID:8842590',
                                       'PMID:9526099',
                                       'PMID:9561094'],
                          'all_positive_examples': []},
    'config': None,
    'code_statistics': None,
    'message': '\n'
               'Attempt failed: F1 score of 0.40310077519379844 is too low.\n'
               'Outcomes:\n'
               '------\n'
               '\n'
               'True positives: SMILES: '
               'CCCCC\\C=C/C[C@@H](O)\\C=C\\C=C\\C=C/[C@@H](O)CCCC(O)=O NAME: '
               'leukotriene B4 REASON: CORRECT Molecule matches criteria for a '
               'leukotriene\n'
               ' * SMILES: CCCCC[C@@H]1O[C@H]1\\C=C\\C=C\\C=C/C\\C=C/CCCC(O)=O '
               'NAME: eoxin A4 REASON: CORRECT Molecule matches criteria for a '
               'leukotriene\n'
               ' * SMILES: OC(CCCC(O)=O)/C=C/C=C/C=C/C(O)C\\C=C\\CCCCC NAME: '
               '5,12-dihydroxy-6,8,10,14-eicosatetraenoic acid REASON: CORRECT '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: '
               'CCCCC\\C=C/C[C@H](O)\\C=C\\C=C\\C=C\\[C@@H](O)CCCC(O)=O NAME: '
               'Delta(6)-trans-12-epi-leukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'CCCCC\\C=C/C[C@H](O)\\C=C\\C=C\\C=C/[C@@H](O)CCCC(O)=O NAME: '
               '12-epi-leukotriene B4 REASON: CORRECT Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: CCCCC\\C=C/C\\C=C/C=C/C=C/[C@@H]1O[C@H]1CCCC(O)=O '
               'NAME: leukotriene A4 REASON: CORRECT Molecule matches criteria '
               'for a leukotriene\n'
               ' * SMILES: '
               'CC[C@@H](O)\\C=C\\C=C/C\\C=C\\C=C\\C=C\\C1OC1CCCC(O)=O NAME: '
               '5,6-Epoxy-18R-HEPE REASON: CORRECT Molecule matches criteria '
               'for a leukotriene\n'
               ' * SMILES: '
               'C([C@H](/C=C/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCCC(O)O '
               'NAME: 20,20-dihydroxyleukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'O[C@@H](CCCC(O)=O)\\C=C/C=C/C=C/[C@H](O)C/C=C\\C/C=C\\CC NAME: '
               'leukotriene B5 REASON: CORRECT Molecule matches criteria for a '
               'leukotriene\n'
               ' * SMILES: '
               'O[C@@H](CCCC(O)=O)\\C=C/C=C/C=C/[C@H](O)C\\C=C/CCCCC=O NAME: '
               '20-oxoleukotriene B4 REASON: CORRECT Molecule matches criteria '
               'for a leukotriene\n'
               ' * SMILES: C(C(/C=C/C=C/C=C\\[C@H](CCCC(O)=O)O)=O)/C=C\\CCCCCO '
               'NAME: 12-oxo-20-hydroxyleukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: C(C(/C=C/C=C/C=C/[C@H](CCCC(O)=O)O)=O)/C=C\\CCCCC '
               'NAME: 12-oxo-6-trans-leukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'OC(O)(O)CCCC/C=C\\CC(=O)\\C=C\\C=C\\C=C/[C@H](O)CCCC(O)=O '
               'NAME: 12-Oxo-20-trihydroxy-leukotriene B4 REASON: CORRECT '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: O[C@H](C/C=C\\CCCCC)/C=C/C=C/C=C/C(=O)CCCC(O)=O '
               'NAME: 5-Oxo-6-trans-leukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'O[C@H](CCCC(O)=O)/C=C\\C=C\\C=C\\C(=O)C/C=C\\CCCCC(O)O NAME: '
               '12-oxo-20-dihydroxy-leukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C([C@H](/C=C/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCC(CC)O NAME: '
               '18-hydroxyleukotriene B4 REASON: CORRECT Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: '
               'CCCCC\\C=C/C[C@@H](O)\\C=C\\C=C/C=C/[C@@H](O)CCCC(O)=O NAME: '
               'Delta(6)-trans,Delta(8)-cis-leukotriene B4 REASON: CORRECT '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: O[C@@H](CCCC(O)=O)\\C=C/C=C/C=C/[C@H](O)CC#CCCCCC '
               'NAME: 14,15-dehydro-LTB4 REASON: CORRECT Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: '
               'C([C@H](/C=C/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCCC(O)(O)O '
               'NAME: 20,20,20-trihydroxyleukotriene B4 REASON: CORRECT '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: CCCCC\\C=C/C\\C=C/C=C/C=C/C[C@H]1O[C@@H]1CCC(O)=O '
               'NAME: 4,5-leukotriene A4 REASON: CORRECT Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: '
               'CCCCC\\C=C/C[C@@H](O)\\C=C\\C=C\\C=C\\[C@@H](O)CCCC(O)=O NAME: '
               '6-trans-leukotriene B4 REASON: CORRECT Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: '
               'O[C@@H](CCCC(O)=O)\\C=C/C=C/C=C/[C@H](O)C\\C=C/CCCCC(O)=O '
               'NAME: 20-hydroxy-20-oxoleukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: CCCCC\\C=C/CC(=O)\\C=C\\C=C\\C=C/[C@@H](O)CCCC(O)=O '
               'NAME: 12-dehydro-leukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C([C@H](/C=C/C=C/C=C/[C@H](CCCC(O)=O)O)O)/C=C\\CCCCCO NAME: '
               '20-hydroxy-6-trans-leukotriene B4 REASON: CORRECT Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C([C@H](/C=C/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCC(C)O NAME: '
               '19-hydroxyleukotriene B4 REASON: CORRECT Molecule matches '
               'criteria for a leukotriene\n'
               'False positives: SMILES: '
               'C(\\[C@H](CCCC(O)=O)O)=C\\C=C\\C=C\\[C@@H](C/C=C\\C=C\\C(CC)=O)O '
               'NAME: 18-oxoresolvin E1 REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C(\\[C@H](CCCC(O)=O)O)=C\\C=C\\C=C\\[C@@H](C\\C=C/C=C/[C@@H](CC)O)O '
               'NAME: resolvin E1 REASON: WRONGLY CLASSIFIED Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: '
               'CCCCC[C@H](O)\\C=C\\C=C/C=C/C=C/[C@@H]1O[C@H]1CCCC(O)=O NAME: '
               '5,6-Ep-15S-HETE REASON: WRONGLY CLASSIFIED Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: '
               '[C@H]1(CCCC(O)=O)[C@H](/C=C/C=C/C=C\\C\\C=C/C=C/[C@@H](CC)O)O1 '
               'NAME: '
               '5(S),6(S)-epoxy-18(R)-hydroxy-(7E,9E,11Z,14Z,16E)-icosapentaenoic '
               'acid REASON: WRONGLY CLASSIFIED Molecule matches criteria for '
               'a leukotriene\n'
               ' * SMILES: '
               'C(\\C=C/C=C/[C@@H](C/C=C\\CC)O)=C/C=C/C(C(CCCC(=O)O)O)O NAME: '
               '(7E,9E,11Z,13E,15R,17Z)-5,6,15-trihydroxyicosapentaenoic acid '
               'REASON: WRONGLY CLASSIFIED Molecule matches criteria for a '
               'leukotriene\n'
               ' * SMILES: CCCCCCC\\C=C\\C=C\\C=C/C\\C=C/CCCC(O)=O NAME: '
               '(5Z,8Z,10E,12E)-icosatetraenoic acid REASON: WRONGLY '
               'CLASSIFIED Molecule matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C(\\[C@H](CCCC(O)=O)O)=C\\C=C\\C=C\\[C@@H](C\\C=C/C=C/[C@H](CC)O)O '
               'NAME: (18S)-resolvin E1 REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'O[C@H](CCCC(O)=O)/C=C/C=C\\C=C\\[C@@H](O)C/C=C\\C=C\\[C@H](O)CC '
               'NAME: 5,12,18R-TriHEPE REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               '[C@@H](CC)(C(/C=C/C=C/C=C\\C/C=C\\C/C=C\\CCCC(=O)O)O)O NAME: '
               '(18R)-resolvin E3 REASON: WRONGLY CLASSIFIED Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: OC(C/C=C\\C/C=C\\CCCC(O)=O)/C=C/C=C\\C=C\\C(O)CC '
               'NAME: 11,18-Di-HEPE REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C(/C=C/C=C\\C=C\\[C@@H](OO)CCCCC)=C\\[C@H]([C@H](CCCC(O)=O)O)O '
               'NAME: '
               '(5S,6R)-dihydroxy-(15S)-hydroperoxy-(7E,9E,11Z,13E)-icosatetraenoic '
               'acid REASON: WRONGLY CLASSIFIED Molecule matches criteria for '
               'a leukotriene\n'
               ' * SMILES: '
               'C(/C=C\\C=C\\[C@H](CCCCCO)O)=C\\C=C\\[C@H]([C@H](CCCC(=O)O)O)O '
               'NAME: 20-hydroxylipoxin A4 REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'CCCCC[C@H](O)CC\\C=C/C=C/C=C/[C@@H](O)[C@@H](O)CCCC(O)=O NAME: '
               '13,14-dihydrolipoxin A4 REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C(=C\\C=C\\[C@H](CCCCC)OO)\\C=C\\[C@H](C/C=C\\CCCC(O)=O)OO '
               'NAME: 8(S),15(S)-DiHPETE REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'O[C@H](CCCC(O)=O)/C=C\\C=C\\C=C\\C(=O)C/C=C\\CCCCC(O)=O NAME: '
               '12-Oxo-20-carboxy-leukotriene B4 REASON: WRONGLY CLASSIFIED '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: '
               'OC(CCCC(O)=O)\\C=C\\C=C\\C=C\\C(O)C\\C=C/C=C\\C(O)CC NAME: '
               '6-trans-5,12,18-triHEPE REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: [H]C(CCC=C([H])C=C([H])C=C([H])CCCCC(O)=O)=CCCCCC '
               'NAME: icosa-6,8,10,14-tetraenoic acid REASON: WRONGLY '
               'CLASSIFIED Molecule matches criteria for a leukotriene\n'
               ' * SMILES: '
               'CC(\\C=C\\C=C(/C)C(O)=O)=C/C=C/C=C(C)/C=C/C=C(\\C)C(O)=O NAME: '
               'crocetin REASON: WRONGLY CLASSIFIED Molecule matches criteria '
               'for a leukotriene\n'
               ' * SMILES: '
               '[C@H](CC)(C(/C=C/C=C/C=C\\C/C=C\\C/C=C\\CCCC(=O)O)O)O NAME: '
               'resolvin E3 REASON: WRONGLY CLASSIFIED Molecule matches '
               'criteria for a leukotriene\n'
               ' * SMILES: O[C@H](CCCCCCCC=CC=CC=CC(O)=O)CCCCC NAME: '
               '15(S)-Hydroxyeicosatrienoic acid REASON: WRONGLY CLASSIFIED '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: OC(CCCC(O)=O)C(O)/C=C/C=C/C=C/C\\C=C/C=C\\C(O)CC '
               'NAME: 5,6,18-TriHEPE REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: '
               'O[C@@H](CCCC(O)=O)/C=C\\C=C\\C=C\\[C@H](O)C/C=C\\C=C\\[C@@H](O)C(O)C '
               'NAME: 19-Hydroxy-Resolvin E1 REASON: WRONGLY CLASSIFIED '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: O[C@H](CCCCCCCC)C=CC=CC=C[C@@H](O)CCCC(O)=O NAME: '
               '12-epi Leukotriene B3 REASON: WRONGLY CLASSIFIED Molecule '
               'matches criteria for a leukotriene\n'
               ' * SMILES: OC(CCCCCCCC)CCCC/C=C/C=C/C=C/C(O)=O NAME: '
               '12(r)-Hydroxyeicosatrienoic acid REASON: WRONGLY CLASSIFIED '
               'Molecule matches criteria for a leukotriene\n'
               ' * SMILES: '
               'C(O)(=O)CCC[C@H]1[C@@H](/C=C/C=C/C=C\\C/C=C\\C=C\\[C@H](CC)O)O1 '
               'NAME: 5(S)6-epoxy-18(S)-HEPE REASON: WRONGLY CLASSIFIED '
               'Molecule matches criteria for a leukotriene\n'
               'False negatives: SMILES: '
               'C(/C=C\\CCCCC)[C@H](C/C=C/C=C/[C@H]([C@@H](O)CCCC(=O)O)SC[C@H](N)C(=O)O)O '
               'NAME: (12R)-hydroxy-10,11-dihydroleukotriene E4 REASON: MISSED '
               'Molecule has 23 carbon atoms, expected exactly 20 for '
               'leukotriene\n'
               ' * SMILES: '
               'CCCCC\\C=C/C\\C=C/C=C/C=C/[C@@H](SC[C@H](N)C(=O)NCC(O)=O)[C@@H](O)CCCC(O)=O '
               'NAME: leukotriene D4 REASON: MISSED Molecule has 25 carbon '
               'atoms, expected exactly 20 for leukotriene\n'
               ' * SMILES: '
               'C([C@H](CC/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCCC(O)O NAME: '
               '10,11-dihydro-20,20-dihydroxyleukotriene B4 REASON: MISSED No '
               'conjugated triene (three conjugated double bonds) found\n'
               ' * SMILES: '
               'C([C@H](NC(CC[C@@H](C(=O)O)N)=O)C(NCC(=O)O)=O)S[C@H](\\C=C\\C=C\\C[C@@H](C/C=C\\CCCCC)O)[C@@H](O)CCCC(=O)O '
               'NAME: 11,12-dihydro-(12R)-hydroxyleukotriene C4 REASON: MISSED '
               'Molecule has 30 carbon atoms, expected exactly 20 for '
               'leukotriene\n'
               ' * SMILES: '
               'CCCCC[C@H](O)[C@H](SC[C@H](NC(=O)CC[C@H](N)C(O)=O)C(=O)NCC(O)=O)\\C=C\\C=C\\C=C/C\\C=C/CCCC(O)=O '
               'NAME: eoxin C4 REASON: MISSED Molecule has 30 carbon atoms, '
               'expected exactly 20 for leukotriene\n'
               ' * SMILES: '
               'S([C@@H]([C@@H](O)CCCC(O)=O)/C=C/C=C/C=C\\C/C=C\\CCCCC)C[C@H](N)C(OC[C@@H](OC(=O)CC/C=C\\C/C=C\\C/C=C\\C/C=C\\C/C=C\\CCCCC)COP(OC[C@H](N)C(O)=O)(O)=O)=O '
               'NAME: PS(LTE4/22:5(4Z,7Z,10Z,13Z,16Z)) REASON: MISSED Molecule '
               'has 51 carbon atoms, expected exactly 20 for leukotriene\n'
               ' * SMILES: '
               'C(/C=C\\CCCCC)\\C=C/C=C/C=C/[C@H]([C@@H](O)CCCC(=O)OC)SC[C@H](NC(=O)CC[C@H](N)C(O)=O)C(NCC(=O)O)=O '
               'NAME: leukotriene C4 methyl ester REASON: MISSED Molecule has '
               '31 carbon atoms, expected exactly 20 for leukotriene\n'
               ' * SMILES: '
               'C(/C=C\\CCCCC)/C=C\\C=C\\C=C\\[C@H]([C@@H](O)CCCC(=O)OC)SC[C@H](N)C(=O)O '
               'NAME: leukotriene E4 methyl ester REASON: MISSED Molecule has '
               '24 carbon atoms, expected exactly 20 for leukotriene\n'
               ' * SMILES: '
               'S([C@@H]([C@@H](O)CCCC(O)=O)/C=C/C=C/C=C\\C/C=C\\CCCCC)C[C@H](N)C(O[C@H](COC(=O)CCCCCCCCCCCCCCCCCCCCCCC)COP(OCCN)(O)=O)=O '
               'NAME: PE(24:0/LTE4) REASON: MISSED Molecule has 52 carbon '
               'atoms, expected exactly 20 for leukotriene\n'
               ' * SMILES: '
               'S([C@@H]([C@@H](O)CCCC(O)=O)/C=C/C=C/C=C\\C/C=C\\CCCCC)C[C@H](N)C(O[C@H](COC(=O)CCCCCCCCCCCCCCCCCCCCC(CC)C)COP(OC[C@@H](O)CO)(O)=O)=O '
               'NAME: PG(a-25:0/LTE4) REASON: MISSED Molecule has 54 carbon '
               'atoms, expected exactly 20 for leukotriene\n'
               '------\n'
               '\n'
               'In your reasoning step, analyze the previous program and the '
               'above outcomes, hypothesizing about what went wrong, and how '
               'to improve.\n',
    'sample_true_negatives': [   {   'smiles': 'C[C@@H]1CN(S(=O)(=O)C2=C(C=C(C=C2)C3=CC=C(C=C3)C(=O)N(C)C)O[C@@H]1CN(C)C(=O)NC4=CC=CC=C4F)[C@H](C)CO',
                                     'name': '4-[(4R,5S)-5-[[[(2-fluoroanilino)-oxomethyl]-methylamino]methyl]-2-[(2R)-1-hydroxypropan-2-yl]-4-methyl-1,1-dioxo-4,5-dihydro-3H-6,1$l^{6},2-benzoxathiazocin-8-yl]-N,N-dimethylbenzamide',
                                     'reason': 'No conjugated triene system '
                                               'found'},
                                 {   'smiles': 'C[C@H]1[C@H](CN(C(=O)C2=C(C=CC(=C2)C#N)OC[C@H]3[C@H](CC[C@@H](O3)CCN(C1=O)C)OC)C)OC',
                                     'name': 'LSM-10564',
                                     'reason': 'No conjugated triene system '
                                               'found'},
                                 {   'smiles': 'CCC(C(=O)NC1=NN=C(S1)SCC)(C(F)(F)F)C(F)(F)F',
                                     'name': 'N-[5-(ethylthio)-1,3,4-thiadiazol-2-yl]-2,2-bis(trifluoromethyl)butanamide',
                                     'reason': 'Molecule has 10 carbon atoms, '
                                               'expected at least 20 for '
                                               'leukotriene'},
                                 {   'smiles': 'O=C(N[C@@H](CCC(=O)N)C(O)=O)[C@@H](NC(=O)[C@@H](N)CC1=CC=C(O)C=C1)CC(=O)N',
                                     'name': 'Tyr-Asn-Gln',
                                     'reason': 'Molecule has 18 carbon atoms, '
                                               'expected at least 20 for '
                                               'leukotriene'},
                                 {   'smiles': 'OC[C@H]1OC(O)[C@@H](O)[C@@H]1O',
                                     'name': 'D-arabinofuranose',
                                     'reason': 'Molecule has 5 carbon atoms, '
                                               'expected at least 20 for '
                                               'leukotriene'},
                                 {   'smiles': 'O[C@@H]([C@H]([C@@]1([C@@]2([C@@](CC1)(/C(/CCC2)=C/C=C\\3/C[C@@H](O)CCC3=C)[H])C)[H])C)C',
                                     'name': '(5Z,7E)-(3S,22R)- '
                                             '24-nor-9,10-seco-5,7,10(19)-cholatriene-3,22-diol',
                                     'reason': 'No conjugated triene system '
                                               'found'},
                                 {   'smiles': 'O1C(C(CC1=O)C)CCCCCC',
                                     'name': '5-Hexyldihydro-4-methyl-2(3H)-furanone',
                                     'reason': 'Molecule has 11 carbon atoms, '
                                               'expected at least 20 for '
                                               'leukotriene'},
                                 {   'smiles': 'O=C1O[C@]23C(=O)N[C@H]([C@@H]2[C@H](C)C(=C[C@@H]3C=CC[C@@H]([C@H]([C@H](C=C1)C)O)C)C)CC4=CC=CC=C4',
                                     'name': '7-deoxy-cytochalasin Z7',
                                     'reason': 'No conjugated triene system '
                                               'found'},
                                 {   'smiles': '[C@]123[C@H](O1)[C@](C(C=C2[C@]4([C@](CC3)(C([C@@H](C(C4)=O)O)(C)C)[H])C)=O)(C=C)C',
                                     'name': 'yucalexin P8',
                                     'reason': 'No conjugated triene system '
                                               'found'},
                                 {   'smiles': '[H][C@]12CC[C@@]3([H])[C@]4([H])CC[C@]([H])([C@H](C)CCC(O)=O)[C@@]4(C)[C@@H](O)C[C@]3([H])[C@@]1(C)CC[C@@H](O)C2',
                                     'name': 'deoxycholic acid',
                                     'reason': 'No conjugated triene system '
                                               'found'}],
    'sample_false_negatives': [   {   'smiles': 'C(/C=C\\CCCCC)[C@H](C/C=C/C=C/[C@H]([C@@H](O)CCCC(=O)O)SC[C@H](N)C(=O)O)O',
                                      'name': '(12R)-hydroxy-10,11-dihydroleukotriene '
                                              'E4',
                                      'reason': 'No conjugated triene system '
                                                'found'},
                                  {   'smiles': 'C([C@H](CC/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCCC(O)O',
                                      'name': '10,11-dihydro-20,20-dihydroxyleukotriene '
                                              'B4',
                                      'reason': 'No conjugated triene system '
                                                'found'},
                                  {   'smiles': 'C([C@H](NC(CC[C@@H](C(=O)O)N)=O)C(NCC(=O)O)=O)S[C@H](\\C=C\\C=C\\C[C@@H](C/C=C\\CCCCC)O)[C@@H](O)CCCC(=O)O',
                                      'name': '11,12-dihydro-(12R)-hydroxyleukotriene '
                                              'C4',
                                      'reason': 'No conjugated triene system '
                                                'found'},
                                  {   'smiles': 'CC[C@@H](O)\\C=C\\C=C/C\\C=C\\C=C\\C=C\\C1OC1CCCC(O)=O',
                                      'name': '5,6-Epoxy-18R-HEPE',
                                      'reason': 'Molecule has multiple '
                                                'hydroxyl groups, may not be a '
                                                'leukotriene'},
                                  {   'smiles': 'S([C@@H]([C@@H](O)CCCC(O)=O)/C=C/C=C/C=C\\C/C=C\\CCCCC)C[C@H](N)C(O[C@H](COC(=O)CCCCCCCCCCCCCCCCCCCCC(CC)C)COP(OC[C@@H](O)CO)(O)=O)=O',
                                      'name': 'PG(a-25:0/LTE4)',
                                      'reason': 'Molecule has multiple '
                                                'hydroxyl groups, may not be a '
                                                'leukotriene'},
                                  {   'smiles': 'C([C@H](NC(CC[C@@H](C(=O)O)N)=O)C(NCC(=O)O)=O)S[C@@H](\\C=C\\C=C\\CC(C/C=C\\CCCCC)=O)[C@H](O)CCCC(=O)O',
                                      'name': '11,12-dihydro-12-oxoleukotriene '
                                              'C4',
                                      'reason': 'No conjugated triene system '
                                                'found'},
                                  {   'smiles': 'S([C@@H]([C@@H](O)CCCC(O)=O)/C=C/C=C/C=C\\C/C=C\\CCCCC)C[C@H](N)C(OC[C@@H](OC(=O)CCCCCCCCCCCCCCCCC(CC)C)COP(OC[C@@H](O)CO)(O)=O)=O',
                                      'name': 'PG(LTE4/a-21:0)',
                                      'reason': 'Molecule has multiple '
                                                'hydroxyl groups, may not be a '
                                                'leukotriene'},
                                  {   'smiles': 'C([C@H](CC/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCCC',
                                      'name': '10,11-dihydroleukotriene B4',
                                      'reason': 'No conjugated triene system '
                                                'found'},
                                  {   'smiles': 'C([C@H](/C=C/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCC(CC)O',
                                      'name': '18-hydroxyleukotriene B4',
                                      'reason': 'Molecule has multiple '
                                                'hydroxyl groups, may not be a '
                                                'leukotriene'},
                                  {   'smiles': 'C([C@H](CC/C=C/C=C\\[C@H](CCCC(O)=O)O)O)/C=C\\CCCCC(O)(O)O',
                                      'name': '10,11-dihydro-20,20,20-trihydroxyleukotriene '
                                              'B4',
                                      'reason': 'No conjugated triene system '
                                                'found'}],
    'attempt': 3,
    'reasoning': 'Upon analyzing the previous program and the outcomes, '
                 'several issues have become apparent:\n'
                 '\n'
                 '1. **Strict Carbon Count**: The code requires the molecule '
                 'to have exactly 20 carbon atoms. However, leukotriene '
                 'derivatives may have modifications such as additional '
                 'functional groups or conjugates (e.g., glutathione, '
                 'cysteine) that increase the carbon count beyond 20. This '
                 'strict requirement leads to false negatives, as seen with '
                 'molecules like leukotriene D4, which has 25 carbons.\n'
                 '\n'
                 '2. **Insufficient Specificity**: The code uses a general '
                 'pattern to identify conjugated trienes and allows any '
                 'molecule with this feature and a terminal carboxylic acid to '
                 'be classified as a leukotriene. This lack of specificity '
                 'results in false positives, including molecules like '
                 'resolvins and lipoxins, which also contain conjugated '
                 'trienes but are not leukotrienes.\n'
                 '\n'
                 '3. **Lack of Functional Group Checks**: Leukotrienes often '
                 'contain specific functional groups such as hydroxyl groups '
                 'at particular positions, epoxide rings, or '
                 'glutathione/cysteine conjugates. The previous code does not '
                 'check for these features in detail.\n'
                 '\n'
                 '**Improvements to be made:**\n'
                 '\n'
                 '- **Relax Carbon Count Constraint**: Instead of requiring '
                 'exactly 20 carbons, allow molecules with 20 or more carbons '
                 'to account for derivatives.\n'
                 '- **Specific Structural Features**: Implement more specific '
                 'SMARTS patterns to detect the characteristic conjugated '
                 'triene system in leukotrienes. Ensure that this system is '
                 'part of a linear (acyclic) chain to exclude rings.\n'
                 '- **Functional Group Verification**: Check for the presence '
                 'of key functional groups such as terminal carboxylic acid, '
                 'hydroxyl groups at specific positions (e.g., C5, C12), and '
                 'consider possible epoxide formations.\n'
                 '- **Exclude Non-Leukotrienes**: Add checks to exclude other '
                 'eicosanoids like resolvins and lipoxins by identifying '
                 'structural features unique to them.\n'
                 '\n'
                 'Based on these observations, here is the improved code:',
    'success': True,
    'best': True,
    'error': '',
    'stdout': None,
    'num_true_positives': 44,
    'num_false_positives': 70,
    'num_true_negatives': 142174,
    'num_false_negatives': 12,
    'num_negatives': None,
    'precision': 0.38596491228070173,
    'recall': 0.7857142857142857,
    'f1': 0.5176470588235293,
    'accuracy': 0.9994237526352776,
    'negative_predictive_value': 0.9999156035052678}