"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagReportLogger = exports.NagReportFormat = exports.AnnotationLogger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_rules_1 = require("./nag-rules");
/**
 * A NagLogger that outputs to the CDK Annotations system.
 */
class AnnotationLogger {
    constructor(props) {
        this.suppressionId = 'CdkNagSuppression';
        this.verbose = props?.verbose ?? false;
        this.logIgnores = props?.logIgnores ?? false;
    }
    onCompliance(_data) {
        return;
    }
    onNonCompliance(data) {
        const message = this.createMessage(data.ruleId, data.findingId, data.ruleInfo, data.ruleExplanation, this.verbose);
        if (data.ruleLevel == nag_rules_1.NagMessageLevel.ERROR) {
            aws_cdk_lib_1.Annotations.of(data.resource).addError(message);
        }
        else if (data.ruleLevel == nag_rules_1.NagMessageLevel.WARN) {
            aws_cdk_lib_1.Annotations.of(data.resource).addWarning(message);
        }
    }
    onSuppressed(data) {
        if (this.logIgnores) {
            const message = this.createMessage(this.suppressionId, data.findingId, `${data.ruleId} was triggered but suppressed.`, `Provided reason: "${data.suppressionReason}"`, this.verbose);
            aws_cdk_lib_1.Annotations.of(data.resource).addInfo(message);
        }
    }
    onError(data) {
        const information = `'${data.ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. You can suppress the "${nag_rules_1.VALIDATION_FAILURE_ID}" to get rid of this error. For more details enable verbose logging.'`;
        const message = this.createMessage(nag_rules_1.VALIDATION_FAILURE_ID, data.ruleId, information, data.errorMessage, this.verbose);
        aws_cdk_lib_1.Annotations.of(data.resource).addWarning(message);
    }
    onSuppressedError(data) {
        if (this.logIgnores === true) {
            const message = this.createMessage(this.suppressionId, data.ruleId, `${nag_rules_1.VALIDATION_FAILURE_ID} was triggered but suppressed.`, data.errorSuppressionReason, this.verbose);
            aws_cdk_lib_1.Annotations.of(data.resource).addInfo(message);
        }
    }
    onNotApplicable(_data) {
        return;
    }
    createMessage(ruleId, findingId, ruleInfo, ruleExplanation, verbose) {
        let message = findingId
            ? `${ruleId}[${findingId}]: ${ruleInfo}`
            : `${ruleId}: ${ruleInfo}`;
        return verbose ? `${message} ${ruleExplanation}\n` : `${message}\n`;
    }
}
exports.AnnotationLogger = AnnotationLogger;
_a = JSII_RTTI_SYMBOL_1;
AnnotationLogger[_a] = { fqn: "cdk-nag.AnnotationLogger", version: "2.36.36" };
/**
 * Possible output formats of the NagReport
 */
var NagReportFormat;
(function (NagReportFormat) {
    NagReportFormat["CSV"] = "csv";
    NagReportFormat["JSON"] = "json";
})(NagReportFormat || (exports.NagReportFormat = NagReportFormat = {}));
/**
 * A NagLogger that creates compliance reports
 */
class NagReportLogger {
    constructor(props) {
        this.reportStacks = new Map();
        if (props.formats.length === 0) {
            throw new Error('Must provide at least 1 NagReportFormat.');
        }
        this.formats = props.formats;
    }
    onCompliance(data) {
        this.initializeStackReport(data);
        this.writeToStackComplianceReport(data, nag_rules_1.NagRuleCompliance.COMPLIANT);
    }
    onNonCompliance(data) {
        this.initializeStackReport(data);
        this.writeToStackComplianceReport(data, nag_rules_1.NagRuleCompliance.NON_COMPLIANT);
    }
    onSuppressed(data) {
        this.initializeStackReport(data);
        this.writeToStackComplianceReport(data, nag_rules_1.NagRulePostValidationStates.SUPPRESSED);
    }
    onError(data) {
        this.initializeStackReport(data);
        this.writeToStackComplianceReport(data, nag_rules_1.NagRulePostValidationStates.UNKNOWN);
    }
    onSuppressedError(data) {
        this.initializeStackReport(data);
        this.writeToStackComplianceReport(data, nag_rules_1.NagRulePostValidationStates.SUPPRESSED);
    }
    onNotApplicable(data) {
        this.initializeStackReport(data);
    }
    getFormatStacks(format) {
        return this.reportStacks.get(format) ?? [];
    }
    /**
     * Initialize the report for the rule pack's compliance report for the resource's Stack if it doesn't exist
     * @param data
     */
    initializeStackReport(data) {
        for (const format of this.formats) {
            const stackName = data.resource.stack.nested
                ? aws_cdk_lib_1.Names.uniqueId(data.resource.stack)
                : data.resource.stack.stackName;
            const fileName = `${data.nagPackName}-${stackName}-NagReport.${format}`;
            const stacks = this.getFormatStacks(format);
            if (!stacks.includes(fileName)) {
                const filePath = (0, path_1.join)(aws_cdk_lib_1.App.of(data.resource)?.outdir ?? '', fileName);
                this.reportStacks.set(format, [...stacks, fileName]);
                let body = '';
                if (format === NagReportFormat.CSV) {
                    body =
                        'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n';
                }
                else if (format === NagReportFormat.JSON) {
                    body = JSON.stringify({ lines: [] });
                }
                else {
                    throw new Error(`Unrecognized output format ${format} for the NagReportLogger`);
                }
                (0, fs_1.writeFileSync)(filePath, body);
            }
        }
    }
    writeToStackComplianceReport(data, compliance) {
        for (const format of this.formats) {
            const stackName = data.resource.stack.nested
                ? aws_cdk_lib_1.Names.uniqueId(data.resource.stack)
                : data.resource.stack.stackName;
            const fileName = `${data.nagPackName}-${stackName}-NagReport.${format}`;
            const filePath = (0, path_1.join)(aws_cdk_lib_1.App.of(data.resource)?.outdir ?? '', fileName);
            if (format === NagReportFormat.CSV) {
                //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
                const line = Array();
                line.push(data.ruleId);
                line.push(data.resource.node.path);
                if (compliance === nag_rules_1.NagRulePostValidationStates.SUPPRESSED) {
                    line.push(nag_rules_1.NagRulePostValidationStates.SUPPRESSED);
                    if (data.suppressionReason !== undefined) {
                        line.push(data.suppressionReason);
                    }
                    else {
                        line.push(data.errorSuppressionReason);
                    }
                }
                else {
                    line.push(compliance);
                    line.push('N/A');
                }
                line.push(data.ruleLevel);
                line.push(data.ruleInfo);
                (0, fs_1.appendFileSync)(filePath, line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n');
            }
            else if (format === NagReportFormat.JSON) {
                const report = JSON.parse((0, fs_1.readFileSync)(filePath, 'utf8'));
                let exceptionReason = 'N/A';
                if (compliance === nag_rules_1.NagRulePostValidationStates.SUPPRESSED) {
                    if (data.suppressionReason !== undefined) {
                        exceptionReason = data
                            .suppressionReason;
                    }
                    else {
                        exceptionReason = data
                            .errorSuppressionReason;
                    }
                }
                report.lines.push({
                    ruleId: data.ruleId,
                    resourceId: data.resource.node.path,
                    compliance,
                    exceptionReason,
                    ruleLevel: data.ruleLevel,
                    ruleInfo: data.ruleInfo,
                });
                (0, fs_1.writeFileSync)(filePath, JSON.stringify(report));
            }
            else {
                throw new Error(`Unrecognized output format ${format} for the NagReportLogger`);
            }
        }
    }
}
exports.NagReportLogger = NagReportLogger;
_b = JSII_RTTI_SYMBOL_1;
NagReportLogger[_b] = { fqn: "cdk-nag.NagReportLogger", version: "2.36.36" };
//# sourceMappingURL=data:application/json;base64,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