"""Signing helpers for Hyperliquid API.

1. Construct EIP-712 typed data

   Use :func:`.construct_l1_action` or :func:`.construct_user_signed_action`
2. Sign EIP-712 typed data

   Use :func:`.sign_typed_data`

.. autofunction:: construct_l1_action

.. autofunction:: construct_user_signed_action

.. autofunction:: sign_typed_data

.. autofunction:: generate_message_types

.. autofunction:: get_timestamp_ms
"""

from __future__ import annotations

import time

__all__ = [
    "EIP712Domain",
    "MessageData",
    "MessageType",
    "MessageTypes",
    "PhantomAgentMessage",
    "Signature",
    "construct_l1_action",
    "construct_user_signed_action",
    "generate_message_types",
    "get_timestamp_ms",
    "sign_typed_data",
]

import hashlib
from collections.abc import Mapping, Sequence
from typing import TYPE_CHECKING, Any, TypedDict, cast

from pybotters._static_dependencies import keccak, msgpack
from pybotters._static_dependencies.ecdsa import SECP256k1, SigningKey
from pybotters._static_dependencies.ecdsa.util import (
    sigencode_strings_canonize,
    string_to_number,
)
from pybotters._static_dependencies.ethereum.account.messages import (
    encode_typed_data,
)

if TYPE_CHECKING:
    import sys

    if sys.version_info >= (3, 11):
        from typing import NotRequired
    else:
        from typing_extensions import NotRequired

    if sys.version_info >= (3, 10):
        from typing import TypeAlias
    else:
        from typing_extensions import TypeAlias


class EIP712Domain(TypedDict):
    """TypedDict for EIP-712 domain data."""

    name: str
    version: str
    chainId: int
    verifyingContract: str
    salt: NotRequired[bytes]


class MessageType(TypedDict):
    """TypedDict for EIP-712 message type."""

    name: str
    type: str


MessageTypes: TypeAlias = Mapping[str, Sequence[MessageType]]

MessageData: TypeAlias = Mapping[str, object]


class PhantomAgentMessage(TypedDict):
    """TypedDict for Phantom Agent message."""

    source: str
    connectionId: bytes


class Signature(TypedDict):
    """TypedDict for signature."""

    r: str
    s: str
    v: int


def construct_l1_action(
    action: MessageData,
    nonce: int,
    is_mainnet: bool,
    vault_address: str | None = None,
    expires_after: int | None = None,
) -> tuple[EIP712Domain, MessageTypes, PhantomAgentMessage]:
    """Construct EIP-712 typed data for Hyperliquid L1 action.

    Args:
        action: Hyperqilid ``action`` object
        nonce: nonce for signing
        is_mainnet: True for mainnet, False for testnet
        vault_address: (optional) vault address

    Returns:
        tuple[EIP712Domain, MessageTypes, PhantomAgentMessage]
    """

    # Ref: hyperliquid.utils.signing.action_hash
    data: bytes = msgpack.packb(action)
    data += nonce.to_bytes(8, "big")
    if vault_address is None:
        data += b"\x00"
    else:
        data += b"\x01"
        data += bytes.fromhex(f"{int(vault_address, 16):x}")  # address_to_bytes
    if expires_after is not None:
        data += b"\x00"
        data += expires_after.to_bytes(8, "big")
    hash_val = keccak.SHA3(data)

    # Ref: hyperliquid.utils.signing.construct_phantom_agent
    phantom_agent: PhantomAgentMessage = {
        "source": "a" if is_mainnet else "b",
        "connectionId": hash_val,
    }

    # Ref: hyperliquid.utils.signing.sign_l1_action
    domain_data: EIP712Domain = {
        "name": "Exchange",
        "version": "1",
        "chainId": 1337,
        "verifyingContract": "0x0000000000000000000000000000000000000000",
    }
    message_types: MessageTypes = {
        "Agent": [
            {"name": "source", "type": "string"},
            {"name": "connectionId", "type": "bytes32"},
        ],
    }

    return (domain_data, message_types, phantom_agent)


def construct_user_signed_action(
    action: MessageData, message_types: MessageTypes | None = None
) -> tuple[EIP712Domain, MessageTypes, MessageData]:
    """Construct EIP-712 typed data for Hyperliquid user signed action.

    Args:
        action: Hyperqilid ``action`` object
        message_types: (optional) message types. If not provided, it will be generated by :func:`.generate_message_types`.

    Returns:
        tuple[EIP712Domain, MessageTypes, PhantomAgentMessage]
    """

    # Ref: hyperliquid.utils.signing.sign_user_signed_action
    action = dict(action)
    action.setdefault("signatureChainId", "0x66eee")
    domain_data: EIP712Domain = {
        "name": "HyperliquidSignTransaction",
        "version": "1",
        "chainId": 421614,
        "verifyingContract": "0x0000000000000000000000000000000000000000",
    }

    # Generate message types if not provided
    if message_types is None:
        message_types = generate_message_types(action)

    return (domain_data, message_types, action)


def generate_message_types(message_data: MessageData) -> MessageTypes:
    """Generate message types from message data.

    Args:
        message_data: Hyperqilid message data

    Returns:
        MessageTypes

    Infer Solidity Types from Python types. This is a basic conversion and not complete.
    """

    primary_type: str | None = None
    payload_types: list[MessageType] = []
    for key, value in message_data.items():
        # Determine primary type
        if key == "type" and isinstance(value, str):
            # capitalize first letter and remove digits
            # e.g. "usdSend" -> "HyperliquidTransaction:UsdSend", "withdraw3" -> "HyperliquidTransaction:Withdraw"
            primary_type = "HyperliquidTransaction:" + "".join(
                s.upper() if i == 0 else s
                for i, s in enumerate(value)
                if not s.isdigit()
            )
            continue
        elif key == "signatureChainId":
            continue

        # Determine Solidity Types
        if isinstance(value, bool):
            payload_types.append({"name": key, "type": "bool"})
        elif isinstance(value, int):
            payload_types.append({"name": key, "type": "uint64"})
        elif isinstance(value, str):
            payload_types.append({"name": key, "type": "string"})
        elif isinstance(value, (bytes, bytearray)):
            payload_types.append({"name": key, "type": "bytes32"})
        else:
            raise TypeError(f"Unsupported type: {type(value)}")

    if primary_type is None:
        raise ValueError(f"Primary type not found: {message_data}")

    return {primary_type: payload_types}


def _cast_to_any_dict(like_typed_dict: Mapping[str, object]) -> dict[str, Any]:
    """Cast Mapping[str, object] to dict[str, Any].

    TypedDict is a Mapping[str, object], but many signatures that do not handle this correctly require dict[str, Any].
    """
    return cast("dict[str, Any]", like_typed_dict)


# Ref: eth_account.Account.sign_typed_data
def sign_typed_data(
    private_key: str,
    domain_data: EIP712Domain,
    message_types: MessageTypes,
    message_data: MessageData,
) -> Signature:
    """Sign EIP-712 typed data with private key.

    Args:
        private_key: Your API wallet private key
        domain_data: EIP-712 domain data
        message_types: EIP-712 message types
        message_data: EIP-712 message data

    Returns:
        Signature
    """

    # Ref: eth_account.Account.encode_typed_data
    signable_message = encode_typed_data(
        _cast_to_any_dict(domain_data),
        _cast_to_any_dict(message_types),
        _cast_to_any_dict(message_data),
    )

    # Ref: eth_account.messages._hash_eip191_message
    joined = (
        b"\x19"
        + signable_message.version
        + signable_message.header
        + signable_message.body
    )
    message_hash = keccak.SHA3(joined)

    # Ref: eth_account.Account._sign_hash
    signing_key = SigningKey.from_secret_exponent(int(private_key, 16), SECP256k1)
    r_binary, s_binary, v = signing_key.sign_digest_deterministic(
        message_hash, hashlib.sha256, sigencode_strings_canonize
    )
    return Signature(
        {
            "r": hex(string_to_number(r_binary)),
            "s": hex(string_to_number(s_binary)),
            "v": 27 + v,
        }
    )


def get_timestamp_ms() -> int:
    """Get current timestamp in milliseconds that can be used as `time` or `nonce`.

    Returns:
        int
    """

    return int(time.time() * 1000)
