"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnvironmentPlaceholders = void 0;
/**
 * Placeholders which can be used manifests
 *
 * These can occur both in the Asset Manifest as well as the general
 * Cloud Assembly manifest.
 */
class EnvironmentPlaceholders {
    /**
     * Replace the environment placeholders in all strings found in a complex object.
     */
    static replace(object, values) {
        return this.recurse(object, value => {
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_REGION, values.region);
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_ACCOUNT, values.accountId);
            return value;
        });
    }
    /**
     * Like 'replace', but asynchronous
     */
    static async replaceAsync(object, provider) {
        let needRegion = false;
        let needAccountId = false;
        this.recurse(object, value => {
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_REGION) > 1) {
                needRegion = true;
            }
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_ACCOUNT) > 1) {
                needAccountId = true;
            }
            return value;
        });
        const region = needRegion ? await provider.region() : undefined;
        const accountId = needAccountId ? await provider.accountId() : undefined;
        return this.recurse(object, value => {
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_REGION, region ?? 'WONTHAPPEN');
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_ACCOUNT, accountId ?? 'WONTHAPPEN');
            return value;
        });
    }
    static recurse(value, cb) {
        if (typeof value === 'string') {
            return cb(value);
        }
        if (typeof value !== 'object' || value === null) {
            return value;
        }
        if (Array.isArray(value)) {
            return value.map(x => this.recurse(x, cb));
        }
        const ret = {};
        for (const [key, inner] of Object.entries(value)) {
            ret[key] = this.recurse(inner, cb);
        }
        return ret;
    }
}
exports.EnvironmentPlaceholders = EnvironmentPlaceholders;
/**
 * Insert this into the destination fields to be replaced with the current region
 */
EnvironmentPlaceholders.CURRENT_REGION = '${ALIYUN::Region}';
/**
 * Insert this into the destination fields to be replaced with the current account
 */
EnvironmentPlaceholders.CURRENT_ACCOUNT = '${ALIYUN::AccountId}';
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
//# sourceMappingURL=data:application/json;base64,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