#!/usr/bin/env python3
"""
Clyrdia CLI - Zero-Knowledge AI Benchmarking Platform
The most advanced local-first AI model benchmarking tool
"""

# ============================================================================
# Environment Loading - THIS IS THE FIX
# ============================================================================
from dotenv import load_dotenv
load_dotenv() # Load environment variables from .env file FIRST
# ============================================================================

import os
import sys
import json
import time
import asyncio
import sqlite3
import uuid
import webbrowser
from pathlib import Path
from typing import Optional, List, Dict, Any, Tuple, Callable
from datetime import datetime, timedelta
from dataclasses import dataclass, field, asdict
from enum import Enum
import hashlib
import yaml
import functools

import typer
from rich.console import Console
from rich.table import Table
from rich.progress import Progress, SpinnerColumn, TextColumn, BarColumn, TaskProgressColumn, TimeRemainingColumn
from rich.panel import Panel
from rich.layout import Layout
from rich.live import Live
from rich.text import Text
from rich.columns import Columns
from rich.align import Align
from rich.rule import Rule
from rich.tree import Tree
from rich.syntax import Syntax
from rich.markdown import Markdown
from rich import box
from rich.prompt import Prompt, Confirm, IntPrompt, FloatPrompt

import numpy as np
import pandas as pd
from concurrent.futures import ThreadPoolExecutor, as_completed
import aiohttp
import plotly.graph_objects as go
from plotly.subplots import make_subplots
import requests

# AI Provider imports
import openai
import anthropic

# Local imports
from .dashboard import DashboardManager

app = typer.Typer(
    name="clyrdia-cli",
    help="🚀 Clyrdia - Zero-Knowledge AI Benchmarking Platform",
    pretty_exceptions_show_locals=False,
    rich_markup_mode="rich",  # Enable Rich markup in help text
    add_completion=False,  # Disable completion to avoid formatting issues
    no_args_is_help=True,  # Show help when no args provided
)

# Custom help callback to ensure proper formatting
def custom_help_callback(ctx: typer.Context, value: bool):
    if value:
        # Get help text and clean it up
        help_text = ctx.get_help()
        # Remove excessive blank lines at the beginning
        help_lines = help_text.split('\n')
        while help_lines and help_lines[0].strip() == '':
            help_lines.pop(0)
        # Ensure we have one clean line at the top
        clean_help = '\n' + '\n'.join(help_lines)
        typer.echo(clean_help)
        raise typer.Exit()

# ============================================================================
# Rich Console Configuration
# ============================================================================

# Configure Rich console for better text alignment and box formatting
console = Console(
    width=None,  # Auto-detect terminal width
    force_terminal=True,  # Force terminal mode for consistent output
    color_system="auto",  # Auto-detect color support
    markup=True,  # Enable markup parsing
    highlight=True,  # Enable syntax highlighting
    soft_wrap=True,  # Enable soft wrapping for better text flow
    no_color=False,  # Enable colors
    tab_size=4,  # Set tab size for consistent indentation
    legacy_windows=False,  # Use modern Windows terminal features
    safe_box=True,  # Use safe box characters for better compatibility
    record=False,  # Don't record output to avoid extra formatting
)

def _display_welcome_screen():
    """Display a beautiful ASCII art welcome screen for first-time users"""
    # Clear screen for clean presentation
    console.clear()
    
    # ASCII art for CLYRDIA in bubble letters
    ascii_art = """
[bold cyan]
  ██████╗██╗     ██╗   ██╗██████╗ ██████╗ ██╗ █████╗ 
 ██╔════╝██║     ╚██╗ ██╔╝██╔══██╗██╔══██╗██║██╔══██╗
 ██║     ██║      ╚████╔╝ ██████╔╝██║  ██║██║███████║
 ██║     ██║       ╚██╔╝  ██╔══██╗██║  ██║██║██╔══██║
 ╚██████╗███████╗   ██║   ██║  ██║██████╔╝██║██║  ██║
  ╚═════╝╚══════╝   ╚═╝   ╚═╝  ╚═╝╚═════╝ ╚═╝╚═╝  ╚═╝
[/bold cyan]
"""
    
    # Subtitle and tagline
    subtitle = """
[bold bright_white]Zero-Knowledge AI Benchmarking Platform[/bold bright_white]
[dim bright_blue]The most advanced local-first AI model benchmarking tool[/dim bright_blue]
"""
    
    # Feature highlights
    features = """
[bold bright_green]✨ What makes Clyrdia special:[/bold bright_green]

[bright_cyan]🚀 Dual-Mode Workflow[/bright_cyan]   Production & Developer modes for every use case
[bright_cyan]💰 Smart Caching[/bright_cyan]      Save costs with intelligent result caching  
[bright_cyan]📊 Rich Analytics[/bright_cyan]     Beautiful dashboards & deep insights
[bright_cyan]🔒 Zero-Knowledge[/bright_cyan]     Your data stays local, always
[bright_cyan]🏆 Multi-Provider[/bright_cyan]     OpenAI & Anthropic
[bright_cyan]⚡ Lightning Fast[/bright_cyan]     Optimized for speed & efficiency
"""
    
    # Create the welcome panel with gradient border
    welcome_content = ascii_art + subtitle + features
    
    # Display with beautiful formatting
    console.print()
    console.print(Panel.fit(
        Align.center(welcome_content),
        border_style="bright_blue",
        padding=(2, 4),
        title="[bold bright_white]🌟 Welcome to the Future of AI Benchmarking 🌟[/bold bright_white]",
        title_align="center"
    ))
    console.print("[dim bright_blue]Let's get you started with your first benchmark...[/dim bright_blue]")
    console.print()

def format_help_text(text: str, title: str = "", border_style: str = "cyan") -> str:
    """Format help text with proper alignment and borders"""
    # Clean up the text to ensure proper alignment
    lines = text.strip().split('\n')
    
    # Find the maximum line length for proper box sizing
    max_length = max(len(line.strip()) for line in lines if line.strip())
    
    # Ensure minimum width for better readability
    box_width = max(max_length + 4, 60)
    
    # Format each line to ensure consistent width
    formatted_lines = []
    for line in lines:
        line = line.strip()
        if line:
            # Pad the line to ensure consistent width
            formatted_lines.append(line.ljust(box_width - 2))
    
    # Create the formatted text
    if title:
        formatted_text = f"\n{title}\n" + "\n".join(formatted_lines)
    else:
        formatted_text = "\n".join(formatted_lines)
    
    return formatted_text

# ============================================================================
# Authentication & First-Run Flow
# ============================================================================

def require_auth(func: Callable) -> Callable:
    """
    Authentication decorator that gatekeeps all user-facing commands.
    Implements the "First-Run Authentication Flow" for seamless onboarding.
    """
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        # Check if user is already authenticated
        licensing_manager = LicensingManager()
        
        if licensing_manager.is_authenticated():
            # User is authenticated, proceed with command
            return func(*args, **kwargs)
        
        # User is not authenticated - trigger first-run flow
        return _handle_first_run_auth(func, *args, **kwargs)
    
    return wrapper

def _handle_first_run_auth(func: Callable, *args, **kwargs):
    """
    Handles the first-run authentication flow with seamless handoff to web signup.
    """
    # Display beautiful ASCII art welcome screen
    _display_welcome_screen()
    
    console.print(Panel.fit(
        "[bold cyan]👋 Welcome to Clyrdia![/bold cyan]\n"
        "To run your first benchmark, let's connect your CLI to your free account.\n"
        "This will unlock your 250 monthly credits.",
        border_style="cyan",
        padding=(1, 2),
        title="Welcome",
        title_align="center"
    ))
    
    console.print("\n[bold]🔗 Please visit clyrdia.com to create your account:[/bold]")
    console.print("[bold bright_blue]https://clyrdia.com/auth[/bold bright_blue]")
    console.print("\n[yellow]Complete the signup process and get your API key, then paste it below.[/yellow]")
    
    console.print("\n[bold]Steps to complete signup:[/bold]")
    console.print("1. Complete the signup form on clyrdia.com")
    console.print("2. Verify your email address")
    console.print("3. Get your API key from your dashboard")
    console.print("4. Paste it below when ready")
    
    console.print("\n[bold]Waiting for API key...[/bold] (paste it here when you're done)")
    
    # Wait for user to paste their API key
    while True:
        try:
            api_key = Prompt.ask("API Key", password=True)
            
            if not api_key or len(api_key.strip()) < 10:
                console.print("[red]❌ Invalid API key format. Please try again.[/red]")
                continue
            
            # Validate the API key
            licensing_manager = LicensingManager()
            try:
                user_status = asyncio.run(licensing_manager.login(api_key.strip()))
                
                console.print(f"\n[green]✅ Successfully authenticated![/green]")
                console.print(f"Welcome, [bold]{user_status.user_name}[/bold]!")
                console.print(f"Plan: [bold]{user_status.plan.upper()}[/bold]")
                console.print(f"Credits: [bold]{user_status.credits_remaining:,}[/bold]")
                
                # Success! Now execute the original command
                console.print(f"\n[bold green]🚀 Continuing with your original command...[/bold green]")
                return func(*args, **kwargs)
                
            except Exception as e:
                console.print(f"[red]❌ Authentication failed: {str(e)}[/red]")
                console.print("\n[bold]Troubleshooting:[/bold]")
                console.print("  • Verify you completed the signup process")
                console.print("  • Check your internet connection")
                console.print("  • Try copying the API key again")
                console.print("  • Visit [bold]https://clyrdia.com[/bold] to start over")
                
                if not Confirm.ask("Try again with a different API key?"):
                    console.print("[dim]Authentication cancelled. Run your command again when ready.[/dim]")
                    raise typer.Exit(0)
                
                continue
                
        except KeyboardInterrupt:
            console.print("\n\n[yellow]Authentication cancelled.[/yellow]")
            console.print("Run your command again when you're ready to authenticate.")
            raise typer.Exit(0)
        except Exception as e:
            console.print(f"[red]❌ Unexpected error: {str(e)}[/red]")
            raise typer.Exit(1)

def _is_first_run() -> bool:
    """Check if this is the user's first run of Clyrdia"""
    config_file = Path.home() / ".clyrdia" / "config.json"
    return not config_file.exists()

def _get_original_command() -> str:
    """Get the original command the user was trying to run"""
    # This is a simplified approach - in practice, you might want to store
    # the original command in a more sophisticated way
    return " ".join(sys.argv[1:])

# ============================================================================
# Data Models & Configuration
# ============================================================================

class ModelProvider(Enum):
    OPENAI = "openai"
    ANTHROPIC = "anthropic"

@dataclass
class ModelConfig:
    """Configuration for an AI model"""
    name: str
    provider: ModelProvider
    input_cost: float  # per 1M tokens
    output_cost: float  # per 1M tokens
    max_tokens: int
    context_window: int
    capabilities: List[str] = field(default_factory=list)
    speed_tier: str = "standard"  # fast, standard, slow
    tier: str = "balanced"  # flagship, balanced, speed_cost
    
    # Additional cost parameters for different token ranges (only for live API calls)
    input_cost_over_200k: Optional[float] = None  # per 1M tokens
    output_cost_over_200k: Optional[float] = None  # per 1M tokens
    
@dataclass
class BenchmarkResult:
    """Result from a single benchmark test"""
    model: str
    provider: str
    test_name: str
    prompt: str
    response: str
    latency_ms: int
    input_tokens: int
    output_tokens: int
    cost: float
    success: bool
    error: Optional[str] = None
    quality_scores: Dict[str, float] = field(default_factory=dict)
    timestamp: datetime = field(default_factory=datetime.now)
    metadata: Dict[str, Any] = field(default_factory=dict)

@dataclass
class TestCase:
    """Definition of a benchmark test case"""
    name: str
    prompt: str
    expected_output: Optional[str] = None
    max_tokens: int = 1000
    temperature: float = 0.7
    evaluation_criteria: List[str] = field(default_factory=list)
    tags: List[str] = field(default_factory=list)
    weight: float = 1.0

@dataclass
class CachedResult:
    """Cached benchmark result"""
    cache_key: str
    model: str
    test_name: str
    prompt: str
    response: str
    input_tokens: int
    output_tokens: int
    quality_scores: Dict[str, float]
    timestamp: datetime
    metadata: Dict[str, Any]

class CacheManager:
    """Manages caching for benchmark results"""
    
    def __init__(self, cache_dir: Optional[Path] = None):
        self.cache_dir = cache_dir or (Path.home() / ".clyrdia" / "cache")
        self.cache_dir.mkdir(parents=True, exist_ok=True)
        self.cache_db = self.cache_dir / "cache.db"
        self._init_cache_db()
    
    def _init_cache_db(self):
        """Initialize the cache database"""
        with sqlite3.connect(self.cache_db) as conn:
            conn.execute("""
                CREATE TABLE IF NOT EXISTS cached_results (
                    cache_key TEXT PRIMARY KEY,
                    model TEXT NOT NULL,
                    test_name TEXT NOT NULL,
                    prompt TEXT NOT NULL,
                    response TEXT NOT NULL,
                    input_tokens INTEGER NOT NULL,
                    output_tokens INTEGER NOT NULL,
                    quality_scores TEXT NOT NULL,
                    timestamp TEXT NOT NULL,
                    metadata TEXT NOT NULL
                )
            """)
            conn.commit()
    
    def _generate_cache_key(self, model: str, test_name: str, prompt: str, 
                           max_tokens: int, temperature: float) -> str:
        """Generate a unique cache key for a test run"""
        # Create a hash based on all parameters that affect the result
        content = f"{model}:{test_name}:{prompt.strip()}:{max_tokens}:{temperature}"
        return hashlib.sha256(content.encode()).hexdigest()
    
    def get_cached_result(self, model: str, test_name: str, prompt: str,
                         max_tokens: int, temperature: float) -> Optional[CachedResult]:
        """Get a cached result if it exists"""
        cache_key = self._generate_cache_key(model, test_name, prompt, max_tokens, temperature)
        
        with sqlite3.connect(self.cache_db) as conn:
            cursor = conn.execute(
                "SELECT * FROM cached_results WHERE cache_key = ?",
                (cache_key,)
            )
            row = cursor.fetchone()
            
            if row:
                return CachedResult(
                    cache_key=row[0],
                    model=row[1],
                    test_name=row[2],
                    prompt=row[3],
                    response=row[4],
                    input_tokens=row[5],
                    output_tokens=row[6],
                    quality_scores=json.loads(row[7]),
                    timestamp=datetime.fromisoformat(row[8]),
                    metadata=json.loads(row[9])
                )
        return None
    
    def cache_result(self, model: str, test_name: str, prompt: str,
                    response: str, input_tokens: int, output_tokens: int,
                    quality_scores: Dict[str, float], max_tokens: int,
                    temperature: float, metadata: Dict[str, Any]):
        """Cache a benchmark result"""
        cache_key = self._generate_cache_key(model, test_name, prompt, max_tokens, temperature)
        
        with sqlite3.connect(self.cache_db) as conn:
            conn.execute("""
                INSERT OR REPLACE INTO cached_results 
                (cache_key, model, test_name, prompt, response, input_tokens, 
                 output_tokens, quality_scores, timestamp, metadata)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            """, (
                cache_key, model, test_name, prompt, response, input_tokens,
                output_tokens, json.dumps(quality_scores), 
                datetime.now().isoformat(), json.dumps(metadata)
            ))
            conn.commit()
    
    def clear_cache(self, model: Optional[str] = None):
        """Clear all cache or cache for a specific model"""
        with sqlite3.connect(self.cache_db) as conn:
            if model:
                conn.execute("DELETE FROM cached_results WHERE model = ?", (model,))
            else:
                conn.execute("DELETE FROM cached_results")
            conn.commit()
    
    def get_cache_stats(self) -> Dict[str, Any]:
        """Get cache statistics"""
        with sqlite3.connect(self.cache_db) as conn:
            # Total entries
            total = conn.execute("SELECT COUNT(*) FROM cached_results").fetchone()[0]
            
            # Size by model
            model_counts = {}
            cursor = conn.execute(
                "SELECT model, COUNT(*) FROM cached_results GROUP BY model"
            )
            for row in cursor.fetchall():
                model_counts[row[0]] = row[1]
            
            # Cache size on disk
            cache_size = self.cache_db.stat().st_size if self.cache_db.exists() else 0
            
            return {
                "total_entries": total,
                "model_counts": model_counts,
                "cache_size_bytes": cache_size,
                "cache_size_mb": round(cache_size / (1024 * 1024), 2)
            }
    
    def get_cache_hit_rate(self, test_cases: List[TestCase], models: List[str]) -> Dict[str, float]:
        """Calculate potential cache hit rate for a benchmark run"""
        total_tests = len(test_cases) * len(models)
        cacheable_tests = 0
        
        for test_case in test_cases:
            for model in models:
                if self.get_cached_result(model, test_case.name, test_case.prompt, 
                                        test_case.max_tokens, test_case.temperature):
                    cacheable_tests += 1
        
        return {
            "total_tests": total_tests,
            "cacheable_tests": cacheable_tests,
            "hit_rate": cacheable_tests / total_tests if total_tests > 0 else 0
        }

class ClyrdiaConfig:
    """Global configuration management"""
    
    # Model catalog with 2025 latest models
    # All prices are USD per 1M tokens (or per 1K searches where noted).
    MODELS = {
    # ---------- OpenAI ----------
    "gpt-5": ModelConfig(
        name="gpt-5",
        provider=ModelProvider.OPENAI,
        input_cost=1.25,
        output_cost=10.00,
        max_tokens=128000, context_window=128000,
        capabilities=["chat","code","vision","function_calling"], speed_tier="standard", tier="flagship"
    ),
    "gpt-5-mini": ModelConfig(
        name="gpt-5-mini",
        provider=ModelProvider.OPENAI,
        input_cost=0.25,
        output_cost=2.00,
        max_tokens=128000, context_window=128000,
        capabilities=["chat","code"], speed_tier="fast", tier="balanced"
    ),
    "gpt-5-nano": ModelConfig(
        name="gpt-5-nano",
        provider=ModelProvider.OPENAI,
        input_cost=0.05,
        output_cost=0.40,
        max_tokens=128000, context_window=128000,
        capabilities=["chat"], speed_tier="fastest", tier="speed_cost"
    ),
    "gpt-4.1": ModelConfig(
        name="gpt-4.1",
        provider=ModelProvider.OPENAI,
        input_cost=2.00,
        output_cost=8.00,
        max_tokens=128000, context_window=128000,
        capabilities=["chat","code","vision"], speed_tier="standard", tier="flagship"
    ),
    "gpt-4.1-mini": ModelConfig(
        name="gpt-4.1-mini",
        provider=ModelProvider.OPENAI,
        input_cost=0.40,
        output_cost=1.60,
        max_tokens=128000, context_window=128000,
        capabilities=["chat","code"], speed_tier="fast", tier="balanced"
    ),
    "gpt-4.1-nano": ModelConfig(
        name="gpt-4.1-nano",
        provider=ModelProvider.OPENAI,
        input_cost=0.10,
        output_cost=0.40,
        max_tokens=128000, context_window=128000,
        capabilities=["chat"], speed_tier="fast", tier="speed_cost"
    ),
    "gpt-4o": ModelConfig(
        name="gpt-4o",
        provider=ModelProvider.OPENAI,
        input_cost=2.50,
        output_cost=10.00,
        max_tokens=128000, context_window=128000,
        capabilities=["chat","code","vision","function_calling","multimodal"], speed_tier="fast", tier="flagship"
    ),
    "gpt-4o-mini": ModelConfig(
        name="gpt-4o-mini",
        provider=ModelProvider.OPENAI,
        input_cost=0.15,
        output_cost=0.60,
        max_tokens=128000, context_window=128000,
        capabilities=["chat","code"], speed_tier="fast", tier="balanced"
    ),

    # ---------- Anthropic (Claude) ----------
    "claude-opus-4.1": ModelConfig(
        name="claude-opus-4-1-20250805",
        provider=ModelProvider.ANTHROPIC,
        input_cost=15.00,
        output_cost=75.00,
        max_tokens=200000, context_window=200000,
        capabilities=["chat","code","analysis","creative","multimodal"], speed_tier="standard", tier="flagship"
    ),
    "claude-sonnet-4": ModelConfig(
        name="claude-sonnet-4-20250514",
        provider=ModelProvider.ANTHROPIC,
        input_cost=3.00,                 # ≤200k prompt
        output_cost=15.00,               # ≤200k prompt
        input_cost_over_200k=6.00,       # >200k prompt
        output_cost_over_200k=22.50,     # >200k prompt
        max_tokens=200000, context_window=200000,
        capabilities=["chat","code","analysis","multimodal"], speed_tier="fast", tier="balanced"
    ),
    "claude-haiku-3.5": ModelConfig(
        name="claude-3-5-haiku-20241022",
        provider=ModelProvider.ANTHROPIC,
        input_cost=0.80,
        output_cost=4.00,
        max_tokens=200000, context_window=200000,
        capabilities=["chat","code"], speed_tier="fastest", tier="speed_cost"
    ),


    }
    
    @classmethod
    def get_model(cls, name: str) -> Optional[ModelConfig]:
        return cls.MODELS.get(name)
    
    @classmethod
    def list_models(cls) -> List[str]:
        return list(cls.MODELS.keys())

@dataclass
class UserStatus:
    """User subscription and credit status"""
    user_name: str
    plan: str  # "free" or "pro"
    credits_remaining: int
    resets_on: str
    api_key: str

@dataclass
class CreditEstimate:
    """Credit cost estimation for a benchmark run"""
    total_tests: int
    cache_hits: int
    live_api_calls: int
    estimated_credits: int
    current_balance: int
    test_breakdown: Dict[str, int]

class LicensingManager:
    """Manages SaaS licensing and credit system"""
    
    def __init__(self):
        from .config import config
        self.config = config
        self.config_dir = Path.home() / ".clyrdia"
        self.config_file = self.config_dir / "config.json"
        self.api_key = self._load_api_key()
    
    def _load_api_key(self) -> Optional[str]:
        """Load API key from config file"""
        if self.config_file.exists():
            try:
                with open(self.config_file, 'r') as f:
                    config = json.load(f)
                    return config.get('api_key')
            except (json.JSONDecodeError, KeyError):
                pass
        return None
    
    def is_first_run(self) -> bool:
        """Check if this is the user's first run"""
        return not self.config_file.exists()
    
    def _save_api_key(self, api_key: str):
        """Save API key to config file with secure permissions"""
        self.config_dir.mkdir(parents=True, exist_ok=True)
        
        config = {'api_key': api_key}
        with open(self.config_file, 'w') as f:
            json.dump(config, f, indent=2)
        
        # Set secure file permissions (600)
        os.chmod(self.config_file, 0o600)
    
    async def _make_api_request(self, endpoint: str, method: str = "GET", 
                         data: Optional[Dict] = None) -> Dict[str, Any]:
        """Make authenticated API request to Clyrdia backend"""
        if not self.api_key:
            raise Exception("No API key configured. Please run 'clyrdia-cli login' first.")
        
        headers = {
            "Authorization": f"Bearer {self.api_key}",
            "Content-Type": "application/json"
        }
        
        url = self.config.get_api_url(endpoint)
        
        try:
            import httpx
            async with httpx.AsyncClient(timeout=30.0) as client:
                if method == "GET":
                    response = await client.get(url, headers=headers)
                elif method == "POST":
                    response = await client.post(url, headers=headers, json=data)
                else:
                    raise ValueError(f"Unsupported HTTP method: {method}")
                
                response.raise_for_status()
                return response.json()
                
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 401:
                raise Exception("Invalid API key. Please run 'clyrdia-cli login' again.")
            elif e.response.status_code == 402:
                return e.response.json()  # Return payment required response
            else:
                raise Exception(f"API request failed: {str(e)}")
        except Exception as e:
            raise Exception(f"API request failed: {str(e)}")
    
    async def login(self, api_key: str) -> UserStatus:
        """Login with API key and validate subscription"""
        # Test the API key by calling the status endpoint
        self._save_api_key(api_key)
        self.api_key = api_key
        
        try:
            status = await self.get_status()
            return status
        except Exception as e:
            # Remove invalid key
            self._remove_api_key()
            raise e
    
    async def get_status(self) -> UserStatus:
        """Get current user status and credit balance"""
        response = await self._make_api_request("/cli-status")
        
        return UserStatus(
            user_name=response["user_name"],
            plan=response["plan"],
            credits_remaining=response["credits_remaining"],
            resets_on=response["resets_on"],
            api_key=self.api_key
        )
    
    async def debit_credits(self, credits_to_debit: int, run_id: str) -> Dict[str, Any]:
        """Debit credits for a benchmark run"""
        data = {
            "credits_to_debit": credits_to_debit,
            "run_id": run_id
        }
        
        response = await self._make_api_request("/cli-usage-debit", method="POST", data=data)
        return response
    
    def _remove_api_key(self):
        """Remove API key from config"""
        if self.config_file.exists():
            self.config_file.unlink()
        self.api_key = None
    
    def logout(self):
        """Logout and remove API key"""
        self._remove_api_key()
    
    def is_authenticated(self) -> bool:
        """Check if user is authenticated"""
        return self.api_key is not None
    
    async def estimate_credits(self, test_cases: List['TestCase'], models: List[str], 
                        use_cache: bool = False) -> CreditEstimate:
        """Estimate credit cost for a benchmark run"""
        cache_manager = CacheManager()
        total_credits = 0
        test_breakdown = {}
        
        for test_case in test_cases:
            for model in models:
                test_key = f"{test_case.name}_{model}"
                
                if use_cache and cache_manager.get_cached_result(
                    model, test_case.name, test_case.prompt, 
                    test_case.max_tokens, test_case.temperature
                ):
                    test_breakdown[test_key] = 0  # Cache hit = 0 credits
                else:
                    # Estimate credits based on test case
                    estimated_tokens = len(test_case.prompt.split()) * 1.3  # Rough estimate
                    estimated_credits = max(1, int(estimated_tokens / 1000))  # 1 credit per 1K tokens
                    total_credits += estimated_credits
                    test_breakdown[test_key] = estimated_credits
        
        # Get current balance if authenticated
        current_balance = 0
        if self.is_authenticated():
            try:
                status = await self.get_status()
                current_balance = status.credits_remaining
            except:
                pass
        
        cache_hits = sum(1 for cost in test_breakdown.values() if cost == 0)
        live_api_calls = len(test_breakdown) - cache_hits
        
        return CreditEstimate(
            total_tests=len(test_breakdown),
            cache_hits=cache_hits,
            live_api_calls=live_api_calls,
            estimated_credits=total_credits,
            current_balance=current_balance,
            test_breakdown=test_breakdown
        )
    
    def display_credit_summary(self, before_credits: int, after_credits: int):
        """Display credit usage summary"""
        used = before_credits - after_credits
        console.print(f"[bold]💸 Credits Used: {used}[/bold]")
        console.print(f"[bold] Remaining: {after_credits} credits[/bold]")
    
    def get_credit_usage(self, run_id: str) -> Dict[str, Any]:
        """Get credit usage for a specific run"""
        try:
            response = self._make_api_request(f"/api/v1/usage/{run_id}")
            return response
        except Exception as e:
            return {"error": str(e), "run_id": run_id}
    
    def show_credit_balance(self):
        """Display current credit balance"""
        try:
            status = asyncio.run(self.get_status())
            console.print(f"[bold]💰 Current Balance: {status.credits_remaining} credits[/bold]")
            console.print(f"[dim]Plan: {status.plan.upper()}[/dim]")
            console.print(f"[dim]Resets on: {status.resets_on}[/dim]")
            
            # Show upgrade message if credits are low
            if status.credits_remaining <= 50:
                if status.plan == "free":
                    console.print(f"\n[yellow]⚠️  Low credits![/yellow]")
                    console.print(f"• You have {status.credits_remaining} credits remaining")
                    console.print(f"• Free plan includes 250 credits/month")
                    console.print(f"• Upgrade to Pro for 10,000 credits/month")
                    console.print(f"• Visit [bold]https://clyrdia.com[/bold] to upgrade")
                else:
                    console.print(f"\n[yellow]⚠️  Low credits![/yellow]")
                    console.print(f"• You have {status.credits_remaining} credits remaining")
                    console.print(f"• Consider upgrading for more credits")
                    console.print(f"• Visit [bold]https://clyrdia.com[/bold] to upgrade")
            
            return status.credits_remaining
        except Exception as e:
            console.print(f"[red]❌ Could not fetch credit balance: {str(e)}[/red]")
            return 0
    
    def check_credits_sufficient(self, required_credits: int) -> Tuple[bool, int]:
        """Check if user has sufficient credits for an operation"""
        try:
            status = asyncio.run(self.get_status())
            if status.credits_remaining >= required_credits:
                return True, status.credits_remaining
            else:
                return False, status.credits_remaining
        except Exception as e:
            console.print(f"[red]❌ Could not check credit balance: {str(e)}[/red]")
            return False, 0
    
    def show_upgrade_message(self):
        """Display upgrade message when credits are insufficient"""
        console.print(f"\n[red]🚫 Out of Credits![/red]")
        console.print(f"• You've used all your available credits")
        console.print(f"• Upgrade to Pro for 10,000 credits/month")
        console.print(f"• Priority support and advanced features")
        console.print(f"• Visit [bold]https://clyrdia.com[/bold] to upgrade")
        console.print(f"• No credit card required for free plan")

# ============================================================================
# Database Layer (SQLite Local Storage)
# ============================================================================

class LocalDatabase:
    """Local SQLite database for zero-knowledge storage"""
    
    def __init__(self):
        self.db_path = Path.home() / ".clyrdia" / "clyrdia.db"
        self.db_path.parent.mkdir(parents=True, exist_ok=True)
        self._init_database()
    
    def _init_database(self):
        """Initialize database schema"""
        with sqlite3.connect(self.db_path) as conn:
            # Create the main benchmark_results table that the dashboard expects
            conn.execute("""
                CREATE TABLE IF NOT EXISTS benchmark_results (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    benchmark_id TEXT NOT NULL,
                    benchmark_name TEXT,
                    model TEXT NOT NULL,
                    provider TEXT NOT NULL,
                    test_name TEXT NOT NULL,
                    prompt TEXT,
                    response TEXT,
                    latency_ms INTEGER,
                    input_tokens INTEGER,
                    output_tokens INTEGER,
                    cost REAL,
                    success BOOLEAN,
                    error TEXT,
                    quality_score REAL,
                    metadata TEXT,
                    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """)
            
            conn.execute("""
                CREATE TABLE IF NOT EXISTS benchmarks (
                    id TEXT PRIMARY KEY,
                    name TEXT NOT NULL,
                    description TEXT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    config TEXT,
                    tags TEXT
                )
            """)
            
            conn.execute("""
                CREATE TABLE IF NOT EXISTS results (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    benchmark_id TEXT,
                    model TEXT NOT NULL,
                    provider TEXT NOT NULL,
                    test_name TEXT NOT NULL,
                    prompt TEXT,
                    response TEXT,
                    latency_ms INTEGER,
                    input_tokens INTEGER,
                    output_tokens INTEGER,
                    cost REAL,
                    success BOOLEAN,
                    error TEXT,
                    quality_scores TEXT,
                    metadata TEXT,
                    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (benchmark_id) REFERENCES benchmarks(id)
                )
            """)
            
            conn.execute("""
                CREATE TABLE IF NOT EXISTS baselines (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    name TEXT UNIQUE NOT NULL,
                    model TEXT NOT NULL,
                    metrics TEXT NOT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """)
            
            conn.execute("""
                CREATE TABLE IF NOT EXISTS drift_history (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    model TEXT NOT NULL,
                    test_hash TEXT NOT NULL,
                    drift_score REAL,
                    details TEXT,
                    detected_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """)
            
            # Create indexes for performance
            conn.execute("CREATE INDEX IF NOT EXISTS idx_benchmark_results_benchmark ON benchmark_results(benchmark_id)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_benchmark_results_model ON benchmark_results(model)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_benchmark_results_timestamp ON benchmark_results(timestamp)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_results_benchmark ON results(benchmark_id)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_results_model ON results(model)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_results_timestamp ON results(timestamp)")
    
    def save_benchmark(self, benchmark_id: str, name: str, description: str, config: Dict, tags: List[str]) -> str:
        """Save benchmark configuration"""
        with sqlite3.connect(self.db_path) as conn:
            conn.execute(
                "INSERT INTO benchmarks (id, name, description, config, tags) VALUES (?, ?, ?, ?, ?)",
                (benchmark_id, name, description, json.dumps(config), json.dumps(tags))
            )
        return benchmark_id
    
    def save_result(self, result: BenchmarkResult, benchmark_id: Optional[str] = None) -> int:
        """Save benchmark result to both tables for compatibility"""
        with sqlite3.connect(self.db_path) as conn:
            # Save to the original results table
            cursor = conn.execute(
                """INSERT INTO results 
                   (benchmark_id, model, provider, test_name, prompt, response, 
                    latency_ms, input_tokens, output_tokens, cost, success, error, 
                    quality_scores, metadata, timestamp)
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
                (
                    benchmark_id,
                    result.model,
                    result.provider,
                    result.test_name,
                    result.prompt,
                    result.response,
                    result.latency_ms,
                    result.input_tokens,
                    result.output_tokens,
                    result.cost,
                    result.success,
                    result.error,
                    json.dumps(result.quality_scores),
                    json.dumps(result.metadata),
                    result.timestamp
                )
            )
            
            # Also save to the benchmark_results table that the dashboard expects
            # Extract quality score (use first score if multiple, or 0.0 if none)
            quality_score = 0.0
            if result.quality_scores and len(result.quality_scores) > 0:
                if isinstance(result.quality_scores, dict):
                    quality_score = list(result.quality_scores.values())[0]
                elif isinstance(result.quality_scores, list):
                    quality_score = result.quality_scores[0]
                else:
                    quality_score = float(result.quality_scores)
            
            # Get benchmark name if available
            benchmark_name = None
            if benchmark_id:
                try:
                    name_cursor = conn.execute(
                        "SELECT name FROM benchmarks WHERE id = ?",
                        (benchmark_id,)
                    )
                    name_row = name_cursor.fetchone()
                    if name_row:
                        benchmark_name = name_row[0]
                except:
                    pass
            
            conn.execute(
                """INSERT INTO benchmark_results 
                   (benchmark_id, benchmark_name, model, provider, test_name, prompt, response, 
                    latency_ms, input_tokens, output_tokens, cost, success, error, 
                    quality_score, metadata, timestamp)
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
                (
                    benchmark_id,
                    benchmark_name,
                    result.model,
                    result.provider,
                    result.test_name,
                    result.prompt,
                    result.response,
                    result.latency_ms,
                    result.input_tokens,
                    result.output_tokens,
                    result.cost,
                    result.success,
                    result.error,
                    quality_score,
                    json.dumps(result.metadata),
                    result.timestamp
                )
            )
            
            return cursor.lastrowid
    
    def get_recent_benchmarks(self, limit: int = 10) -> List[Dict]:
        """Get recent benchmark runs"""
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.execute(
                "SELECT * FROM benchmarks ORDER BY created_at DESC LIMIT ?",
                (limit,)
            )
            return [dict(row) for row in cursor.fetchall()]
    
    def get_model_performance(self, model: str, days: int = 30) -> pd.DataFrame:
        """Get model performance over time"""
        with sqlite3.connect(self.db_path) as conn:
            query = """
                SELECT timestamp, latency_ms, cost, success, quality_scores
                FROM results
                WHERE model = ? AND timestamp > datetime('now', '-{} days')
                ORDER BY timestamp
            """.format(days)
            return pd.read_sql_query(query, conn, params=(model,))
    
    def migrate_existing_data(self) -> int:
        """Migrate existing data from results table to benchmark_results table"""
        migrated_count = 0
        with sqlite3.connect(self.db_path) as conn:
            # Check if there's data to migrate
            cursor = conn.execute("SELECT COUNT(*) FROM results")
            total_results = cursor.fetchone()[0]
            
            if total_results == 0:
                return 0
            
            # Check if benchmark_results already has data
            cursor = conn.execute("SELECT COUNT(*) FROM benchmark_results")
            existing_benchmark_results = cursor.fetchone()[0]
            
            if existing_benchmark_results > 0:
                return 0  # Already migrated
            
            console.print(f"[yellow]🔄 Migrating {total_results} existing results to dashboard format...[/yellow]")
            
            # Get all results and migrate them
            cursor = conn.execute("""
                SELECT r.*, b.name as benchmark_name
                FROM results r
                LEFT JOIN benchmarks b ON r.benchmark_id = b.id
                ORDER BY r.timestamp
            """)
            
            for row in cursor.fetchall():
                # Extract quality score
                quality_score = 0.0
                if row['quality_scores']:
                    try:
                        scores = json.loads(row['quality_scores'])
                        if isinstance(scores, dict) and scores:
                            quality_score = list(scores.values())[0]
                        elif isinstance(scores, list) and scores:
                            quality_score = scores[0]
                        else:
                            quality_score = float(scores)
                    except:
                        quality_score = 0.0
                
                # Insert into benchmark_results
                conn.execute("""
                    INSERT INTO benchmark_results 
                    (benchmark_id, benchmark_name, model, provider, test_name, prompt, response, 
                     latency_ms, input_tokens, output_tokens, cost, success, error, 
                     quality_score, metadata, timestamp)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                """, (
                    row['benchmark_id'],
                    row['benchmark_name'],
                    row['model'],
                    row['provider'],
                    row['test_name'],
                    row['prompt'],
                    row['response'],
                    row['latency_ms'],
                    row['input_tokens'],
                    row['output_tokens'],
                    row['cost'],
                    row['success'],
                    row['error'],
                    quality_score,
                    row['metadata'],
                    row['timestamp']
                ))
                migrated_count += 1
            
            conn.commit()
            console.print(f"[green]✅ Successfully migrated {migrated_count} results[/green]")
        
        return migrated_count

# ============================================================================
# AI Provider Interfaces
# ============================================================================

class ModelInterface:
    """Unified interface for all AI model providers"""
    
    def __init__(self):
        self.openai_client = None
        self.anthropic_client = None
        self._init_clients()
    
    def _init_clients(self):
        """Initialize AI provider clients"""
        # OpenAI
        if os.getenv("OPENAI_API_KEY"):
            self.openai_client = openai.OpenAI(api_key=os.getenv("OPENAI_API_KEY"))
        
        # Anthropic
        if os.getenv("ANTHROPIC_API_KEY"):
            self.anthropic_client = anthropic.Anthropic(api_key=os.getenv("ANTHROPIC_API_KEY"))
        
        # Initialize tiktoken for accurate token counting
        try:
            import tiktoken
            self.tiktoken_encoder = tiktoken.get_encoding("cl100k_base")  # OpenAI's encoding
        except ImportError:
            self.tiktoken_encoder = None
    
    async def call_model(self, model_name: str, prompt: str, max_tokens: int = 1000, 
                         temperature: float = 0.7) -> Tuple[str, int, int, float, int]:
        """Call AI model and return response with metrics"""
        model_config = ClyrdiaConfig.get_model(model_name)
        if not model_config:
            raise ValueError(f"Unknown model: {model_name}")
        
        start_time = time.perf_counter()
        
        try:
            if model_config.provider == ModelProvider.OPENAI:
                response, input_tokens, output_tokens = await self._call_openai(
                    model_config.name, prompt, max_tokens, temperature
                )
            elif model_config.provider == ModelProvider.ANTHROPIC:
                response, input_tokens, output_tokens = await self._call_anthropic(
                    model_config.name, prompt, max_tokens, temperature
                )

            else:
                raise ValueError(f"Unsupported provider: {model_config.provider}")
            
            latency_ms = int((time.perf_counter() - start_time) * 1000)
            
            # Calculate cost with proper token range handling (per 1M tokens)
            total_tokens = input_tokens + output_tokens
            
            # Handle models with different costs for different token ranges
            if total_tokens > 200000 and model_config.input_cost_over_200k and model_config.output_cost_over_200k:
                # Use higher cost for >200k tokens (per 1M tokens)
                input_cost = (input_tokens / 1_000_000) * model_config.input_cost_over_200k
                output_cost = (output_tokens / 1_000_000) * model_config.output_cost_over_200k
            else:
                # Use standard costs (per 1M tokens)
                input_cost = (input_tokens / 1_000_000) * model_config.input_cost
                output_cost = (output_tokens / 1_000_000) * model_config.output_cost
            
            cost = input_cost + output_cost
            
            # Ensure cost is never negative and has reasonable precision
            cost = max(0.0, round(cost, 6))
            
            return response, input_tokens, output_tokens, cost, latency_ms
            
        except Exception as e:
            latency_ms = int((time.perf_counter() - start_time) * 1000)
            raise Exception(f"Model call failed: {str(e)}")
    
    async def _call_openai(self, model: str, prompt: str, max_tokens: int, temperature: float):
        """Call OpenAI model with accurate token counting"""
        if not self.openai_client:
            raise Exception("OpenAI API key not configured")
        
        try:
            response = await asyncio.to_thread(
                self.openai_client.chat.completions.create,
                model=model,
                messages=[{"role": "user", "content": prompt}],
                max_tokens=max_tokens,
                temperature=temperature
            )
            
            # Extract precise token counts from OpenAI response
            input_tokens = response.usage.prompt_tokens
            output_tokens = response.usage.completion_tokens
            
            return (
                response.choices[0].message.content,
                input_tokens,
                output_tokens
            )
        except Exception as e:
            raise Exception(f"OpenAI API call failed: {str(e)}")
    
    async def _call_anthropic(self, model: str, prompt: str, max_tokens: int, temperature: float):
        """Call Anthropic model with accurate token counting"""
        if not self.anthropic_client:
            raise Exception("Anthropic API key not configured")
        
        try:
            response = await asyncio.to_thread(
                self.anthropic_client.messages.create,
                model=model,
                max_tokens=max_tokens,
                temperature=temperature,
                messages=[{"role": "user", "content": prompt}]
            )
            
            # Extract precise token counts from Anthropic response
            input_tokens = response.usage.input_tokens
            output_tokens = response.usage.output_tokens
            
            return (
                response.content[0].text,
                input_tokens,
                output_tokens
            )
        except Exception as e:
            raise Exception(f"Anthropic API call failed: {str(e)}")
    

    

    
    def validate_cost_calculation(self, model_name: str, input_tokens: int, output_tokens: int, calculated_cost: float):
        """Validate that cost calculation matches expected values (per 1M tokens)"""
        model_config = ClyrdiaConfig.get_model(model_name)
        if not model_config:
            return False, f"Unknown model: {model_name}"
        
        # Calculate expected cost (per 1M tokens)
        total_tokens = input_tokens + output_tokens
        
        if total_tokens > 200000 and model_config.input_cost_over_200k and model_config.output_cost_over_200k:
            expected_input_cost = (input_tokens / 1_000_000) * model_config.input_cost_over_200k
            expected_output_cost = (output_tokens / 1_000_000) * model_config.output_cost_over_200k
        else:
            expected_input_cost = (input_tokens / 1_000_000) * model_config.input_cost
            expected_output_cost = (output_tokens / 1_000_000) * model_config.output_cost
        
        expected_cost = expected_input_cost + expected_output_cost
        expected_cost = max(0.0, round(expected_cost, 6))
        
        # Use appropriate tolerance based on cost magnitude
        if expected_cost < 0.0001:
            # For very small costs, use absolute tolerance
            tolerance = 0.000001
        else:
            # For larger costs, use percentage tolerance (0.1%)
            tolerance = expected_cost * 0.001
        
        if abs(calculated_cost - expected_cost) <= tolerance:
            return True, f"Cost validation passed: ${calculated_cost:.6f} (expected: ${expected_cost:.6f}) [per 1M tokens]"
        else:
            return False, f"Cost validation failed: calculated ${calculated_cost:.6f}, expected ${expected_cost:.6f} [per 1M tokens]"
    
    def get_detailed_cost_breakdown(self, model_name: str, input_tokens: int, output_tokens: int) -> Dict[str, Any]:
        """Get detailed cost breakdown for transparency and debugging"""
        model_config = ClyrdiaConfig.get_model(model_name)
        if not model_config:
            return {"error": f"Unknown model: {model_name}"}
        
        total_tokens = input_tokens + output_tokens
        
        # Determine which pricing tier to use
        if total_tokens > 200000 and model_config.input_cost_over_200k and model_config.output_cost_over_200k:
            input_rate = model_config.input_cost_over_200k
            output_rate = model_config.output_cost_over_200k
            pricing_tier = "over_200k"
        else:
            input_rate = model_config.input_cost
            output_rate = model_config.output_cost
            pricing_tier = "standard"
        
        # Calculate costs
        input_cost = (input_tokens / 1_000_000) * input_rate
        output_cost = (output_tokens / 1_000_000) * output_rate
        total_cost = input_cost + output_cost
        
        # Format for display
        breakdown = {
            "model": model_name,
            "provider": model_config.provider.value,
            "pricing_tier": pricing_tier,
            "input_tokens": input_tokens,
            "output_tokens": output_tokens,
            "total_tokens": total_tokens,
            "input_rate_per_1M": input_rate,
            "output_rate_per_1M": output_rate,
            "input_cost": round(input_cost, 6),
            "output_cost": round(output_cost, 6),
            "total_cost": round(total_cost, 6),
            "token_breakdown": {
                "input_percentage": round((input_tokens / total_tokens) * 100, 1) if total_tokens > 0 else 0,
                "output_percentage": round((output_tokens / total_tokens) * 100, 1) if total_tokens > 0 else 0
            }
        }
        
        return breakdown

# ============================================================================
# Quality Evaluation Engine
# ============================================================================

class QualityEvaluator:
    """Advanced quality evaluation system with improved accuracy"""
    
    @staticmethod
    def evaluate(prompt: str, response: str, expected: Optional[str] = None) -> Dict[str, float]:
        """Evaluate response quality across multiple dimensions with improved accuracy"""
        scores = {}
        
        # Relevance Score - Improved keyword matching with semantic similarity
        prompt_words = set(prompt.lower().split())
        response_words = set(response.lower().split())
        
        # Remove common stop words for better relevance scoring
        stop_words = {'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might', 'can', 'this', 'that', 'these', 'those'}
        prompt_keywords = prompt_words - stop_words
        response_keywords = response_words - stop_words
        
        if prompt_keywords:
            relevance_score = len(prompt_keywords & response_keywords) / len(prompt_keywords)
            scores['relevance'] = min(1.0, relevance_score * 1.2)  # Boost relevance score
        else:
            scores['relevance'] = 0.5
        
        # Completeness Score - More nuanced based on prompt complexity
        prompt_complexity = len(prompt.split()) / 10  # Normalize by expected response length
        response_length = len(response.split())
        
        if prompt_complexity > 0:
            expected_length = prompt_complexity * 50  # Rough estimate of expected response length
            completeness = min(1.0, response_length / expected_length)
            scores['completeness'] = completeness
        else:
            scores['completeness'] = min(1.0, response_length / 100)
        
        # Clarity Score - Improved sentence structure analysis
        sentences = response.count('.') + response.count('!') + response.count('?')
        words_per_sentence = response_length / max(sentences, 1)
        
        # Optimal sentence length is 15-25 words
        if 15 <= words_per_sentence <= 25:
            clarity_score = 1.0
        elif 10 <= words_per_sentence <= 30:
            clarity_score = 0.8
        elif 5 <= words_per_sentence <= 40:
            clarity_score = 0.6
        else:
            clarity_score = 0.4
        
        scores['clarity'] = clarity_score
        
        # Accuracy Score - Enhanced comparison with expected output
        if expected:
            from difflib import SequenceMatcher
            # Use both exact match and fuzzy matching
            exact_match = response.lower().strip() == expected.lower().strip()
            fuzzy_similarity = SequenceMatcher(None, response.lower(), expected.lower()).ratio()
            
            if exact_match:
                scores['accuracy'] = 1.0
            else:
                # Weight fuzzy matching more heavily for partial matches
                scores['accuracy'] = fuzzy_similarity * 0.9 + 0.1
        else:
            # For responses without expected output, use content quality indicators
            has_structure = any(marker in response.lower() for marker in ['first', 'second', 'finally', 'therefore', 'however', 'moreover'])
            has_examples = any(marker in response.lower() for marker in ['for example', 'such as', 'like', 'including'])
            has_conclusion = any(marker in response.lower() for marker in ['in conclusion', 'to summarize', 'overall', 'in summary'])
            
            structure_score = sum([has_structure, has_examples, has_conclusion]) / 3
            scores['accuracy'] = 0.5 + (structure_score * 0.3)
        
        # Coherence Score - Enhanced logical flow detection
        coherence_markers = [
            'therefore', 'however', 'moreover', 'furthermore', 'because', 'since', 'as a result',
            'consequently', 'thus', 'hence', 'additionally', 'further', 'meanwhile', 'nevertheless',
            'nonetheless', 'in contrast', 'on the other hand', 'similarly', 'likewise'
        ]
        
        coherence_count = sum(1 for marker in coherence_markers if marker in response.lower())
        # Normalize by response length to avoid bias toward longer responses
        normalized_coherence = min(1.0, coherence_count / max(response_length / 50, 1))
        scores['coherence'] = normalized_coherence
        
        # Technical Accuracy Score (for technical/code content)
        if any(keyword in prompt.lower() for keyword in ['code', 'function', 'algorithm', 'technical', 'scientific']):
            # Check for technical indicators in response
            technical_indicators = [
                'def ', 'class ', 'import ', 'function', 'algorithm', 'complexity', 'efficiency',
                'optimization', 'implementation', 'methodology', 'framework', 'architecture'
            ]
            tech_score = sum(1 for indicator in technical_indicators if indicator in response.lower())
            scores['technical_accuracy'] = min(1.0, tech_score / 5)
        else:
            scores['technical_accuracy'] = 0.5
        
        # Overall Score - Weighted average with improved weights
        scores['overall'] = (
            scores['relevance'] * 0.25 +
            scores['completeness'] * 0.20 +
            scores['clarity'] * 0.15 +
            scores['accuracy'] * 0.25 +
            scores['coherence'] * 0.10 +
            scores['technical_accuracy'] * 0.05
        )
        
        # Ensure all scores are within valid range and rounded appropriately
        return {k: round(max(0.0, min(1.0, v)), 3) for k, v in scores.items()}

# ============================================================================
# Benchmarking Engine
# ============================================================================

class BenchmarkEngine:
    """Core benchmarking engine with advanced features"""
    
    def __init__(self):
        self.db = LocalDatabase()
        self.model_interface = ModelInterface()
        self.evaluator = QualityEvaluator()
    
    async def run_benchmark(self, test_cases: List[TestCase], models: List[str], 
                           benchmark_name: str = "Unnamed Benchmark") -> str:
        """Run comprehensive benchmark across models and test cases"""
        benchmark_id = hashlib.md5(f"{benchmark_name}_{datetime.now()}".encode()).hexdigest()
        
        # Save benchmark configuration
        self.db.save_benchmark(
            benchmark_id=benchmark_id,
            name=benchmark_name,
            description=f"Benchmark with {len(models)} models and {len(test_cases)} tests",
            config={"models": models, "test_cases": [asdict(tc) for tc in test_cases]},
            tags=[]
        )
        
        results = []
        total_tests = len(test_cases) * len(models)
        
        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            BarColumn(),
            TaskProgressColumn(),
            TimeRemainingColumn(),
            console=console
        ) as progress:
            task = progress.add_task(f"[cyan]Running {total_tests} tests...", total=total_tests)
            
            for test_case in test_cases:
                for model in models:
                    progress.update(task, description=f"[cyan]Testing {model} on '{test_case.name}'")
                    
                    try:
                        # Call model
                        response, input_tokens, output_tokens, cost, latency_ms = await self.model_interface.call_model(
                            model, test_case.prompt, test_case.max_tokens, test_case.temperature
                        )
                        
                        # Evaluate quality
                        quality_scores = self.evaluator.evaluate(
                            test_case.prompt, response, test_case.expected_output
                        )
                        
                        # Create result
                        result = BenchmarkResult(
                            model=model,
                            provider=ClyrdiaConfig.get_model(model).provider.value,
                            test_name=test_case.name,
                            prompt=test_case.prompt,
                            response=response,
                            latency_ms=latency_ms,
                            input_tokens=input_tokens,
                            output_tokens=output_tokens,
                            cost=cost,
                            success=True,
                            quality_scores=quality_scores
                        )
                        
                    except Exception as e:
                        result = BenchmarkResult(
                            model=model,
                            provider=ClyrdiaConfig.get_model(model).provider.value if ClyrdiaConfig.get_model(model) else "unknown",
                            test_name=test_case.name,
                            prompt=test_case.prompt,
                            response="",
                            latency_ms=0,
                            input_tokens=0,
                            output_tokens=0,
                            cost=0,
                            success=False,
                            error=str(e)
                        )
                    
                    results.append(result)
                    self.db.save_result(result, benchmark_id)
                    progress.advance(task)
        
        # Display results
        self._display_results(results)
        
        # Validate costs for transparency
        console.print(f"\n[bold]🔍 Cost Validation:[/bold]")
        for result in results:
            if result.success:
                is_valid, message = self.model_interface.validate_cost_calculation(
                    result.model, result.input_tokens, result.output_tokens, result.cost
                )
                status = "[green]✅[/green]" if is_valid else "[red]❌[/red]"
                console.print(f"  {status} {result.model}: {message}")
                
                # Show detailed cost breakdown for transparency
                cost_breakdown = self.model_interface.get_detailed_cost_breakdown(
                    result.model, result.input_tokens, result.output_tokens
                )
                if "error" not in cost_breakdown:
                    console.print(f"    [dim]💰 Cost breakdown: Input ${cost_breakdown['input_cost']:.6f} + Output ${cost_breakdown['output_cost']:.6f} = Total ${cost_breakdown['total_cost']:.6f}[/dim]")
                    console.print(f"    [dim]📊 Token split: {cost_breakdown['token_breakdown']['input_percentage']}% input, {cost_breakdown['token_breakdown']['output_percentage']}% output[/dim]")
        
        # Show API key setup instructions if any models failed due to missing keys
        self._show_api_key_setup_instructions(results)
        
        # Overall accuracy summary
        total_successful = sum(1 for r in results if r.success)
        total_tests = len(results)
        success_rate = (total_successful / total_tests) * 100 if total_tests > 0 else 0
        
        console.print(f"\n[bold]📊 Overall Accuracy Summary:[/bold]")
        console.print(f"  • Total tests: {total_tests}")
        console.print(f"  • Successful: {total_successful}")
        console.print(f"  • Success rate: {success_rate:.1f}%")
        
        if success_rate < 100:
            failed_models = [r.model for r in results if not r.success]
            console.print(f"  • Failed models: {', '.join(set(failed_models))}")
        
        return benchmark_id
    
    def _display_results(self, results: List[BenchmarkResult]):
        """Display benchmark results in a beautiful table"""
        # Group by model
        model_stats = {}
        for result in results:
            if result.model not in model_stats:
                model_stats[result.model] = {
                    'total_cost': 0,
                    'avg_latency': [],
                    'quality_scores': [],
                    'success_rate': [],
                    'total_tokens': 0
                }
            
            stats = model_stats[result.model]
            stats['total_cost'] += result.cost
            if result.success:
                stats['avg_latency'].append(result.latency_ms)
                stats['quality_scores'].append(result.quality_scores.get('overall', 0))
                stats['success_rate'].append(1)
                stats['total_tokens'] += result.input_tokens + result.output_tokens
            else:
                stats['success_rate'].append(0)
        
        # Create summary table
        table = Table(title="📊 Benchmark Results Summary", box=box.ROUNDED)
        table.add_column("Model", style="cyan", no_wrap=True)
        table.add_column("Success Rate", justify="center", style="green")
        table.add_column("Avg Quality", justify="center", style="yellow")
        table.add_column("Avg Latency", justify="center", style="blue")
        table.add_column("Total Cost", justify="right", style="red")
        table.add_column("Total Tokens", justify="right", style="magenta")
        
        # Sort by quality score
        sorted_models = sorted(model_stats.items(), 
                              key=lambda x: np.mean(x[1]['quality_scores']) if x[1]['quality_scores'] else 0, 
                              reverse=True)
        
        for model, stats in sorted_models:
            success_rate = np.mean(stats['success_rate']) * 100
            avg_quality = np.mean(stats['quality_scores']) if stats['quality_scores'] else 0
            avg_latency = np.mean(stats['avg_latency']) if stats['avg_latency'] else 0
            
            # Add quality indicator
            if avg_quality >= 0.8:
                quality_indicator = "🏆"
            elif avg_quality >= 0.6:
                quality_indicator = "✅"
            else:
                quality_indicator = "⚠️"
            
            table.add_row(
                model,
                f"{success_rate:.1f}%",
                f"{quality_indicator} {avg_quality:.3f}",
                f"{avg_latency:.0f}ms",
                f"${stats['total_cost']:.4f}",
                f"{stats['total_tokens']:,}"
            )
        
        console.print("\n")
        console.print(table)
        
        # Find winner
        if sorted_models:
            winner = sorted_models[0][0]
            console.print(f"\n🥇 [bold green]Best Overall Performance: {winner}[/bold green]")
    
    def _show_api_key_setup_instructions(self, results: List[BenchmarkResult]):
        """Show clear instructions for setting up missing API keys"""
        failed_models = [r for r in results if not r.success and any(
            key_phrase in str(r.error).lower() for key_phrase in [
                "api key not configured", "missing api key", "no api key"
            ]
        )]
        
        if failed_models:
            console.print(f"\n[bold red]🔑 API Key Setup Required[/bold red]")
            console.print("Some models failed because API keys are missing. Here's how to fix this:")
            
            console.print(f"\n[bold]📝 Create a .env file in your project directory:[/bold]")
            console.print("```bash")
            console.print("# .env file")
            
            if any("openai" in str(r.error).lower() for r in failed_models):
                console.print("OPENAI_API_KEY=your_openai_api_key_here")
            if any("anthropic" in str(r.error).lower() for r in failed_models):
                console.print("ANTHROPIC_API_KEY=your_anthropic_api_key_here")
            
            console.print("```")
            
            console.print(f"\n[bold]🔗 Get your API keys from:[/bold]")
            if any("openai" in str(r.error).lower() for r in failed_models):
                console.print("• OpenAI: https://platform.openai.com/api-keys")
            if any("anthropic" in str(r.error).lower() for r in failed_models):
                console.print("• Anthropic: https://console.anthropic.com/")
            
            console.print(f"\n[bold]💡 After adding API keys:[/bold]")
            console.print("• Restart your terminal or run: source .env")
            console.print("• Run your benchmark command again")
            console.print("• The CLI will automatically detect and use your API keys")
    
    async def run_benchmark_with_cache(self, test_cases: List[TestCase], models: List[str], 
                                       benchmark_name: str = "Unnamed Benchmark") -> str:
        """Run benchmark with caching enabled"""
        benchmark_id = hashlib.md5(f"{benchmark_name}_{datetime.now()}".encode()).hexdigest()
        
        # Save benchmark configuration
        self.db.save_benchmark(
            benchmark_id=benchmark_id,
            name=benchmark_name,
            description=f"Benchmark with {len(models)} models and {len(test_cases)} tests",
            config={"models": models, "test_cases": [asdict(tc) for tc in test_cases]},
            tags=[]
        )
        
        results = []
        total_tests = len(test_cases) * len(models)
        cache_manager = CacheManager()
        cached_count = 0
        
        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            BarColumn(),
            TaskProgressColumn(),
            TimeRemainingColumn(),
            console=console
        ) as progress:
            task = progress.add_task(f"[cyan]Running {total_tests} tests...", total=total_tests)
            
            for test_case in test_cases:
                for model in models:
                    progress.update(task, description=f"[cyan]Testing {model} on '{test_case.name}'")
                    
                    try:
                        # Check cache first
                        cached_result = cache_manager.get_cached_result(
                            model, test_case.name, test_case.prompt, 
                            test_case.max_tokens, test_case.temperature
                        )
                        
                        if cached_result:
                            # Use cached result
                            cached_count += 1
                            result = BenchmarkResult(
                                model=model,
                                provider=ClyrdiaConfig.get_model(model).provider.value,
                                test_name=test_case.name,
                                prompt=test_case.prompt,
                                response=cached_result.response,
                                latency_ms=1,  # Near-zero latency for cached results
                                input_tokens=cached_result.input_tokens,
                                output_tokens=cached_result.output_tokens,
                                cost=0.0,  # Cached results cost $0
                                success=True,
                                quality_scores=cached_result.quality_scores,
                                metadata={"cached": True, "cache_timestamp": cached_result.timestamp.isoformat()}
                            )
                        else:
                            # Call model and cache result
                            response, input_tokens, output_tokens, cost, latency_ms = await self.model_interface.call_model(
                                model, test_case.prompt, test_case.max_tokens, test_case.temperature
                            )
                            
                            # Evaluate quality
                            quality_scores = self.evaluator.evaluate(
                                test_case.prompt, response, test_case.expected_output
                            )
                            
                            # Create result
                            result = BenchmarkResult(
                                model=model,
                                provider=ClyrdiaConfig.get_model(model).provider.value,
                                test_name=test_case.name,
                                prompt=test_case.prompt,
                                response=response,
                                latency_ms=latency_ms,
                                input_tokens=input_tokens,
                                output_tokens=output_tokens,
                                cost=cost,
                                success=True,
                                quality_scores=quality_scores,
                                metadata={"cached": False}
                            )
                            
                            # Cache the result for future use
                            cache_manager.cache_result(
                                model, test_case.name, test_case.prompt, response,
                                input_tokens, output_tokens, quality_scores,
                                test_case.max_tokens, test_case.temperature, test_case.tags
                            )
                        
                    except Exception as e:
                        result = BenchmarkResult(
                            model=model,
                            provider=ClyrdiaConfig.get_model(model).provider.value if ClyrdiaConfig.get_model(model) else "unknown",
                            test_name=test_case.name,
                            prompt=test_case.prompt,
                            response="",
                            latency_ms=0,
                            input_tokens=0,
                            output_tokens=0,
                            cost=0,
                            success=False,
                            error=str(e)
                        )
                    
                    results.append(result)
                    self.db.save_result(result, benchmark_id)
                    progress.advance(task)
        
        # Display results with cache information
        self._display_results_with_cache(results, cached_count, total_tests)
        
        return benchmark_id
    
    def _display_results_with_cache(self, results: List[BenchmarkResult], cached_count: int, total_tests: int):
        """Display benchmark results with cache information"""
        # Group by model
        model_stats = {}
        for result in results:
            if result.model not in model_stats:
                model_stats[result.model] = {
                    'total_cost': 0,
                    'avg_latency': [],
                    'quality_scores': [],
                    'success_rate': [],
                    'total_tokens': 0,
                    'cached_tests': 0,
                    'live_tests': 0
                }
            
            stats = model_stats[result.model]
            stats['total_cost'] += result.cost
            if result.success:
                stats['avg_latency'].append(result.latency_ms)
                stats['quality_scores'].append(result.quality_scores.get('overall', 0))
                stats['success_rate'].append(1)
                stats['total_tokens'] += result.input_tokens + result.output_tokens
                
                # Track cache vs live
                if result.metadata.get('cached', False):
                    stats['cached_tests'] += 1
                else:
                    stats['live_tests'] += 1
            else:
                stats['success_rate'].append(0)
        
        # Create summary table
        table = Table(title="📊 Benchmark Results Summary (with Caching)", box=box.ROUNDED)
        table.add_column("Model", style="cyan", no_wrap=True)
        table.add_column("Status", justify="center", style="magenta")
        table.add_column("Success Rate", justify="center", style="green")
        table.add_column("Avg Quality", justify="center", style="yellow")
        table.add_column("Avg Latency", justify="center", style="blue")
        table.add_column("Total Cost", justify="right", style="red")
        table.add_column("Total Tokens", justify="right", style="magenta")
        
        # Sort by quality score
        sorted_models = sorted(model_stats.items(), 
                              key=lambda x: np.mean(x[1]['quality_scores']) if x[1]['quality_scores'] else 0, 
                              reverse=True)
        
        for model, stats in sorted_models:
            success_rate = np.mean(stats['success_rate']) * 100
            avg_quality = np.mean(stats['quality_scores']) if stats['quality_scores'] else 0
            avg_latency = np.mean(stats['avg_latency']) if stats['avg_latency'] else 0
            
            # Add quality indicator
            if avg_quality >= 0.8:
                quality_indicator = "🏆"
            elif avg_quality >= 0.6:
                quality_indicator = "✅"
            else:
                quality_indicator = "⚠️"
            
            # Status indicator (cached vs live)
            if stats['cached_tests'] > 0 and stats['live_tests'] > 0:
                status = f"[yellow]MIXED[/yellow] ({stats['cached_tests']}C/{stats['live_tests']}L)"
            elif stats['cached_tests'] > 0:
                status = "[green]CACHED[/green]"
            else:
                status = "[blue]LIVE[/blue]"
            
            table.add_row(
                model,
                status,
                f"{success_rate:.1f}%",
                f"{quality_indicator} {avg_quality:.3f}",
                f"{avg_latency:.0f}ms",
                f"${stats['total_cost']:.4f}",
                f"{stats['total_tokens']:,}"
            )
        
        console.print("\n")
        console.print(table)
        
        # Cache summary
        if cached_count > 0:
            cache_percentage = (cached_count / total_tests) * 100
            live_cost = sum(r.cost for r in results if not r.metadata.get('cached', False))
            
            console.print(f"\n[bold]📦 Cache Summary:[/bold]")
            console.print(f"  • {cached_count}/{total_tests} results served from cache ({cache_percentage:.1f}%)")
            console.print(f"  • Cached results cost: $0.00")
            console.print(f"  • Live results cost: ${live_cost:.4f}")
            console.print(f"  • Total cost for this run: ${live_cost:.4f}")
            
            if cache_percentage > 50:
                console.print(f"  • [green]💰 Significant cost savings achieved![/green]")
            elif cache_percentage > 20:
                console.print(f"  • [yellow]💰 Moderate cost savings achieved[/yellow]")
            else:
                console.print(f"  • [blue]💰 Small cost savings achieved[/blue]")
        else:
            console.print(f"\n[bold]📦 Cache Summary:[/bold]")
            console.print(f"  • No cached results found - all tests ran live")
            console.print(f"  • Total cost: ${sum(r.cost for r in results):.4f}")
            console.print(f"  • [dim]💡 Future runs with --use-cache will cache these results[/dim]")
        
        # Find winner
        if sorted_models:
            winner = sorted_models[0][0]
            console.print(f"\n🥇 [bold green]Best Overall Performance: {winner}[/bold green]")

# ============================================================================
# Ratchet System (Performance Regression Detection)
# ============================================================================

class RatchetSystem:
    """Ensure performance never regresses"""
    
    def __init__(self):
        self.db = LocalDatabase()
    
    def save_baseline(self, model: str, metrics: Dict[str, float], name: str = "default"):
        """Save performance baseline"""
        with sqlite3.connect(self.db.db_path) as conn:
            conn.execute(
                """INSERT OR REPLACE INTO baselines (name, model, metrics, updated_at) 
                   VALUES (?, ?, ?, CURRENT_TIMESTAMP)""",
                (f"{name}_{model}", model, json.dumps(metrics))
            )
        console.print(f"[green]✅ Baseline saved for {model}[/green]")
    
    def check_regression(self, model: str, current_metrics: Dict[str, float], 
                        name: str = "default") -> Tuple[bool, List[str]]:
        """Check if current performance regresses from baseline"""
        with sqlite3.connect(self.db.db_path) as conn:
            cursor = conn.execute(
                "SELECT metrics FROM baselines WHERE name = ?",
                (f"{name}_{model}",)
            )
            row = cursor.fetchone()
        
        if not row:
            return True, ["No baseline found"]
        
        baseline = json.loads(row[0])
        regressions = []
        
        # Check each metric
        for metric, current_value in current_metrics.items():
            if metric in baseline:
                baseline_value = baseline[metric]
                
                # For latency, lower is better
                if metric == 'latency' and current_value > baseline_value * 1.1:
                    regressions.append(f"Latency regression: {current_value:.0f}ms > {baseline_value:.0f}ms")
                
                # For quality, higher is better
                elif metric == 'quality' and current_value < baseline_value * 0.95:
                    regressions.append(f"Quality regression: {current_value:.3f} < {baseline_value:.3f}")
                
                # For cost, lower is better
                elif metric == 'cost' and current_value > baseline_value * 1.2:
                    regressions.append(f"Cost regression: ${current_value:.4f} > ${baseline_value:.4f}")
        
        return len(regressions) == 0, regressions

# ============================================================================
# Canary System (Drift Detection)
# ============================================================================

class CanarySystem:
    """Detect model behavior changes and drift"""
    
    def __init__(self):
        self.db = LocalDatabase()
        self.model_interface = ModelInterface()
    
    async def run_canary_test(self, model: str, test_prompts: List[str], 
                             baseline_responses: Optional[Dict[str, str]] = None) -> Dict[str, Any]:
        """Run canary tests to detect drift"""
        drift_detected = False
        drift_details = []
        
        for prompt in test_prompts:
            try:
                # Get current response
                response, _, _, _, _ = await self.model_interface.call_model(model, prompt)
                
                # Calculate test hash
                test_hash = hashlib.md5(f"{model}_{prompt}".encode()).hexdigest()
                
                # Compare with baseline if available
                if baseline_responses and prompt in baseline_responses:
                    baseline = baseline_responses[prompt]
                    
                    # Calculate similarity
                    from difflib import SequenceMatcher
                    similarity = SequenceMatcher(None, response.lower(), baseline.lower()).ratio()
                    
                    if similarity < 0.8:  # Significant change detected
                        drift_detected = True
                        drift_score = 1 - similarity
                        
                        drift_details.append({
                            'prompt': prompt[:100] + '...' if len(prompt) > 100 else prompt,
                            'drift_score': drift_score,
                            'similarity': similarity
                        })
                        
                        # Save to database
                        with sqlite3.connect(self.db.db_path) as conn:
                            conn.execute(
                                "INSERT INTO drift_history (model, test_hash, drift_score, details) VALUES (?, ?, ?, ?)",
                                (model, test_hash, drift_score, json.dumps({'prompt': prompt, 'similarity': similarity}))
                            )
                
            except Exception as e:
                drift_details.append({
                    'prompt': prompt[:100] + '...' if len(prompt) > 100 else prompt,
                    'error': str(e)
                })
        
        return {
            'model': model,
            'drift_detected': drift_detected,
            'details': drift_details,
            'timestamp': datetime.now().isoformat()
        }

# ============================================================================
# CLI Commands
# ============================================================================

@app.command()
@require_auth
def init():
    """🚀 Initialize Clyrdia configuration and create sample benchmark
    
    This command will:
    1. Set up your local environment
    2. Create a sample benchmark configuration
    3. Check your API key setup
    4. Guide you through next steps
    """
    # Show credit balance at start
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    # Check if this is truly the first time
    config_dir = Path.home() / ".clyrdia"
    if not config_dir.exists():
        # Show the beautiful welcome screen for first-time users
        _display_welcome_screen()
        console.print("[bold green]🎉 Welcome to Clyrdia! Let's set up your environment...[/bold green]\n")
    else:
        console.print(Panel.fit(
            "[bold cyan]🚀 Initializing Clyrdia[/bold cyan]\n"
            "Zero-Knowledge AI Benchmarking Platform",
            border_style="cyan",
            padding=(1, 2),
            title="Initialization",
            title_align="center"
        ))
    
    # Create config directory
    config_dir = Path.home() / ".clyrdia"
    config_dir.mkdir(parents=True, exist_ok=True)
    
    # Create sample benchmark file
    sample_benchmark = {
        'name': 'Sample Benchmark',
        'description': 'Compare AI models on various tasks',
                    'models': ['gpt-4o-mini', 'claude-sonnet-4'],
        'test_cases': [
            {
                'name': 'Code Generation',
                'prompt': 'Write a Python function to calculate the fibonacci sequence',
                'max_tokens': 500,
                'temperature': 0.3,
                'expected_output': 'def fibonacci'
            },
            {
                'name': 'Creative Writing',
                'prompt': 'Write a haiku about artificial intelligence',
                'max_tokens': 100,
                'temperature': 0.8
            },
            {
                'name': 'Analysis',
                'prompt': 'Explain the difference between supervised and unsupervised learning',
                'max_tokens': 300,
                'temperature': 0.5
            }
        ]
    }
    
    benchmark_file = Path("benchmark.yaml")
    with open(benchmark_file, 'w') as f:
        yaml.dump(sample_benchmark, f, default_flow_style=False)
    
    console.print(f"[green]✅ Created sample benchmark: {benchmark_file}[/green]")
    
    # Check for API keys
    console.print("\n[yellow]Checking API keys...[/yellow]")
    
    api_keys = {
        'OPENAI_API_KEY': '🤖 OpenAI',
        'ANTHROPIC_API_KEY': '🧠 Anthropic',
        
    }
    
    configured = []
    missing = []
    
    for key, name in api_keys.items():
        if os.getenv(key):
            configured.append(name)
        else:
            missing.append(name)
    
    if configured:
        console.print(f"[green]✅ Configured: {', '.join(configured)}[/green]")
    
    if missing:
        console.print(f"[red]❌ Missing: {', '.join(missing)}[/red]")
        console.print("\n[dim]Set API keys in your environment or .env file[/dim]")
        console.print("You can still run benchmarks with the models you have configured!")
    
    console.print("\n[bold cyan]🎯 Next steps:[/bold cyan]")
    console.print("1. Edit benchmark.yaml to customize your tests")
    console.print("2. Run: [bold]clyrdia-cli benchmark benchmark.yaml[/bold]")
    console.print("3. View results: [bold]clyrdia-cli dashboard[/bold]")
    console.print("4. Learn more: [bold]clyrdia-cli tutorial[/bold]")
    
    console.print("\n[bold]💡 Dual-Mode Workflow:[/bold]")
    console.print("• [bold]Production Mode:[/bold] clyrdia-cli benchmark file.yaml")
    console.print("  - 100% accuracy, live APIs, perfect for CI/CD & canary tests")
    console.print("  - Use when you need the truth before deployment")
    console.print("  - Higher credit cost but guaranteed fresh results")
    
    console.print("• [bold]Developer Mode:[/bold] clyrdia-cli benchmark file.yaml --use-cache")
    console.print("  - Smart caching, cost savings, perfect for rapid iteration")
    console.print("  - Use when tweaking prompts during development")
    console.print("  - Lower credit cost with intelligent result reuse")
    
    console.print("\n[bold]📦 Cache Management:[/bold]")
    console.print("• Check status: [bold]clyrdia-cli cache status[/bold]")
    console.print("• Clear cache: [bold]clyrdia-cli cache clear[/bold]")
    console.print("• Model-specific: [bold]clyrdia-cli cache clear --model gpt-4o[/bold]")
    
    console.print("\n[bold]🔧 CI/CD Integration:[/bold]")
    console.print("• Perfect for automated testing in deployment pipelines")
    console.print("• Use environment variables for API keys")
    console.print("• Run [bold]clyrdia-cli tutorial[/bold] for detailed examples")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def benchmark(
    config_file: str = typer.Argument("benchmark.yaml", help="Benchmark configuration file"),
    models: Optional[List[str]] = typer.Option(None, "--model", "-m", help="Override models to test"),
    save: bool = typer.Option(True, "--save/--no-save", help="Save results to database"),
    compare: bool = typer.Option(False, "--compare", "-c", help="Show comparison after benchmark"),
    use_cache: bool = typer.Option(False, "--use-cache", help="Use cached results when available")
):
    """Run AI model benchmarks"""
    
    # Load configuration
    if not Path(config_file).exists():
        console.print(f"[red]❌ Configuration file not found: {config_file}[/red]")
        console.print("[yellow]Run 'clyrdia-cli init' to create a sample configuration[/yellow]")
        raise typer.Exit(1)
    
    with open(config_file, 'r') as f:
        config = yaml.safe_load(f)
    
    # Override models if specified
    if models:
        config['models'] = models
    
    # Validate models
    invalid_models = [m for m in config['models'] if m not in ClyrdiaConfig.list_models()]
    if invalid_models:
        console.print(f"[red]❌ Unknown models: {', '.join(invalid_models)}[/red]")
        console.print(f"[yellow]Available models: {', '.join(ClyrdiaConfig.list_models())}[/yellow]")
        raise typer.Exit(1)
    
    # Create test cases
    test_cases = []
    for tc in config.get('test_cases', []):
        test_cases.append(TestCase(
            name=tc['name'],
            prompt=tc['prompt'],
            expected_output=tc.get('expected_output'),
            max_tokens=tc.get('max_tokens', 1000),
            temperature=tc.get('temperature', 0.7)
        ))
    
    # ============================================================================
    # Credit Estimation & Licensing Integration
    # ============================================================================
    
    licensing_manager = LicensingManager()
    credit_estimate = None
    
    if licensing_manager.is_authenticated():
        # Estimate credit cost before running
        credit_estimate = asyncio.run(licensing_manager.estimate_credits(
            test_cases, config['models'], use_cache
        ))
        
        console.print(Panel.fit(
            "[bold cyan]💰 Credit Estimation & Cost Analysis[/bold cyan]",
            border_style="cyan",
            padding=(1, 2),
            title="Credits",
            title_align="center"
        ))
        
        # Show detailed breakdown
        console.print(f"[bold]📊 Test Breakdown:[/bold]")
        console.print(f"• Total Tests: {credit_estimate.total_tests}")
        console.print(f"• Cache Hits: {credit_estimate.cache_hits} (0 credits)")
        console.print(f"• Live API Calls: {credit_estimate.live_api_calls}")
        console.print(f"• Estimated credit cost: [bold]{credit_estimate.estimated_credits}[/bold] credits")
        console.print(f"• Your current balance: [bold]{credit_estimate.current_balance}[/bold] credits")
        
        # Show cost savings from caching
        if use_cache and credit_estimate.cache_hits > 0:
            savings_percent = (credit_estimate.cache_hits / credit_estimate.total_tests) * 100
            console.print(f"\n[bold]💰 Cache Savings:[/bold]")
            console.print(f"• Cached results: {credit_estimate.cache_hits} tests")
            console.print(f"• Savings: {savings_percent:.1f}% of total tests")
            console.print(f"• [green]Smart caching is saving you credits![/green]")
        
        # Check if user has enough credits
        if credit_estimate.estimated_credits > credit_estimate.current_balance:
            console.print(f"\n[red]⚠️  Insufficient credits![/red]")
            console.print(f"You need {credit_estimate.estimated_credits - credit_estimate.current_balance} more credits")
            
            if credit_estimate.current_balance == 0:
                console.print("\n[bold]🔑 Get your free credits:[/bold]")
                console.print("• Visit [bold]https://clyrdia.com[/bold] to sign up")
                console.print("• Free plan includes 250 credits/month")
                console.print("• No credit card required")
            else:
                console.print("\n[bold]🚀 Upgrade to Pro:[/bold]")
                console.print("• Get 10,000 credits/month")
                console.print("• Priority support and advanced features")
                console.print("• Visit [bold]https://clyrdia.com[/bold] to upgrade")
            
            if not Confirm.ask("Continue anyway? (will fail if credits insufficient)"):
                console.print("[dim]Benchmark cancelled[/dim]")
                raise typer.Exit(0)
        else:
            # Show cost breakdown by model and test
            console.print(f"\n[bold]📋 Cost Breakdown:[/bold]")
            for test_key, cost in credit_estimate.test_breakdown.items():
                if cost > 0:  # Only show tests that cost credits
                    test_name, model_name = test_key.rsplit('_', 1)
                    console.print(f"  • {test_name} ({model_name}): {cost} credits")
            
            console.print(f"\n[bold]💡 Cost Optimization Tips:[/bold]")
            if use_cache:
                console.print("• [green]Caching enabled[/green] - you're already optimizing!")
                console.print("• Consider using smaller models for development")
                console.print("• Use --use-cache for repeated tests")
            else:
                console.print("• [yellow]Caching disabled[/yellow] - consider --use-cache for development")
                console.print("• Use smaller models for initial testing")
                console.print("• Cache results for cost savings")
            
            if not Confirm.ask("Continue with this benchmark?"):
                console.print("[dim]Benchmark cancelled[/dim]")
                raise typer.Exit(0)
    else:
        console.print(Panel.fit(
            "[bold yellow]🔑 No License Configured[/bold yellow]\n"
            "Running in offline mode - no credit tracking",
            border_style="yellow",
            padding=(1, 2),
            title="Offline Mode",
            title_align="center"
        ))
        console.print("\n[bold]To enable credit tracking and get free credits:[/bold]")
        console.print("  • Run [bold]clyrdia-cli login[/bold] to connect your account")
        console.print("  • Get 250 free credits at [bold]https://clyrdia.com[/bold]")
        console.print("  • No credit card required")
        
        if not Confirm.ask("Continue without credit tracking?"):
            console.print("[dim]Benchmark cancelled[/dim]")
            raise typer.Exit(0)
    
    # Display benchmark info
    cache_status = "[yellow]CACHE ENABLED[/yellow]" if use_cache else "[dim]No caching[/dim]"
    
    if use_cache:
        # Show cache hit rate information
        cache_manager = CacheManager()
        cache_stats = cache_manager.get_cache_hit_rate(test_cases, config['models'])
        hit_rate_percent = cache_stats['hit_rate'] * 100
        
        cache_info = f"Cache: {cache_status}\n"
        cache_info += f"Potential savings: {cache_stats['cacheable_tests']}/{cache_stats['total_tests']} tests ({hit_rate_percent:.1f}%)"
        
        console.print(Panel.fit(
            f"[bold cyan]🏁 Starting Benchmark (Developer Mode)[/bold cyan]\n"
            f"Name: {config.get('name', 'Unnamed')}\n"
            f"Models: {', '.join(config['models'])}\n"
            f"Tests: {len(test_cases)}\n"
            f"{cache_info}",
            border_style="cyan",
            padding=(1, 2),

            title="Developer Mode",
            title_align="center"
        ))
        
        console.print("\n[dim]💡 Developer Mode: Cached results cost $0 and have near-zero latency[/dim]")
        console.print("[dim]   Use this mode for rapid iteration and cost savings during development[/dim]")
    else:
        console.print(Panel.fit(
            f"[bold cyan]🏁 Starting Benchmark (Production Mode)[/bold cyan]\n"
            f"Name: {config.get('name', 'Unnamed')}\n"
            f"Models: {', '.join(config['models'])}\n"
            f"Tests: {len(test_cases)}\n"
            f"Cache: {cache_status}",
            border_style="cyan",
            padding=(1, 2),

            title="Production Mode",
            title_align="center"
        ))
        
        console.print("\n[dim]🔒 Production Mode: All tests run against live APIs for maximum accuracy[/dim]")
        console.print("[dim]   Use this mode for CI/CD, canary tests, and final validation[/dim]")
    
    # Show credit balance before running benchmark
    console.print("\n" + "="*50)
    console.print("[bold]💰 Pre-Benchmark Credit Check[/bold]")
    before_credits = licensing_manager.show_credit_balance()
    console.print("="*50)
    
    # Run benchmark with caching if enabled
    engine = BenchmarkEngine()
    if use_cache:
        benchmark_id = asyncio.run(engine.run_benchmark_with_cache(
            test_cases=test_cases,
            models=config['models'],
            benchmark_name=config.get('name', 'Unnamed Benchmark')
        ))
    else:
        benchmark_id = asyncio.run(engine.run_benchmark(
            test_cases=test_cases,
            models=config['models'],
            benchmark_name=config.get('name', 'Unnamed Benchmark')
        ))
    
    # Show final credit balance and usage summary
    console.print("\n" + "="*50)
    console.print("[bold]💰 Post-Benchmark Credit Summary[/bold]")
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)
    
    console.print(f"\n[green]✅ Benchmark completed![/green]")
    console.print(f"[dim]Benchmark ID: {benchmark_id}[/dim]")
    
    # ============================================================================
    # Post-Run Credit Debit
    # ============================================================================
    
    if licensing_manager.is_authenticated() and credit_estimate:
        try:
            # Calculate actual credits used (only for live API calls)
            actual_credits = credit_estimate.estimated_credits
            
            # Generate unique run ID for idempotency
            run_id = str(uuid.uuid4())
            
            # Debit credits from account
            debit_result = asyncio.run(licensing_manager.debit_credits(actual_credits, run_id))
            
            if debit_result.get("success"):
                console.print(f"\n[green]💰 Credits debited successfully![/green]")
                console.print(f"Credits used: [bold]{actual_credits}[/bold]")
                console.print(f"New balance: [bold]{debit_result['credits_remaining']}[/bold]")
            else:
                console.print(f"\n[yellow]⚠️  Credit debit failed[/yellow]")
                console.print(f"Error: {debit_result.get('message', 'Unknown error')}")
                console.print(f"Credits remaining: {debit_result.get('credits_remaining', 'Unknown')}")
                
                if debit_result.get('error_code') == 'insufficient_credits':
                    console.print("\n[bold]💡 Upgrade your plan:[/bold]")
                    console.print("Visit [bold]https://clyrdia.com[/bold] for more credits")
        
        except Exception as e:
            console.print(f"\n[yellow]⚠️  Credit tracking failed: {str(e)}[/yellow]")
            console.print("Benchmark results saved, but credits not debited")
            console.print("Contact support if this persists")
    
    if compare:
        # Show detailed comparison
        _show_comparison(benchmark_id)

@app.command()
@require_auth
def ratchet(
    model: str = typer.Argument(..., help="Model name to set baseline for"),
    update: bool = typer.Option(False, "--update", "-u", help="Update baseline with current performance"),
    check: bool = typer.Option(False, "--check", "-c", help="Check for regressions")
):
    """Manage performance baselines (ratchet system)"""
    # Show credit balance before ratchet operations
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    ratchet_system = RatchetSystem()
    db = LocalDatabase()
    
    if update:
        # Get recent performance metrics
        with sqlite3.connect(db.db_path) as conn:
            cursor = conn.execute(
                """SELECT AVG(latency_ms) as avg_latency, AVG(cost) as avg_cost, 
                          AVG(json_extract(quality_scores, '$.overall')) as avg_quality
                   FROM results 
                   WHERE model = ? AND success = 1
                   ORDER BY timestamp DESC LIMIT 10""",
                (model,)
            )
            row = cursor.fetchone()
        
        if row and row[0]:
            metrics = {
                'latency': row[0],
                'cost': row[1],
                'quality': row[2] or 0.5
            }
            
            ratchet_system.save_baseline(model, metrics)
            
            console.print(Panel.fit(
                f"[green]✅ Baseline Updated[/green]\n"
                f"Model: {model}\n"
                f"Latency: {metrics['latency']:.0f}ms\n"
                f"Cost: ${metrics['cost']:.4f}\n"
                f"Quality: {metrics['quality']:.3f}",
                border_style="green"
            ))
        else:
            console.print(f"[red]❌ No performance data found for {model}[/red]")
    
    elif check:
        # Check for regressions
        with sqlite3.connect(db.db_path) as conn:
            cursor = conn.execute(
                """SELECT AVG(latency_ms) as avg_latency, AVG(cost) as avg_cost,
                          AVG(json_extract(quality_scores, '$.overall')) as avg_quality
                   FROM results 
                   WHERE model = ? AND success = 1
                   ORDER BY timestamp DESC LIMIT 5""",
                (model,)
            )
            row = cursor.fetchone()
        
        if row and row[0]:
            current_metrics = {
                'latency': row[0],
                'cost': row[1],
                'quality': row[2] or 0.5
            }
            
            passed, regressions = ratchet_system.check_regression(model, current_metrics)
            
            if passed:
                console.print(f"[green]✅ No regressions detected for {model}[/green]")
            else:
                console.print(f"[red]❌ Regressions detected for {model}:[/red]")
                for regression in regressions:
                    console.print(f"  [red]• {regression}[/red]")
        else:
            console.print(f"[red]❌ No performance data found for {model}[/red]")
    
    else:
        console.print("[yellow]Use --update to save baseline or --check to test for regressions[/yellow]")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def canary(
    model: str = typer.Argument(..., help="Model to test for drift"),
    prompts_file: Optional[str] = typer.Option(None, "--prompts", "-p", help="File with test prompts"),
    baseline_file: Optional[str] = typer.Option(None, "--baseline", "-b", help="Baseline responses file")
):
    """Run canary tests to detect model drift"""
    # Show credit balance before canary tests
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    canary_system = CanarySystem()
    
    # Default test prompts
    test_prompts = [
        "What is 2+2?",
        "Complete this sentence: The capital of France is",
        "Is water wet? Answer yes or no.",
        "What color is the sky on a clear day?"
    ]
    
    # Load custom prompts if provided
    if prompts_file and Path(prompts_file).exists():
        with open(prompts_file, 'r') as f:
            test_prompts = [line.strip() for line in f if line.strip()]
    
    # Load baseline if provided
    baseline_responses = None
    if baseline_file and Path(baseline_file).exists():
        with open(baseline_file, 'r') as f:
            baseline_responses = json.load(f)
    
    console.print(Panel.fit(
        f"[bold cyan]🐤 Running Canary Tests[/bold cyan]\n"
        f"Model: {model}\n"
        f"Test prompts: {len(test_prompts)}",
        border_style="cyan"
    ))
    
    # Run canary tests
    results = asyncio.run(canary_system.run_canary_test(model, test_prompts, baseline_responses))
    
    if results['drift_detected']:
        console.print(f"\n[red]⚠️  Drift detected for {model}![/red]")
        
        table = Table(title="Drift Details", box=box.ROUNDED)
        table.add_column("Prompt", style="cyan")
        table.add_column("Drift Score", justify="center", style="red")
        table.add_column("Similarity", justify="center", style="yellow")
        
        for detail in results['details']:
            if 'drift_score' in detail:
                table.add_row(
                    detail['prompt'],
                    f"{detail['drift_score']:.3f}",
                    f"{detail['similarity']:.3f}"
                )
        
        console.print(table)
    else:
        console.print(f"\n[green]✅ No significant drift detected for {model}[/green]")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)


def _show_comparison(benchmark_id: str):
    """Show detailed model comparison"""
    db = LocalDatabase()
    
    with sqlite3.connect(db.db_path) as conn:
        query = """
            SELECT model, test_name, latency_ms, cost, quality_scores
            FROM results
            WHERE benchmark_id = ?
            ORDER BY model, test_name
        """
        df = pd.read_sql_query(query, conn, params=(benchmark_id,))
    
    if df.empty:
        console.print("[yellow]No results found for comparison[/yellow]")
        return
    
    # Parse quality scores
    df['quality'] = df['quality_scores'].apply(lambda x: json.loads(x).get('overall', 0) if x else 0)
    
    # Create comparison table
    table = Table(title="🔍 Detailed Model Comparison", box=box.DOUBLE_EDGE)
    table.add_column("Test", style="cyan")
    
    models = df['model'].unique()
    for model in models:
        table.add_column(model, justify="center")
    
    tests = df['test_name'].unique()
    for test in tests:
        row = [test]
        for model in models:
            data = df[(df['model'] == model) & (df['test_name'] == test)]
            if not data.empty:
                quality = data.iloc[0]['quality']
                latency = data.iloc[0]['latency_ms']
                cost = data.iloc[0]['cost']
                
                # Format with color coding
                if quality >= 0.8:
                    quality_str = f"[green]{quality:.2f}[/green]"
                elif quality >= 0.6:
                    quality_str = f"[yellow]{quality:.2f}[/yellow]"
                else:
                    quality_str = f"[red]{quality:.2f}[/red]"
                
                row.append(f"{quality_str}\n{latency}ms\n${cost:.4f}")
            else:
                row.append("N/A")
        
        table.add_row(*row)
    
    console.print("\n")
    console.print(table)

@app.command()
@require_auth
def models(
    provider: Optional[str] = typer.Option(None, "--provider", "-p", help="Filter by provider"),
    show_pricing: bool = typer.Option(False, "--pricing", help="Show detailed pricing"),
    show_capabilities: bool = typer.Option(False, "--capabilities", help="Show capabilities")
):
    """List available AI models"""
    
    # Show credit balance before listing models
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    models_to_show = ClyrdiaConfig.MODELS.values()
    
    # Filter by provider
    if provider:
        provider_enum = ModelProvider(provider.lower())
        models_to_show = [m for m in models_to_show if m.provider == provider_enum]
    
    if show_pricing:
        # Detailed pricing table
        table = Table(title="💰 Model Pricing (per 1M tokens)", box=box.ROUNDED)
        table.add_column("Model", style="cyan")
        table.add_column("Provider", style="yellow")
        table.add_column("Tier", justify="center", style="magenta")
        table.add_column("Input", justify="right", style="green")
        table.add_column("Output", justify="right", style="green")
        table.add_column("1M In + 1M Out", justify="right", style="red")
        
        for model in sorted(models_to_show, key=lambda x: x.input_cost + x.output_cost):
            total = model.input_cost + model.output_cost
            
            # Add tier emoji and color
            tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
            tier_color = {"flagship": "red", "balanced": "yellow", "speed_cost": "green"}
            
            tier_display = f"{tier_emoji.get(model.tier, '❓')} {model.tier.replace('_', ' ').title()}"
            
            table.add_row(
                model.name,
                model.provider.value.capitalize(),
                f"[{tier_color.get(model.tier, 'white')}]{tier_display}[/{tier_color.get(model.tier, 'white')}]",
                f"${model.input_cost:.6f}",
                f"${model.output_cost:.6f}",
                f"${total:.6f}"
            )
    
    elif show_capabilities:
        # Capabilities table
        table = Table(title="🎯 Model Capabilities", box=box.ROUNDED)
        table.add_column("Model", style="cyan")
        table.add_column("Tier", justify="center", style="magenta")
        table.add_column("Context", justify="right", style="yellow")
        table.add_column("Speed", justify="center", style="green")
        table.add_column("Capabilities", style="blue")
        
        for model in models_to_show:
            # Format context window
            if model.context_window >= 1000000:
                context = f"{model.context_window // 1000000}M"
            else:
                context = f"{model.context_window // 1000}K"
            
            # Add tier emoji and color
            tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
            tier_color = {"flagship": "red", "balanced": "yellow", "speed_cost": "green"}
            
            tier_display = f"{tier_emoji.get(model.tier, '❓')} {model.tier.replace('_', ' ').title()}"
            
            table.add_row(
                model.name,
                f"[{tier_color.get(model.tier, 'white')}]{tier_display}[/{tier_color.get(model.tier, 'white')}]",
                context,
                model.speed_tier,
                ", ".join(model.capabilities)
            )
    
    else:
        # Simple list grouped by tier
        table = Table(title="🤖 Available Models by Tier", box=box.ROUNDED)
        table.add_column("Tier", style="magenta")
        table.add_column("Description", style="yellow")
        table.add_column("Models", style="cyan")
        
        # Group by tier
        tier_groups = {
            "flagship": {
                "description": "🏆 Highest quality, best capabilities",
                "models": []
            },
            "balanced": {
                "description": "⚖️ Good balance of quality and cost",
                "models": []
            },
            "speed_cost": {
                "description": "🚀 Fast and cost-effective",
                "models": []
            }
        }
        
        for model in models_to_show:
            tier_groups[model.tier]["models"].append(model.name)
        
        for tier, info in tier_groups.items():
            if info["models"]:  # Only show tiers that have models
                tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
                tier_color = {"flagship": "red", "balanced": "yellow", "speed_cost": "green"}
                
                tier_display = f"{tier_emoji.get(tier, '❓')} {tier.replace('_', ' ').title()}"
                
                table.add_row(
                    f"[{tier_color.get(tier, 'white')}]{tier_display}[/{tier_color.get(tier, 'white')}]",
                    info["description"],
                    "\n".join(sorted(info["models"]))
                )
    
    console.print(table)
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def export(
    benchmark_id: Optional[str] = typer.Argument(None, help="Benchmark ID to export"),
    format: str = typer.Option("json", "--format", "-f", help="Export format (json, csv, yaml)"),
    output: Optional[str] = typer.Option(None, "--output", "-o", help="Output file path")
):
    """Export benchmark results"""
    # Show credit balance before export
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    db = LocalDatabase()
    
    # Get data
    with sqlite3.connect(db.db_path) as conn:
        if benchmark_id:
            query = "SELECT * FROM results WHERE benchmark_id = ?"
            df = pd.read_sql_query(query, conn, params=(benchmark_id,))
        else:
            query = "SELECT * FROM results ORDER BY timestamp DESC LIMIT 100"
            df = pd.read_sql_query(query, conn)
    
    if df.empty:
        console.print("[red]No results found to export[/red]")
        return
    
    # Determine output file
    if not output:
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        output = f"clyrdia_export_{timestamp}.{format}"
    
    # Export based on format
    if format == "csv":
        df.to_csv(output, index=False)
    elif format == "json":
        df.to_json(output, orient='records', indent=2)
    elif format == "yaml":
        data = df.to_dict(orient='records')
        with open(output, 'w') as f:
            yaml.dump(data, f, default_flow_style=False)
    else:
        console.print(f"[red]Unsupported format: {format}[/red]")
        return
    
    console.print(f"[green]✅ Exported {len(df)} results to {output}[/green]")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def analyze(
    model: Optional[str] = typer.Option(None, "--model", "-m", help="Analyze specific model"),
    days: int = typer.Option(7, "--days", "-d", help="Number of days to analyze"),
    metric: str = typer.Option("all", "--metric", help="Metric to analyze (cost, latency, quality, all)")
):
    """Analyze benchmark trends and patterns"""
    # Show credit balance before analysis
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    db = LocalDatabase()
    
    with sqlite3.connect(db.db_path) as conn:
        if model:
            query = f"""
                SELECT * FROM results 
                WHERE model = ? AND timestamp > datetime('now', '-{days} days')
                ORDER BY timestamp
            """
            df = pd.read_sql_query(query, conn, params=(model,))
        else:
            query = f"""
                SELECT * FROM results 
                WHERE timestamp > datetime('now', '-{days} days')
                ORDER BY timestamp
            """
            df = pd.read_sql_query(query, conn)
    
    if df.empty:
        console.print("[yellow]No data available for analysis[/yellow]")
        return
    
    # Parse quality scores
    df['quality'] = df['quality_scores'].apply(
        lambda x: json.loads(x).get('overall', 0) if pd.notna(x) else 0
    )
    
    console.print(Panel.fit(
        f"[bold cyan]📊 Performance Analysis[/bold cyan]\n"
        f"Period: Last {days} days\n"
        f"Total tests: {len(df)}",
        border_style="cyan"
    ))
    
    if metric in ['cost', 'all']:
        # Cost analysis with detailed breakdown
        total_cost = df['cost'].sum()
        avg_cost = df['cost'].mean()
        
        console.print(f"\n💰 [bold]Cost Analysis[/bold]")
        console.print(f"  Total: ${total_cost:.6f}")
        console.print(f"  Average: ${avg_cost:.6f}")
        
        if not model:
            cost_by_model = df.groupby('model')['cost'].agg(['sum', 'mean', 'count']).sort_values('sum', ascending=False)
            console.print(f"  Most expensive: {cost_by_model.index[0]} (${cost_by_model.iloc[0]['sum']:.6f})")
            console.print(f"  Least expensive: {cost_by_model.index[-1]} (${cost_by_model.iloc[-1]['sum']:.6f})")
            
            # Show cost breakdown by model
            console.print(f"\n  [bold]Cost Breakdown by Model:[/bold]")
            for model_name, stats in cost_by_model.iterrows():
                console.print(f"    • {model_name}: ${stats['sum']:.6f} total, ${stats['mean']:.6f} avg, {stats['count']} tests")
        
        # Show cost distribution
        if len(df) > 1:
            cost_std = df['cost'].std()
            cost_min = df['cost'].min()
            cost_max = df['cost'].max()
            console.print(f"  Cost range: ${cost_min:.6f} - ${cost_max:.6f}")
            console.print(f"  Cost std dev: ${cost_std:.6f}")
    
    if metric in ['latency', 'all']:
        # Latency analysis
        successful = df[df['success'] == True]
        if not successful.empty:
            avg_latency = successful['latency_ms'].mean()
            p95_latency = successful['latency_ms'].quantile(0.95)
            
            console.print(f"\n⚡ [bold]Latency Analysis[/bold]")
            console.print(f"  Average: {avg_latency:.0f}ms")
            console.print(f"  P95: {p95_latency:.0f}ms")
            
            if not model:
                latency_by_model = successful.groupby('model')['latency_ms'].mean().sort_values()
                console.print(f"  Fastest: {latency_by_model.index[0]} ({latency_by_model.iloc[0]:.0f}ms)")
                console.print(f"  Slowest: {latency_by_model.index[-1]} ({latency_by_model.iloc[-1]:.0f}ms)")
    
    if metric in ['quality', 'all']:
        # Quality analysis
        successful = df[df['success'] == True]
        if not successful.empty:
            avg_quality = successful['quality'].mean()
            
            console.print(f"\n🏆 [bold]Quality Analysis[/bold]")
            console.print(f"  Average score: {avg_quality:.3f}")
            
            if not model:
                quality_by_model = successful.groupby('model')['quality'].mean().sort_values(ascending=False)
                console.print(f"  Best quality: {quality_by_model.index[0]} ({quality_by_model.iloc[0]:.3f})")
                console.print(f"  Lowest quality: {quality_by_model.index[-1]} ({quality_by_model.iloc[-1]:.3f})")
    
    # Success rate
    success_rate = df['success'].mean() * 100
    console.print(f"\n✅ [bold]Success Rate:[/bold] {success_rate:.1f}%")
    
    # Trend analysis
    if len(df) > 10:
        df['timestamp'] = pd.to_datetime(df['timestamp'])
        df_daily = df.groupby(df['timestamp'].dt.date).agg({
            'cost': 'sum',
            'latency_ms': 'mean',
            'quality': 'mean',
            'success': 'mean'
        })
        
        if len(df_daily) > 1:
            # Calculate trends
            cost_trend = "📈" if df_daily['cost'].iloc[-1] > df_daily['cost'].iloc[0] else "📉"
            latency_trend = "📈" if df_daily['latency_ms'].iloc[-1] > df_daily['latency_ms'].iloc[0] else "📉"
            quality_trend = "📈" if df_daily['quality'].iloc[-1] > df_daily['quality'].iloc[0] else "📉"
            
            console.print(f"\n📊 [bold]Trends:[/bold]")
            console.print(f"  Cost: {cost_trend}")
            console.print(f"  Latency: {latency_trend}")
            console.print(f"  Quality: {quality_trend}")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def optimize(
    target_cost: float = typer.Option(50.0, "--target-cost", "-t", help="Target daily cost in USD"),
    min_quality: float = typer.Option(0.7, "--min-quality", "-q", help="Minimum quality score (0-1)"),
    max_latency: int = typer.Option(2000, "--max-latency", "-l", help="Maximum latency in ms"),
    use_case: str = typer.Option("general", "--use-case", "-u", help="Use case (general, code, code, analysis)")
):
    """Find optimal model configuration for your requirements"""
    # Show credit balance before optimization
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    db = LocalDatabase()
    
    console.print(Panel.fit(
        f"[bold cyan]🎯 Model Optimization[/bold cyan]\n"
        f"Target daily cost: ${target_cost}\n"
        f"Min quality: {min_quality}\n"
        f"Max latency: {max_latency}ms\n"
        f"Use case: {use_case}",
        border_style="cyan"
    ))
    
    # Get recent performance data
    with sqlite3.connect(db.db_path) as conn:
        query = """
            SELECT model, 
                   AVG(cost) as avg_cost,
                   AVG(latency_ms) as avg_latency,
                   AVG(json_extract(quality_scores, '$.overall')) as avg_quality,
                   COUNT(*) as samples,
                   SUM(CASE WHEN success = 1 THEN 1 ELSE 0 END) * 1.0 / COUNT(*) as success_rate
            FROM results
            WHERE timestamp > datetime('now', '-7 days')
            GROUP BY model
            HAVING samples > 5
        """
        df = pd.read_sql_query(query, conn)
    
    if df.empty:
        console.print("[yellow]Insufficient data for optimization. Run more benchmarks first.[/yellow]")
        return
    
    # Calculate daily cost estimate (assuming 1000 requests/day)
    df['daily_cost'] = df['avg_cost'] * 1000
    
    # Filter by requirements
    candidates = df[
        (df['daily_cost'] <= target_cost) &
        (df['avg_quality'] >= min_quality) &
        (df['avg_latency'] <= max_latency) &
        (df['success_rate'] >= 0.95)
    ]
    
    if candidates.empty:
        console.print("\n[yellow]⚠️  No models meet all requirements[/yellow]")
        console.print("\nClosest alternatives:")
        
        # Show best alternatives
        df['score'] = (
            (1 - df['daily_cost'] / target_cost).clip(0, 1) * 0.3 +
            df['avg_quality'] * 0.4 +
            (1 - df['avg_latency'] / max_latency).clip(0, 1) * 0.3
        )
        
        alternatives = df.nlargest(3, 'score')
    else:
        # Score and rank candidates
        candidates['score'] = (
            (1 - candidates['daily_cost'] / target_cost) * 0.3 +
            candidates['avg_quality'] * 0.4 +
            (1 - candidates['avg_latency'] / max_latency) * 0.3
        )
        
        alternatives = candidates.nlargest(3, 'score')
        console.print("\n[green]✅ Recommended models:[/green]")
    
    # Display recommendations
    table = Table(box=box.ROUNDED)
    table.add_column("Rank", style="cyan")
    table.add_column("Model", style="yellow")
    table.add_column("Tier", justify="center", style="magenta")
    table.add_column("Daily Cost", justify="right", style="green")
    table.add_column("Quality", justify="center", style="blue")
    table.add_column("Latency", justify="right", style="red")
    table.add_column("Score", justify="center", style="bold")
    
    for i, (_, row) in enumerate(alternatives.iterrows(), 1):
        # Add emoji for rank
        rank_emoji = "🥇" if i == 1 else "🥈" if i == 2 else "🥉"
        
        # Get tier information
        model_config = ClyrdiaConfig.get_model(row['model'])
        tier_display = "❓ Unknown"
        if model_config:
            tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
            tier_color = {"flagship": "red", "balanced": "yellow", "speed_cost": "green"}
            tier_display = f"{tier_emoji.get(model_config.tier, '❓')} {model_config.tier.replace('_', ' ').title()}"
        
        table.add_row(
            f"{rank_emoji} {i}",
            row['model'],
            f"[{tier_color.get(model_config.tier, 'white') if model_config else 'white'}]{tier_display}[/{tier_color.get(model_config.tier, 'white') if model_config else 'white'}]",
            f"${row['daily_cost']:.2f}",
            f"{row['avg_quality']:.3f}",
            f"{row['avg_latency']:.0f}ms",
            f"{row['score']:.3f}"
        )
    
    console.print(table)
    
    # Provide specific recommendations based on use case
    best_model = alternatives.iloc[0]['model']
    savings = target_cost - alternatives.iloc[0]['daily_cost']
    
    console.print(f"\n💡 [bold]Recommendation:[/bold]")
    console.print(f"Use [bold cyan]{best_model}[/bold cyan] for {use_case} tasks")
    console.print(f"Estimated daily savings: [green]${savings:.2f}[/green]")
    
    # Use case specific advice
    use_case_advice = {
        'code': "Consider using temperature=0.3 for more deterministic code generation",
        'chat': "Use streaming responses for better user experience",
        'analysis': "Increase max_tokens for comprehensive analysis",
        'general': "Monitor quality scores and adjust temperature as needed"
    }
    
    console.print(f"\n💡 [dim]{use_case_advice.get(use_case, '')}[/dim]")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def compare(
    models: List[str] = typer.Argument(..., help="Models to compare"),
    test_prompt: Optional[str] = typer.Option(None, "--prompt", "-p", help="Custom test prompt"),
    interactive: bool = typer.Option(False, "--interactive", "-i", help="Interactive comparison mode")
):
    """Quick side-by-side model comparison"""
    
    # Show credit balance before comparison
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    # Validate models
    invalid = [m for m in models if m not in ClyrdiaConfig.list_models()]
    if invalid:
        console.print(f"[red]Unknown models: {', '.join(invalid)}[/red]")
        return
    
    if interactive:
        # Interactive mode - get prompt from user
        test_prompt = Prompt.ask("Enter your test prompt")
    elif not test_prompt:
        # Default comparison prompt
        test_prompt = "Explain quantum computing in simple terms"
    
    console.print(Panel.fit(
        f"[bold cyan]⚖️  Model Comparison[/bold cyan]\n"
        f"Models: {', '.join(models)}\n"
        f"Prompt: {test_prompt[:100]}{'...' if len(test_prompt) > 100 else ''}",
        border_style="cyan"
    ))
    
    # Run comparison
    model_interface = ModelInterface()
    results = []
    
    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console
    ) as progress:
        task = progress.add_task("[cyan]Comparing models...", total=len(models))
        
        for model in models:
            progress.update(task, description=f"[cyan]Testing {model}...")
            
            try:
                response, input_tokens, output_tokens, cost, latency = asyncio.run(
                    model_interface.call_model(model, test_prompt, max_tokens=500)
                )
                
                results.append({
                    'model': model,
                    'response': response,
                    'latency': latency,
                    'cost': cost,
                    'tokens': input_tokens + output_tokens,
                    'success': True
                })
            except Exception as e:
                results.append({
                    'model': model,
                    'response': f"Error: {str(e)}",
                    'latency': 0,
                    'cost': 0,
                    'tokens': 0,
                    'success': False
                })
            
            progress.advance(task)
    
    # Display results
    console.print("\n")
    
    # Summary table
    table = Table(title="📊 Comparison Summary", box=box.ROUNDED)
    table.add_column("Model", style="cyan")
    table.add_column("Tier", justify="center", style="magenta")
    table.add_column("Latency", justify="right", style="yellow")
    table.add_column("Cost", justify="right", style="green")
    table.add_column("Tokens", justify="right", style="blue")
    table.add_column("Status", justify="center")
    
    for result in results:
        status = "[green]✅[/green]" if result['success'] else "[red]❌[/red]"
        
        # Get tier information
        model_config = ClyrdiaConfig.get_model(result['model'])
        tier_display = "❓ Unknown"
        if model_config:
            tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
            tier_color = {"flagship": "red", "balanced": "yellow", "speed_cost": "green"}
            tier_display = f"{tier_emoji.get(model_config.tier, '❓')} {model_config.tier.replace('_', ' ').title()}"
        
        table.add_row(
            result['model'],
            f"[{tier_color.get(model_config.tier, 'white') if model_config else 'white'}]{tier_display}[/{tier_color.get(model_config.tier, 'white') if model_config else 'white'}]",
            f"{result['latency']}ms" if result['success'] else "N/A",
            f"${result['cost']:.6f}" if result['success'] else "N/A",
            str(result['tokens']) if result['success'] else "N/A",
            status
        )
    
    console.print(table)
    
    # Detailed responses
    console.print("\n[bold]📝 Responses:[/bold]\n")
    
    for result in results:
        if result['success']:
            # Truncate long responses
            response = result['response']
            if len(response) > 500:
                response = response[:500] + "...\n[dim](truncated)[/dim]"
            
            console.print(Panel(
                response,
                title=f"[cyan]{result['model']}[/cyan]",
                border_style="cyan if result['success'] else red"
            ))
        else:
            console.print(Panel(
                f"[red]{result['response']}[/red]",
                title=f"[red]{result['model']}[/red]",
                border_style="red"
            ))
    
    # Find winner
    successful = [r for r in results if r['success']]
    if successful:
        fastest = min(successful, key=lambda x: x['latency'])
        cheapest = min(successful, key=lambda x: x['cost'])
        
        console.print("\n[bold]🏆 Winners:[/bold]")
        console.print(f"  Fastest: [green]{fastest['model']}[/green] ({fastest['latency']}ms)")
        console.print(f"  Cheapest: [green]{cheapest['model']}[/green] (${cheapest['cost']:.6f})")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
def version():
    """Show Clyrdia version and system info"""
    import platform
    
    # ASCII art logo
    logo = """
    ╔═══════════════════════════════════╗
    ║       🚀 CLYRDIA v1.0.0 🚀        ║
    ║   Zero-Knowledge AI Benchmarking  ║
    ╚═══════════════════════════════════╝
    """
    
    console.print(logo, style="bold cyan")
    
    # System info
    info = Table(box=box.SIMPLE)
    info.add_column("Component", style="cyan")
    info.add_column("Version/Status", style="yellow")
    
    info.add_row("Python", platform.python_version())
    info.add_row("Platform", platform.platform())
    info.add_row("Database", "SQLite (Local)")
    
    # Check API keys
    api_status = {
        'OpenAI': '✅' if os.getenv('OPENAI_API_KEY') else '❌',
        'Anthropic': '✅' if os.getenv('ANTHROPIC_API_KEY') else '❌',
        
    }
    
    for provider, status in api_status.items():
        info.add_row(f"{provider} API", status)
    
    # Database stats
    db = LocalDatabase()
    with sqlite3.connect(db.db_path) as conn:
        cursor = conn.execute("SELECT COUNT(*) FROM benchmarks")
        benchmark_count = cursor.fetchone()[0]
        
        cursor = conn.execute("SELECT COUNT(*) FROM results")
        result_count = cursor.fetchone()[0]
    
    info.add_row("Benchmarks", str(benchmark_count))
    info.add_row("Test Results", str(result_count))
    info.add_row("Database Location", str(db.db_path))
    
    console.print(info)
    
    console.print("\n[dim]Your prompts. Your data. Your control.[/dim]")

@app.command()
@require_auth
def tiers(
    tier: Optional[str] = typer.Option(None, "--tier", "-t", help="Show specific tier (flagship, balanced, speed_cost)"),
    show_details: bool = typer.Option(False, "--details", "-d", help="Show detailed tier information"),
    compare: bool = typer.Option(False, "--compare", "-c", help="Compare tiers side by side")
):
    """Show AI model tiers to help with quick decision making"""
    
    # Show credit balance before showing tiers
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    if compare:
        # Show tier comparison table
        table = Table(title="🏆 Model Tier Comparison", box=box.DOUBLE_EDGE)
        table.add_column("Tier", style="cyan", no_wrap=True)
        table.add_column("Description", style="yellow")
        table.add_column("Models", style="magenta")
        table.add_column("Best For", style="green")
        table.add_column("Typical Use Case", style="blue")
        
        tier_info = {
            "flagship": {
                "description": "🏆 Highest quality, best capabilities",
                "models": ["gpt-5", "claude-opus-4.1"],
                "best_for": "Complex reasoning, Research, Enterprise",
                "use_case": "High-stakes applications where quality is paramount"
            },
            "balanced": {
                "description": "⚖️ Good balance of quality and cost",
                "models": ["gpt-5-mini", "claude-sonnet-4"],
                "best_for": "General use, Development, Business",
                "use_case": "Daily tasks requiring good performance and reasonable cost"
            },
            "speed_cost": {
                "description": "🚀 Fast and cost-effective",
                "models": ["gpt-5-nano", "claude-haiku-3.5"],
                "best_for": "High-volume, Real-time, Prototyping",
                "use_case": "Tasks where speed and cost matter more than perfect quality"
            }
        }
        
        for tier, info in tier_info.items():
            table.add_row(
                f"[bold]{tier.title()}[/bold]",
                info["description"],
                "\n".join(info["models"]),
                info["best_for"],
                info["use_case"]
            )
        
        console.print(table)
        
        # Show recommendation
        console.print("\n💡 [bold]Quick Decision Guide:[/bold]")
        console.print("• [red]Flagship[/red]: When you need the absolute best quality")
        console.print("• [yellow]Balanced[/yellow]: For most everyday use cases")
        console.print("• [green]Speed/Cost[/green]: When you need fast, cheap processing")
        
        return
    
    if tier:
        # Show specific tier
        tier = tier.lower()
        if tier not in ["flagship", "balanced", "speed_cost"]:
            console.print(f"[red]❌ Unknown tier: {tier}[/red]")
            console.print("[yellow]Available tiers: flagship, balanced, speed_cost[/yellow]")
            return
        
        tier_models = [m for m in ClyrdiaConfig.MODELS.values() if m.tier == tier]
        if not tier_models:
            console.print(f"[yellow]No models found in {tier} tier[/yellow]")
            return
        
        tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
        tier_color = {"flagship": "red", "balanced": "yellow", "speed_cost": "green"}
        
        console.print(Panel.fit(
            f"[bold {tier_color[tier]}]{tier_emoji[tier]} {tier.title()} Tier[/bold {tier_color[tier]}]\n"
            f"Models: {len(tier_models)}",
            border_style=tier_color[tier]
        ))
        
        # Show models in this tier
        table = Table(title=f"Models in {tier.title()} Tier", box=box.ROUNDED)
        table.add_column("Model", style="cyan")
        table.add_column("Provider", style="yellow")
        table.add_column("Capabilities", style="magenta")
        table.add_column("Context", justify="right", style="blue")
        table.add_column("Cost per 1M tokens", justify="right", style="green")
        
        for model in sorted(tier_models, key=lambda x: x.input_cost + x.output_cost):
            context = f"{model.context_window // 1000}K" if model.context_window < 1000000 else f"{model.context_window // 1000000}M"
            total_cost = model.input_cost + model.output_cost
            
            table.add_row(
                model.name,
                model.provider.value.capitalize(),
                ", ".join(model.capabilities[:3]) + ("..." if len(model.capabilities) > 3 else ""),
                context,
                f"${total_cost:.6f}"
            )
        
        console.print(table)
        
        # Show use case recommendations
        use_cases = {
            "flagship": [
                "Complex reasoning and analysis",
                "High-stakes business decisions",
                "Research and development",
                "Enterprise applications",
                "When quality is paramount"
            ],
            "balanced": [
                "General business use",
                "Content creation and editing",
                "Software development",
                "Data analysis",
                "Daily productivity tasks"
            ],
            "speed_cost": [
                "High-volume processing",
                "Real-time applications",
                "Prototyping and testing",
                "Budget-conscious projects",
                "When speed matters most"
            ]
        }
        
        console.print(f"\n🎯 [bold]Best for {tier.title()} tier:[/bold]")
        for use_case in use_cases[tier]:
            console.print(f"  • {use_case}")
        
    else:
        # Show all tiers overview
        console.print(Panel.fit(
            "[bold cyan]🏆 AI Model Tiers[/bold cyan]\n"
            "Quick decision guide for choosing the right model",
            border_style="cyan"
        ))
        
        # Create tier overview table
        table = Table(title="Model Tier Overview", box=box.ROUNDED)
        table.add_column("Tier", style="cyan", no_wrap=True)
        table.add_column("Description", style="yellow")
        table.add_column("Model Count", justify="center", style="magenta")
        table.add_column("Example Models", style="green")
        
        tier_counts = {}
        for model in ClyrdiaConfig.MODELS.values():
            tier_counts[model.tier] = tier_counts.get(model.tier, 0) + 1
        
        tier_info = {
            "flagship": {
                "description": "🏆 Highest quality, best capabilities",
                "example": "gpt-5, claude-opus-4.1"
            },
            "balanced": {
                "description": "⚖️ Good balance of quality and cost",
                "example": "gpt-5-mini, claude-sonnet-4"
            },
            "speed_cost": {
                "description": "🚀 Fast and cost-effective",
                "example": "gpt-5-nano, claude-haiku-3.5"
            }
        }
        
        for tier, info in tier_info.items():
            count = tier_counts.get(tier, 0)
            tier_emoji = {"flagship": "🏆", "balanced": "⚖️", "speed_cost": "🚀"}
            tier_display = f"{tier_emoji[tier]} {tier.replace('_', ' ').title()}"
            
            table.add_row(
                tier_display,
                info["description"],
                str(count),
                info["example"]
            )
        
        console.print(table)
        
        console.print("\n💡 [bold]Quick Commands:[/bold]")
        console.print("• [bold]clyrdia tiers --tier flagship[/bold] - Show flagship models")
        console.print("• [bold]clyrdia tiers --compare[/bold] - Compare all tiers")
        console.print("• [bold]clyrdia tiers --details[/bold] - Show detailed information")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def cache(
    action: str = typer.Argument(..., help="Cache action: status, clear"),
    model: Optional[str] = typer.Option(None, "--model", "-m", help="Specific model to clear cache for")
):
    """Manage benchmark result cache"""
    # Show credit balance before cache operations
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    cache_manager = CacheManager()
    
    if action == "status":
        stats = cache_manager.get_cache_stats()
        
        console.print(Panel.fit(
            "[bold cyan]📦 Cache Status[/bold cyan]",
            border_style="cyan"
        ))
        
        console.print(f"Total entries: [bold]{stats['total_entries']}[/bold]")
        console.print(f"Cache size: [bold]{stats['cache_size_mb']} MB[/bold]")
        
        if stats['model_counts']:
            console.print("\n[bold]Entries by model:[/bold]")
            for model_name, count in stats['model_counts'].items():
                console.print(f"  • {model_name}: {count}")
        else:
            console.print("\n[dim]No cached results[/dim]")
        
        console.print("\n[bold]💡 Cache Workflow:[/bold]")
        console.print("  • [bold]Production Mode:[/bold] clyrdia-cli benchmark file.yaml (no cache, 100% accuracy)")
        console.print("  • [bold]Developer Mode:[/bold] clyrdia-cli benchmark file.yaml --use-cache (smart caching)")
        console.print("  • [bold]Cache Management:[/bold] clyrdia-cli cache clear (start fresh)")
    
    elif action == "clear":
        if model:
            cache_manager.clear_cache(model)
            console.print(f"[green]✅ Cleared cache for {model}[/bold]")
            console.print(f"[dim]Next benchmark run will test {model} against live APIs[/dim]")
        else:
            cache_manager.clear_cache()
            console.print("[green]✅ Cleared all cache[/bold]")
            console.print("[dim]Next benchmark run will test all models against live APIs[/dim]")
        
        console.print("\n[bold]💡 When to clear cache:[/bold]")
        console.print("  • Testing against updated models")
        console.print("  • Suspecting provider changes")
        console.print("  • Starting fresh development cycle")
        console.print("  • Debugging unexpected results")
    
    else:
        console.print(f"[red]❌ Unknown action: {action}[/red]")
        console.print("Available actions: status, clear")
        raise typer.Exit(1)
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

@app.command()
@require_auth
def workflow():
    """Explain Clyrdia's dual-mode workflow for different use cases"""
    # Show credit balance before explaining workflow
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    console.print(Panel.fit(
        "[bold cyan]🔄 Clyrdia Dual-Mode Workflow[/bold cyan]\n"
        "Understanding when to use Production vs. Developer mode",
        border_style="cyan"
    ))
    
    console.print("\n[bold]🔒 Production Mode (Default):[/bold]")
    console.print("  Command: [bold]clyrdia-cli benchmark file.yaml[/bold]")
    console.print("  Use Case: Final validation, CI/CD, canary tests")
    console.print("  Behavior: All tests run against live APIs")
    console.print("  Cost: Full API costs")
    console.print("  Accuracy: 100% - always reflects current model behavior")
    console.print("  When: Before deployment, detecting drift, performance validation")
    
    console.print("\n[bold]💻 Developer Mode (Opt-in):[/bold]")
    console.print("  Command: [bold]clyrdia-cli benchmark file.yaml --use-cache[/bold]")
    console.print("  Use Case: Rapid iteration, prompt development, cost optimization")
    console.print("  Behavior: Smart hybrid - cached results + live API calls")
    console.print("  Cost: Only for changed tests (cached = $0)")
    console.print("  Accuracy: High - cached results from previous live runs")
    console.print("  When: Developing prompts, debugging, iterative testing")
    
    console.print("\n[bold]📊 Example Workflow:[/bold]")
    console.print("  1. [bold]First Run:[/bold] clyrdia-cli benchmark file.yaml (establish baseline)")
    console.print("  2. [bold]Development:[/bold] clyrdia-cli benchmark file.yaml --use-cache (iterate fast)")
    console.print("  3. [bold]Final Check:[/bold] clyrdia-cli benchmark file.yaml (verify before deploy)")
    
    console.print("\n[bold]💡 Pro Tips:[/bold]")
    console.print("  • Start without cache to establish truth")
    console.print("  • Use cache during development iterations")
    console.print("  • Clear cache when testing model updates")
    console.print("  • Monitor cache hit rates for cost optimization")
    
    console.print("\n[bold]🛡️ Trust & Integrity:[/bold]")
    console.print("  • Caching is OFF by default - your gatekeeper is always accurate")
    console.print("  • Cached results are clearly marked [CACHE] vs [LIVE]")
    console.print("  • Cache costs $0 - no hidden charges")
    console.print("  • Full control with cache clear commands")
    
    console.print("\n[bold]🔍 Built-in Accuracy:[/bold]")
    console.print("  • Every benchmark automatically validates cost calculations")
    console.print("  • Detailed cost breakdowns for transparency")
    console.print("  • Token counting uses official API responses when available")
    console.print("  • Quality evaluation with content-aware scoring")
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)

# ============================================================================
# SaaS Licensing Commands
# ============================================================================

@app.command()
def login():
    """🔑 Login to Clyrdia with your API key
    
    This command will:
    1. Open your browser to clyrdia.com for signup/login
    2. Guide you through getting your API key
    3. Store your credentials securely
    4. Show your current credit balance
    
    Get your free 250 credits at https://clyrdia.com
    """
    console.print(Panel.fit(
        "[bold cyan]🔑 Clyrdia Login[/bold cyan]\n"
        "Connect your CLI to your Clyrdia account",
        border_style="cyan",
        padding=(1, 2),
        title="Login",
        title_align="center"
    ))
    
    # Check if already authenticated
    licensing_manager = LicensingManager()
    if licensing_manager.is_authenticated():
        console.print("[green]✅ You are already logged in![/green]")
        try:
            status = asyncio.run(licensing_manager.get_status())
            console.print(f"Welcome back, [bold]{status.user_name}[/bold]!")
            console.print(f"Plan: [bold]{status.plan.upper()}[/bold]")
            console.print(f"Credits: [bold]{status.credits_remaining:,}[/bold]")
            
            if Confirm.ask("Would you like to log out and login with a different account?"):
                licensing_manager.logout()
                console.print("[green]✅ Logged out successfully[/green]")
            else:
                return
        except Exception as e:
            console.print(f"[yellow]⚠️  Could not verify current login: {str(e)}[/yellow]")
            if Confirm.ask("Would you like to login again?"):
                licensing_manager.logout()
            else:
                return
    
    # Show the beautiful welcome screen
    _display_welcome_screen()
    
    console.print(Panel.fit(
        "[bold cyan]👋 Welcome to Clyrdia![/bold cyan]\n"
        "To get started, you'll need to create a free account.\n"
        "This will unlock your 250 monthly credits.",
        border_style="cyan",
        padding=(1, 2),
        title="Account Setup",
        title_align="center"
    ))
    
    console.print("\n[bold]🔗 Please visit clyrdia.com to create your account:[/bold]")
    console.print("[bold bright_blue]https://clyrdia.com/auth[/bold bright_blue]")
    console.print("\n[yellow]Complete the signup process and get your API key, then paste it below.[/yellow]")
    
    console.print("\n[bold]Steps to complete signup:[/bold]")
    console.print("1. Complete the signup form on clyrdia.com")
    console.print("2. Verify your email address")
    console.print("3. Get your API key from your dashboard")
    console.print("4. Paste it below when ready")
    
    console.print("\n[bold]Waiting for API key...[/bold] (paste it here when you're done)")
    
    # Wait for user to paste their API key
    while True:
        try:
            api_key = Prompt.ask("API Key", password=True)
            
            if not api_key or len(api_key.strip()) < 10:
                console.print("[red]❌ Invalid API key format. Please try again.[/red]")
                continue
            
            # Validate the API key
            try:
                user_status = asyncio.run(licensing_manager.login(api_key.strip()))
                
                console.print(f"\n[green]✅ Successfully authenticated![/green]")
                console.print(f"Welcome, [bold]{user_status.user_name}[/bold]!")
                console.print(f"Plan: [bold]{user_status.plan.upper()}[/bold]")
                console.print(f"Credits: [bold]{user_status.credits_remaining:,}[/bold]")
                console.print(f"Resets on: [dim]{user_status.resets_on}[/dim]")
                
                console.print(f"\n[bold green]🎉 You're all set![/bold green]")
                console.print("Your CLI is now connected to your Clyrdia account.")
                
                console.print(f"\n[bold]Next steps:[/bold]")
                console.print("1. Run [bold]clyrdia-cli init[/bold] to set up your environment")
                console.print("2. Run [bold]clyrdia-cli tutorial[/bold] to learn how to use Clyrdia")
                console.print("3. Run [bold]clyrdia-cli benchmark[/bold] to start benchmarking")
                
                break
                
            except Exception as e:
                console.print(f"[red]❌ Authentication failed: {str(e)}[/red]")
                console.print("\n[bold]Troubleshooting:[/bold]")
                console.print("  • Verify you completed the signup process")
                console.print("  • Check your internet connection")
                console.print("  • Try copying the API key again")
                console.print("  • Visit [bold]https://clyrdia.com[/bold] to start over")
                
                if not Confirm.ask("Try again with a different API key?"):
                    console.print("[dim]Login cancelled. Run 'clyrdia-cli login' when ready.[/dim]")
                    raise typer.Exit(0)
                
                continue
                
        except KeyboardInterrupt:
            console.print("\n\n[yellow]Login cancelled.[/yellow]")
            console.print("Run 'clyrdia-cli login' when you're ready to authenticate.")
            raise typer.Exit(0)
        except Exception as e:
            console.print(f"[red]❌ Unexpected error: {str(e)}[/red]")
            raise typer.Exit(1)

@app.command()
@require_auth
def tutorial():
    """📚 Interactive tutorial and guide to using Clyrdia
    
    Learn how to:
    • Run benchmarks and compare models
    • Use caching for cost savings
    • Integrate with CI/CD pipelines
    • Monitor model drift and performance
    • Analyze results and optimize costs
    """
    # Show credit balance at start
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    console.print(Panel.fit(
        "[bold cyan]📚 Clyrdia Tutorial[/bold cyan]\n"
        "Learn how to use the most advanced AI benchmarking platform",
        border_style="cyan",
        padding=(1, 2),
        title="Tutorial",
        title_align="center"
    ))
    
    # Welcome and overview
    console.print("\n[bold]🌟 Welcome to Clyrdia![/bold]")
    console.print("Clyrdia is a zero-knowledge AI benchmarking platform that gives you complete control over your AI model testing.")
    
    # Core concepts
    console.print("\n[bold]🔑 Core Concepts:[/bold]")
    console.print("• [bold]Zero-Knowledge:[/bold] Your data stays local, always")
    console.print("• [bold]Dual-Mode:[/bold] Production (live) vs Developer (cached) workflows")
    console.print("• [bold]Smart Caching:[/bold] Save costs by reusing previous results")
    console.print("• [bold]Credit System:[/bold] Pay only for what you use")
    
    # Basic workflow
    console.print("\n[bold]🚀 Basic Workflow:[/bold]")
    console.print("1. [bold]Initialize:[/bold] clyrdia-cli init - Set up your environment")
    console.print("2. [bold]Configure:[/bold] Edit benchmark.yaml with your test cases")
    console.print("3. [bold]Run:[/bold] clyrdia-cli benchmark - Execute your benchmarks")
    console.print("4. [bold]Analyze:[/bold] clyrdia-cli dashboard - View results visually")
    console.print("5. [bold]Optimize:[/bold] clyrdia-cli optimize - Find best models for your needs")
    
    # Dual-mode explanation
    console.print("\n[bold]💡 Dual-Mode Workflow:[/bold]")
    console.print("[bold]Production Mode:[/bold] clyrdia-cli benchmark file.yaml")
    console.print("  • 100% accuracy, live APIs, perfect for CI/CD & canary tests")
    console.print("  • Use when you need the truth before deployment")
    console.print("  • Higher credit cost but guaranteed fresh results")
    
    console.print("\n[bold]Developer Mode:[/bold] clyrdia-cli benchmark file.yaml --use-cache")
    console.print("  • Smart caching, cost savings, perfect for rapid iteration")
    console.print("  • Use when tweaking prompts during development")
    console.print("  • Lower credit cost with intelligent result reuse")
    
    # CI/CD Integration
    console.print("\n[bold]🔧 CI/CD Integration:[/bold]")
    console.print("Clyrdia is perfect for automated testing in your deployment pipeline:")
    
    console.print("\n[bold]GitHub Actions Example:[/bold]")
    console.print("```yaml")
    console.print("name: AI Model Testing")
    console.print("on: [push, pull_request]")
    console.print("jobs:")
    console.print("  test:")
    console.print("    runs-on: ubuntu-latest")
    console.print("    steps:")
    console.print("      - uses: actions/checkout@v3")
    console.print("      - name: Setup Python")
    console.print("        uses: actions/setup-python@v3")
    console.print("        with:")
    console.print("          python-version: '3.9'")
    console.print("      - name: Install Clyrdia")
    console.print("        run: pip install clyrdia-cli")
    console.print("      - name: Setup API Key")
    console.print("        run: echo '${{ secrets.CLYRDIA_API_KEY }}' > ~/.clyrdia/config.json")
    console.print("      - name: Run Benchmark")
    console.print("        run: clyrdia-cli benchmark benchmark.yaml")
    console.print("```")
    
    console.print("\n[bold]Environment Variables:[/bold]")
    console.print("• CLYRDIA_API_KEY: Your Clyrdia API key")
    console.print("• OPENAI_API_KEY: OpenAI API key (if using OpenAI models)")
    console.print("• ANTHROPIC_API_KEY: Anthropic API key (if using Claude)")
    
    
    # Advanced features
    console.print("\n[bold]🚀 Advanced Features:[/bold]")
    console.print("• [bold]Drift Detection:[/bold] clyrdia-cli canary - Monitor model behavior changes")
    console.print("• [bold]Performance Ratchet:[/bold] clyrdia-cli ratchet - Ensure no regressions")
    console.print("• [bold]Cost Optimization:[/bold] clyrdia-cli optimize - Find best models for your budget")
    console.print("• [bold]Model Comparison:[/bold] clyrdia-cli compare - Quick side-by-side testing")
    
    # Cost management
    console.print("\n[bold]💰 Cost Management:[/bold]")
    console.print("• [bold]Credit System:[/bold] Pay only for live API calls")
    console.print("• [bold]Smart Caching:[/bold] Cached results cost 0 credits")
    console.print("• [bold]Cost Estimation:[/bold] See costs before running benchmarks")
    console.print("• [bold]Usage Tracking:[/bold] Monitor your credit consumption")
    
    # Best practices
    console.print("\n[bold]💡 Best Practices:[/bold]")
    console.print("• Use [bold]--use-cache[/bold] during development to save costs")
    console.print("• Run production benchmarks without caching for accuracy")
    console.print("• Set up canary tests to monitor model drift")
    console.print("• Use the dashboard to analyze trends and patterns")
    console.print("• Export results for external analysis and reporting")
    
    # Troubleshooting
    console.print("\n[bold]🔧 Troubleshooting:[/bold]")
    console.print("• [bold]API Key Issues:[/bold] Run 'clyrdia-cli login' to refresh")
    console.print("• [bold]Missing Models:[/bold] Check 'clyrdia-cli models' for available options")
    console.print("• [bold]Cache Problems:[/bold] Use 'clyrdia-cli cache clear' to reset")
    console.print("• [bold]Dashboard Issues:[/bold] Check 'clyrdia-cli dashboard-status'")
    
    # Next steps
    console.print("\n[bold]🎯 Next Steps:[/bold]")
    console.print("1. Run [bold]clyrdia-cli init[/bold] to create your first benchmark")
    console.print("2. Edit the generated benchmark.yaml with your test cases")
    console.print("3. Run [bold]clyrdia-cli benchmark benchmark.yaml[/bold]")
    console.print("4. View results with [bold]clyrdia-cli dashboard[/bold]")
    console.print("5. Explore advanced features like drift detection and optimization")
    
    console.print("\n[bold]📚 Additional Resources:[/bold]")
    console.print("• Documentation: [bold]https://docs.clyrdia.com[/bold]")
    console.print("• Community: [bold]https://community.clyrdia.com[/bold]")
    console.print("• Support: [bold]https://clyrdia.com/support[/bold]")
    
    # Interactive demo
    if Confirm.ask("\nWould you like to see a live demo of a benchmark?"):
        console.print("\n[bold]🎬 Live Demo:[/bold]")
        console.print("Let's run a quick benchmark to see Clyrdia in action!")
        
        # Create a simple demo benchmark
        demo_config = {
            'name': 'Tutorial Demo',
            'description': 'Quick demo to show Clyrdia capabilities',
            'models': ['gpt-4o-mini', 'claude-sonnet-4'],
            'test_cases': [
                {
                    'name': 'Simple Question',
                    'prompt': 'What is the capital of Japan?',
                    'max_tokens': 50,
                    'temperature': 0.3
                }
            ]
        }
        
        # Save demo config
        demo_file = Path("tutorial_demo.yaml")
        with open(demo_file, 'w') as f:
            yaml.dump(demo_config, f, default_flow_style=False)
        
        console.print(f"[green]✅ Created demo benchmark: {demo_file}[/green]")
        
        if Confirm.ask("Run the demo benchmark now?"):
            console.print("\n[bold]🚀 Running demo benchmark...[/bold]")
            try:
                # Run the demo benchmark directly
                console.print("\n[green]✅ Demo benchmark file created![/green]")
                console.print("To run it manually, use:")
                console.print(f"[bold]clyrdia-cli benchmark {demo_file}[/bold]")
                console.print("\nYou can also view your results with: [bold]clyrdia-cli dashboard[/bold]")
            except Exception as e:
                console.print(f"\n[yellow]⚠️  Demo setup had an issue: {str(e)}[/yellow]")
                console.print("Try running it manually with:")
                console.print(f"[bold]clyrdia-cli benchmark {demo_file}[/bold]")
        
        # Clean up demo file
        if Confirm.ask("Remove the demo file?"):
            demo_file.unlink()
            console.print("[green]✅ Demo file removed[/green]")
        
        # Show final credit balance
        console.print("\n" + "="*50)
        after_credits = licensing_manager.show_credit_balance()
        licensing_manager.display_credit_summary(before_credits, after_credits)
        console.print("="*50)

@app.command()
@require_auth
def logout():
    """🔓 Logout from Clyrdia and remove stored credentials"""
    licensing_manager = LicensingManager()
    
    if not licensing_manager.is_authenticated():
        console.print("[yellow]⚠️  You are not currently logged in[/yellow]")
        return
        
    try:
        # Get current status before logout
        status = asyncio.run(licensing_manager.get_status())
        user_name = status.user_name
        plan = status.plan
        
        # Logout
        licensing_manager.logout()
        
        console.print(f"[green]✅ Successfully logged out[/green]")
        console.print(f"Goodbye, [bold]{user_name}[/bold]!")
        console.print(f"Your {plan.upper()} plan credentials have been removed.")
        
        console.print("\n[bold]To login again:[/bold]")
        console.print("• Run [bold]clyrdia-cli login[/bold]")
        console.print("• Or visit [bold]https://clyrdia.com[/bold]")
        
    except Exception as e:
        console.print(f"[red]❌ Logout failed: {str(e)}[/red]")
        # Force logout anyway
        licensing_manager.logout()
        console.print("[yellow]⚠️  Credentials removed locally, but server logout failed[/yellow]")

@app.command()
@require_auth
def status():
    """📊 Show your current Clyrdia account status and credit balance"""
    licensing_manager = LicensingManager()
    
    try:
        status = asyncio.run(licensing_manager.get_status())
        
        console.print(Panel.fit(
            "[bold cyan]📊 Account Status[/bold cyan]",
            border_style="cyan",
            padding=(1, 2),
            title="Status",
            title_align="center"
        ))
        
        # Account info
        console.print(f"[bold]👤 User:[/bold] {status.user_name}")
        console.print(f"[bold]📋 Plan:[/bold] {status.plan.upper()}")
        console.print(f"[bold]💰 Credits:[/bold] {status.credits_remaining:,}")
        console.print(f"[bold]🔄 Resets:[/bold] {status.resets_on}")
        
        # Plan details
        if status.plan == "free":
            console.print(f"\n[bold]🎁 Free Plan Benefits:[/bold]")
            console.print("• 250 credits per month")
            console.print("• All CLI features")
            console.print("• Local dashboard")
            console.print("• Basic support")
            
            console.print(f"\n[bold]🚀 Upgrade to Pro:[/bold]")
            console.print("• 10,000 credits per month")
            console.print("• Priority support")
            console.print("• Advanced analytics")
            console.print("• Team collaboration")
            console.print("• Visit [bold]https://clyrdia.com[/bold] to upgrade")
        else:
            console.print(f"\n[bold]💎 Pro Plan Benefits:[/bold]")
            console.print("• 10,000 credits per month")
            console.print("• Priority support")
            console.print("• Advanced analytics")
            console.print("• Team collaboration")
        
        # Recent activity
        console.print(f"\n[bold]📈 Recent Activity:[/bold]")
        db = LocalDatabase()
        with sqlite3.connect(db.db_path) as conn:
            cursor = conn.execute("""
                SELECT COUNT(*) as total_benchmarks,
                       COUNT(DISTINCT model) as unique_models,
                       SUM(CASE WHEN success = 1 THEN 1 ELSE 0 END) as successful_tests,
                       COUNT(*) as total_tests
                FROM results
                WHERE timestamp > datetime('now', '-30 days')
            """)
            row = cursor.fetchone()
            
            if row and row[0] > 0:
                success_rate = (row[2] / row[3]) * 100 if row[3] > 0 else 0
                console.print(f"• Benchmarks run: {row[0]}")
                console.print(f"• Models tested: {row[1]}")
                console.print(f"• Tests completed: {row[2]}/{row[3]} ({success_rate:.1f}% success)")
            else:
                console.print("• No recent activity")
                console.print("• Run [bold]clyrdia-cli init[/bold] to get started")
        
        # API key info
        console.print(f"\n[bold]🔑 API Key:[/bold]")
        masked_key = status.api_key[:8] + "..." + status.api_key[-4:] if len(status.api_key) > 12 else "***"
        console.print(f"• Key: {masked_key}")
        console.print(f"• Status: [green]✅ Active[/green]")
        
        # Quick actions
        console.print(f"\n[bold]⚡ Quick Actions:[/bold]")
        console.print("• [bold]clyrdia-cli init[/bold] - Set up your environment")
        console.print("• [bold]clyrdia-cli benchmark[/bold] - Run benchmarks")
        console.print("• [bold]clyrdia-cli dashboard[/bold] - View results")
        console.print("• [bold]clyrdia-cli tutorial[/bold] - Learn more")
        
    except Exception as e:
        console.print(f"[red]❌ Could not fetch status: {str(e)}[/red]")
        console.print("Try running [bold]clyrdia-cli login[/bold] to refresh your credentials")

@app.command()
def commands():
    """📚 Show all available Clyrdia commands and their descriptions
    
    This command provides a comprehensive overview of all CLI functionality,
    perfect for learning how to use Clyrdia effectively.
    """
    console.print(Panel.fit(
        "[bold cyan]📚 Clyrdia CLI Commands[/bold cyan]\n"
        "Complete command reference and usage guide",
        border_style="cyan",
        padding=(1, 2),
        title="Commands",
        title_align="center"
    ))
    
    # Group commands by category
    command_categories = {
        "🔑 Authentication & Setup": [
            ("login", "Connect your CLI to your Clyrdia account and get free credits"),
            ("logout", "Disconnect your CLI and remove stored credentials"),
            ("status", "Show your current account status and credit balance"),
            ("init", "Initialize Clyrdia and create sample benchmark configuration")
        ],
        "📚 Learning & Help": [
            ("tutorial", "Interactive tutorial and guide to using Clyrdia"),
            ("commands", "Show this command reference"),
            ("version", "Show Clyrdia version and system information")
        ],
        "🚀 Core Benchmarking": [
            ("benchmark", "Run AI model benchmarks with customizable test cases"),
            ("compare", "Quick side-by-side model comparison"),
            ("models", "List available AI models with pricing and capabilities")
        ],
        "📊 Analysis & Results": [
            ("dashboard", "Launch local web dashboard for viewing results"),
            ("analyze", "Analyze benchmark trends and performance patterns"),
            ("export", "Export benchmark results in various formats")
        ],
        "🔒 Quality Assurance": [
            ("ratchet", "Ensure performance never regresses (CI/CD ready)"),
            ("canary", "Detect model drift and behavior changes"),
            ("optimize", "Find optimal model configuration for your needs")
        ],
        "📦 System Management": [
            ("cache", "Manage intelligent caching for cost savings"),
            ("migrate_data", "Migrate existing data to dashboard format"),
            ("tiers", "Show AI model tiers for quick decision making")
        ]
    }
    
    for category, commands_list in command_categories.items():
        console.print(f"\n[bold]{category}[/bold]")
        for cmd, description in commands_list:
            console.print(f"  • [bold]clyrdia {cmd}[/bold] - {description}")
    
    console.print(f"\n[bold]💡 Pro Tips:[/bold]")
    console.print("• Use [bold]clyrdia-cli --help[/bold] for detailed command help")
    console.print("• Use [bold]clyrdia-cli <command> --help[/bold] for specific command help")
    console.print("• Run [bold]clyrdia-cli tutorial[/bold] for interactive learning")
    console.print("• Get 250 free credits at [bold]https://clyrdia.com[/bold]")
    
    console.print(f"\n[bold]🔧 CI/CD Integration:[/bold]")
    console.print("• Perfect for automated testing in deployment pipelines")
    console.print("• Use environment variables for API keys")
    console.print("• Run [bold]clyrdia-cli tutorial[/bold] for detailed examples")

@app.command()


# ============================================================================
# Dashboard System
# ============================================================================

# Remove duplicate migrate_data command - it's defined later

@app.command()
@require_auth
def dashboard_status():
    """Check the status of the local dashboard server"""
    # Show credit balance before checking dashboard status
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    try:
        dashboard_manager = DashboardManager()
        
        # Use the new dashboard manager method
        dashboard_manager.check_dashboard_status()
        
        # Show final credit balance
        console.print("\n" + "="*50)
        after_credits = licensing_manager.show_credit_balance()
        licensing_manager.display_credit_summary(before_credits, after_credits)
        console.print("="*50)
        
    except Exception as e:
        console.print(f"[red]❌ Status check failed: {str(e)}[/red]")
        raise typer.Exit(1)

@app.command()
@require_auth
def dashboard(
    port: int = typer.Option(3000, "--port", "-p", help="Port to run dashboard on"),
    no_open: bool = typer.Option(False, "--no-open", help="Don't open browser automatically"),
    open_browser: bool = typer.Option(False, "--open", "-o", help="Open dashboard in browser")
):
    """🚀 Launch the local web dashboard for exploring benchmark results
    
    The dashboard provides:
    • 📊 Interactive charts and performance metrics
    • 🏆 Model comparison and ranking
    • 💰 Cost analysis and optimization insights
    • 🔍 Detailed result inspection and search
    • 📈 Historical trend analysis
    • 📋 Export and sharing capabilities
    
    Note: The dashboard is a local application that runs on your machine.
    This command will start the local dashboard server for you.
    """
    # Show credit balance before dashboard operations
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    try:
        dashboard_manager = DashboardManager()
        dashboard_manager.set_port(port)
        
        # Show dashboard instructions
        dashboard_manager.show_dashboard_instructions()
        
        # Check if dashboard is running locally
        if dashboard_manager.is_dashboard_running():
            console.print(f"[green]✅ Dashboard is running on port {port}[/green]")
            if open_browser or (not no_open):
                dashboard_manager.open_dashboard_url()
        else:
            console.print(f"[yellow]⚠️  Dashboard is not running locally on port {port}[/yellow]")
            if open_browser or (not no_open):
                console.print("Starting local dashboard...")
                if dashboard_manager.start_dashboard():
                    dashboard_manager.open_dashboard_url()
        
    except Exception as e:
        console.print(f"[red]❌ Dashboard operation failed: {str(e)}[/red]")
        raise typer.Exit(1)
    
    # Show final credit balance
    console.print("\n" + "="*50)
    after_credits = licensing_manager.show_credit_balance()
    licensing_manager.display_credit_summary(before_credits, after_credits)
    console.print("="*50)



# ============================================================================
# Additional CLI Commands
# ============================================================================

@app.command()
@require_auth
def migrate_data():
    """🔄 Migrate existing benchmark data to dashboard format
    
    This command will:
    1. Scan your existing benchmark results
    2. Convert them to the new dashboard format
    3. Ensure compatibility with the web interface
    4. Preserve all your historical data
    """
    # Show credit balance at start
    licensing_manager = LicensingManager()
    console.print(Panel.fit(
        "[bold cyan]💰 Credit Balance Check[/bold cyan]",
        border_style="cyan",
        padding=(1, 2),
        title="Credits",
        title_align="center"
    ))
    before_credits = licensing_manager.show_credit_balance()
    
    console.print(Panel.fit(
        "[bold cyan]🔄 Data Migration[/bold cyan]\n"
        "Migrate existing benchmark data to dashboard format",
        border_style="cyan",
        padding=(1, 2),
        title="Migration",
        title_align="center"
    ))
    
    try:
        db_manager = LocalDatabase()
        migrated_count = db_manager.migrate_existing_data()
        
        if migrated_count == 0:
            console.print("[green]✅ No data migration needed - all data is already in the correct format[/green]")
        else:
            console.print(f"[green]✅ Successfully migrated {migrated_count} benchmark results[/green]")
            console.print("\n[bold]Next steps:[/bold]")
            console.print("• Run [bold]clyrdia-cli dashboard[/bold] to view your results")
            console.print("• Your data is now compatible with the web dashboard")
        
        # Show final credit balance
        console.print("\n" + "="*50)
        after_credits = licensing_manager.show_credit_balance()
        licensing_manager.display_credit_summary(before_credits, after_credits)
        console.print("="*50)
        
    except Exception as e:
        console.print(f"[red]❌ Migration failed: {str(e)}[/red]")
        raise typer.Exit(1)





# ============================================================================
# Main Entry Point
# ============================================================================

def _display_quick_start():
    """Display a quick start guide when no arguments are provided"""
    console.print()
    
    # Display CLYRDIA ASCII art
    console.print(Panel.fit(
        """      ██████╗██╗     ██╗   ██╗██████╗ ██████╗ ██╗ █████╗
     ██╔════╝██║     ╚██╗ ██╔╝██╔══██╗██╔══██╗██║██╔══██╗
     ██║     ██║      ╚████╔╝ ██████╔╝██║  ██║██║███████║
     ██║     ██║       ╚██╔╝  ██╔══██╗██║  ██║██║██╔══██║
     ╚██████╗███████╗   ██║   ██║  ██║██████╔╝██║██║  ██║
      ╚═════╝╚══════╝   ╚═╝   ╚═╝  ╚═╝╚═════╝ ╚═╝╚═╝  ╚═╝""",
        border_style="bright_cyan",
        padding=(1, 2),
        title="[bold bright_cyan]🚀 CLYRDIA[/bold bright_cyan]",
        title_align="center"
    ))
    
    console.print(Panel.fit(
        "[bold bright_cyan]🚀 CLYRDIA[/bold bright_cyan] - [bold white]Zero-Knowledge AI Benchmarking[/bold white]\n\n"
        "[bold bright_green]Quick Start:[/bold bright_green]\n"
        "[bright_cyan]1.[/bright_cyan] [bold]clyrdia-cli login[/bold]       - Connect your account and get free credits\n"
        "[bright_cyan]2.[/bright_cyan] [bold]clyrdia-cli init[/bold]        - Initialize and create sample benchmark\n"
        "[bright_cyan]3.[/bright_cyan] [bold]clyrdia-cli tutorial[/bold]   - Learn how to use Clyrdia\n"
        "[bright_cyan]4.[/bright_cyan] [bold]clyrdia-cli benchmark[/bold]   - Run your first benchmark\n"
        "[bright_cyan]5.[/bright_cyan] [bold]clyrdia-cli dashboard[/bold]   - View results in web interface\n\n"
        "[bold bright_yellow]💡 Pro Tips:[/bold bright_yellow]\n"
        "[bright_yellow]•[/bright_yellow] Use [bold]--use-cache[/bold] for cost savings during development\n"
        "[bright_yellow]•[/bright_yellow] Run [bold]clyrdia-cli models[/bold] to see all available AI models\n"
        "[bright_yellow]•[/bright_yellow] Use [bold]clyrdia-cli --help[/bold] for full command reference\n"
        "[bright_yellow]•[/bright_yellow] Get 250 free credits at [bold]https://clyrdia.com[/bold]",
        border_style="bright_blue",
        padding=(1, 2),
        title="[bold bright_white]Welcome to Clyrdia![/bold bright_white]",
        title_align="center"
    ))
    console.print()


