from gai.lib.logging import getLogger

logger = getLogger(__name__)
from gai.messages.typing import OrchStepPydantic, OrchPlanPydantic

# A "turn" represents a complete request-response interaction between the user and the assistant.
# - The "turn request" is initiated by the user, where the user acts as the sender and the assistant as the designated recipient.
# - The subsequent "turn response" is generated by the assistant, which then becomes the sender while the original user becomes the recipient.
# - In cases where the active participant is not the intended agent (i.e., it is not the agent's turn), the specific flow design determines whether third-party listeners or non-recipients are allowed to observe the exchange.

# Since a turn is a two step request-response pair, we will track the step number for each round rather than the turn number.

# A flow_diagram describes the steps that takes place in one round of dialogue
# A dialogue contains multiple rounds of chat and each round contains multiple steps

# Flow Types:
#
# "Poll" Flow - In a Poll Flow, User message is sent once to all agents and each agent responds only to the original user message.
# An agent's do not see the other agent's response and cannot influence the next agent's response.
#
# Example - Polling:
# User->>Agent1
# User->>Agent2
#
# Round:
#        ├── Step 0: User -> *: "Everybody, please introduce yourself."
#        ├── Step 1: Node1 -> User: "Hi, my name is Sara. Nice to meet you!"
#        └── Step 2: Node2 -> User: "Hello, I'm Diana. Pleasure to meet you!"
#
# "Chain" Flow - In a Chain Flow, User message is sent only to the first agent and each agent responds to the the earlier agents message.
# An agent's can see the previous agent's response and can influence the next agent's response.
#
# Example - Chaining:
# User->>Agent1
# Agent1->>Agent2
# Agent2->>User
#
# Round:
#        ├── Step 0: User -> Node1: "Sara, tell me a one paragraph story"
#        ├── Step 1: Node1 -> User: "Sure! Once upon a time..."
#        ├── Step 2: User -> Node2: "Please continue."
#        └── Step 3: Node2 -> User: "And they lived happily ever after."


class InterimTurn:
    def __init__(self, sender, recipient):
        self.sender = sender
        self.recipient = recipient


def parse_flow(dialogue_id: str, round_no: int, flow: str) -> OrchPlanPydantic:
    turns = flow.replace("->>", "-->").split("\n")
    turns = [turn.strip() for turn in turns if turn.strip()]

    # 1st pass: Convert flow into interim turns

    interim_turns = []
    for i, turn in enumerate(turns):
        if "-->" in turn:
            sender, recipient = turn.split("-->")
            sender = sender.strip()
            recipient = recipient.strip()
            interim_turns.append(InterimTurn(sender, recipient))

    # 2nd pass: Validate and classify the flow as poll or chain

    is_poll = True
    for i, turn in enumerate(interim_turns):
        # Always start with User
        # If subsequent turns are from User, then it is a poll

        if i == 0 and turn.sender != "User":
            raise ValueError("First turn must be from User")
        if turn.sender != "User":
            is_poll = False
    flow_type = "poll"
    if not is_poll:
        # We know it is not a poll, so we need to check if it is a chain

        is_chain = True
        for i, turn in enumerate(interim_turns):
            if i > 0 and interim_turns[i - 1].recipient != turn.sender:
                is_chain = False

        if not is_chain:
            raise ValueError(
                "Invalid flow. Previous recipient must be the sender in a chain flow."
            )
        flow_type = "chain"

    # 3rd pass: Convert interim turns into OrchPlanPydantic

    plan = OrchPlanPydantic(
        dialogue_id=dialogue_id, round_no=round_no, participants=[], flow_type=flow_type
    )
    step_no = 0
    participants = set()

    for i, turn in enumerate(interim_turns):
        turn.sender = "User"
        step = OrchStepPydantic(
            step_type="chat.send",
            sender=turn.sender,
            recipient=turn.recipient,
            step_no=step_no,
            is_pm=(flow_type == "poll"),
        )
        plan.steps.append(step)
        step_no += 1

        step = OrchStepPydantic(
            step_type="chat.reply",
            sender=turn.recipient,
            recipient=turn.sender,
            step_no=step_no,
            is_pm=(flow_type == "poll"),
        )
        plan.steps.append(step)
        step_no += 1

        participants.add(turn.sender)
        participants.add(turn.recipient)

    plan.participants = list(participants)
    if "User" in plan.participants:
        plan.participants.remove("User")
    return plan
