# -*- coding: utf-8 -*-

import json
import os
import yaml
from pathlib import Path

import pytest

from tuxlava.__main__ import main
from tuxlava.devices import Device
from tuxlava.devices.fvp import FVPLAVA, FVPMorelloAndroid
from tuxlava.devices.qemu import QemuArmv5
from tuxlava.exceptions import InvalidArgument

BASE = (Path(__file__) / "..").resolve()


def test_select():
    assert Device.select("qemu-armv5") == QemuArmv5
    assert Device.select("fvp-morello-android") == FVPMorelloAndroid
    assert Device.select("fvp-lava") == FVPLAVA

    with pytest.raises(InvalidArgument):
        Device.select("Hello")


ARTEFACTS = [
    "bzImage.gz",
    "zImage.xz",
    "modules.tar.xz",
    "tf-bl1.bin",
    "mcp_fw.bin",
    "mcp_romfw.bin",
    "android-nano.img.xz",
    "busybox.img.xz",
    "debian.img.xz",
    "core-image-minimal-morello-fvp.wic",
    "scp_fw.bin",
    "scp_romfw.bin",
    "fip.bin",
    "ubuntu.satadisk.xz",
    "startup.nsh",
    "kselftest.tar.xz",
]

FVP_MORELLO_ANDROID = [
    "--ap-romfw",
    "tf-bl1.bin",
    "--mcp-fw",
    "mcp_fw.bin",
    "--mcp-romfw",
    "mcp_romfw.bin",
    "--rootfs",
    "android-nano.img.xz",
    "--scp-fw",
    "scp_fw.bin",
    "--scp-romfw",
    "scp_romfw.bin",
    "--fip",
    "fip.bin",
]


metadata = {
    "results": {
        "artifacts": {"kernel": ["bzImage.gz"], "modules": ["modules.tar.xz"]},
    },
    "build": {"target_arch": "arm64"},
}


@pytest.fixture
def artefacts(tmp_path):
    os.chdir(tmp_path)
    for art in ARTEFACTS:
        (tmp_path / art).touch()
    (tmp_path / "metadata.json").write_text(json.dumps(metadata), encoding="utf-8")
    return tmp_path


@pytest.mark.parametrize(
    "args,filename",
    [
        (
            ["--device", "qemu-arm64"],
            "qemu-arm64.yaml",
        ),
        (
            ["--device", "qemu-arm64", "--timeouts", "deploy=4", "boot=12"],
            "qemu-arm64-timeouts.yaml",
        ),
        (
            ["--device", "qemu-arm64", "--tests", "ltp-fcntl-locktests"],
            "qemu-arm64-ltp-fcntl-locktests.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--tests",
                "kselftest-arm64",
            ],
            "qemu-arm64-kselftest-arm64.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian/bookworm/arm64/rootfs.ext4.xz",
                "--tests",
                "hacking-session",
                "--parameters",
                "PUB_KEY=ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABgQDNNtK2rxgzZ5qQHteYg3zh_rSD46m72eI8eL0nOoA_6dnLSQ_yHQsJkbtucJRQS1hu_hlgtNGNrURmXPUk5ZiWvSSpFinsnSmjbCRj5k1B2cbhuxeqdgcgOm5vvydVrPnRjekV0_DULCH_W2mooMH9cz1ouxUs08cdQivalmn6svrcoe3ozUklvEX7rDquv_vgx1_mcdmNuMNvQ6_k9PF2Q0Fi_qzYQbB_MSQGnNUx15lljOEKla5PwuwQsPWbHKh_r_XBz_lrK47PTRoxE5HPgpeCLxWx6L32BYLAxiCXIOBxNomfj__lFk0ZzQobAhvgAwSmEcVJAcotgch0bUN7N_Nr_RfBiqTguk0ADn1HPo4scwbYMl5MWNFg0CPZjIyYsKpeXZN2uM_Q_mUVC_WNKTjsmMWPcnoSjF8BbJsF1rzObg_QwJu0oNMkVj77bEDzSFsuQ8y9ovdaaXn26aYiDn2SKFC9MJTjZg22ncAK9MZmLoF3761mucWQjD51MXM_ anders@monster",
            ],
            "qemu-arm64-hacking-session.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian/bookworm/arm64/rootfs.ext4.xz",
                "--tests",
                "modules",
                "--parameters",
                "MODULES_SUBDIRS=net",
            ],
            "qemu-arm64-modules.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian/bookworm/arm64/rootfs.ext4.xz",
                "--tests",
                "ltp-syscalls",
                "--parameters",
                "RUNNER=kirk",
            ],
            "qemu-arm64-ltp-syscalls-kirk.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://example.com/script.sh",
                "/",
                "--tests",
                "hacking-session",
                "--parameters",
                "PUB_KEY=ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABgQDNNtK2rxgzZ5qQHteYg3zh_rSD46m72eI8eL0nOoA_6dnLSQ_yHQsJkbtucJRQS1hu_hlgtNGNrURmXPUk5ZiWvSSpFinsnSmjbCRj5k1B2cbhuxeqdgcgOm5vvydVrPnRjekV0_DULCH_W2mooMH9cz1ouxUs08cdQivalmn6svrcoe3ozUklvEX7rDquv_vgx1_mcdmNuMNvQ6_k9PF2Q0Fi_qzYQbB_MSQGnNUx15lljOEKla5PwuwQsPWbHKh_r_XBz_lrK47PTRoxE5HPgpeCLxWx6L32BYLAxiCXIOBxNomfj__lFk0ZzQobAhvgAwSmEcVJAcotgch0bUN7N_Nr_RfBiqTguk0ADn1HPo4scwbYMl5MWNFg0CPZjIyYsKpeXZN2uM_Q_mUVC_WNKTjsmMWPcnoSjF8BbJsF1rzObg_QwJu0oNMkVj77bEDzSFsuQ8y9ovdaaXn26aYiDn2SKFC9MJTjZg22ncAK9MZmLoF3761mucWQjD51MXM_ anders@monster",
            ],
            "qemu-arm64-overlay-script.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/lib/",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/trixie/arm64/rootfs.ext4.xz",
            ],
            "qemu-arm64-module-path.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-db-sqlite-insert-small.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-db-sqlite-insert-small",
            ],
            "qemu-arm64-mmtests-db-sqlite-insert-small.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-hpc-scimarkc-small.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-hpc-scimarkc-small",
            ],
            "qemu-arm64-mmtests-hpc-scimarkc-small.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-blogbench.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-io-blogbench",
            ],
            "qemu-arm64-mmtests-io-blogbench.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-fio-randread-async-randwrite.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-io-fio-randread-async-randwrite",
            ],
            "qemu-arm64-mmtests-io-fio-randread-async-randwrite.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-fio-randread-async-seqwrite.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-io-fio-randread-async-seqwrite",
            ],
            "qemu-arm64-mmtests-io-fio-randread-async-seqwrite.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-fio-randread-sync-heavywrite.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-io-fio-randread-sync-heavywrite",
            ],
            "qemu-arm64-mmtests-io-fio-randread-sync-heavywrite.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-fio-randread-sync-randwrite.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-io-fio-randread-sync-randwrite",
            ],
            "qemu-arm64-mmtests-io-fio-randread-sync-randwrite.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-fsmark-small-file-stream.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-io-fsmark-small-file-stream",
            ],
            "qemu-arm64-mmtests-io-fsmark-small-file-stream.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-memdb-redis-benchmark-small.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-memdb-redis-benchmark-small",
            ],
            "qemu-arm64-mmtests-memdb-redis-benchmark-small.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-memdb-redis-memtier-small.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-memdb-redis-memtier-small",
            ],
            "qemu-arm64-mmtests-memdb-redis-memtier-small.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-scheduler-schbench.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-scheduler-schbench",
            ],
            "qemu-arm64-mmtests-scheduler-schbench.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-scheduler-sysbench-cpu.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-scheduler-sysbench-cpu",
            ],
            "qemu-arm64-mmtests-scheduler-sysbench-cpu.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-scheduler-sysbench-thread.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-scheduler-sysbench-thread",
            ],
            "qemu-arm64-mmtests-scheduler-sysbench-thread.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-aim9-disk.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-aim9-disk",
            ],
            "qemu-arm64-mmtests-workload-aim9-disk.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-coremark.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-coremark",
            ],
            "qemu-arm64-mmtests-workload-coremark.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-cyclictest-fine-hackbench.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-cyclictest-fine-hackbench",
            ],
            "qemu-arm64-mmtests-workload-cyclictest-fine-hackbench.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-cyclictest-hackbench.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-cyclictest-hackbench",
            ],
            "qemu-arm64-mmtests-workload-cyclictest-hackbench.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-ebizzy.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-ebizzy",
            ],
            "qemu-arm64-mmtests-workload-ebizzy.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-pmqtest-hackbench.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-pmqtest-hackbench",
            ],
            "qemu-arm64-mmtests-workload-pmqtest-hackbench.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-af-alg.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-af-alg",
            ],
            "qemu-arm64-mmtests-workload-stressng-af-alg.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-bad-altstack.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-bad-altstack",
            ],
            "qemu-arm64-mmtests-workload-stressng-bad-altstack.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-class-io-parallel.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-class-io-parallel",
            ],
            "qemu-arm64-mmtests-workload-stressng-class-io-parallel.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-context.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-context",
            ],
            "qemu-arm64-mmtests-workload-stressng-context.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-fork.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-fork",
            ],
            "qemu-arm64-mmtests-workload-stressng-fork.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-get.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-get",
            ],
            "qemu-arm64-mmtests-workload-stressng-get.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-getdent.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-getdent",
            ],
            "qemu-arm64-mmtests-workload-stressng-getdent.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-madvise.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-madvise",
            ],
            "qemu-arm64-mmtests-workload-stressng-madvise.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-mmap.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-mmap",
            ],
            "qemu-arm64-mmtests-workload-stressng-mmap.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-usemem.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-usemem",
            ],
            "qemu-arm64-mmtests-workload-usemem.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-vm-splice.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-vm-splice",
            ],
            "qemu-arm64-mmtests-workload-stressng-vm-splice.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-zombie.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-zombie",
            ],
            "qemu-arm64-mmtests-workload-stressng-zombie.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-will-it-scale-io-processes.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-will-it-scale-io-processes",
            ],
            "qemu-arm64-mmtests-workload-will-it-scale-io-processes.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-will-it-scale-io-threads.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-will-it-scale-io-threads",
            ],
            "qemu-arm64-mmtests-workload-will-it-scale-io-threads.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-will-it-scale-pf-processes.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-will-it-scale-pf-processes",
            ],
            "qemu-arm64-mmtests-workload-will-it-scale-pf-processes.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-will-it-scale-pf-threads.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-will-it-scale-pf-threads",
            ],
            "qemu-arm64-mmtests-workload-will-it-scale-pf-threads.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-will-it-scale-sys-processes.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-will-it-scale-sys-processes",
            ],
            "qemu-arm64-mmtests-workload-will-it-scale-sys-processes.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-will-it-scale-sys-threads.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-will-it-scale-sys-threads",
            ],
            "qemu-arm64-mmtests-workload-will-it-scale-sys-threads.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-get.tar.xz",
                "/mmtests",
                "--tests",
                "mmtests-workload-stressng-get",
                "--parameters",
                "ITERATIONS=2",
            ],
            "qemu-arm64-mmtests-workload-stressng-get-iterations-2.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-stressng-get.tar.xz",
                "/opt/mmtests",
                "--tests",
                "mmtests-workload-stressng-get",
                "--parameters",
                "ITERATIONS=4",
                "--parameters",
                "MMTESTS_PATH=/opt/mmtests",
            ],
            "qemu-arm64-mmtests-workload-stressng-get-iterations-4-mmtests_path.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-workload-unixbench.tar.xz",
                "/opt/mmtests",
                "--tests",
                "mmtests-workload-unixbench",
                "--parameters",
                "ITERATIONS=4",
                "--parameters",
                "MMTESTS_PATH=/opt/mmtests",
            ],
            "qemu-arm64-mmtests-workload-unixbench-get-iterations-4-mmtests_path.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian-mmtests/bookworm/arm64/rootfs.ext4.xz",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/xfstests/master/xfstests.tar.xz",
                "/opt/",
                "--tests",
                "xfstests-ext4",
            ],
            "qemu-arm64-xfstests-ext4.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian/bookworm/arm64/rootfs.ext4.xz",
                "--tests",
                "systemd-analyze",
            ],
            "qemu-arm64-systemd-analyze.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "rw",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://storage.tuxboot.com/debian/trixie/arm64/rootfs.ext4.xz",
                "--tests",
                "tcpreplay",
            ],
            "qemu-arm64-tcpreplay.yaml",
        ),
        (
            ["--device", "qemu-arm64be"],
            "qemu-arm64be.yaml",
        ),
        (
            ["--device", "qemu-arm64be", "--qemu-image", "docker.io/qemu/qemu:latest"],
            "qemu-arm64be-qemu-image.yaml",
        ),
        (
            ["--device", "qemu-armv5"],
            "qemu-armv5.yaml",
        ),
        (
            ["--device", "qemu-armv5", "--tests", "ltp-fs_bind"],
            "qemu-armv5-ltp-fs_bind.yaml",
        ),
        (
            ["--device", "qemu-armv7"],
            "qemu-armv7.yaml",
        ),
        (
            [
                "--device",
                "qemu-armv7",
                "--tests",
                "kselftest-ipc",
                "--parameters",
                "KSELFTEST=https://example.com/kselftest.tar.xz",
            ],
            "qemu-armv7-kselftest-ipc.yaml",
        ),
        (
            [
                "--device",
                "qemu-armv7",
                "--tests",
                "kselftest-ipc",
                "--parameters",
                "KSELFTEST=https://example.com/kselftest.tar.xz",
                "SKIPFILE=/skipfile.yaml",
            ],
            "qemu-armv7-kselftest-ipc-skipfile.yaml",
        ),
        (
            [
                "--device",
                "qemu-armv7",
                "--tests",
                "kselftest-ipc",
                "--parameters",
                "CPUPOWER=https://example.com/cpupower.tar.xz",
                "KSELFTEST=https://example.com/kselftest.tar.xz",
            ],
            "qemu-armv7-kselftest-ipc-cpupower.yaml",
        ),
        (
            [
                "--device",
                "qemu-armv7",
                "--tests",
                "ltp-fs_perms_simple",
                "--tests",
                "ltp-fsx",
                "--tests",
                "ltp-nptl",
                "--timeouts",
                "ltp-fs_perms_simple=4",
                "--timeouts",
                "ltp-fsx=3",
            ],
            "qemu-armv7-ltp-mutliple-tests.yaml",
        ),
        (
            [
                "--device",
                "qemu-armv7",
                "--tests",
                "ltp-fs_perms_simple",
                "ltp-fsx",
                "ltp-nptl",
                "--timeouts",
                "ltp-fs_perms_simple=4",
                "ltp-fsx=3",
            ],
            "qemu-armv7-ltp-timeouts.yaml",
        ),
        (
            ["--device", "qemu-armv7", "--kernel", "zImage.xz"],
            "qemu-armv7-kernel-xz.yaml",
        ),
        (
            ["--device", "qemu-armv7be"],
            "qemu-armv7be.yaml",
        ),
        (
            [
                "--device",
                "qemu-armv7be",
                "--qemu-image",
                "docker.io/linaro/tuxlava-qemu:latest",
            ],
            "qemu-armv7be-qemu-image.yaml",
        ),
        (
            ["--device", "qemu-armv7be", "--prompt", "root@tuxlava"],
            "qemu-armv7be-prompt.yaml",
        ),
        (
            ["--device", "qemu-i386"],
            "qemu-i386.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "kunit"],
            "qemu-i386-kunit.yaml",
        ),
        (
            [
                "--device",
                "qemu-i386",
                "--tests",
                "kunit",
                "--parameters",
                "KUNIT_TEST_MODULE=test.ko",
            ],
            "qemu-i386-kunit-module.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "libgpiod"],
            "qemu-i386-libgpiod.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "libhugetlbfs"],
            "qemu-i386-libhugetlbfs.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "perf"],
            "qemu-i386-perf.yaml",
        ),
        (
            [
                "--device",
                "qemu-i386",
                "--tests",
                "perf",
                "--overlay",
                "http://example.com/overlay1.tar.xz",
                "--overlay",
                "http://example.com/overlay2.tar.xz",
                "--parameters",
                "PERF=https://example.com/perf.tar.xz",
            ],
            "qemu-i386-perf-parameters.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "rcutorture"],
            "qemu-i386-rcutorture.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "v4l2"],
            "qemu-i386-v4l2.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "vdso"],
            "qemu-i386-vdso.yaml",
        ),
        (
            [
                "--device",
                "qemu-i386",
                "--tests",
                "kunit",
                "--overlay",
                "http://example.com/overlay1.tar.xz",
                "--overlay",
                "http://example.com/overlay2.tar.xz",
            ],
            "qemu-i386-kunit-overlays.yaml",
        ),
        (
            [
                "--device",
                "qemu-i386",
                "--tests",
                "kunit",
                "--overlay",
                "http://example.com/overlay1.tar.xz",
                "/overlay1/",
                "--overlay",
                "http://example.com/overlay2.tar.xz",
                "/overlay2/",
            ],
            "qemu-i386-kunit-overlays-paths.yaml",
        ),
        (
            ["--device", "qemu-i386", "--tests", "kvm-unit-tests"],
            "qemu-i386-kvm-unit-tests.yaml",
        ),
        (
            ["--device", "qemu-i386", "--kernel", "bzImage.gz"],
            "qemu-i386-kernel-gz.yaml",
        ),
        (
            ["--device", "qemu-i386", "--boot-args", "bla blo"],
            "qemu-i386-boot-args.yaml",
        ),
        (
            [
                "--device",
                "qemu-i386",
                "--tests",
                "kselftest-ipc",
                "--parameters",
                "CPUPOWER=https://example.com/cpupower.tar.xz",
                "KSELFTEST=https://example.com/kselftest.tar.xz",
            ],
            "qemu-i386-kselftest-ipc.yaml",
        ),
        (
            ["--device", "qemu-mips32"],
            "qemu-mips32.yaml",
        ),
        (
            [
                "--device",
                "qemu-mips32",
                "--modules",
                "https://example.com/modules.tar.xz",
            ],
            "qemu-mips32-modules.yaml",
        ),
        (
            [
                "--device",
                "qemu-mips32",
                "--modules",
                "https://example.com/modules.tgz",
            ],
            "qemu-mips32-modules-tgz.yaml",
        ),
        (
            [
                "--device",
                "qemu-mips32",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--overlay",
                "http://example.com/overlay2.tar.xz",
                "--tests",
                "kunit",
            ],
            "qemu-mips32-modules-overlays-kunit.yaml",
        ),
        (
            ["--device", "qemu-mips32", "--", "cat", "/proc/cpuinfo"],
            "qemu-mips32-command.yaml",
        ),
        (
            ["--device", "qemu-mips32el"],
            "qemu-mips32el.yaml",
        ),
        (
            ["--device", "qemu-mips64"],
            "qemu-mips64.yaml",
        ),
        (
            ["--device", "qemu-mips64"],
            "qemu-mips64el.yaml",
        ),
        (
            ["--device", "qemu-ppc32"],
            "qemu-ppc32.yaml",
        ),
        (
            ["--device", "qemu-ppc64"],
            "qemu-ppc64.yaml",
        ),
        (
            ["--device", "qemu-ppc64le"],
            "qemu-ppc64le.yaml",
        ),
        (
            ["--device", "qemu-s390"],
            "qemu-s390.yaml",
        ),
        (
            ["--device", "qemu-s390", "--tests", "ltp-smoke"],
            "qemu-s390-ltp-smoke.yaml",
        ),
        (
            [
                "--device",
                "qemu-s390",
                "--tests",
                "ltp-smoke",
                "--parameters",
                "SKIPFILE=skipfile-lkft.yaml",
            ],
            "qemu-s390-ltp-smoke-skipfile.yaml",
        ),
        (
            ["--device", "qemu-riscv32"],
            "qemu-riscv32.yaml",
        ),
        (
            ["--device", "qemu-riscv64"],
            "qemu-riscv64.yaml",
        ),
        (
            ["--device", "qemu-sh4"],
            "qemu-sh4.yaml",
        ),
        (
            ["--device", "qemu-sh4", "--boot-args", "hello"],
            "qemu-sh4-boot-args.yaml",
        ),
        (
            ["--device", "qemu-sparc64"],
            "qemu-sparc64.yaml",
        ),
        (
            ["--device", "qemu-x86_64"],
            "qemu-x86_64.yaml",
        ),
        (
            ["--device", "qemu-x86_64", "--enable-kvm"],
            "qemu-x86_64-enable-kvm.yaml",
        ),
        (
            [
                "--device",
                "qemu-x86_64",
                "--rootfs",
                "https://example.com/rootfs.ext4.zst",
            ],
            "qemu-x86_64-rootfs-zst.yaml",
        ),
        (
            [
                "--device",
                "qemu-x86_64",
                "--rootfs",
                "https://example.com/rootfs.ext4",
            ],
            "qemu-x86_64-rootfs.yaml",
        ),
        (
            [
                "--device",
                "qemu-x86_64",
                "--rootfs",
                "https://example.com/rootfs.ext4.gz",
            ],
            "qemu-x86_64-rootfs-gz.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--bl1",
                "tf-bl1.bin",
                "--fip",
                "fip.bin",
                "--kernel",
                "zImage.xz",
                "--rootfs",
                "https://example.com/rootfs.ext4.zst",
                "--uefi",
                "https://example.com/uefi.bin",
            ],
            "fvp-aemva-kernel-xz.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--bl1",
                "tf-bl1.bin",
                "--fip",
                "fip.bin",
                "--rootfs",
                "https://example.com/rootfs.ext4.zst",
                "--tests",
                "ltp-smoke",
            ],
            "fvp-aemva-ltp-smoke.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--bl1",
                "tf-bl1.bin",
                "--fip",
                "fip.bin",
                "--rootfs",
                "https://example.com/rootfs.ext4.zst",
            ],
            "fvp-aemva.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--prompt",
                "root@tuxlava",
            ],
            "fvp-aemva-prompt.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--bl1",
                "tf-bl1.bin",
                "--fip",
                "fip.bin",
                "--modules",
                "https://example.com/modules.tar.xz",
            ],
            "fvp-aemva-modules.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--bl1",
                "tf-bl1.bin",
                "--fip",
                "fip.bin",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
            ],
            "fvp-aemva-modules-path-usr.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
                "--bl1",
                "tf-bl1.bin",
                "--fip",
                "fip.bin",
                "--modules",
                "https://example.com/modules.tar.gz",
            ],
            "fvp-aemva-modules-tar-gz.yaml",
        ),
        (
            [
                "--device",
                "fvp-aemva",
            ],
            "fvp-aemva-defaults.yaml",
        ),
        (
            ["--device", "fvp-morello-android", *FVP_MORELLO_ANDROID],
            "fvp-morello-android.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "binder",
            ],
            "fvp-morello-android-binder.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "bionic",
            ],
            "fvp-morello-android-bionic.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "bionic",
                "--parameters",
                "GTEST_FILTER=hello",
                "BIONIC_TEST_TYPE=dynamic",
            ],
            "fvp-morello-android-bionic-params.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "boringssl",
                "--parameters",
                "SYSTEM_URL=system.tar.xz",
            ],
            "fvp-morello-android-boringssl.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "boringssl",
                "--parameters",
                "SYSTEM_URL=system.tar.xz",
                "--timeouts",
                "boringssl=4212",
            ],
            "fvp-morello-android-boringssl-timeouts.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "compartment",
                "--parameters",
                "USERDATA=userdata.tar.xz",
            ],
            "fvp-morello-android-compartment.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "device-tree",
            ],
            "fvp-morello-android-device-tree.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "dvfs",
            ],
            "fvp-morello-android-dvfs.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "libjpeg-turbo",
                "--parameters",
                "SYSTEM_URL=system.tar.xz",
                "LIBJPEG_TURBO_URL=libjpeg.tar.xz",
            ],
            "fvp-morello-android-libjpeg-turbo.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "libpdfium",
                "--parameters",
                "SYSTEM_URL=system.tar.xz",
                "PDFIUM_URL=pdfium.tar.xz",
            ],
            "fvp-morello-android-libpdfium.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "libpng",
                "--parameters",
                "SYSTEM_URL=system.tar.xz",
                "PNG_URL=png.tar.xz",
            ],
            "fvp-morello-android-libpng.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "lldb",
                "--parameters",
                "LLDB_URL=lldb.tar.xz",
                "TC_URL=toolchain.tar.xz",
            ],
            "fvp-morello-android-lldb.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "logd",
                "--parameters",
                "USERDATA=userdata.tar.xz",
            ],
            "fvp-morello-android-logd.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "multicore",
            ],
            "fvp-morello-android-multicore.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "smc91x",
            ],
            "fvp-morello-android-smc91x.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "virtio_net",
            ],
            "fvp-morello-android-virtio_net.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-android",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "zlib",
                "--parameters",
                "SYSTEM_URL=system.tar.xz",
            ],
            "fvp-morello-android-zlib.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-busybox",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "busybox.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
            ],
            "fvp-morello-busybox.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-busybox",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "busybox.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "purecap",
            ],
            "fvp-morello-busybox-purecap.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-grub",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "debian.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "boot-debian-dt",
            ],
            "fvp-morello-grub-boot-debian-dt.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-grub",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "debian.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "boot-busybox-dt",
            ],
            "fvp-morello-grub-boot-busybox-dt.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-grub",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "debian.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "boot-debian-acpi",
            ],
            "fvp-morello-grub-boot-debian-acpi.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-grub",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "debian.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "boot-busybox-acpi",
            ],
            "fvp-morello-grub-boot-busybox-acpi.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-debian",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "debian.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "debian-purecap",
            ],
            "fvp-morello-debian-purecap.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-busybox",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "busybox.img.xz",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
                "--tests",
                "smc91x",
            ],
            "fvp-morello-busybox-smc91x.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-oe",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--rootfs",
                "core-image-minimal-morello-fvp.wic",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
            ],
            "fvp-morello-oe.yaml",
        ),
        (
            [
                "--device",
                "fvp-morello-ubuntu",
                "--ap-romfw",
                "tf-bl1.bin",
                "--mcp-fw",
                "mcp_fw.bin",
                "--mcp-romfw",
                "mcp_romfw.bin",
                "--scp-fw",
                "scp_fw.bin",
                "--scp-romfw",
                "scp_romfw.bin",
                "--fip",
                "fip.bin",
            ],
            "fvp-morello-ubuntu.yaml",
        ),
        (
            ["--tuxmake", ".", "--tests", "kselftest-cgroup"],
            "tuxmake.yaml",
        ),
        (
            [
                "--tuxmake",
                ".",
                "--tests",
                "kselftest-cgroup",
                "--parameters",
                "KSELFTEST=$BUILD/kselftest.tar.xz",
            ],
            "tuxmake-parameters.yaml",
        ),
        (
            [
                "--device",
                "fvp-lava",
                "--fvp-ubl-license",
                "test-license-key",
                "--job-definition",
                f"{BASE}/refs/definitions/fvp-lava-job-definition.yaml",
            ],
            "fvp-lava-job-definition.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/trixie/arm64/rt-tests/v2.8/rt-tests.tar.xz",
                "/",
                "--tests",
                "rt-tests-cyclicdeadline",
            ],
            "qemu-arm64-rt-tests-cyclicdeadline.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/trixie/arm64/rt-tests/v2.8/rt-tests.tar.xz",
                "/",
                "--tests",
                "rt-tests-rt-migrate-test",
            ],
            "qemu-arm64-rt-tests-rt-migrate-test.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/trixie/arm64/rt-tests/v2.8/rt-tests.tar.xz",
                "/",
                "--tests",
                "rt-tests-pi-stress",
            ],
            "qemu-arm64-rt-tests-pi-stress.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/trixie/arm64/rt-tests/v2.8/rt-tests.tar.xz",
                "/",
                "--tests",
                "rt-tests-pmqtest",
            ],
            "qemu-arm64-rt-tests-pmqtest.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/trixie/arm64/rt-tests/v2.8/rt-tests.tar.xz",
                "/",
                "--tests",
                "rt-tests-signaltest",
            ],
            "qemu-arm64-rt-tests-signaltest.yaml",
        ),
        (
            [
                "--device",
                "ssh-device",
                "--ssh-host",
                "127.0.0.1",
                "--ssh-user",
                "testuser",
                "--ssh-identity-file",
                f"{BASE}/refs/dummy.pem",
                "--ssh-prompt",
                "testuser",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-blogbench2.tar.xz",
                "/mmtests",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-blogbench.tar.xz",
                "/mmtests",
                "--tests",
                "ltp-smoke",
            ],
            "ssh-device.yaml",
        ),
        (
            [
                "--device",
                "avh-rpi4b",
                "--secrets",
                "avh_api_token=avhapitoken",
                "--kernel",
                "https://example.com/Image",
                "--dtb",
                "https://example.com/devicetree",
                "--rootfs",
                "https://example.com/rootfs.ext4",
            ],
            "avh-rpi4b.yaml",
        ),
        (
            [
                "--device",
                "avh-imx93",
                "--secrets",
                "avh_api_token=avhapitoken",
                "--kernel",
                "https://example.com/Image",
                "--dtb",
                "https://example.com/devicetree",
                "--rootfs",
                "https://example.com/rootfs.ext4",
            ],
            "avh-imx93.yaml",
        ),
        (
            [
                "--device",
                "avh-imx93",
                "--secrets",
                "avh_api_token=avhapitoken",
                "--kernel",
                "https://example.com/Image",
                "--dtb",
                "https://example.com/devicetree",
                "--rootfs",
                "https://example.com/rootfs.ext4",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--partition",
                "1",
            ],
            "avh-imx93-module-overlay.yaml",
        ),
        (
            [
                "--device",
                "avh-imx93",
                "--secrets",
                "avh_api_token=avhapitoken",
                "--kernel",
                "https://example.com/Image",
                "--dtb",
                "https://example.com/devicetree",
                "--rootfs",
                "https://example.com/rootfs.ext4",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--partition",
                "1",
                "--boot-args",
                "console=ttyLP0,115200 earlycon= root=/dev/mmcblk0p2",
            ],
            "avh-imx93-boot-args.yaml",
        ),
        (
            [
                "--device",
                "avh-imx93",
                "--secrets",
                "avh_api_token=avhapitoken",
                "--kernel",
                "https://example.com/Image",
                "--dtb",
                "https://example.com/devicetree",
                "--rootfs",
                "https://example.com/rootfs.ext4",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--partition",
                "1",
                "--boot-args",
                "console=ttyLP0,115200 earlycon= root=/dev/mmcblk0p2",
                "--tests",
                "ltp-smoke",
            ],
            "avh-imx93-ltp-smoke.yaml",
        ),
        (
            [
                "--device",
                "avh-imx93",
                "--secrets",
                "avh_api_token=avhapitoken",
                "--kernel",
                "https://example.com/Image",
                "--dtb",
                "https://example.com/devicetree",
                "--rootfs",
                "https://example.com/rootfs.ext4",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--partition",
                "1",
                "--boot-args",
                "console=ttyLP0,115200 earlycon= root=/dev/mmcblk0p2",
                "--tests",
                "ltp-smoke",
                "--tests",
                "kselftest-arm64",
            ],
            "avh-imx93-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--boot-args",
                "earlycon",
                "--kernel",
                "https://example.com/Image",
                "--bios",
                "https://example.com/Bios",
                "--enable-trustzone",
                "--tests",
                "tfa-tests",
            ],
            "tfa-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-ampereone",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "--lava-definition",
            ],
            "nfs-ampereone-ltp-smoke.yaml",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
            ],
            "nfs-x86-64-module-path.yaml",
        ),
        (
            [
                "--device",
                "nfs-ampereone",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
            ],
            "nfs-ampereone-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-orion-o6",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "--lava-definition",
            ],
            "nfs-orion-o6-ltp-smoke.yaml",
        ),
        (
            [
                "--device",
                "nfs-orion-o6",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
            ],
            "nfs-orion-o6-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-x86",
                "kunit",
            ],
            "nfs-x86-64-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-i386",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
            ],
            "nfs-i386-module-path.yaml",
        ),
        (
            [
                "--device",
                "nfs-i386",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-x86",
                "kunit",
            ],
            "nfs-i386-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-bcm2711-rpi-4-b",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
            ],
            "nfs-bcm2711-rpi-4-b-module-path.yaml",
        ),
        (
            [
                "--device",
                "nfs-bcm2711-rpi-4-b",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
            ],
            "nfs-bcm2711-rpi-4-b-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-juno-r2",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
            ],
            "nfs-juno-r2-module-path.yaml",
        ),
        (
            [
                "--device",
                "nfs-juno-r2",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
            ],
            "nfs-juno-r2-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-s32g399a-rdb3",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
            ],
            "nfs-s32g399a-rdb3-module-path.yaml",
        ),
        (
            [
                "--device",
                "nfs-rk3399-rock-pi-4b",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
            ],
            "nfs-rk3399-rock-pi-4b-module-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-e850-96",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/exynos/exynos850-e850-96.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
            ],
            "fastboot-e850-96-module-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-e850-96",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/exynos/exynos850-e850-96.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
            ],
            "fastboot-e850-96-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/qcom/apq8016-sbc.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
            ],
            "fastboot-dragonboard-410c-module-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/qcom/apq8016-sbc.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
            ],
            "fastboot-dragonboard-410c-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-845c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/qcom/sdm845-db845c.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
            ],
            "fastboot-dragonboard-845c-module-path-no-tags.yaml",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-845c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/qcom/sdm845-db845c.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--parameters",
                "TAGS=lts",
            ],
            "fastboot-dragonboard-845c-module-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-845c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/qcom/sdm845-db845c.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--parameters",
                "TAGS=lts,production,development",
            ],
            "fastboot-dragonboard-845c-module-path-multiple-tags.yaml",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-845c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/qcom/sdm845-db845c.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
                "--parameters",
                "TAGS=lts",
            ],
            "fastboot-dragonboard-845c-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "fastboot-gs101-oriole",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.ext4.xz",
                "--dtb",
                "https://example.com/dtbs/google/gs101-oriole.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--parameters",
                "TAGS=lts",
            ],
            "fastboot-gs101-oriole-module-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-gs101-oriole",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.ext4.xz",
                "--dtb",
                "https://example.com/dtbs/google/gs101-oriole.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--parameters",
                "TAGS=lts,production,development",
            ],
            "fastboot-gs101-oriole-module-path-multiple-tags.yaml",
        ),
        (
            [
                "--device",
                "fastboot-gs101-oriole",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.ext4.xz",
                "--dtb",
                "https://example.com/dtbs/google/gs101-oriole.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-arm64",
                "kunit",
                "--parameters",
                "TAGS=lts",
            ],
            "fastboot-gs101-oriole-multiple-tests.yaml",
        ),
        (
            [
                "--device",
                "fastboot-x15",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/dtbs/am57xx-beagle-x15.dtb",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--deploy-os",
                "debian",
            ],
            "fastboot-x15-module-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-dragonboard-845c",
                "--parameters",
                "LAVA_QUEUE_TIMEOUT=100",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
            ],
            "fastboot-aosp-dragonboard-845c-boot-test.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--boot-args",
                "blah blah",
                "--prompt",
                "blaah",
                "--parameters",
                "LAVA_JOB_PRIORITY=99",
                "LAVA_QUEUE_TIMEOUT=100",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
            ],
            "fastboot-aosp-qrb5165-rb5-boot-test.yaml",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--tests",
                "network-basic",
            ],
            "network-basic-tests.yaml",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
            ],
            "test-modules-default-path.yaml",
        ),
        (
            [
                "--device",
                "fastboot-qrb5165-rb5",
                "commands",
                "cd /opt/ && ls",
                "--bios",
                "https://bios/url",
            ],
            "fastboot-qrb5165-rb5.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-dragonboard-845c",
                "--tests",
                "android-cts",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_CTS_URL=https://test/cts/url",
            ],
            "fastboot-aosp-dragonboard-845c-cts.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-dragonboard-845c",
                "--tests",
                "android-vts",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_VTS_URL=https://test/cts/url",
            ],
            "fastboot-aosp-dragonboard-845c-vts.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-dragonboard-845c",
                "--tests",
                "android-vts-kernel-v7a",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_VTS_URL=https://test/cts/url",
                "SQUAD_URL=https://squad/url",
            ],
            "fastboot-aosp-dragonboard-845c-vts-kernel-v7a.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-dragonboard-845c",
                "--tests",
                "android-vts-kernel-v8a",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_VTS_URL=https://test/cts/url",
            ],
            "fastboot-aosp-dragonboard-845c-vts-kernel-v8a.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--tests",
                "android-cts",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_CTS_URL=https://test/cts/url",
            ],
            "fastboot-arb5165-rb5-cts.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--tests",
                "android-vts",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_VTS_URL=https://test/cts/url",
            ],
            "fastboot-arb5165-rb5-vts.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--tests",
                "android-vts-kernel-v7a",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_VTS_URL=https://test/cts/url",
            ],
            "fastboot-arb5165-rb5-vts-kernel-v7a.yaml",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--tests",
                "android-vts-kernel-v8a",
                "--parameters",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/url",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "LKFT_BUILD_CONFIG=lkft-config",
                "TEST_VTS_URL=https://test/cts/url",
                "SQUAD_URL=https://squad/url",
            ],
            "fastboot-arb5165-rb5-vts-kernel-v8a.yaml",
        ),
        (
            [
                "--device",
                "qemu-arm64",
                "--tests",
                "usb-gadget-framework",
            ],
            "qemu-arm64-pheripherals.yaml",
        ),
        (
            [
                "--device",
                "flasher-debian-qcs6490-rb3gen2-core-kit",
                "--tests",
                "smoke",
                "--parameters",
                'TESTS="foo bar"',
                "--rootfs",
                "http://example.com/flasher.tar.gz",
            ],
            "flasher-debian-qcs6490-rb3gen2-core-kit-smoke.yaml",
        ),
        (
            [
                "--device",
                "flasher-qcom-distro-qcs6490-rb3gen2-core-kit",
                "--tests",
                "smoke",
                "--parameters",
                'TESTS="foo bar"',
                "--rootfs",
                "http://example.com/flasher.tar.gz",
            ],
            "flasher-qcom-distro-qcs6490-rb3gen2-core-kit-smoke.yaml",
        ),
    ],
)
def test_definition(monkeypatch, mocker, capsys, tmpdir, artefacts, args, filename):
    monkeypatch.setattr("tuxlava.__main__.sys.argv", ["tuxlava"] + args)
    mocker.patch("tempfile.mkdtemp", return_value=tmpdir)
    main()
    output, error = capsys.readouterr()

    for art in ARTEFACTS:
        output = output.replace(f"file://{artefacts}/{art}", f"/DATA/{art}")
    output = output.replace(
        f'container_name: "{os.path.basename(tmpdir)}"', 'container_name: "tuxlava-ci"'
    )
    output = output.replace(
        f'network_from: "{os.path.basename(tmpdir)}"', 'network_from: "tuxlava-ci"'
    )

    if os.environ.get("TUXLAVA_RENDER"):
        (BASE / "refs" / "definitions" / filename).write_text(output, encoding="utf-8")
    assert output == (BASE / "refs" / "definitions" / filename).read_text(
        encoding="utf-8"
    )
    try:
        with open((BASE / "refs" / "definitions" / filename), "r") as yaml_file:
            yaml.load(yaml_file, Loader=yaml.SafeLoader)
    except yaml.YAMLError as e:
        pytest.fail(f"YAML file '{filename}' is invalid. Error: {e}")


@pytest.mark.parametrize(
    "args,error_str",
    [
        (
            [
                "--device",
                "nfs-juno-r2",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--overlay",
                "https://example.com/ltp.tar.xz",
                "--tests",
                "ltp-smoke",
                "ltp-io",
                "ltp-math",
                "perf",
                "kselftest-x86",
                "kunit",
            ],
            "Test 'kselftest-x86' not supported on device",
        ),
        (
            [
                "--device",
                "nfs-juno-r2",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--boot-args",
                'blah"blah',
            ],
            'argument --boot-args should not contain "',
        ),
        (
            [
                "--device",
                "nfs-juno-r2",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--prompt",
                'blah"blah',
            ],
            'argument --prompt should not contain "',
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--dtb",
                "https://example.com/blah.dtb",
            ],
            "argument --dtb is only valid for ",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/xyz.xyz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
            ],
            "argument --modules should be a .tar.gz, tar.xz or .tgz",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--parameters",
                "LAVA_JOB_PRIORITY=101",
            ],
            "argument --parameters LAVA_JOB_PRIORITY must be a value between 1-100",
        ),
        (
            [
                "--device",
                "nfs-x86_64",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--ramdisk",
                "https://example.com/ramdisk.gz",
            ],
            "Invalid option(s) for nfs devices:",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-845c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--enable-kvm",
            ],
            "Invalid option(s) for fastboot devices:",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--bios",
                "https://example.com/bios.img",
            ],
            "argument --bios is only valid for 'fastboot-dragonboard-845c' and 'fastboot-qrb5165-rb5' device",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--ramdisk",
                "https://example.com/ramdisk.gz",
            ],
            "argument --ramdisk is only valid for 'fastboot-dragonboard-845c' and 'fastboot-qrb5165-rb5' device",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--boot-args",
                'blah"blah',
            ],
            'argument --boot-args should not contain "',
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--prompt",
                'blah"blah',
            ],
            'argument --prompt should not contain "',
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xyz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
            ],
            "argument --modules should be a .tar.gz, tar.xz or .tgz",
        ),
        (
            [
                "--device",
                "fastboot-dragonboard-410c",
                "--kernel",
                "https://example.com/Image.gz",
                "--modules",
                "https://example.com/modules.tar.xz",
                "/usr/",
                "--rootfs",
                "https://example.com/rootfs.tar.xz",
                "--parameters",
                "LAVA_JOB_PRIORITY=101",
            ],
            "argument --parameters LAVA_JOB_PRIORITY must be a value between 1-100",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
                "--ramdisk",
                "https://example.com/ramdisk.gz",
            ],
            "Invalid option(s) for fastboot aosp devices:",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
                "--boot-args",
                'blah"blah',
            ],
            'argument --boot-args should not contain "',
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
                "--prompt",
                'blah"blah',
            ],
            'argument --prompt should not contain "',
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
                "--parameters",
                "LAVA_JOB_PRIORITY=101",
            ],
            "argument --parameters LAVA_JOB_PRIORITY must be a value between 1-100",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-db845c-aosp-master-mainline-gki",
                "IMAGE_SUPPORTED_VENDOR_BOOT='true'",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://stylesen.dev.storage.tuxsuite.com/public/tuxsuite/senthil/oebuilds/2k0BQ4PP7MJG21NqvHmWZpCA6RC/Image.gz",
            ],
            "argument --parameters TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL must be provided",
        ),
        (
            [
                "--device",
                "fastboot-aosp-dragonboard-845c",
                "--tests",
                "android-cts",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-config",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/download/url",
            ],
            "argument missing --parameters 'TEST_CTS_URL='...'",
        ),
        (
            [
                "--device",
                "fastboot-aosp-qrb5165-rb5",
                "--tests",
                "android-vts-kernel-v7a",
                "--parameters",
                "LKFT_BUILD_CONFIG=lkft-config",
                "BUILD_REFERENCE_IMAGE_GZ_URL=https://reference/image/url",
                "TUXSUITE_BAKE_VENDOR_DOWNLOAD_URL=https://vendor/download/url",
            ],
            "argument missing --parameters 'TEST_VTS_URL='...'",
        ),
        (
            [
                "--device",
                "ssh-device",
                "--ssh-user",
                "testuser",
                "--ssh-identity-file",
                f"{BASE}/refs/dummy.pem",
                "--ssh-prompt",
                "testuser",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-blogbench2.tar.xz",
                "/mmtests",
                "--overlay",
                "https://storage.tuxboot.com/overlays/debian/bookworm/arm64/mmtests/master/config-io-blogbench.tar.xz",
                "/mmtests",
                "--tests",
                "ltp-smoke",
            ],
            "'ssh-host', ssh-user', 'ssh-identity-file' are required argument for ssh device",
        ),
        (["--device", "fvp-lava"], "Missing argument --job-definition"),
        (
            [
                "--device",
                "avh-imx93",
            ],
            "'--secrets' is required by AVH device",
        ),
        (
            ["--device", "fvp-aemva", "--boot-args", '"rw'],
            '--boot-args should not contains "',
        ),
        (
            ["--device", "fvp-morello-android"],
            "Missing option(s) for fvp devices: --ap-romfw, --fip, --mcp-fw, --mcp-romfw, --rootfs, --scp-fw, --scp-romfw",
        ),
        (
            [
                "--device",
                "fvp-morello-baremetal",
                *FVP_MORELLO_ANDROID,
                "--tests",
                "binder",
            ],
            "Tests are not supported on this device",
        ),
        (
            [
                "--device",
                "fvp-morello-ubuntu",
                *FVP_MORELLO_ANDROID,
                "--rootfs",
                "https://url/rootfs",
            ],
            "Invalid option for this fvp device: --rootfs",
        ),
        (
            ["--device", "fastboot-dragonboard-845c", "--shared"],
            "--shared options is only available for qemu devices",
        ),
        (
            ["--device", "qemu-arm64", "--tests", "ltp-smoke", "ltp-smoke"],
            "each test should appear only once",
        ),
        (
            [
                "--device",
                "flasher-debian-qcs6490-rb3gen2-core-kit",
                "--tests",
                "smoke",
                "--parameters",
                'TESTS="foo bar"',
            ],
            "Missing --rootfs argument. Can't render the template",
        ),
    ],
)
def test_failures(monkeypatch, mocker, capsys, tmpdir, args, error_str):
    monkeypatch.setattr("tuxlava.__main__.sys.argv", ["tuxlava"] + args)
    mocker.patch("tempfile.mkdtemp", return_value=tmpdir)
    with pytest.raises(SystemExit):
        main()
    _, error = capsys.readouterr()
    assert error_str in error


def test_fvp_aemva_extra_assets(tmpdir):
    device = Device.select("fvp-aemva")()

    # 1/ default case
    asset = device.extra_assets(dtb=None, kernel=None, tmpdir=tmpdir, tux_boot_args="")
    assert len(asset) == 1
    assert asset[0] == f"file://{tmpdir / 'startup.nsh'}"
    assert (tmpdir / "startup.nsh").read_text(
        encoding="utf-8"
    ) == "Image dtb=fvp-base-revc.dtb systemd.log_level=warning console=ttyAMA0 earlycon=pl011,0x1c090000 root=/dev/vda ip=dhcp"

    # 2/ custom urls
    asset = device.extra_assets(
        dtb="file://hello/world/fdt.dtb",
        kernel="http://example.com/kernel",
        tmpdir=tmpdir,
        tux_boot_args="",
    )
    assert len(asset) == 1
    assert asset[0] == f"file://{tmpdir / 'startup.nsh'}"
    assert (tmpdir / "startup.nsh").read_text(
        encoding="utf-8"
    ) == "kernel dtb=fdt.dtb systemd.log_level=warning console=ttyAMA0 earlycon=pl011,0x1c090000 root=/dev/vda ip=dhcp"

    # 3/ compression
    asset = device.extra_assets(
        dtb="file://tmp/my-dtb", kernel="Image.gz", tmpdir=tmpdir, tux_boot_args=None
    )
    assert len(asset) == 1
    assert asset[0] == f"file://{tmpdir / 'startup.nsh'}"
    assert (tmpdir / "startup.nsh").read_text(
        encoding="utf-8"
    ) == "Image dtb=my-dtb systemd.log_level=warning console=ttyAMA0 earlycon=pl011,0x1c090000 root=/dev/vda ip=dhcp"

    # 4/ custom boot-args
    asset = device.extra_assets(
        dtb=None, kernel=None, tmpdir=tmpdir, tux_boot_args="debug"
    )
    assert len(asset) == 1
    assert asset[0] == f"file://{tmpdir / 'startup.nsh'}"
    assert (tmpdir / "startup.nsh").read_text(
        encoding="utf-8"
    ) == "Image dtb=fvp-base-revc.dtb systemd.log_level=warning debug console=ttyAMA0 earlycon=pl011,0x1c090000 root=/dev/vda ip=dhcp"
