"""Client for interacting with LLMs."""

from __future__ import annotations

import sys
import time
from typing import TYPE_CHECKING

import pyperclip
from rich.live import Live

from agent_cli.core.utils import console, live_timer, print_error_message, print_output_panel

if TYPE_CHECKING:
    import logging

    from pydantic_ai import Agent
    from pydantic_ai.models.openai import OpenAIModel
    from pydantic_ai.tools import Tool

    from agent_cli import config


def _openai_llm_model(openai_config: config.OpenAILLM) -> OpenAIModel:
    from pydantic_ai.models.openai import OpenAIModel  # noqa: PLC0415
    from pydantic_ai.providers.openai import OpenAIProvider  # noqa: PLC0415

    if not openai_config.openai_api_key:
        msg = "OpenAI API key is not set."
        raise ValueError(msg)
    provider = OpenAIProvider(api_key=openai_config.openai_api_key)
    model_name = openai_config.openai_llm_model
    return OpenAIModel(model_name=model_name, provider=provider)


def _ollama_llm_model(ollama_config: config.Ollama) -> OpenAIModel:
    from pydantic_ai.models.openai import OpenAIModel  # noqa: PLC0415
    from pydantic_ai.providers.openai import OpenAIProvider  # noqa: PLC0415

    provider = OpenAIProvider(base_url=f"{ollama_config.ollama_host}/v1")
    model_name = ollama_config.ollama_model
    return OpenAIModel(model_name=model_name, provider=provider)


def build_agent(
    provider_config: config.ProviderSelection,
    ollama_config: config.Ollama,
    openai_config: config.OpenAILLM,
    *,
    system_prompt: str | None = None,
    instructions: str | None = None,
    tools: list[Tool] | None = None,
) -> Agent:
    """Construct and return a PydanticAI agent."""
    from pydantic_ai import Agent  # noqa: PLC0415

    if provider_config.llm_provider == "openai":
        llm_model = _openai_llm_model(openai_config)
    elif provider_config.llm_provider == "local":
        llm_model = _ollama_llm_model(ollama_config)

    return Agent(
        model=llm_model,
        system_prompt=system_prompt or (),
        instructions=instructions,
        tools=tools or [],
    )


# --- LLM (Editing) Logic ---

INPUT_TEMPLATE = """
<original-text>
{original_text}
</original-text>

<instruction>
{instruction}
</instruction>
"""


async def get_llm_response(
    *,
    system_prompt: str,
    agent_instructions: str,
    user_input: str,
    provider_config: config.ProviderSelection,
    ollama_config: config.Ollama,
    openai_config: config.OpenAILLM,
    logger: logging.Logger,
    live: Live | None = None,
    tools: list[Tool] | None = None,
    quiet: bool = False,
    clipboard: bool = False,
    show_output: bool = False,
    exit_on_error: bool = False,
) -> str | None:
    """Get a response from the LLM with optional clipboard and output handling."""
    agent = build_agent(
        provider_config=provider_config,
        ollama_config=ollama_config,
        openai_config=openai_config,
        system_prompt=system_prompt,
        instructions=agent_instructions,
        tools=tools,
    )

    start_time = time.monotonic()

    try:
        model_name = (
            ollama_config.ollama_model
            if provider_config.llm_provider == "local"
            else openai_config.openai_llm_model
        )

        async with live_timer(
            live or Live(console=console),
            f"🤖 Applying instruction with {model_name}",
            style="bold yellow",
            quiet=quiet,
        ):
            result = await agent.run(user_input)

        elapsed = time.monotonic() - start_time
        result_text = result.output

        if clipboard:
            pyperclip.copy(result_text)
            logger.info("Copied result to clipboard.")

        if show_output and not quiet:
            print_output_panel(
                result_text,
                title="✨ Result (Copied to Clipboard)" if clipboard else "✨ Result",
                subtitle=f"[dim]took {elapsed:.2f}s[/dim]",
            )
        elif quiet and clipboard:
            print(result_text)

        return result_text

    except Exception as e:
        logger.exception("An error occurred during LLM processing.")
        if provider_config.llm_provider == "openai":
            msg = "Please check your OpenAI API key."
        else:
            msg = f"Please check your Ollama server at [cyan]{ollama_config.ollama_host}[/cyan]"
        print_error_message(f"An unexpected LLM error occurred: {e}", msg)
        if exit_on_error:
            sys.exit(1)
        return None


async def process_and_update_clipboard(
    system_prompt: str,
    agent_instructions: str,
    *,
    provider_config: config.ProviderSelection,
    ollama_config: config.Ollama,
    openai_config: config.OpenAILLM,
    logger: logging.Logger,
    original_text: str,
    instruction: str,
    clipboard: bool,
    quiet: bool,
    live: Live,
) -> None:
    """Processes the text with the LLM, updates the clipboard, and displays the result."""
    user_input = INPUT_TEMPLATE.format(original_text=original_text, instruction=instruction)

    await get_llm_response(
        system_prompt=system_prompt,
        agent_instructions=agent_instructions,
        user_input=user_input,
        provider_config=provider_config,
        ollama_config=ollama_config,
        openai_config=openai_config,
        logger=logger,
        quiet=quiet,
        clipboard=clipboard,
        live=live,
        show_output=True,
        exit_on_error=True,
    )
