"""Knowledge configuration."""

from __future__ import annotations

from pydantic import ConfigDict, Field
from schemez import Schema

from llmling_agent.prompts.prompts import PromptType


class Knowledge(Schema):
    """Collection of context sources for an agent.

    Supports both simple paths and rich resource types for content loading,
    plus LLMling's prompt system for dynamic content generation.

    Docs: https://phil65.github.io/llmling-agent/YAML%20Configuration/knowledge_configuration/
    """

    paths: list[str] = Field(
        default_factory=list,
        examples=[["docs/readme.md", "https://api.example.com/docs"], ["/data/context.txt"]],
        title="Resource paths",
    )
    """Quick access to files and URLs."""

    prompts: list[PromptType] = Field(default_factory=list, title="Dynamic prompts")
    """Prompts for dynamic content generation:
    - StaticPrompt: Fixed message templates
    - DynamicPrompt: Python function-based
    - FilePrompt: File-based with template support
    """

    convert_to_markdown: bool = Field(default=False, title="Convert to markdown")
    """Whether to convert content to markdown when possible."""

    model_config = ConfigDict(frozen=True)

    def get_resources(self) -> list[PromptType | str]:
        """Get all resources."""
        return self.prompts + self.paths
