"""
This file applies a trained KitNET model onto the stream of connection data generated by Zeek.
The script then outputs the anomaly results to a file.

Authors: 
- Zoe Hsu <wh2405@nyu.edu> 
- Olive Song <js10417@nyu.edu> 
- Diego Lopez <dtl310@nyu.edu>
- Zihang Xia <zx961@nyu.edu>
"""
import argparse
import random
import pandas as pd
import numpy as np
import pysad
from pysad.models.kitnet import KitNet
from sklearn.preprocessing import OneHotEncoder, StandardScaler
from sklearn.pipeline import Pipeline
import sys
import logging
import os 
from utils import *
import pyod 
from joblib import dump, load
import json 
import tailer
from datetime import datetime
from multiprocessing import Process


logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s (%(filename)s)')

# create logger for writing anomaly to file
anomaly_logger = logging.getLogger('anomaly')
anomaly_logger.setLevel(logging.INFO)

# create file handler
fh = logging.FileHandler('anomalies.log')
file_formatter = logging.Formatter('%(asctime)s - %(message)s')
fh.setLevel(logging.INFO)
fh.setFormatter(file_formatter)
anomaly_logger.addHandler(fh)

CONN_AD_ENABLED=False
HTTP_AD_ENABLED=False
DNS_AD_ENABLED=False

def main():
    """
    The main control flow for the application. This script takes in as arguments the 
    location of the zeek spool directory (also found in the zeek/logs/current symlink)
    and applies the model to the conn.log stream. This expects that this stream is in JSON format, 
    which can be applied to Zeek by adding "@load policy/tuning/json-logs.zeek" to local.zeek.
    """
    global CONN_AD_ENABLED, DNS_AD_ENABLED, HTTP_AD_ENABLED 
    parser = argparse.ArgumentParser(
        description='Apply a KitNET model to current data and output anomaly scores. The logs MUST have been stored '
                    'in JSON format.')
    parser.add_argument('--model-path', type=str, default='kit.joblib',  
                        help='The path to the model file to load.') 
    parser.add_argument('--log-dir', type=str, required=True, 
                        help='Zeek logdir variable, where this script can find Zeek data. Please point to the logs/ '
                             'directory.')
    parser.add_argument('--modules', nargs='+', required=True, choices=['CONN', 'DNS', 'HTTP'],
                        help='List of modules to enable. Choose from CONN, DNS, or HTTP. At least one module is required.')
    args = parser.parse_args()
    log_dir = args.log_dir
    # At least 1 module must be specified
    logging.info(f"Using Modules {args.modules}")
    logging.info(f"Using logdir: {log_dir}")
    spool_path = os.path.join(log_dir, "current") 
    logging.info(f"Using spool path: {spool_path}")
    logging.info(f"Using model path: {args.model_path}")
    if 'CONN' in args.modules:
        CONN_AD_ENABLED = True
        conn_log_path = os.path.join(spool_path, "conn.log")
        conn_model_path = "conn_" + args.model_path
        try:
            kit_conn_model = load(conn_model_path)
        except FileNotFoundError:
            logging.error(f"Could not find model path {conn_model_path}")
            sys.exit(1)
        conn_process = Process(target=follow, args=(conn_log_path, kit_conn_model, "conn"))
        conn_process.start()
        logging.info("Started CONN Monitoring Process")
    if 'DNS' in args.modules:
        DNS_AD_ENABLED = True
        dns_log_path = os.path.join(spool_path, "dns.log")
        dns_model_path = "dns_" + args.model_path
        try:
            kit_dns_model = load(dns_model_path)
        except FileNotFoundError:
            logging.error(f"Could not find model path {dns_model_path}")
            sys.exit(1)
        dns_process = Process(target=follow, args=(dns_log_path, kit_dns_model, "dns"))
        dns_process.start()
        logging.info("Started DNS Monitoring Process")
    if 'HTTP' in args.modules:
        HTTP_AD_ENABLED = True
        http_log_path = os.path.join(spool_path, "http.log")
        http_model_path = "http_" + args.model_path
        try:
            kit_http_model = load(http_model_path)
        except FileNotFoundError:
            logging.error(f"Could not find model path {http_model_path}")
            sys.exit(1)
        http_process = Process(target=follow, args=(http_log_path, kit_http_model, "http"))
        http_process.start()
        logging.info("Started HTTP Monitoring Process")
    logging.info("Awaiting events...")

def follow(log_path, kit_model, type):
    for line in tailer.follow(open(log_path)):
        result = score_json(kit_model, line, type)
        anomaly_logger.info(result)

def score_json(kit, line, type):
    """
    Given a line from the processed json, fit the model using new data and return the anomaly score.
    The result is returned as a dictionary containing the original json, and anomaly score
    """
    if type == "conn":
        line_processed = preprocess_json_conn(line)
    elif type == "dns":
        line_processed = preprocess_json_dns(line)
    elif type == "http":
        line_processed = preprocess_json_http(line)
    else:
        logging.error(f"Invalid scoring type {type}")
        sys.exit(1)
    assert len(line_processed) == 1
    features = ['uid', "id.resp_h", "id.orig_h", "id.orig_p", "id.resp_p", "proto"]
    features_obj = json.loads(line)
    features_dict = dict(zip(features, [features_obj[feature] for feature in features]))
    assert len(features_dict) == 6
    anomaly_score = kit.score_partial(line_processed[0])
    features_dict['anomaly_score'] = anomaly_score
    return features_dict

if __name__ == "__main__":
    main()