import asyncio
from typing import List

import logging
import os

from eval_protocol.dataset_logger import default_logger
from eval_protocol.models import EvaluationRow, Message, ChatCompletionMessageToolCall
from eval_protocol.pytest.types import RolloutProcessorConfig


async def default_single_turn_rollout_processor(
    rows: List[EvaluationRow], config: RolloutProcessorConfig
) -> List[EvaluationRow]:
    """Generate a single response from any supported model provider using LiteLLM."""

    # Quiet LiteLLM logs in test runs unless user overrode
    try:
        if os.environ.get("LITELLM_LOG") is None:
            os.environ["LITELLM_LOG"] = "ERROR"
        _llog = logging.getLogger("LiteLLM")
        _llog.setLevel(logging.CRITICAL)
        _llog.propagate = False
        for _h in list(_llog.handlers):
            _llog.removeHandler(_h)
    except Exception:
        pass

    # Do not modify global LiteLLM cache. Disable caching per-request instead.

    async def process_row(row: EvaluationRow) -> EvaluationRow:
        """Process a single row asynchronously."""
        if len(row.messages) == 0:
            raise ValueError("Messages is empty. Please provide a non-empty dataset")

        messages_payload = [{"role": m.role, "content": m.content} for m in row.messages]

        request_params = {"model": config.model, "messages": messages_payload, **config.input_params}
        # Ensure caching is disabled only for this request (review feedback)
        request_params["cache"] = {"no-cache": True}
        # Allow passing reasoning effort to Fireworks via LiteLLM using extra_body
        # Expected: config.input_params may contain {"reasoning": {"effort": "low|medium|high"}}
        if "reasoning" in config.input_params:
            request_params.setdefault("extra_body", {})
            request_params["extra_body"]["reasoning"] = config.input_params["reasoning"]

        if row.tools is not None:
            request_params["tools"] = row.tools

        # Dynamic import to avoid static dependency/lint errors if LiteLLM isn't installed yet
        import importlib
        _litellm = importlib.import_module("litellm")
        acompletion = getattr(_litellm, "acompletion")
        response = await acompletion(**request_params)

        assistant_content = response.choices[0].message.content or ""
        tool_calls = response.choices[0].message.tool_calls if response.choices[0].message.tool_calls else None

        converted_tool_calls = None
        if tool_calls:
            converted_tool_calls = [
                ChatCompletionMessageToolCall(
                    id=tool_call.id,
                    type=tool_call.type,
                    function={
                        "name": tool_call.function.name,
                        "arguments": tool_call.function.arguments,
                    },
                )
                for tool_call in tool_calls
            ]

        messages = list(row.messages) + [
            Message(
                role="assistant",
                content=assistant_content,
                tool_calls=converted_tool_calls,
            )
        ]

        row.messages = messages
        default_logger.log(row)
        return row

    # Process rows with bounded concurrency if configured
    max_concurrent = getattr(config, "max_concurrent_rollouts", 8) or 8
    semaphore = asyncio.Semaphore(max_concurrent)

    async def _sem_wrapper(r: EvaluationRow) -> EvaluationRow:
        async with semaphore:
            return await process_row(r)

    tasks = [_sem_wrapper(row) for row in rows]
    dataset = list(await asyncio.gather(*tasks))

    return dataset
