"""The Nexus Simulator class is the main class for interacting with Nexus models."""
from __future__ import annotations

import os
import warnings
from typing import Any, Union, Optional, Sequence

import resqpy.model as rq
from datetime import datetime

from ResSimpy.Nexus.NexusIPRMethods import NexusIprMethods

from ResSimpy.Enums.FluidTypeEnums import PvtType
from ResSimpy.Nexus.DataModels.NexusOptions import NexusOptions
import ResSimpy.Nexus.nexus_file_operations as nfo
import ResSimpy.FileOperations.file_operations as fo
from ResSimpy.Nexus.DataModels.FcsFile import FcsNexusFile
from ResSimpy.Nexus.DataModels.NexusFile import NexusFile
from ResSimpy.Nexus.NexusPVTMethods import NexusPVTMethods
from ResSimpy.Nexus.NexusSeparatorMethods import NexusSeparatorMethods
from ResSimpy.Nexus.NexusWaterMethods import NexusWaterMethods
from ResSimpy.Nexus.NexusEquilMethods import NexusEquilMethods
from ResSimpy.Nexus.NexusRockMethods import NexusRockMethods
from ResSimpy.Nexus.NexusRelPermMethods import NexusRelPermMethods
from ResSimpy.Nexus.NexusValveMethods import NexusValveMethods
from ResSimpy.Nexus.NexusAquiferMethods import NexusAquiferMethods
from ResSimpy.Nexus.NexusHydraulicsMethods import NexusHydraulicsMethods
from ResSimpy.Nexus.NexusGasliftMethods import NexusGasliftMethods
from ResSimpy.Nexus.DataModels.StructuredGrid.NexusGrid import NexusGrid
from ResSimpy.Nexus.NexusEnums.DateFormatEnum import DateFormat
from ResSimpy.Enums.UnitsEnum import UnitSystem
from ResSimpy.Nexus.NexusNetwork import NexusNetwork
from ResSimpy.Nexus.NexusReporting import NexusReporting
from ResSimpy.Nexus.NexusWells import NexusWells
from ResSimpy.Nexus.nexus_model_file_generator import NexusModelFileGenerator
from ResSimpy.Nexus.runcontrol_operations import SimControls
from ResSimpy.Nexus.logfile_operations import Logging
from ResSimpy.Nexus.structured_grid_operations import StructuredGridOperations
from ResSimpy.DataModelBaseClasses.Simulator import Simulator
from ResSimpy.Time.ISODateTime import ISODateTime


class NexusSimulator(Simulator):

    def __init__(self, origin: Optional[str] = None, destination: Optional[str] = None,
                 root_name: Optional[str] = None, nexus_data_name: str = "data", write_times: bool = False,
                 manual_fcs_tidy_call: bool = False, lazy_loading: bool = True, start_date: None | str = None,
                 run_units: None | UnitSystem = None, default_units: None | UnitSystem = None,
                 pvt_type: None | PvtType = None, assume_loaded: bool = False,
                 eos_details: None | str = None, date_format: DateFormat = DateFormat.MM_DD_YYYY) -> None:
        """Nexus simulator class. Inherits from the Simulator super class.

        Args:
            origin (Optional[str], optional): file path to the fcs file. Defaults to None.
            destination (Optional[str], optional): The location to move the model to.
            root_name (Optional[str], optional): Root file name of the fcs. Defaults to None.
            nexus_data_name (str, optional): Folder name for the nexus data files to be stored in. Defaults to "data".
            write_times (bool, optional): Sets whether the runcontrol file will expand the include files with time \
                cards in. Defaults to True.
            manual_fcs_tidy_call (bool, optional): Determines whether fcs_tidy should be called - Currently not used. \
                Defaults to False.
            lazy_loading(bool): If set to True, parts of the model will only be loaded in when requested via \
                properties on the object.
            start_date: (str, optional): The start date of the model. If not provided, it will be inferred from a read \
                in reservoir deck, specifically the runcontrol file.
            run_units (Optional[UnitSystem], optional): The run units of the model. If not provided, it will be read \
                from the fcs file or defaulted to the Nexus default ENGLISH.
            default_units (Optional[UnitSystem], optional): The default units of the model. If not provided, it will \
                be read from the fcs file or defaulted to the Nexus default ENGLISH.
            pvt_type (Optional[PvtType], optional): The PVT type of the model. If not provided, it will be read from \
                the fcs file. Defaults to None.
            assume_loaded (bool, optional): If True, assumes that the model is already loaded and does not attempt to \
                load the fcs file. Defaults to False.
            eos_details (None | str, optional): A string containing the EOS details. If not provided, \
                it will be set to None and read from the fcs file if applicable. Defaults to None.
            date_format (DateFormat, optional): The date format to use for the model. Defaults to MM_DD_YYYY.

        Attributes:
            run_control_file_path (Optional[str]): file path to the run control file - derived from the fcs file
            __destination (Optional[str]): output path for the simulation. Currently not used.
            date_format (DateFormat): Enum value representing the date format.
            __original_fcs_file_path (str): Path to the original fcs file path supplied
            __new_fcs_file_path (str): Where the new fcs will be saved to
            __force_output (bool): private attribute of force_output
            __origin (str): private attribute of origin. File path to the fcs file.
            __root_name (str): private attribute of root_name. Root file name of the fcs.
            __nexus_data_name (str): private attribute of nexus_data_name. Folder name for the nexus data files to be \
                stored in.
            __structured_grid_file_path (Optional[str]): file path to the structured grid.
            __structured_grid_file (Optional[NexusGrid]): StructuredGridFile object representing the \
                structured grid used in Nexus
            __run_units (Optional[str]): Unit system used in the Nexus model
            use_american_run_units (bool): True if an American unit system is used equivalent to 'ENGLISH'. \
                False otherwise. For the RUN_UNITS keyword.
            use_american_input_units (bool): True if an American unit system is used equivalent to 'ENGLISH'. \
                False otherwise. For the DEFAULT_UNITS keyword.
            __write_times (bool): private attribute for write_times. Sets whether the runcontrol file will expand \
                the include files with time cards in.
            __manual_fcs_tidy_call (bool): private attribute for manual_fcs_tidy_call. Determines whether fcs_tidy \
                should be called

        Raises:
            ValueError: If the FCS file path is not given

        Examples:
            Example of calling a NexusSimulator with a file path to an fcs file:

            >>> from ResSimpy.Nexus.NexusSimulator import NexusSimulator
            >>> model = NexusSimulator(origin="/path/to/fcs_file.fcs")

        """
        if origin is None:
            raise ValueError(f'Origin path to model fcs file is required. Instead got {origin}.')
        self.origin = origin
        self.assume_loaded: bool = assume_loaded
        self._model_files: FcsNexusFile = FcsNexusFile(location=self.origin, origin=None)

        self._start_date: str = '' if start_date is None else start_date.strip()
        self.run_control_file_path: Optional[str] = ''
        self.__destination: Optional[str] = None
        self.date_format: DateFormat = date_format  # Nexus default
        self.__original_fcs_file_path: str = self.origin
        self.__new_fcs_file_path: str = self.origin
        self.__nexus_data_name: str = nexus_data_name
        self.__run_units: UnitSystem = run_units if run_units is not None else UnitSystem.ENGLISH  # The Nexus default
        self.root_name: str = root_name
        self.use_american_run_units: bool = False
        self.use_american_input_units: bool = False
        self.__write_times: bool = write_times
        self.__manual_fcs_tidy_call: bool = manual_fcs_tidy_call

        self._default_units: UnitSystem = default_units if default_units is not None else (
            UnitSystem.ENGLISH)  # The Nexus default

        self._pvt_type: PvtType = pvt_type if pvt_type is not None else PvtType.BLACKOIL
        self._eos_details: None | str = eos_details

        self._network: NexusNetwork = NexusNetwork(model=self, assume_loaded=assume_loaded)
        self._wells: NexusWells = NexusWells(self)
        self._grid: Optional[NexusGrid] = None
        self._options: Optional[NexusOptions] = None
        # Model dynamic properties
        self._pvt: NexusPVTMethods = NexusPVTMethods(model_unit_system=self.default_units,
                                                     model_files=self.model_files)
        self._separator: NexusSeparatorMethods = NexusSeparatorMethods(model_unit_system=self.default_units,
                                                                       model_files=self.model_files)
        self._water: NexusWaterMethods = NexusWaterMethods(model_unit_system=self.default_units,
                                                           model_files=self.model_files)
        self._equil: NexusEquilMethods = NexusEquilMethods(model_unit_system=self.default_units,
                                                           model_files=self.model_files)
        self._rock: NexusRockMethods = NexusRockMethods(model_unit_system=self.default_units,
                                                        model_files=self.model_files)
        self._relperm: NexusRelPermMethods = NexusRelPermMethods(model_unit_system=self.default_units,
                                                                 model_files=self.model_files)
        self._valve: NexusValveMethods = NexusValveMethods(model_unit_system=self.default_units,
                                                           model_files=self.model_files)
        self._aquifer: NexusAquiferMethods = NexusAquiferMethods(model_unit_system=self.default_units,
                                                                 model_files=self.model_files)
        self._hydraulics: NexusHydraulicsMethods = NexusHydraulicsMethods(model_unit_system=self.default_units,
                                                                          model_files=self.model_files)
        self._gaslift: NexusGasliftMethods = NexusGasliftMethods(model_unit_system=self.default_units,
                                                                 model_files=self.model_files)
        # Nexus operations modules
        self.logging: Logging = Logging(self)
        self._reporting: NexusReporting = NexusReporting(self)
        self._structured_grid_operations: StructuredGridOperations = StructuredGridOperations(self)
        self.__lazy_loading: bool = lazy_loading
        self._sim_controls: SimControls = SimControls(self)
        self.__ipr_methods: NexusIprMethods = NexusIprMethods(self)

        if destination is not None and destination != '':
            self.set_output_path(path=destination.strip())

        # Check the status of any existing or completed runs related to this model
        self.get_simulation_status(from_startup=True)

        self.__is_multi_reservoir: bool = False  # Flag to indicate if the model is a multi-reservoir model
        self.__reservoir_paths: dict[str, str] = {}
        self.__multi_reservoirs: dict[str, NexusSimulator] = {}

        # Load in the model
        if not assume_loaded:
            self.__load_fcs_file()

    def __repr__(self) -> str:
        """Pretty printing NexusSimulator data."""
        printable_str = f'Simulation name: {self.root_name}\n'
        printable_str += super().__repr__()
        printable_str += f'Run units: {str(self.default_units)}\n'
        # add details from the fcsfile
        printable_str += self.model_files.__repr__()
        return printable_str

    @staticmethod
    def _attr_info_to_tuple(sim_attr: Union[dict, Sequence]) \
            -> tuple[tuple[tuple[str, Union[str, bool, float]], ...], ...]:
        """Convert the network constraints/wells completions attribute to a tuple of tuples so that it is hashable.

        Args: sim_attr (Union[dict, Sequence]): dict of {well name: [NexusConstraint]} for
        self.network.constraints.get_all(), list of [NexusWell objects] for self.wells.get_all()

        Returns: Tuple[Tuple[Tuple[str, Union[str, float, bool]], ...], ...]: tuple of tuples
        """
        lst_of_tuples = []
        if isinstance(sim_attr, dict):
            for wells, nexus_constraint in sim_attr.items():
                for constraints in nexus_constraint:
                    network_dict = constraints.to_dict(add_units=False, include_nones=False)
                    lst_of_tuples.append(tuple(network_dict.items()))

        elif isinstance(sim_attr, list):
            for nexus_completion in sim_attr:
                for el in nexus_completion.completions:
                    well_dict = el.to_dict(add_units=False, include_nones=False)
                    lst_of_tuples.append(tuple(well_dict.items()))

        return tuple(lst_of_tuples)

    def network_wells_tuple(self) -> tuple:
        """Returns a tuple of the network constraints and wells completions attributes.

        Returns:
            tuple: tuple of the network constraints and wells completions attributes
        """
        network_attr = self.network.constraints.get_all()
        wells_attr = self.wells.get_all()

        network_tuple = self._attr_info_to_tuple(network_attr)
        wells_tuple = self._attr_info_to_tuple(wells_attr)
        return network_tuple, wells_tuple

    def hash_network_wells(self) -> int:
        """Hashes the network constraints and wells completions attributes.

        Returns:
            int: hash value of the network constraints and wells completions attributes
        """
        hash_attr_tuple = self.network_wells_tuple()
        return hash(hash_attr_tuple)

    def wells_and_network_equal(self, other: NexusSimulator) -> bool:
        """Compares the network constraints and wells completions of two NexusSimulator objects.

        Args:
            other (NexusSimulator): NexusSimulator object to compare with

        Returns:
            Union[bool]: Returns True if the network constraints and wells completions are equal, False otherwise.

        Raises:
            ValueError: if both models have no network constraints or wells completions.
            TypeError: if the other object is not a NexusSimulator object.
        """
        if isinstance(other, NexusSimulator):
            base_class_tuple = self.network_wells_tuple()
            other_class_tuple = other.network_wells_tuple()
            if base_class_tuple == ((), ()) and other_class_tuple == ((), ()):
                # if base_class_tuple and other_class_tuple both return empty tuple, that means both of them /
                # have no network constraints or wells completions
                raise ValueError("Both models have empty network constraints or wells completions. Unable to compare.")
            return self.network_wells_tuple() == other.network_wells_tuple()
        raise TypeError(f"Unable to compare {type(self)} with {other}. Ensure that {other} is of type NexusSimulator. "
                        f"{other} has {type(other)}")

    def remove_temp_from_properties(self) -> None:
        """Updates model values if the files are moved from a temp directory.

        Replaces the first instance of temp/ in the file paths in the nexus simulation file paths.

        Raises:
            ValueError: if any of [__structured_grid_file_path, __new_fcs_file_path, __surface_file_path] are None.
        """
        if self.model_files.structured_grid_file is None or self.model_files.structured_grid_file.location is None:
            raise ValueError(
                "No structured_grid_file_path found, can't remove temporary properties from file path")
        if self.__new_fcs_file_path is None:
            raise ValueError(
                "No __new_fcs_file_path found, can't remove temporary properties from file path")
        if self.model_files.surface_files is None or self.model_files.surface_files[1] is None \
                or self.model_files.surface_files[1].location is None:
            raise ValueError(
                "No __surface_file_path found, can't remove temporary properties from file path")

        self._origin = self._origin.replace('temp/', '', 1)
        self.__root_name = '' if self.root_name is None else self.root_name.replace('temp/', '', 1)
        self.model_files.structured_grid_file.location = \
            self.model_files.structured_grid_file.location.replace('temp/', '', 1)
        self.__new_fcs_file_path = self.__new_fcs_file_path.replace('temp/', '', 1)
        self.model_files.surface_files[1].location = self.model_files.surface_files[1].location.replace('temp/', '', 1)

    def get_simulation_status(self, from_startup: bool = False) -> Optional[str]:
        """Returns run status from the latest simulation run.

        Args:
            from_startup(bool = False): Searches the same directory as the original_fcs_file_path if true.
            Otherwise searches the destination folder path, failing this then searches the original_fcs_file_path
            if false. Defaults to false.
        """
        return self.logging.get_simulation_status(from_startup)

    def get_simulation_progress(self) -> float:
        """Returns simulation progress from log files."""
        return self.logging.get_simulation_progress()

    def get_users_linked_with_files(self) -> Optional[list[tuple[Optional[str], Optional[str], Optional[datetime]]]]:
        """Returns list of users linked with each model file."""
        return self.model_files.files_info

    @property
    def model_files(self) -> FcsNexusFile:
        """Returns model files associated with instance of 'FcsNexusfile'."""
        return self._model_files

    @property
    def network(self) -> NexusNetwork:
        """Returns an instance of Nexus network class."""
        return self._network

    @property
    def structured_grid_path(self) -> Optional[str]:
        """Returns the location of the structured grid file."""
        return None if self.model_files.structured_grid_file is None else self.model_files.structured_grid_file.location

    @property
    def default_units(self) -> UnitSystem:
        """Returns the default units."""
        return self._default_units

    @staticmethod
    def sim_default_unit_system() -> UnitSystem:
        """Returns the default unit system used by the Simulator."""
        return UnitSystem.ENGLISH

    @property
    def run_units(self) -> UnitSystem:
        """Returns the run units."""
        return self.__run_units

    @property
    def pvt_type(self) -> PvtType:
        """Returns the PVT type."""
        return self._pvt_type

    def set_pvt_type(self, value: str) -> None:
        """Sets the PVT type for the model.

        Args:
            value (str): The PVT type to set from the fluid_type string (e.g., 'BLACKOIL', 'WATEROIL', etc.).

        Returns:
            PvtType: The PVT type set for the model.
        """
        if 'EOS' in value.upper():
            self._pvt_type = PvtType.EOS
        else:
            self._pvt_type = PvtType(value)

    @property
    def eos_details(self) -> Optional[str]:
        """Returns the EOS details."""
        return self._eos_details

    @property
    def new_fcs_name(self) -> Optional[str]:
        """Returns the new name for the FCS file without the fcs extension."""
        return self.__root_name

    @property
    def write_times(self) -> bool:
        """Sets whether the runcontrol file will expand the include files with time cards in."""
        return self.__write_times

    @property
    def original_fcs_file_path(self) -> str:
        """Returns path to the original fcs file path supplied."""
        return self.__original_fcs_file_path

    @property
    def root_name(self) -> Optional[str]:
        """Returns root file name of the fcs."""
        return self.__root_name

    @root_name.setter
    def root_name(self, value: Optional[str]) -> None:
        """Returns the name of the fcs file without the .fcs extension.

        Returns:
            str: string of the fcs file without the .fcs extension.
        """
        if value is not None:
            rootname = value
        else:
            rootname = os.path.basename(self._origin)
            rootname = rootname.split(".fcs")[0]
        self.__root_name = rootname

    @property
    def start_iso_date(self) -> ISODateTime:
        """Returns the start date of the model in ISO format."""
        return ISODateTime.convert_to_iso(date=self._start_date, date_format=self.date_format)

    @staticmethod
    def get_check_run_input_units_for_models(models: list[str]) -> tuple[Optional[bool], Optional[bool]]:
        # TODO: add LAB units
        """Returns the run and input unit formats for all the supplied models.

        Supported model formats:
            RESQML type epc files ending in ".epc"
            Nexus files containing a line identifying the "RUN_UNITS" or "DEFAULT_UNITS".

        Supported units: ENGLISH, METRIC

        Args:
            models (list[str]): list of paths to supported reservoir models

        Raises:
            ValueError: if a model in the list is using inconcistent run/default units

        Returns:
            Tuple[Optional[Bool], Optional[Bool]]: If all units are consistent between models,
                Returns (True, True) if 'ft' is the length unit in an epc or Nexus specifies "ENGLISH" as the
                (RUN_UNITS,DEFAULT_UNITS) respectively and False, False otherwise.
                Returns (None, None) if it can't find a (RUN_UNITS, DEFAULT_UNITS) in the supplied files.
        """
        oilfield_run_units: Optional[bool] = None
        oilfield_default_units: Optional[bool] = None

        for model in models:
            # If we're checking the units of a RESQML model, read it in and get the units. Otherwise, read the units
            # from the fcs file
            if os.path.splitext(model)[1] == '.epc':
                resqpy_model = rq.Model(epc_file=model)

                # Load in the RESQML grid
                grid = resqpy_model.grid()

                # Check the grid units
                grid_length_unit = grid.xy_units()
                model_oilfield_default_units = grid_length_unit == 'ft'
                model_oilfield_run_units = grid_length_unit == 'ft'
            else:
                simulator = NexusSimulator(origin=model)
                model_oilfield_default_units = simulator.default_units == UnitSystem.ENGLISH
                model_oilfield_run_units = simulator.run_units == UnitSystem.ENGLISH

            # If not defined, assign it to model_oilfield_default_units
            if oilfield_default_units is None:
                oilfield_default_units = model_oilfield_default_units
            # Raise ValueError if default units are inconsistent with the other models
            elif model_oilfield_default_units != oilfield_default_units:
                raise ValueError(f"Model at {model} using inconsistent default units")

            # If not defined, assign it to model_oilfield_run_units
            if oilfield_run_units is None:
                oilfield_run_units = model_oilfield_run_units
            # Raise ValueError if run units are inconsistent with the other models
            elif model_oilfield_run_units != oilfield_run_units:
                raise ValueError(f"Model at {model} using inconsistent run units")

        return oilfield_run_units, oilfield_default_units

    @staticmethod
    def get_check_oil_gas_types_for_models(models: list[str]) -> Optional[str]:
        """Checks for fluid types within a list of paths to models.

        Currently limited to checking for the first SURFACE network in a file.

        Args:
            models (list[str]): a list of paths to models to check for fluid types.

        Raises:
            ValueError: If fluid types are inconsistent between models

        Returns:
            Optional[str]: The fluid type used for the model for the first surface network
        """
        fluid_type = None
        for model in models:
            model_obj = NexusSimulator(origin=model)
            model_fluid_type = None
            # get the first surface method and expand out all include files
            if model_obj.model_files.surface_files is None or model_obj.model_files.surface_files[1].location is None:
                raise ValueError(f"No surface file found for {model}")
            surface_file_content = model_obj.model_files.surface_files[1].get_flat_list_str_file
            if surface_file_content is not None:
                model_fluid_type = NexusSimulator.get_fluid_type(
                    surface_file_content=surface_file_content)

            if fluid_type is None:
                fluid_type = model_fluid_type
            elif fluid_type != model_fluid_type:
                raise ValueError(
                    f"Inconsistent Oil / Gas types: {model_fluid_type} found for {model}")

        return fluid_type

    @staticmethod
    def get_eos_details(surface_file: list[str]) -> str:
        """Gets all the information about an EOS from a Nexus model.

        Args:
            surface_file (list[str]): path to the surface file in a Nexus model

        Returns:
            str: a concatenated string of EOS components
        """
        eos_string: str = ''
        eos_found: bool = False
        for line in surface_file:
            if nfo.check_token("EOS", line):
                eos_string += line
                eos_found = True
            elif eos_found and nfo.get_next_value(0, [line]) is not None:
                eos_string += line
            if nfo.check_token("COMPONENTS", line):
                break

        return eos_string

    @staticmethod
    def get_fluid_type(surface_file_content: list[str]) -> str:
        """Gets the fluid type for a single model from a surface file.

        Defaults to BLACKOIL if no explicit fluid type is found.

        Args:
            surface_file_content (str): list of strings with a new line per entry from the surface file

        Returns:
            str: fluid type as one of [BLACKOIL, WATEROIL, GASWATER, API] or the full details from an EOS model
        """
        fluid_type = None

        for line in surface_file_content:
            if nfo.check_token("BLACKOIL", line):
                fluid_type = "BLACKOIL"
                break
            elif nfo.check_token("WATEROIL", line):
                fluid_type = "WATEROIL"
                break
            elif nfo.check_token("GASWATER", line):
                fluid_type = "GASWATER"
                break
            elif nfo.check_token("EOS", line):
                fluid_type = NexusSimulator.get_eos_details(surface_file_content)
            elif nfo.check_token("API", line):
                fluid_type = "API"
        if fluid_type is None:
            warnings.warn("No explicit fluid type found in the file. Defaulting to BLACKOIL", UserWarning)
            fluid_type = "BLACKOIL"
        return fluid_type

    def get_model_oil_type(self) -> str:
        """Returns the get_fluid_type method on the existing NexusSimulator instance.

        Raises:
            ValueError: If no file path is provided for the surface file path

        Returns:
            str: fluid type as one of [BLACKOIL, WATEROIL, GASWATER,] or the full details from an EOS model
        """
        if self.model_files.surface_files is None or self.model_files.surface_files[1].location is None:
            raise ValueError("No value found for the path to the surface file")
        return NexusSimulator.get_fluid_type(self.model_files.surface_files[1].get_flat_list_str_file)

    def check_output_path(self) -> None:
        """Confirms that the output path has been set.

        (used to stop accidental writing operations in the original directory).

        Raises:
            ValueError: if the destination provided is set to None.
        """
        if self.__destination is None:
            raise ValueError("Destination is required for this operation. Currently set to: ", self.__destination)

    @property
    def destination(self) -> Optional[str]:
        """Returns the destination to move the model to (either as a string or none)."""
        return self.__destination

    def set_output_path(self, path: str) -> None:
        """Initialises the output to the declared output location.

        If the file is a different directory to the origin path location the function will set the origin
        to the new destination.
        """
        self.__destination = path
        if self.__destination is not None and os.path.dirname(self._origin) != os.path.dirname(self.__destination):
            self._origin = self.__destination + "/" + os.path.basename(self.__original_fcs_file_path)

    def __load_fcs_file(self) -> None:
        """Loads in the information from the supplied FCS file into the class instance.

        Loads in the paths for runcontrol, structured grid and the first surface network.
        Loads in the values for dateformat and run units.
        Attempts to load the run_control_file.
        Loads the wellspec and dynamic property files.
        """
        # fcs_content_with_includes is used to scan only the fcs file and files specifically called with the INCLUDE
        # token in front of it to prevent it from reading through all the other files. We need this here to extract the
        # fcs properties only. The FcsFile structure is then generated and stored in the object (with all the nesting of
        # the NexusFiles as self.model_files (e.g. STRUCTURED_GRID, RUNCONTROL etc.)
        fcs_content_with_includes = (NexusFile.generate_file_include_structure(simulator_type=NexusFile,
                                                                               origin=self.origin,
                                                                               file_path=self.__new_fcs_file_path)
                                     .get_flat_list_str_file)
        self._model_files = FcsNexusFile.generate_fcs_structure(self.__new_fcs_file_path)
        if fcs_content_with_includes is None:
            raise ValueError(f'FCS file not found, no content for {self.__new_fcs_file_path}')
        for line in fcs_content_with_includes:
            if nfo.check_token('DATEFORMAT', line) or nfo.check_token('DATE_FORMAT', line):
                format_token = 'DATEFORMAT' if nfo.check_token('DATEFORMAT', line) else 'DATE_FORMAT'
                value = fo.get_token_value(format_token, line, fcs_content_with_includes)
                if value is not None:
                    self.date_format = DateFormat.DD_MM_YYYY if value == 'DD/MM/YYYY' else DateFormat.MM_DD_YYYY

                self._sim_controls.date_format_string = "%m/%d/%Y" if self.date_format is DateFormat.MM_DD_YYYY \
                    else "%d/%m/%Y"
            elif nfo.check_token('RUN_UNITS', line):
                value = fo.get_token_value('RUN_UNITS', line, fcs_content_with_includes)
                if value is not None:
                    self.__run_units = UnitSystem(value.upper())
            elif nfo.check_token('DEFAULT_UNITS', line):
                value = fo.get_token_value('DEFAULT_UNITS', line, fcs_content_with_includes)
                if value is not None:
                    self._default_units = UnitSystem(value.upper())

        if self._model_files.multi_reservoir_files:
            self.__is_multi_reservoir = True
            self.__process_multi_reservoir_model()

        # Load in the Nexus options information
        if self.model_files.options_file is not None:
            self._options = NexusOptions(file=self.model_files.options_file,
                                         model_unit_system=self.default_units)

        # === Load in dynamic properties ===
        # Read in PVT properties from Nexus PVT method files
        if self.model_files.pvt_files is not None and \
                len(self.model_files.pvt_files) > 0:
            self._pvt = NexusPVTMethods(files=self.model_files.pvt_files,
                                        model_unit_system=self.default_units,
                                        model_files=self.model_files)

        # Read in separator properties from Nexus separator method files
        if self.model_files.separator_files is not None and \
                len(self.model_files.separator_files) > 0:
            self._separator = NexusSeparatorMethods(files=self.model_files.separator_files,
                                                    model_unit_system=self.default_units,
                                                    model_files=self.model_files)

        # Read in water properties from Nexus water method files
        if self.model_files.water_files is not None and \
                len(self.model_files.water_files) > 0:
            self._water = NexusWaterMethods(files=self.model_files.water_files,
                                            model_unit_system=self.default_units,
                                            model_files=self.model_files)

        # Read in equilibration properties from Nexus equil method files
        if self.model_files.equil_files is not None and \
                len(self.model_files.equil_files) > 0:
            self._equil = NexusEquilMethods(files=self.model_files.equil_files,
                                            model_unit_system=self.default_units,
                                            model_files=self.model_files)

        # Read in rock properties from Nexus rock method files
        if self.model_files.rock_files is not None and \
                len(self.model_files.rock_files) > 0:
            self._rock = NexusRockMethods(files=self.model_files.rock_files,
                                          model_unit_system=self.default_units,
                                          model_files=self.model_files)

        # Read in relative permeability and capillary pressure properties from Nexus relperm method files
        if self.model_files.relperm_files is not None and \
                len(self.model_files.relperm_files) > 0:
            self._relperm = NexusRelPermMethods(files=self.model_files.relperm_files,
                                                model_unit_system=self.default_units,
                                                model_files=self.model_files)

        # Read in valve and choke properties from Nexus valve method files
        if self.model_files.valve_files is not None and \
                len(self.model_files.valve_files) > 0:
            self._valve = NexusValveMethods(files=self.model_files.valve_files,
                                            model_unit_system=self.default_units,
                                            model_files=self.model_files)

        # Read in aquifer properties from Nexus aquifer method files
        if self.model_files.aquifer_files is not None and \
                len(self.model_files.aquifer_files) > 0:
            self._aquifer = NexusAquiferMethods(files=self.model_files.aquifer_files,
                                                model_unit_system=self.default_units,
                                                model_files=self.model_files)

        # Read in hydraulics properties from Nexus hyd method files
        if self.model_files.hyd_files is not None and \
                len(self.model_files.hyd_files) > 0:
            self._hydraulics = NexusHydraulicsMethods(files=self.model_files.hyd_files,
                                                      model_unit_system=self.default_units,
                                                      model_files=self.model_files)

        # Read in gaslift properties from Nexus gaslift method files
        if self.model_files.gaslift_files is not None and \
                len(self.model_files.gaslift_files) > 0:
            self._gaslift = NexusGasliftMethods(files=self.model_files.gaslift_files,
                                                model_unit_system=self.default_units,
                                                model_files=self.model_files)

        # === End of dynamic properties loading ===

        # Load in Runcontrol
        if self.model_files.runcontrol_file is not None:
            self.run_control_file_path = self.model_files.runcontrol_file.location
            self._sim_controls.load_run_control_file()

        if self.model_files.structured_grid_file is not None:
            self._grid = NexusGrid.load_structured_grid_file(self.model_files.structured_grid_file,
                                                             lazy_loading=self.__lazy_loading,
                                                             model_unit_system=self.default_units)

        # Load in wellspec files
        if self.model_files.well_files is not None and \
                len(self.model_files.well_files) > 0:
            for well_file in self.model_files.well_files.values():
                if well_file.location is None:
                    warnings.warn(f'Well file location has not been found for {well_file}')
                    continue

        # load the pvt type
        if self.model_files.surface_files is not None and self.model_files.surface_files[1] is not None:
            surface_file = self.model_files.surface_files[1].get_flat_list_str_file
            self.set_pvt_type(NexusSimulator.get_fluid_type(surface_file_content=surface_file))
            if self.pvt_type == PvtType.EOS:
                self._eos_details = self.get_eos_details(surface_file)

    @staticmethod
    def update_file_value(file_path: str, token: str, new_value: str, add_to_start: bool = False) -> None:
        """Updates a value in a file if it is present and in the format {TOKEN} {VALUE}.

        If the token isn't present, it will add the token + value to either the start or end of the file.

        Args:
            file_path (str): path to a file to update the token/value pair in
            token (str): Keyword token to find in the given file (e.g. KX)
            new_value (str): Value following the TOKEN to be replaced
            add_to_start (bool, optional): Inserts the token/value pair to the start of the file. Defaults to False.

        Raises:
            ValueError: If no value is found after the token
        """

        file = nfo.load_file_as_list(file_path)

        line_counter = 0
        token_found = False
        for line in file:
            modified_line = line.lower().replace('/t', ' ')
            modified_line = ' '.join(modified_line.split())
            if token.lower() in modified_line:
                # We've found the token, now replace the value
                token_location = modified_line.find(token.lower())
                line_before_token_value = line[0: token_location]
                line_after_token = line[token_location:]
                current_value = nfo.get_next_value(0, [line], line_after_token[len(token) + 1:])
                if current_value is None:
                    raise ValueError(f"No value found after the supplied {token=}, \
                        please check the following line for that token: {line}")
                new_line_after = line_after_token.replace(
                    current_value, new_value, 1)
                file[line_counter] = line_before_token_value + new_line_after
                token_found = True
                break
            line_counter += 1

        if token_found is False:
            token_line = f"{token} {new_value}"
            if add_to_start:
                file.insert(0, token_line + '\n')
            else:
                file.append('\n' + token_line)

        new_file_str = "".join(file)

        with open(file_path, "w") as text_file:
            text_file.write(new_file_str)

    def update_fcs_file_value(self, token: str, new_value: str, add_to_start: bool = False) -> None:
        """Updates a value in the FCS file."""
        self.update_file_value(self.__new_fcs_file_path, token=token, new_value=new_value, add_to_start=add_to_start)

    @staticmethod
    def comment_out_file_value(token: str, file_path: str) -> None:
        """Comments out an uncommented line containing the specified token.

        Args:
            token (str): Keyword token to find in the given file (e.g. KX)
            file_path (str): path to a file containing the token
        """
        file = nfo.load_file_as_list(file_path)

        line_counter = 0
        for line in file:
            modified_line = line.lower().replace('/t', ' ')
            modified_line = ' '.join(modified_line.split())
            # If we've found the token, and it isn't already commented, comment it out
            if token.lower() in modified_line and \
                    (modified_line.find(token.lower()) < modified_line.find("!") or modified_line.find("!") == -1):
                file[line_counter] = f"! {file[line_counter]}"
                break
            line_counter += 1

        new_file_str = "".join(file)

        with open(file_path, "w") as text_file:
            text_file.write(new_file_str)

    def get_date_format(self) -> str:
        """Returns the date format being used by the model.

        formats used: ('MM/DD/YYYY', 'DD/MM/YYYY').
        """
        return self._sim_controls.get_date_format(self.date_format)

    def modify(self, operation: str, section: str, keyword: str, content: list[str]) -> None:
        """Generic modify method to modify part of the input deck.

        Operations are dependent on the section being modified.

        Args:
            operation (str): operation to perform on the section of the input deck (e.g. 'merge')
            section (str): file type from the input deck provided (e.g. RUNCONTROL)
            keyword (str): which keyword/token to find within the deck provided (e.g. TIME)
            content (list[str]): The content to modify using the above operation, \
            represented as a list of strings with a new entry per line of the file

        Raises:
            NotImplementedError: if the functionality is not yet implemented
        """
        section = section.upper()
        keyword = keyword.upper()
        operation = operation.lower()

        if section == "RUNCONTROL":
            if keyword == "TIME":
                self._sim_controls.modify_times(content=content, operation=operation)
            else:
                raise NotImplementedError(keyword, "not yet implemented")
        else:
            raise NotImplementedError(section, "not yet implemented")

    def get_content(self, section: str, keyword: str) -> Union[list[str], None]:
        """Returns the requested input information.

        Args:
            section (str): Section to retreive information from
            keyword (str): Keyword/token to retrieve the information for

        Raises:
            NotImplementedError: if the functionality is not yet implemented

        Returns:
            Union[list[str], None]: the requested information
        """
        section = section.upper()
        keyword = keyword.upper()
        if section == "RUNCONTROL":
            if keyword == "TIME":
                return self._sim_controls.times
            else:
                raise NotImplementedError(keyword, "not yet implemented")
        else:
            raise NotImplementedError(section, "not yet implemented")

    def get_structured_grid_dict(self) -> dict[str, Any]:
        """Convert the structured grid info to a dictionary and pass it to the front end."""
        if self._grid is None:
            return {}
        return self._grid.to_dict()

    def get_abs_structured_grid_path(self, filename: str) -> str:
        """Returns the absolute path to the Structured Grid file."""
        if self.model_files.structured_grid_file is None:
            raise ValueError(
                f"No structured grid file found within simulator class: {self.model_files.structured_grid_file}")
        grid_path = self.model_files.structured_grid_file.location
        if grid_path is None:
            raise ValueError("No path found for structured grid file path. \
                Please provide a path to the structured grid")
        return os.path.dirname(grid_path) + '/' + filename

    def get_surface_file_path(self) -> str:
        """Get the surface file path."""
        if self.model_files.surface_files is None or self.model_files.surface_files[1] is None:
            raise ValueError('No path found for surface file.')
        return self.model_files.surface_files[1].location

    def load_network(self) -> None:
        """Populates nodes and connections from a surface file."""
        self._network.load()

    def write_out_case(self, new_file_path: str, new_include_file_location: str = 'include_files',
                       case_suffix: str = '') -> None:
        """Creates a new case of a simulator run by writing out the modified include files to a new file.

        Args:
            new_file_path (str): Path to save the new simulator to.
            new_include_file_location (str): Saves included files to a path either absolute or relative to the
            file path provided.
            case_suffix (str): Suffix to append to the case name. Defaults to ''.
        """

        self.model_files.write_out_case(new_file_path=new_file_path,
                                        new_include_file_location=new_include_file_location,
                                        case_suffix=case_suffix)

    def update_simulator_files(self) -> None:
        """Updates the simulator with any changes to the included files. Overwrites existing files.

        IMPORTANT: No changes to the model will be saved until this method is called!
        """

        self.model_files.update_model_files()

    def move_simulator_files(self, new_file_path: str, new_include_file_location: str,
                             overwrite_files: bool = False) -> None:
        """Creates a set of simulator files.

        Args:
            new_file_path (str): Path to save the new simulator to.
            new_include_file_location (str): Saves included files to a path either absolute or relative to the
            file path provided.
            overwrite_files (bool): Overwrite files if they already exist. Defaults to False.
        """

        self.model_files.move_model_files(new_file_path, new_include_file_location, overwrite_files)

    def write_out_new_model(self, new_location: str, new_model_name: str,
                            new_include_file_location: str | None = None,
                            overwrite_files: bool = True) -> None:
        """Writes out a new model at a new location with a new_model_name.fcs.

        This method is used for creating an entirely new model based on the ResSimpy internal memory objects and will
        not include any comments or other content that ResSimpy does not yet interpret. Often used with the
        "assume_loaded" attribute set to True for creating new NexusSimulator instances from scratch.

        Args:
            new_location (str): The location to write the new model to.
            new_model_name (str): The name of the new model without the .fcs extension.
            new_include_file_location (str | None): Optional path to save include files to. If None, defaults to
                'include_files' in the new_location.
            overwrite_files (bool): Overwrite files if they already exist. Defaults to True.

        Example usage:
        >>> from ResSimpy import NexusSimulator
        >>> nexus_sim = NexusSimulator(origin='path/to/original_model.fcs', assume_loaded=True)
        >>> # Modify the nexus_sim object as needed
        >>> nexus_sim.write_out_new_model(new_location='path/to/new_model_directory',
        ... new_model_name='new_model_name', new_include_file_location='optional_include_path')
        """

        def update_model_file(file: NexusFile, new_content: str,
                              new_folder_path: str, new_name: str, suffix: str) -> None:
            """Updates a model file with new content and location."""
            file.file_content_as_list = new_content.splitlines(keepends=True)
            file.location = os.path.join(new_folder_path, new_name + suffix)
            file.origin = new_model_path
            file.write_to_file(new_file_path=file.location, overwrite_file=overwrite_files)

        # ensure the full path is made
        if new_include_file_location is None:
            new_include_file_location = os.path.join(new_location, 'include_files')
        elif not os.path.isabs(new_include_file_location):
            new_include_file_location = os.path.join(new_location, new_include_file_location)
        new_model_name = new_model_name.replace('.fcs', '')  # remove .fcs if it exists
        new_model_path = os.path.join(new_location, new_model_name + '.fcs')

        if not os.path.exists(new_include_file_location):
            # make the folders if they don't already exist
            os.makedirs(new_include_file_location)

        model_file_generator = NexusModelFileGenerator(model=self, model_name=new_model_name)

        # update the content in the surface file:
        if self.model_files.surface_files is not None and self.model_files.surface_files.get(1, None) is not None:
            surface_file = self.model_files.surface_files[1]
        else:
            # create an empty surface file if it doesn't exist
            surface_file = NexusFile(file_content_as_list=[], location='', origin=None)
            self.model_files.surface_files = {1: surface_file}
        surface_content = model_file_generator.output_surface_section()
        update_model_file(file=surface_file, new_content=surface_content, new_folder_path=new_include_file_location,
                          new_name=new_model_name, suffix='_surface.dat')

        # update the structured grid file
        warnings.warn('Structured grid file is not yet implemented in NexusSimulator.write_out_new_model. ')

        # update the wells file
        if self.model_files.well_files is not None and self.model_files.well_files.get(1, None) is not None:
            wells_file = self.model_files.well_files[1]
        else:
            # create an empty wells file if it doesn't exist
            wells_file = NexusFile(file_content_as_list=[], location='', origin=None)
            self.model_files.well_files = {1: wells_file}
        wells_content = model_file_generator.output_wells_section()
        update_model_file(file=wells_file, new_content=wells_content, new_folder_path=new_include_file_location,
                          new_name=new_model_name, suffix='_wells.dat')

        # update each of the dynamic properties files
        dynamic_props = {
            'pvt': (self.pvt, self.model_files.pvt_files),
            'separator': (self.separator, self.model_files.separator_files),
            'water': (self.water, self.model_files.water_files),
            'equil': (self.equil, self.model_files.equil_files),
            'rock': (self.rock, self.model_files.rock_files),
            'relperm': (self.relperm, self.model_files.relperm_files),
            'valve': (self.valve, self.model_files.valve_files),
            'aquifer': (self.aquifer, self.model_files.aquifer_files),
            'hyd': (self.hydraulics, self.model_files.hyd_files),
            'gaslift': (self.gaslift, self.model_files.gaslift_files)
        }

        for suffix, (prop, dyn_files) in dynamic_props.items():
            for method_no, method in prop.inputs.items():
                file_name = f"{new_model_name}_{suffix}_{method_no}.dat"
                method_file_path = os.path.join(new_include_file_location, file_name)
                new_dyn_file = NexusFile(
                        file_content_as_list=method.to_string().splitlines(keepends=True),
                        location=method_file_path,
                        origin=new_model_path
                    )
                if dyn_files is None:
                    # ensure the dynamic files collection exists and if it doesn't then set it
                    files_collection_name = f"{suffix}_files"
                    self.model_files.__setattr__(files_collection_name, {method_no: new_dyn_file})
                else:
                    dyn_files[method_no] = new_dyn_file
                method.write_to_file(new_file_path=method_file_path, overwrite_file=False)

        # update the options file
        options_file_name = f"{new_model_name}_options.dat"
        if self.model_files.options_file is None and self._options is not None:
            self.set_options(self._options, os.path.join(new_include_file_location, options_file_name))
        elif self.model_files.options_file is not None:
            # update the existing options file
            options_file_content = model_file_generator.output_options_section()

            self.model_files.options_file.location = os.path.join(new_include_file_location, options_file_name)
            self.model_files.options_file.origin = new_model_path
            self.model_files.options_file.file_content_as_list = options_file_content.splitlines(keepends=True)

            self.model_files.options_file.write_to_file(
                new_file_path=self.model_files.options_file.location, overwrite_file=overwrite_files)

        # update runcontrol
        runcontrol_path = os.path.join(new_include_file_location, f"{new_model_name}_reporting.dat")
        if self.model_files.runcontrol_file is None and self.reporting is not None and self.sim_controls is not None:
            self.set_reporting_controls(reporting=self.reporting, new_file_path=runcontrol_path)
        elif self.model_files.runcontrol_file is not None:
            runcontrol_file_content = model_file_generator.output_runcontrol_section()
            update_model_file(file=self.model_files.runcontrol_file, new_content=runcontrol_file_content,
                              new_folder_path=new_include_file_location,
                              new_name=new_model_name, suffix='_runcontrol.dat')

        # create new fcsfile
        fcs_content = model_file_generator.generate_base_model_file_contents()
        self.model_files.file_content_as_list = fcs_content.splitlines(keepends=True)
        self.model_files.location = new_model_path
        self.model_files.write_to_file(new_file_path=new_model_path, overwrite_file=overwrite_files)

    @property
    def summary(self) -> str:
        """Returns a summary of the model contents."""
        fluid_type = ''
        # Verify if 'surface_file' exists.
        if (self.model_files.surface_files is not None and
                self.model_files.surface_files[1].file_content_as_list is not None):
            fluid_type = self.get_fluid_type(
                surface_file_content=self.model_files.surface_files[1].file_content_as_list
            )

        # Create a summary of number of completions per well:
        list_of_wells = self.wells.get_all()
        list_of_well_names = [well.well_name for well in list_of_wells]
        number_of_completions = [len(well.completions) for well in list_of_wells]
        well_summary = [f'{y} has: {z} completions' for y, z in zip(list_of_well_names, number_of_completions)]

        model_reporting_date = self.sim_controls.times[-1]
        range_x = None
        range_y = None
        range_z = None
        if self.grid is not None:
            range_x = self.grid.range_x
            range_y = self.grid.range_y
            range_z = self.grid.range_z

        relperm_summary = self.relperm.summary
        pvt_summary = self.pvt.summary
        equil_summary = self.equil.summary
        hyd_summary = self.hydraulics.summary
        model_summary = f"""    Start Date: {self.start_date}
    Last reporting date: {model_reporting_date}
    Grid Dimensions (x y z) : {range_x} x {range_y} x {range_z}
    Well summary: Well names: {well_summary}
    Fluid type: {fluid_type}
    Relperm:
        {relperm_summary.strip()}
    PVT:
        {pvt_summary.strip()}
    Hydraulics:
        {hyd_summary.strip()}
    Equil:
        {equil_summary.strip()}
    """
        return model_summary

    @property
    def sim_controls(self) -> SimControls:
        """Returns an instance of Sim controls class."""
        return self._sim_controls

    @property
    def options(self) -> NexusOptions | None:
        """Returns an instance of Nexus options class."""
        return self._options

    @property
    def is_multi_reservoir(self) -> bool:
        """Returns True if the model is a multi-reservoir model, False otherwise."""
        return self.__is_multi_reservoir

    @property
    def reservoir_paths(self) -> dict[str, str]:
        """Returns a dictionary of reservoir names and their corresponding file paths."""
        return self.__reservoir_paths

    @property
    def multi_reservoirs(self) -> dict[str, NexusSimulator]:
        """Returns a dictionary of multi-reservoir names and their corresponding NexusSimulator instances."""
        return self.__multi_reservoirs

    def __process_multi_reservoir_model(self) -> None:
        """Processes a multi-reservoir model by extracting reservoir paths and creating NexusSimulator instances
        for each.
        """
        warnings.warn('Multi-reservoir models are partially supported. '
                      'Some features may not work as expected.')
        if self.model_files.multi_reservoir_files is None:
            return

        for reservoir_name, reservoir_file in self.model_files.multi_reservoir_files.items():
            if reservoir_file.location is None:
                warnings.warn(f'Reservoir file location for {reservoir_name} not found.')
                continue

            # Store the reservoir path
            self.__reservoir_paths[reservoir_name] = reservoir_file.location

            # Create a NexusSimulator instance for the reservoir
            self.__multi_reservoirs[reservoir_name] = NexusSimulator(
                origin=reservoir_file.location,
                destination=self.destination,
                lazy_loading=self.__lazy_loading
            )

    def set_options(self, options: NexusOptions, options_file_path: Optional[str] = None) -> None:
        """Sets the Nexus options for the simulator.

        Args:
            options (NexusOptions): An instance of NexusOptions to set.
            options_file_path (str): Path to the options file to be set.
        """
        if not isinstance(options, NexusOptions):
            raise TypeError("options must be an instance of NexusOptions")
        if options_file_path is None:
            # pick a default name
            options_file_path = os.path.join(os.path.dirname(self.model_files.location),
                                             f'{self.root_name}_options.dat')
        self._options = options
        # Ensure the options file is set correctly
        new_options_file = NexusFile(
            file_content_as_list=options.to_string().splitlines(keepends=True),
            location=options_file_path,
            origin=self.model_files.location,
        )
        self.model_files.options_file = new_options_file
        self._options.file = new_options_file

    def set_sim_controls(self, sim_controls: SimControls) -> None:
        """Sets the simulation controls for the simulator.

        Args:
            sim_controls (SimControls): An instance of SimControls to set.
        """
        if not isinstance(sim_controls, SimControls):
            raise TypeError("sim_controls must be an instance of SimControls")
        self._sim_controls = sim_controls
        # ensure the model is correctly set in sim_controls
        setattr(self._sim_controls, '_SimControls__model', self)

    @property
    def ipr_methods(self) -> NexusIprMethods:
        """Returns an instance of NexusIPRMethods."""
        return self.__ipr_methods

    @property
    def wells(self) -> NexusWells:
        """Returns the associated NexusWells for the simulator."""
        return self._wells

    @property
    def reporting(self) -> NexusReporting:
        """Returns the associated NexusReporting for the simulator."""
        return self._reporting

    def set_reporting_controls(self, reporting: NexusReporting, new_file_path: str | None = None) -> None:
        """Sets the new reporting controls for the simulator.

        Args:
            reporting (NexusReporting): An instance of NexusReporting to set.
            new_file_path (str | None): Optional path to save the reporting file. If None, a default path is used.
        """
        if not isinstance(reporting, NexusReporting):
            raise TypeError("reporting must be an instance of NexusReporting")
        if new_file_path is None:
            # pick a default name
            new_file_path = os.path.join(os.path.dirname(self.model_files.location), f'{self.root_name}_reporting.dat')
        # Ensure the options file is set correctly

        self._reporting = reporting
        model_name = self.root_name
        if model_name is None:
            model_name = 'default_name.fcs'
        model_file_generator = NexusModelFileGenerator(model=self, model_name=model_name)
        new_reporting_file = NexusFile(
            file_content_as_list=model_file_generator.output_runcontrol_section().splitlines(keepends=True),
            location=new_file_path,
            origin=self.model_files.location,
        )
        self.model_files.runcontrol_file = new_reporting_file

        # ensure the model is correctly set in reporting
        setattr(self._reporting, '_NexusReporting__model', self)
