from hyperscale.kite.checks.core import CheckResult
from hyperscale.kite.checks.core import CheckStatus
from hyperscale.kite.config import Config
from hyperscale.kite.data import get_ec2_instances
from hyperscale.kite.data import get_ecs_clusters
from hyperscale.kite.data import get_eks_clusters
from hyperscale.kite.data import get_guardduty_detectors
from hyperscale.kite.helpers import get_account_ids_in_scope

EC2_FEATURES = [
    "EBS_MALWARE_PROTECTION",
    "RUNTIME_MONITORING",
]
EC2_ADDITIONAL = [
    "EC2_AGENT_MANAGEMENT",
]
EKS_FEATURES = [
    "RUNTIME_MONITORING",
]
EKS_ADDITIONAL = [
    "EKS_ADDON_MANAGEMENT",
]
ECS_FEATURES = [
    "RUNTIME_MONITORING",
]
ECS_ADDITIONAL = [
    "ECS_FARGATE_AGENT_MANAGEMENT",
]


def _summarize_guardduty_runtime_features() -> str:
    summary = "GuardDuty Runtime Monitoring Protection Status:\n\n"
    missing_protection = False
    for account_id in get_account_ids_in_scope():
        for region in Config.get().active_regions:
            detectors = get_guardduty_detectors(account_id, region)
            ec2s = get_ec2_instances(account_id, region) or []
            eks_clusters = get_eks_clusters(account_id, region) or []
            ecs_clusters = get_ecs_clusters(account_id, region) or []
            ecs_fargate = False
            for cluster in ecs_clusters:
                providers = cluster.get("capacityProviders", [])
                if "FARGATE" in providers or "FARGATE_SPOT" in providers:
                    ecs_fargate = True
                    break
            if not (ec2s or eks_clusters or ecs_fargate):
                continue
            if not detectors:
                summary += (
                    f"Account: {account_id}, Region: {region}\n"
                    "  - No GuardDuty detectors found (workloads present)\n"
                )
                missing_protection = True
                continue
            for detector in detectors:
                detector_missing = False
                detector_summary = f"Account: {account_id}, Region: {region}\n"
                features = {f["Name"]: f for f in detector.get("Features", [])}
                if ec2s:
                    for feat in EC2_FEATURES:
                        if features.get(feat, {}).get("Status") != "ENABLED":
                            detector_summary += (
                                f"  - {feat} not enabled (EC2 present)\n"
                            )
                            detector_missing = True
                    runtime = features.get("RUNTIME_MONITORING", {})
                    if runtime:
                        add_configs = {
                            c["Name"]: c
                            for c in runtime.get("AdditionalConfiguration", [])
                        }
                        for add in EC2_ADDITIONAL:
                            if add_configs.get(add, {}).get("Status") != "ENABLED":
                                detector_summary += (
                                    f"  - {add} not enabled (EC2 present)\n"
                                )
                                detector_missing = True
                if eks_clusters:
                    for feat in EKS_FEATURES:
                        if features.get(feat, {}).get("Status") != "ENABLED":
                            detector_summary += (
                                f"  - {feat} not enabled (EKS present)\n"
                            )
                            detector_missing = True
                    runtime = features.get("RUNTIME_MONITORING", {})
                    if runtime:
                        add_configs = {
                            c["Name"]: c
                            for c in runtime.get("AdditionalConfiguration", [])
                        }
                        for add in EKS_ADDITIONAL:
                            if add_configs.get(add, {}).get("Status") != "ENABLED":
                                detector_summary += (
                                    f"  - {add} not enabled (EKS present)\n"
                                )
                                detector_missing = True
                if ecs_fargate:
                    for feat in ECS_FEATURES:
                        if features.get(feat, {}).get("Status") != "ENABLED":
                            detector_summary += (
                                f"  - {feat} not enabled (ECS Fargate present)\n"
                            )
                            detector_missing = True
                    runtime = features.get("RUNTIME_MONITORING", {})
                    if runtime:
                        add_configs = {
                            c["Name"]: c
                            for c in runtime.get("AdditionalConfiguration", [])
                        }
                        for add in ECS_ADDITIONAL:
                            if add_configs.get(add, {}).get("Status") != "ENABLED":
                                detector_summary += (
                                    f"  - {add} not enabled (ECS Fargate present)\n"
                                )
                                detector_missing = True
                if detector_missing:
                    summary += detector_summary
                    missing_protection = True
    if not missing_protection:
        summary += (
            "All required GuardDuty protections are enabled for all workloads "
            "present.\n"
        )
    return summary


class AutomateMalwareAndThreatDetectionCheck:
    def __init__(self):
        self.check_id = "automate-malware-and-threat-detection"
        self.check_name = "Automate Malware and Threat Detection"

    @property
    def question(self) -> str:
        return (
            "Are tools such as GuardDuty used to automate malware and threat detection?"
        )

    @property
    def description(self) -> str:
        return (
            "This check verifies that tools such as GuardDuty are used to automate "
            "malware and threat detection."
        )

    def run(self) -> CheckResult:
        message = (
            "Please review the GuardDuty runtime monitoring protection status "
            "below.\n\n"
        )
        summary = _summarize_guardduty_runtime_features()
        message += summary
        return CheckResult(
            status=CheckStatus.MANUAL,
            context=message,
        )
