"""
This code initializes the FPGA on cloned boards.
"""

import hashlib
import struct

import usb.core
from usb.util import (
    CTRL_OUT,
    CTRL_RECIPIENT_DEVICE,
    CTRL_TYPE_VENDOR,
    build_request_type,
)

VENDER_ID = 0x9588
PRODUCT_ID = 0x9980

request = build_request_type(CTRL_OUT, CTRL_TYPE_VENDOR, CTRL_RECIPIENT_DEVICE)


init = [
    (3654, b"\x00\x01\x02\x02\x03\x03\x04\x04\x05\x05"),
    (818, b"\xe4\xf5,\xf5+\xf5*\xf5)\xc2\x03\xc2\x00\xd2\x02\xc2"),
    (834, b"\x01\x12\x07\xf7~\x0c\x7f\x00\x8e\n\x8f\x0bu\x12\x0cu"),
    (850, b"\x13\x12u\x08\x0cu\t\x1cu\x10\x0cu\x11Ju\x14"),
    (866, b"\x0cu\x15\x86\xeeT\xc0p\x03\x02\x045u-\x00u"),
    (882, b".\x80\x8e/\x8f0\xc3t\xa8\x9f\xfft\x0c\x9e\xcf$"),
    (898, b"\x02\xcf4\x00\xfe\xe4\x8f(\x8e'\xf5&\xf5%\xf5$"),
    (914, b"\xf5#\xf5\"\xf5!\xaf(\xae'\xad&\xac%\xab$"),
    (930, b'\xaa#\xa9"\xa8!\xc3\x12\r\xa9P5\xae#\xaf$'),
    (946, b"\xe50/\xf5\x82\xe5/>\xf5\x83\xe0\xfd\xe5./\xf5"),
    (962, b"\x82\xe5->\xf5\x83\xed\xf0\xe4\xfa\xf9\xf8\xef$\x01\xf5"),
    (978, b'$\xea>\xf5#\xe95"\xf5"\xe85!\xf5!\x80'),
    (994, b"\xb5\x85-\n\x85.\x0bt\x00$\x80\xfft\x0c4\xff"),
    (1010, b"\xfe\xc3\xe5\x13\x9f\xf5\x13\xe5\x12\x9e\xf5\x12\xc3\xe5\r\x9f"),
    (1026, b"\xf5\r\xe5\x0c\x9e\xf5\x0c\xc3\xe5\x0f\x9f\xf5\x0f\xe5\x0e\x9e"),
    (1042, b"\xf5\x0e\xc3\xe5\t\x9f\xf5\t\xe5\x08\x9e\xf5\x08\xc3\xe5\x11"),
    (1058, b"\x9f\xf5\x11\xe5\x10\x9e\xf5\x10\xc3\xe5\x15\x9f\xf5\x15\xe5\x14"),
    (1074, b"\x9e\xf5\x14\xd2\xe8C\xd8 \x90\xe6h\xe0D\t\xf0\x90"),
    (1090, b"\xe6\x82\xe0D\x10\xf0\x90\xe6\\\xe0D=\xf0\xd2\xaf\xd2"),
    (1106, b"\x07\x12\x0c\xe0S\x8e\xf8\xc2\x03\x12\x04\x950\x01\x05\x12"),
    (1122, b"\x00\x80\xc2\x010\x03\xf2\x12\x08\xfdP\xed\xc2\x03\x12\r"),
    (1138, b"\xba \x00\x16\x90\xe6\x82\xe00\xe7\x04\xe0 \xe1\xef\x90"),
    (1154, b"\xe6\x82\xe00\xe6\x04\xe0 \xe0\xe4\x12\r<\x12\x0e\\"),
    (1170, b"\x80\xc7"),
    (1172, b'"'),
    (3643, b'\x90\xe5\r\xe00\xe4\x02\xc3"\xd3"'),
    (128, b"\x90\xe6\xb9\xe0p\x03\x02\x01j\x14p\x03\x02\x02\x13$"),
    (144, b"\xfep\x03\x02\x02\xa9$\xfbp\x03\x02\x01d\x14p\x03"),
    (160, b"\x02\x01^\x14p\x03\x02\x01R\x14p\x03\x02\x01X$"),
    (176, b"\x05`\x03\x02\x03\x1e\x12\x0e^@\x03\x02\x03*\x90\xe6"),
    (192, b"\xbb\xe0$\xfe`;\x14`V$\xfd`\x16\x14`@"),
    (208, b"$\x06pt\xe5\n\x90\xe6\xb3\xf0\xe5\x0b\x90\xe6\xb4\xf0"),
    (224, b"\x02\x03*\x12\x0e;P\x0f\xe5\x12\x90\xe6\xb3\xf0\xe5\x13"),
    (240, b"\x90\xe6\xb4\xf0\x02\x03*\x90\xe6\xa0\xe0D\x01\xf0\x02\x03"),
    (256, b"*\xe5\x0c\x90\xe6\xb3\xf0\xe5\r\x90\xe6\xb4\xf0\x02\x03*"),
    (272, b"\xe5\x0e\x90\xe6\xb3\xf0\xe5\x0f\x90\xe6\xb4\xf0\x02\x03*\x90"),
    (288, b"\xe6\xba\xe0\xff\x12\rh\xaa\x06\xa9\x07{\x01\xeaI`"),
    (304, b"\r\xee\x90\xe6\xb3\xf0\xef\x90\xe6\xb4\xf0\x02\x03*\x90\xe6"),
    (320, b"\xa0\xe0D\x01\xf0\x02\x03*\x90\xe6\xa0\xe0D\x01\xf0\x02"),
    (336, b"\x03*\x12\x0e\x1d\x02\x03*\x12\x0eP\x02\x03*\x12\x0b"),
    (352, b"\xf8\x02\x03*\x12\x0e\x0b\x02\x03*\x12\x0e`@\x03\x02"),
    (368, b"\x03*\x90\xe6\xb8\xe0$\x7f`+\x14`<$\x02`"),
    (384, b"\x03\x02\x02\t\xa2\x00\xe43\xff%\xe0\xff\xa2\x02\xe43"),
    (400, b"O\x90\xe7@\xf0\xe4\xa3\xf0\x90\xe6\x8a\xf0\x90\xe6\x8bt"),
    (416, b"\x02\xf0\x02\x03*\xe4\x90\xe7@\xf0\xa3\xf0\x90\xe6\x8a\xf0"),
    (432, b"\x90\xe6\x8bt\x02\xf0\x02\x03*\x90\xe6\xbc\xe0T~\xff"),
    (448, b"~\x00\xe0\xd3\x94\x80@\x06|\x00}\x01\x80\x04|\x00"),
    (464, b"}\x00\xecN\xfe\xedO$F\xf5\x82t\x0e>\xf5\x83"),
    (480, b"\xe4\x93\xff3\x95\xe0\xfe\xef$\xa1\xff\xee4\xe6\x8f\x82"),
    (496, b"\xf5\x83\xe0T\x01\x90\xe7@\xf0\xe4\xa3\xf0\x90\xe6\x8a\xf0"),
    (512, b"\x90\xe6\x8bt\x02\xf0\x02\x03*\x90\xe6\xa0\xe0D\x01\xf0"),
    (528, b"\x02\x03*\x12\x0eb@\x03\x02\x03*\x90\xe6\xb8\xe0$"),
    (544, b"\xfe`\x1d$\x02`\x03\x02\x03*\x90\xe6\xba\xe0\xb4\x01"),
    (560, b"\x05\xc2\x00\x02\x03*\x90\xe6\xa0\xe0D\x01\xf0\x02\x03*"),
    (576, b"\x90\xe6\xba\xe0pY\x90\xe6\xbc\xe0T~\xff~\x00\xe0"),
    (592, b"\xd3\x94\x80@\x06|\x00}\x01\x80\x04|\x00}\x00\xec"),
    (608, b"N\xfe\xedO$F\xf5\x82t\x0e>\xf5\x83\xe4\x93\xff"),
    (624, b"3\x95\xe0\xfe\xef$\xa1\xff\xee4\xe6\x8f\x82\xf5\x83\xe0"),
    (640, b"T\xfe\xf0\x90\xe6\xbc\xe0T\x80\xff\x13\x13\x13T\x1f\xff"),
    (656, b"\xe0T\x0f/\x90\xe6\x83\xf0\xe0D \xf0\x02\x03*\x90"),
    (672, b"\xe6\xa0\xe0D\x01\xf0\x02\x03*\x12\x0edP|\x90\xe6"),
    (688, b"\xb8\xe0$\xfe` $\x02p[\x90\xe6\xba\xe0\xb4\x01"),
    (704, b"\x04\xd2\x00\x80e\x90\xe6\xba\xe0d\x02`]\x90\xe6\xa0"),
    (720, b"\xe0D\x01\xf0\x80T\x90\xe6\xbc\xe0T~\xff~\x00\xe0"),
    (736, b"\xd3\x94\x80@\x06|\x00}\x01\x80\x04|\x00}\x00\xec"),
    (752, b"N\xfe\xedO$F\xf5\x82t\x0e>\xf5\x83\xe4\x93\xff"),
    (768, b"3\x95\xe0\xfe\xef$\xa1\xff\xee4\xe6\x8f\x82\xf5\x83\xe0"),
    (784, b"D\x01\xf0\x80\x15\x90\xe6\xa0\xe0D\x01\xf0\x80\x0c\x12\r"),
    (800, b"\x0fP\x07\x90\xe6\xa0\xe0D\x01\xf0\x90\xe6\xa0\xe0D\x80"),
    (816, b"\xf0"),
    (817, b'"'),
    (51, b"\x02\x0eX"),
    (3672, b"S\xd8\xef2"),
    (3072, b"\x12\x01\x00\x02\x00\x00\x00@\x88\x95\x99\x98\x00\x00\x01\x02"),
    (3088, b"\x00\x01\n\x06\x00\x02\x00\x00\x00@\x01\x00\t\x02.\x00"),
    (3104, b"\x01\x01\x00@\x05\t\x04\x00\x00\x04\xff\x00\x00\x00\x07\x05"),
    (3120, b"\x01\x02\x00\x02\x00\x07\x05\x81\x02\x00\x02\x00\x07\x05\x02\x02"),
    (3136, b"\x00\x02\x00\x07\x05\x88\x02\x00\x02\x00\t\x02<\x00\x01\x01"),
    (3152, b"\x00@\x05\t\x04\x00\x00\x06\xff\x00\x00\x00\x07\x05\x01\x02"),
    (3168, b"@\x00\x00\x07\x05\x81\x02@\x00\x00\x07\x05\x02\x02@\x00"),
    (3184, b"\x00\x07\x05\x04\x02@\x00\x00\x07\x05\x06\x02@\x00\x00\x07"),
    (3200, b"\x05\x88\x02@\x00\x00\x04\x03\t\x04\x0c\x03B\x00J\x00"),
    (3216, b"J\x00C\x00Z\x00\x12\x03U\x00S\x00B\x00L\x00"),
    (3232, b"M\x00C\x00V\x002\x00\x00\x00"),
    (3388, b"\x90\xe6\x82\xe00\xe0\x04\xe0 \xe6\x0b\x90\xe6\x82\xe00"),
    (3404, b"\xe1\x19\xe00\xe7\x15\x90\xe6\x80\xe0D\x01\xf0\x7f\x14~"),
    (3420, b'\x00\x12\x0b\x18\x90\xe6\x80\xe0T\xfe\xf0"'),
    (3296, b"0\x07\t\x90\xe6\x80\xe0D\n\xf0\x80\x07\x90\xe6\x80\xe0"),
    (3312, b"D\x08\xf0\x7f\xdc~\x05\x12\x0b\x18\x90\xe6]t\xff\xf0"),
    (3328, b'\x90\xe6_\xf0S\x91\xef\x90\xe6\x80\xe0T\xf7\xf0"'),
    (3432, b"\xa9\x07"),
    (3434, b"\xae\x14\xaf\x15\x8f\x82\x8e\x83\xa3\xe0d\x03p\x17\xad\x01"),
    (3450, b'\x19\xedp\x01"\x8f\x82\x8e\x83\xe0|\x00/\xfd\xec>'),
    (3466, b"\xfe\xaf\x05\x80\xdf\xe4\xfe\xff"),
    (3474, b'"'),
    (3514, b"\x90\xe6\x82\xe0D\xc0\xf0\x90\xe6\x81\xf0C\x87\x01\x00\x00"),
    (3530, b'\x00\x00\x00"'),
    (2840, b"\x8e1\x8f2\x90\xe6\x00\xe0T\x18p\x12\xe52$\x01"),
    (2856, b"\xff\xe451\xc3\x13\xf51\xef\x13\xf52\x80\x15\x90\xe6"),
    (2872, b"\x00\xe0T\x18\xff\xbf\x10\x0b\xe52%\xe0\xf52\xe51"),
    (2888, b"3\xf51\xe52\x152\xae1p\x02\x151N`\x05"),
    (2904, b'\x12\r\xce\x80\xee"'),
    (3534, b"t\x00\xf5\x86\x90\xfd\xa5|\x05\xa3\xe5\x82E\x83p\xf9"),
    (3550, b'"'),
    (67, b"\x02\t\x00"),
    (83, b"\x02\t\x00"),
    (2304, b"\x02\x08\xe5\x00\x02\r\xf5\x00\x02\r\xdf\x00\x02\x0b\xe0\x00"),
    (2320, b"\x02\x0c\xaa\x00\x02\x0b\xa3\x00\x02\x08\xff\x00\x02\x0ef\x00"),
    (2336, b"\x02\x0eg\x00\x02\x0eh\x00\x02\x0ei\x00\x02\x0ej\x00"),
    (2352, b"\x02\x0ek\x00\x02\x0el\x00\x02\x0em\x00\x02\x0en\x00"),
    (2368, b"\x02\x0eo\x00\x02\x0ef\x00\x02\x0ep\x00\x02\x0eq\x00"),
    (2384, b"\x02\x0er\x00\x02\x0es\x00\x02\x0et\x00\x02\x0eu\x00"),
    (2400, b"\x02\x0ev\x00\x02\x0ef\x00\x02\x0ef\x00\x02\x0ef\x00"),
    (2416, b"\x02\x0ew\x00\x02\x0ex\x00\x02\x0ey\x00\x02\x0ez\x00"),
    (2432, b"\x02\x0e{\x00\x02\x0e|\x00\x02\x0e}\x00\x02\x0e~\x00"),
    (2448, b"\x02\x0e\x7f\x00\x02\x0e\x80\x00\x02\x0e\x81\x00\x02\x0e\x82\x00"),
    (2464, b"\x02\x0e\x83\x00\x02\x0e\x84\x00\x02\x0e\x85\x00\x02\x0e\x86\x00"),
    (2480, b"\x02\x0e\x87\x00\x02\x0e\x88\x00"),
    (2488, b"\xe4\xf59\xf5:\x90\xe6x\xe0 \xe0/\xae9\xaf:"),
    (2504, b"\xe4\xfc\xfd{\xe8z\xfd\xf9\xf8\xd3\x12\r\x93@\x1c\x00"),
    (2520, b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
    (2536, b"\x00\x05:\xe5:p\x02\x059\x80\xca\x90\xe6x\xe00"),
    (2552, b'\xe2\x03\x7f\x01"\xae9\xaf:\xe4\xfc\xfd{\xe8z\xfd'),
    (2568, b'\xf9\xf8\xd3\x12\r\x93P\x03\x7f\x04"0\x07Q\xe4\xf5'),
    (2584, b"9\xf5:\x90\xe6x\xe0 \xe1/\xae9\xaf:\xe4\xfc"),
    (2600, b"\xfd{Pz\xc3\xf9\xf8\xd3\x12\r\x93@\x1c\x00\x00\x00"),
    (2616, b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05"),
    (2632, b":\xe5:p\x02\x059\x80\xca\xae9\xaf:\xe4\xfc\xfd"),
    (2648, b'{Pz\xc3\xf9\xf8\xd3\x12\r\x93P\x03\x7f\x04"\x7f'),
    (2664, b"\x00"),
    (2665, b'"'),
    (2910, b"\xe4\xf59\xf5:\x90\xe6x\xe00\xe2#\xae9\xaf:"),
    (2926, b"\xe4\xfc\xfd{Pz\xc3\xf9\xf8\xd3\x12\r\x93@\x10\x00"),
    (2942, b"\x00\x00\x00\x00\x00\x05:\xe5:p\x02\x059\x80\xd6\xae"),
    (2958, b"9\xaf:\xe4\xfc\xfd{Pz\xc3\xf9\xf8\xd3\x12\r\x93"),
    (2974, b'P\x01"\xc3'),
    (2978, b'"'),
    (2666, b"\x8f2\x8d3\x8b4"),
    (2672, b"\xe4\xf55\xf56\xf57u8\x04\x90\xe6x\xe0D\x80"),
    (2688, b'\xf0\x12\x0b^P\x03\xaf8"\xe52T\xfe\x90\xe6y'),
    (2704, b"\xf0\xd2\x07\x12\t\xb8\x8f8\xe58\xd3\x94\x00@\x08\xe5"),
    (2720, b"8d\x01p:\x80\xd3\xe4\xf57\xe57\xc3\x954P"),
    (2736, b"+\xe53%7$\x80\xf5\x82\xe44\xe7\xf5\x83\xe0\x90"),
    (2752, b"\xe6y\xf0\xd2\x07\x12\t\xb8\x8f8\xe58\xd3\x94\x00@"),
    (2768, b"\x07\xe58\xb4\x01\x06\x80\xa2\x057\x80\xce\xe4\xf58\x90"),
    (2784, b"\xe6x\xe0D@\xf0\xe4\xf55\xf56\x90\xe6x\xe00"),
    (2800, b"\xe6#\xae5\xaf6\xe4\xfc\xfd{Pz\xc3\xf9\xf8\xd3"),
    (2816, b"\x12\r\x93@\x10\x00\x00\x00\x00\x00\x00\x056\xe56p"),
    (2832, b"\x02\x055\x80\xd6\xaf8"),
    (2839, b'"'),
    (1791, b"u6\x04\x90\xe6x\xe0D\x80\xf0\x12\x0b^P\x03\xaf"),
    (1807, b'6"\x90\xe6yt\xa0\xf0\xd2\x07\x12\t\xb8\x8f6\xe5'),
    (1823, b"6\xd3\x94\x00@\x0b\xe56d\x01`\x03\x02\x07\xf4\x80"),
    (1839, b"\xd2\xe4\x90\xe6y\xf0\xd2\x07\x12\t\xb8\x8f6\xe56\xd3"),
    (1855, b"\x94\x00@\x0b\xe56d\x01`\x03\x02\x07\xf4\x80\xb4\x90"),
    (1871, b"\xe6x\xe0D\x80\xf0\xd2\x07\x12\t\xb8\x8f6\xe56\xd3"),
    (1887, b"\x94\x00@\x0b\xe56d\x01`\x03\x02\x07\xf4\x80\x94\x90"),
    (1903, b"\xe6y\xe0\x90\xe7\xc0\xf0\xc2\x07\x12\t\xb8\x8f6\xe56"),
    (1919, b"\xd3\x94\x00@\t\xe56d\x01pj\x02\x07\x02\xe4\xf5"),
    (1935, b"4\xf55\xc3\xe55\x94?\xe54\x94\x00PA\xe55"),
    (1951, b"d>E4p\x07\x90\xe6x\xe0D \xf0\x90\xe6y"),
    (1967, b"\xe0\xfft\xc0%5\xf5\x82\xe44\xe7\xf5\x83\xef\xf0\xc2"),
    (1983, b"\x07\x12\t\xb8\x8f6\xe56\xd3\x94\x00@\x08\xe56\xb4"),
    (1999, b"\x01\r\x02\x07\x02\x055\xe55p\x02\x054\x80\xb4\x90"),
    (2015, b"\xe6x\xe0D@\xf0\x90\xe6y\xe0\x90\xe7\xff\xf0\xc2\x07"),
    (2031, b"\x12\t\xb8\x8f6\xaf6"),
    (2038, b'"'),
    (1485, b"u3\x04\x90\xe6x\xe0D\x80\xf0\x12\x0b^P\x03\xaf"),
    (1501, b'3"\x90\xe6x\xe0D\x80\xf0\x90\xe6yt\xef\xf0\xd2'),
    (1517, b"\x07\x12\t\xb8\x8f3\xe53\xd3\x94\x00@\x0b\xe53d"),
    (1533, b"\x01`\x03\x02\x06\xfc\x80\xcb\x90\xe6y\xe0\x90\xe7\xc0\xf0"),
    (1549, b"\xc2\x07\x12\t\xb8\x8f3\xe53\xd3\x94\x00@\x0b\xe53"),
    (1565, b"d\x01`\x03\x02\x06\xfc\x80\xaa\x90\xe6y\xe0\x90\xe7\xc0"),
    (1581, b"\xf0\xc2\x07\x12\t\xb8\x8f3\xe53\xd3\x94\x00@\x0b\xe5"),
    (1597, b"3d\x01`\x03\x02\x06\xfc\x80\x89\x90\xe6y\xe0\x90\xe7"),
    (1613, b"\xc1\xf0\xc2\x07\x12\t\xb8\x8f3\xe53\xd3\x94\x00@\x0c"),
    (1629, b"\xe53d\x01`\x03\x02\x06\xfc\x02\x05\xd0\x90\xe7\xc0\xe0"),
    (1645, b"\xd3\x94>P\x06\xa3\xe0\x94\x00@\x06\x90\xe7\xc0t>"),
    (1661, b"\xf0\x90\xe7\xc0\xe0\xd3\x94\x01@G\xe4\xf52\x90\xe7\xc0"),
    (1677, b"\xe0\x14\xff\xe52\xc3\x9fP8\xe0\x14\xb52\x07\x90\xe6"),
    (1693, b"x\xe0D \xf0\x90\xe6y\xe0\xfft\xc2%2\xf5\x82"),
    (1709, b"\xe44\xe7\xf5\x83\xef\xf0\xc2\x07\x12\t\xb8\x8f3\xe53"),
    (1725, b"\xd3\x94\x00@\x08\xe53\xb4\x01\x07\x02\x05\xd0\x052\x80"),
    (1741, b"\xbc\x90\xe6x\xe0D@\xf0\x90\xe7\xc0\xe0\xd3\x94\x00@"),
    (1757, b"\x1e\xe0$\x01\xff\xe43\x90\xe6y\xe0\xfdt\xc0/\xf5"),
    (1773, b"\x82\xe44\xe7\xf5\x83\xed\xf0\xc2\x07\x12\t\xb8\x8f3\xaf"),
    (1789, b"3"),
    (1790, b'"'),
    (2039, b"\x90\xe6\x00\xe0T\xe7D\x10\xf0\x00\x00\x00\x90\xe6\x01t"),
    (2055, b"\x8b\xf0\x00\x00\x00\x90\xe6\x10t\xa0\xf0\x90\xe6\x11\xf0\x00"),
    (2071, b"\x00\x00\x90\xe6\x0bt\x03\xf0\x00\x00\x00\x90\xe6\x12t\xab"),
    (2087, b"\xf0\x00\x00\x00\x90\xe6\x15t\xe2\xf0\x00\x00\x00\x90\xe6\x04"),
    (2103, b"t\x80\xf0\x00\x00\x00t\x02\xf0\x00\x00\x00t\x04\xf0\x00"),
    (2119, b"\x00\x00t\x06\xf0\x00\x00\x00t\x08\xf0\x00\x00\x00\xe4\xf0"),
    (2135, b"\x00\x00\x00\x90\xe6It\x82\xf0\x00\x00\x00\xf0\x00\x00\x00"),
    (2151, b"\xf0\x00\x00\x00\x90\xe6Ht\x88\xf0\x00\x00\x00\xf0\x00\x00"),
    (2167, b"\x00\x90\xe6\x18t\x11\xf0\x00\x00\x00\x90\xe6\x1bt\t\xf0"),
    (2183, b"\x00\x00\x00\xe4\x90\xe6&\xf0\x00\x00\x00\x90\xe6't\x08"),
    (2199, b"\xf0\x00\x00\x00\xe4\x90\xe6\x02\xf0\x00\x00\x00\x90\xe6\x03\xf0"),
    (2215, b"\x00\x00\x00\x90\xe6pt@\xf0\x00\x00\x00\xe4\x90\xe6\t"),
    (2231, b"\xf0\x00\x00\x00\x90\xe66t\x80\xf0\x00\x00\x00\xe4\x90\xe6"),
    (2247, b"7\xf0C\xb2\x03\x00\x00\x00\xc2\x80\x00\x00\x00\xd2\x81\x00"),
    (2263, b'\x00\x00\xc2\x06\xc2\x05\xc2\x04u\xaf\x07\xd2\x00"'),
    (1173, b" \x05\x030\x043\x90\xe6\xa2\xe00\xe1\x03\x02\x05\xcc"),
    (1189, b"0\x05\x17\x12\x06\xff\xefp\x08\x90\xe7\xfft\x04\xf0\x80"),
    (1205, b"\x05\xe4\x90\xe7\xff\xf0\xc2\x05\x80\x05\x12\x05\xcd\xc2\x04\x00"),
    (1221, b'\x00\x00\x90\xe6\x8ft@\xf0"0\x06<\x90\xe6\xa2\xe0'),
    (1237, b"0\xe1\x03\x02\x05\xcc\xe4\xff\xfet\x00/\xf5\x82\xe44"),
    (1253, b" \xf5\x83\xe0\xfdt\xc0/\xf5\x82\xe44\xe7\xf5\x83\xed"),
    (1269, b"\xf0\x0f\xbf\x00\x01\x0e\xefd\nNp\xdd\x00\x00\x00\x90"),
    (1285, b'\xe6\x8ft\x08\xf0\xc2\x06"\x90\xe6\xa1\xe00\xe1\x03\x02'),
    (1301, b"\x05\xcc\x90\xe6\x8d\xe0\xc3\x94\x02@\x06\x90\xe7\x81\xe0`"),
    (1317, b'\x06\xe4\x90\xe6\x8d\xf0"\x90\xe7\x80\xe0\x90 \x00\xf0\x90'),
    (1333, b"\xe7\x81\xe0\x90 \x01\xf0\x90\xe7\x82\xe0\x90 \x02\xf0\x90"),
    (1349, b"\xe7\x83\xe0\x90 \x03\xf0\xe4\xa3\xf0\xa3\xf0\x90\xe7\x80\xe0"),
    (1365, b"d\x07p$\x90\xe7\x82\xe0p\x12{\t}\x04\x7f\xa0"),
    (1381, b"\x12\nj\x90 \x04\xef\xf0\xd2\x06\x80V\x90\xe7\x82\xe0"),
    (1397, b"d\x01pN\xd2\x05\x80J\x90\xe7\x80\xe0\xb4\x08\x17\x90"),
    (1413, b"\xe6\x8d\xe0$\xfe\xfb}\x02\x7f\xee\x12\nj\x90 \x04"),
    (1429, b"\xef\xf0\xd2\x06\x80,\x90\xe7\x80\xe0\xb4\t\x04\xd2\x04\x80"),
    (1445, b"!\x90\xe7\x80\xe0d\np\x19\x90\xe7\x82\xe0p\x07\xc2"),
    (1461, b"\x81\x00\x00\x00\x80\n\xc2\x80\x00\x00\x00\xd2\x81\x00\x00\x00"),
    (1477, b"\xd2\x06\xe4\x90\xe6\x8d\xf0"),
    (1484, b'"'),
    (2301, b'\xd3"'),
    (3676, b'\xd3"'),
    (3678, b'\xd3"'),
    (3064, b'\x90\xe6\xba\xe0\xf5\x17\xd3"'),
    (3595, b"\x90\xe7@\xe5\x17\xf0\xe4\x90\xe6\x8a\xf0\x90\xe6\x8b\x04\xf0"),
    (3611, b'\xd3"'),
    (3664, b'\x90\xe6\xba\xe0\xf5\x16\xd3"'),
    (3613, b"\x90\xe7@\xe5\x16\xf0\xe4\x90\xe6\x8a\xf0\x90\xe6\x8b\x04\xf0"),
    (3629, b'\xd3"'),
    (3680, b'\xd3"'),
    (3682, b'\xd3"'),
    (3684, b'\xd3"'),
    (3343, b"\x90\xe6\xb9\xe0$/`\x12\x04p\x1e\x90\xe6\x04\xe0\xff"),
    (3359, b"C\x07\x80\x00\x00\x00\xef\xf0\x80\x11\x90\xe6\x04\xe0\xffS"),
    (3375, b'\x07\x7f\x00\x00\x00\xef\xf0\x80\x02\xd3"\xc3'),
    (3387, b'"'),
    (2277, b"\xc0\xe0\xc0\x83\xc0\x82\xd2\x01S\x91\xef\x90\xe6]t\x01"),
    (2293, b"\xf0\xd0\x82\xd0\x83\xd0\xe02"),
    (3551, b"\xc0\xe0\xc0\x83\xc0\x82S\x91\xef\x90\xe6]t\x04\xf0\xd0"),
    (3567, b"\x82\xd0\x83\xd0\xe02"),
    (3573, b"\xc0\xe0\xc0\x83\xc0\x82S\x91\xef\x90\xe6]t\x02\xf0\xd0"),
    (3589, b"\x82\xd0\x83\xd0\xe02"),
    (3242, b"\xc0\xe0\xc0\x83\xc0\x82\x85\x10\x0c\x85\x11\r\x85\r\x82\x85"),
    (3258, b"\x0c\x83\xa3t\x02\xf0\x85\x08\x0e\x85\t\x0f\x85\x0f\x82\x85"),
    (3274, b"\x0e\x83\xa3t\x07\xf0S\x91\xef\x90\xe6]t\x10\xf0\xd0"),
    (3290, b"\x82\xd0\x83\xd0\xe02"),
    (3040, b"\xc0\xe0\xc0\x83\xc0\x82\xd2\x03S\x91\xef\x90\xe6]t\x08"),
    (3056, b"\xf0\xd0\x82\xd0\x83\xd0\xe02"),
    (2979, b"\xc0\xe0\xc0\x83\xc0\x82\x90\xe6\x80\xe00\xe7 \x85\x08\x0c"),
    (2995, b"\x85\t\r\x85\r\x82\x85\x0c\x83\xa3t\x02\xf0\x85\x10\x0e"),
    (3011, b"\x85\x11\x0f\x85\x0f\x82\x85\x0e\x83\xa3t\x07\xf0S\x91\xef"),
    (3027, b"\x90\xe6]t \xf0\xd0\x82\xd0\x83\xd0\xe02"),
    (2303, b"2"),
    (3686, b"2"),
    (3687, b"2"),
    (3688, b"2"),
    (3689, b"2"),
    (3690, b"2"),
    (3691, b"2"),
    (3692, b"2"),
    (3693, b"2"),
    (3694, b"2"),
    (3695, b"2"),
    (3696, b"2"),
    (3697, b"2"),
    (3698, b"2"),
    (3699, b"2"),
    (3700, b"2"),
    (3701, b"2"),
    (3702, b"2"),
    (3703, b"2"),
    (3704, b"2"),
    (3705, b"2"),
    (3706, b"2"),
    (3707, b"2"),
    (3708, b"2"),
    (3709, b"2"),
    (3710, b"2"),
    (3711, b"2"),
    (3712, b"2"),
    (3713, b"2"),
    (3714, b"2"),
    (3715, b"2"),
    (3716, b"2"),
    (3717, b"2"),
    (3718, b"2"),
    (3719, b"2"),
    (3720, b"2"),
    (0, b"\x02\x0e/"),
    (3631, b"x\x7f\xe4\xf6\xd8\xfdu\x81:\x02\x032"),
    (3475, b"\xeb\x9f\xf5\xf0\xea\x9eB\xf0\xe9\x9dB\xf0\xecd\x80\xc8"),
    (3491, b'd\x80\x98E\xf0"'),
    (3497, b"\xeb\x9f\xf5\xf0\xea\x9eB\xf0\xe9\x9dB\xf0\xe8\x9cE\xf0"),
    (3513, b'"'),
]


def _firmware(device, start):
    """
    Start and end packet sending data.
    @param device:
    @param start:
    @return:
    """
    device.ctrl_transfer(
        request,
        bRequest=0xA0,
        wValue=0xE600,
        wIndex=0x0,
        data_or_wLength=b"\x01" if start else b"\x00",
    )


def _parse(f):
    """Parses the clone board sys file, for the FPGA code."""
    while True:
        data = f.read(22)
        if data is None or len(data) != 22:
            break
        length, _, value, end, payload, _ = struct.unpack("BBHB16sB", data)
        if end != 0:
            break
        yield value, payload[:length]


def _write_chunks(device, chunks):
    """
    Write chunks to the FPGA.

    @param device:
    @param chunks:
    @return:
    """
    _firmware(device, start=True)
    for value, payload in chunks:
        device.ctrl_transfer(
            request, bRequest=0xA0, wValue=value, wIndex=0x0, data_or_wLength=payload
        )
    _firmware(device, start=False)


def get_offset_by_hash(hash):
    if hash == "b86bc139a88592a4d71e64990133e86f":
        return 0x4440, "LMCV2U2"
    if hash == "f775f53d0cf8c9b3e7855837561a4e6c":
        return 0x11DE0, "LMCUSB2009"
    if hash == "cd408e29768af505d671fb5c216fcdcc":
        return 0x9F60, "LMCUSBdSYS"
    if hash == "2dd5017f569ac9efc292e6c085b08f36":
        return 0x4440, "LMCdotSYS"
    if hash == "48bcb097a38fdd84df43f6ee6e7f0195":
        return 0x5190, "LMCV2U2"
    if hash == "3a5e33b366f6a7af0f530a5ef68f137b":
        return 0x1390, "LMCV4"
    return 0x4440, "Unknown"


def _send_device_sys(device, sys_file, channel=None):
    """
    Open the file, write the parsed chunks.

    @param device:
    @param sys_file:
    @param offset:
    @return:
    """
    if isinstance(sys_file, str):
        with open(sys_file, "rb") as f:
            contents = f.read()
            f.seek(0)

            h = hashlib.md5(contents)
            offset, name = get_offset_by_hash(h.hexdigest())
            if channel:
                channel(
                    f"{sys_file} has hash:{h.hexdigest()} is {name} with offset {offset}"
                )
            f.seek(offset)
            _write_chunks(device, _parse(f))
    else:
        _write_chunks(device, _parse(sys_file))


def load_sys(sys_file, channel=None):
    """
    Find devices, send sys_file to the needed boards.

    @param sys_file:
    @param channel:
    @return:
    """
    try:
        devices = list(
            usb.core.find(idVendor=VENDER_ID, idProduct=PRODUCT_ID, find_all=True)
        )
        if channel:
            channel(f"{len(devices)} devices need initializing.")
        for i, device in enumerate(devices):
            if channel:
                channel(f"Clone board #{i + 1} detected. Sending Initialize.")
            _send_device_sys(device, sys_file, channel=channel)
    except usb.core.USBError as e:
        channel(str(e))
        raise ConnectionRefusedError


def load_chunks(chunks=None, channel=None):
    """
    Find devices, send chunks to the needed boards.
    @param chunks:
    @param channel:
    @return:
    """
    if chunks is None:
        chunks = init
    try:
        devices = list(
            usb.core.find(idVendor=VENDER_ID, idProduct=PRODUCT_ID, find_all=True)
        )
        if channel:
            channel(f"{len(devices)} devices need initializing.")
        for i, device in enumerate(devices):
            if channel:
                channel(f"Clone board #{i + 1} detected. Sending Initialize.")
            _write_chunks(device, chunks)
    except usb.core.USBError as e:
        channel(str(e))
        raise ConnectionRefusedError


if __name__ == "__main__":
    load_sys("Lmcv2u.sys", channel=print)
