"""
Lihuiyu TCP Connection

Governs the TCP out connections for the lihuiyu device. This allows the output controller to write to a particular
network location.
"""

import socket
import threading
import time


class TCPOutput:
    def __init__(self, context, name=None):
        super().__init__()
        self.service = context
        self._stream = None
        self.name = name

        self.lock = threading.RLock()
        self.buffer = bytearray()
        self.thread = None

    def connect(self):
        try:
            self._stream = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            # Make sure port is in a valid range...
            port = min(65535, max(0, self.service.port))
            self._stream.connect((self.service.address, port))
            self.service.signal("tcp;status", "connected")
        except TimeoutError:
            self.disconnect()
            self.service.signal("tcp;status", "timeout connect")
        except ConnectionError:
            self.disconnect()
            self.service.signal("tcp;status", "connection error")
        except (socket.gaierror, OverflowError) as e:
            self.disconnect()
            self.service.signal("tcp;status", "address resolve error")
        except socket.herror as e:
            self.disconnect()
            self.service.signal("tcp;status", f"herror: {str(e)}")
        except OSError as e:
            self.disconnect()
            self.service.signal("tcp;status", f"Host down {str(e)}")

    def disconnect(self):
        self.service.signal("tcp;status", "disconnected")
        if self._stream is not None:
            self._stream.close()
        self._stream = None

    def write(self, data):
        self.service.signal("tcp;write", data)
        with self.lock:
            self.buffer += data
            self.service.signal("tcp;buffer", len(self.buffer))
        self._start()

    realtime_write = write

    @property
    def viewbuffer(self):
        return self.buffer.decode("utf8")

    def _start(self):
        if self.thread is None:
            self.thread = self.service.threaded(
                self._sending,
                thread_name=f"sender-{self.service.port}",
                result=self._stop,
            )

    def _stop(self, *args):
        self.thread = None

    def _sending(self):
        tries = 0
        while True:
            try:
                if len(self.buffer):
                    if self._stream is None:
                        self.connect()
                        if self._stream is None:
                            return
                    with self.lock:
                        sent = self._stream.send(self.buffer)
                        del self.buffer[:sent]
                        self.service.signal("tcp;buffer", len(self.buffer))
                    tries = 0
                else:
                    tries += 1
                    time.sleep(0.1)
            except (ConnectionError, OSError):
                tries += 1
                self.disconnect()
                time.sleep(0.05)
            if tries >= 20:
                with self.lock:
                    if len(self.buffer) == 0:
                        break

    def __repr__(self):
        if self.name is not None:
            return (
                f"TCPOutput('{self.service.address}:{self.service.port}','{self.name}')"
            )
        return f"TCPOutput('{self.service.address}:{self.service.port}')"

    def __len__(self):
        return len(self.buffer)
