# This file is part of django-ca (https://github.com/mathiasertl/django-ca).
#
# django-ca is free software: you can redistribute it and/or modify it under the terms of the GNU General
# Public License as published by the Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# django-ca is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
# implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along with django-ca. If not, see
# <http://www.gnu.org/licenses/>.
#
# Generated by Django 5.1.2 on 2024-10-17 19:04

"""Migration for moving CertificateAuthority.crl_number to a new CertificateRevocationList instance."""

import json
from datetime import timedelta

from cryptography import x509

from django.conf import settings
from django.core.cache import cache
from django.db import migrations
from django.utils import timezone
from django.utils.timezone import make_naive


def migrate_crl_number(apps, schema_editor):  # pragma: no cover
    """Forward operation: CertificateAuthority.crl_number -> CertificateRevocationList."""
    CertificateAuthority = apps.get_model("django_ca", "CertificateAuthority")
    CertificateRevocationList = apps.get_model("django_ca", "CertificateRevocationList")

    # Only migrate enabled certificate authorities
    for ca in CertificateAuthority.objects.filter(enabled=True):
        crl_number_data = json.loads(ca.crl_number)
        for scope, crl_number in crl_number_data.get("scope", {}).items():
            only_contains_ca_certs = False
            only_contains_user_certs = False
            only_contains_attribute_certs = False
            if scope == "ca":
                only_contains_ca_certs = True
            elif scope == "user":
                only_contains_user_certs = True
            elif scope == "attribute":
                only_contains_attribute_certs = True

            # This is how cache keys where computed before 2.1.0:
            cache_key = f"crl_{ca.serial}_DER_{scope}"

            # Retrieve data from cache, if possible.
            try:
                crl_data = cache.get(cache_key)
                crl = x509.load_der_x509_crl(crl_data)
            except Exception:
                crl_data = None
                crl = None

            # If CRL was in the cache, set data, next_update and last_update.
            if crl is not None:
                next_update = crl.next_update_utc
                last_update = crl.last_update_utc

                if settings.USE_TZ is False:
                    next_update = make_naive(next_update)
                    last_update = make_naive(last_update)

            # If we have no data from the cache, we can only set a useless default data
            else:
                last_update = timezone.now() - timedelta(days=2)
                next_update = timezone.now() - timedelta(days=1)

            # Create CRL object
            CertificateRevocationList.objects.filter(only_some_reasons__isnull=True).get_or_create(
                ca=ca,
                number=crl_number,
                only_contains_ca_certs=only_contains_ca_certs,
                only_contains_user_certs=only_contains_user_certs,
                only_contains_attribute_certs=only_contains_attribute_certs,
                defaults={
                    "only_some_reasons": None,
                    "last_update": last_update,
                    "next_update": next_update,
                    "data": crl_data,
                },
            )


class Migration(migrations.Migration):
    dependencies = [
        ("django_ca", "0047_certificaterevocationlist"),
    ]
    operations = [migrations.RunPython(migrate_crl_number, migrations.RunPython.noop, elidable=True)]
