# This file is part of django-ca (https://github.com/mathiasertl/django-ca).
#
# django-ca is free software: you can redistribute it and/or modify it under the terms of the GNU General
# Public License as published by the Free Software Foundation, either version 3 of the License, or (at your
# option) any later version.
#
# django-ca is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
# implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along with django-ca. If not, see
# <http://www.gnu.org/licenses/>.
#
# Generated by Django 5.1.3 on 2024-12-01 19:38

import logging

from cryptography import x509
from cryptography.hazmat.primitives.serialization import Encoding

from django.core.exceptions import ImproperlyConfigured
from django.core.files.storage import InvalidStorageError, storages
from django.db import migrations
from django.utils import timezone

from django_ca.key_backends import ocsp_key_backends
from django_ca.key_backends.storages import StoragesOCSPBackend

log = logging.getLogger(__name__)


def configure_ocsp_responder_key(apps, schema_editor):  # pragma: no cover
    """Migration function to set ocsp responder key if it exists."""
    # We can't import the Person model directly as it may be a newer
    # version than this migration expects. We use the historical version.
    CertificateAuthority = apps.get_model("django_ca", "CertificateAuthority")

    now = timezone.now()
    for ca in CertificateAuthority.objects.filter(enabled=True).filter(not_after__gt=now, not_before__lt=now):
        try:
            ocsp_key_backend = ocsp_key_backends[ca.ocsp_key_backend_alias]
        except ValueError:  # pragma: no cover
            log.exception("%s: Cannot load OCSP key storage backend.")
            continue
        except ImproperlyConfigured:  # pragma: no cover  # not possible to reproduce with SettingsWrapper
            log.warning("%s: Key backend is not configured.", ca.ocsp_key_backend_alias)
            continue  # pragma: no cover

        # COVERAGE NOTE: No other implementations exist at the time of writing
        if not isinstance(ocsp_key_backend, StoragesOCSPBackend):  # pragma: no cover
            continue

        try:
            storage = storages[ocsp_key_backend.storage_alias]

        # COVERAGE NOTE: This exception should never happen, as this misconfiguration is already raised when
        # the ocsp_key_backend is accessed.
        except InvalidStorageError:  # pragma: no cover
            log.error(
                "%s: OCSP key backend storage alias does not refer to a valid storage backend.",
                ocsp_key_backend.storage_alias,
            )
            continue

        private_key_path = f"ocsp/{ca.serial}.key"
        public_key_path = f"ocsp/{ca.serial}.pem"

        if storage.exists(private_key_path) and storage.exists(public_key_path):
            ca.ocsp_key_backend_options["private_key"]["path"] = private_key_path

            # Read public key
            stream = storage.open(public_key_path)

            try:
                public_key_data: bytes = stream.read()  # pragma: no branch
            finally:
                stream.close()

            if not public_key_data.startswith(b"-----BEGIN CERTIFICATE-----"):
                try:
                    public_key = x509.load_der_x509_certificate(public_key_data)
                    public_key_data = public_key.public_bytes(Encoding.PEM)
                except ValueError:
                    log.warning(
                        "%s: Cannot encode certificate. Regenerate OCSP keys manually.", public_key_path
                    )
                    continue

            ca.ocsp_key_backend_options["certificate"]["pem"] = public_key_data.decode()
            ca.save()
        else:
            log.warning("Private or public key not found. Regenerate OCSP keys manually.")


class Migration(migrations.Migration):
    dependencies = [
        ("django_ca", "0050_certificateauthority_ocsp_key_backend_alias_and_more"),
    ]

    operations = [
        migrations.RunPython(configure_ocsp_responder_key, migrations.RunPython.noop, elidable=True)
    ]
