import asyncio
import logging
import uuid
import httpx

from pathlib import Path
from typing import Dict, Optional

from healthchain.sandbox.base import BaseClient
from healthchain.sandbox.utils import ensure_directory_exists, save_data_to_directory
from healthchain.sandbox.workflows import UseCaseType

log = logging.getLogger(__name__)


class SandboxEnvironment:
    """
    Manages the sandbox environment for testing and validation.
    Handles service initialization, client requests, and data management.

    This class provides a controlled environment for testing healthcare services,
    managing the lifecycle of sandbox instances, handling request/response data,
    and providing utilities for data persistence and logging.
    """

    def __init__(
        self,
        api: httpx.URL,
        path: str,
        client: Optional[BaseClient] = None,
        use_case_type: Optional[UseCaseType] = None,
        config: Optional[Dict] = None,
    ):
        """
        Initialize the sandbox environment

        Args:
            api: The API URL to be used for the sandbox
            path: The endpoint path to send requests to
            client: The client to use for sending requests
            use_case_type: Type of use case (clindoc, cds)
            config: Optional configuration dictionary for the sandbox
        """
        self._client = client
        self.type = use_case_type
        self.api = api
        self.path = path
        self.config = config

        self.responses = []
        self.sandbox_id = None
        self.url = None

    def start_sandbox(
        self,
        service_id: Optional[str] = None,
        save_data: bool = True,
        save_dir: str = "./output/",
    ) -> None:
        """
        Starts the sandbox: initializes service and sends request through the client.

        Args:
            service_id: Service identifier (default None)
            save_data: Whether to save request/response data
            save_dir: Directory to save data
            logging_config: Optional logging configuration
        """
        if self._client is None:
            raise RuntimeError(
                "Client is not configured. Please check your class initialization."
            )

        self.sandbox_id = uuid.uuid4()

        log.info(
            f"Starting sandbox {self.sandbox_id} with use case type {self.type.value}..."
        )
        endpoint = self.api.join(self.path)
        if service_id:
            endpoint = endpoint.join(service_id)

        log.info(
            f"Sending {len(self._client.request_data)} requests generated by {self._client.__class__.__name__} to {endpoint}"
        )
        log.debug(f"API: {self.api}, Path: {self.path}, Endpoint: {endpoint}")

        try:
            self.responses = asyncio.run(self._client.send_request(url=endpoint))
        except Exception as e:
            log.error(f"Couldn't start client: {e}", exc_info=True)

        if save_data:
            save_dir = Path(save_dir)
            request_path = ensure_directory_exists(save_dir / "requests")

            if self.type == UseCaseType.clindoc:
                extension = "xml"
                save_data_to_directory(
                    [request.model_dump_xml() for request in self._client.request_data],
                    "request",
                    self.sandbox_id,
                    request_path,
                    extension,
                )
            else:
                extension = "json"
                save_data_to_directory(
                    [
                        request.model_dump(exclude_none=True)
                        for request in self._client.request_data
                    ],
                    "request",
                    self.sandbox_id,
                    request_path,
                    extension,
                )

            log.info(f"Saved request data at {request_path}/")

            response_path = ensure_directory_exists(save_dir / "responses")
            save_data_to_directory(
                self.responses,
                "response",
                self.sandbox_id,
                response_path,
                extension,
            )
            log.info(f"Saved response data at {response_path}/")
