# Configuration file for the Sphinx documentation builder.
#
# This file only contains a selection of the most common options. For a full
# list see the documentation:
# https://www.sphinx-doc.org/en/master/usage/configuration.html

# -- Path setup --------------------------------------------------------------

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
#
import sys, pathlib, importlib, types, re
ROOT = pathlib.Path(__file__).resolve().parents[2]  # repo root
sys.path.insert(0, str(ROOT / "src"))               # src/ layout

# TEMP shim for bad "import Modules" statements
try:
    sys.modules["Modules"] = importlib.import_module("el1xr_opt.Modules")
except Exception:
    sys.modules.setdefault("Modules", types.ModuleType("Modules"))


# -- Project information -----------------------------------------------------

project = ''
copyright = '2025, EL1XR-dev'
author = 'EL1XR-dev Team'

# The short X.Y version
version = 'version 1.0.1'
# The full version, including alpha/beta/rc tags
release = ''


# -- General configuration ---------------------------------------------------

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    'sphinx.ext.todo',
    'sphinx.ext.githubpages',
    "sphinx.ext.autodoc",
    "sphinx.ext.autosummary",
    "sphinx.ext.napoleon",
    "sphinx.ext.viewcode",
    "sphinx.ext.intersphinx",
    "sphinx.ext.mathjax",
    "sphinx_copybutton",
    "sphinx_autodoc_typehints",
]
autosummary_generate = False
napoleon_google_docstring = False
napoleon_numpy_docstring = False

autodoc_mock_imports = [
    "pyomo",
    "gurobipy",
    "gurobi",
    # add any other large/optional deps used only at runtime
]

# Add any paths that contain templates here, relative to this directory.
templates_path = ['_templates']

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
#
# source_suffix = ['.rst', '.md']
source_suffix = '.rst'

# The master toctree document.
master_doc = 'index'

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
# language = None

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path.
exclude_patterns = ['_build', 'Thumbs.db', '.DS_Store']

# The name of the Pygments (syntax highlighting) style to use.
pygments_style = 'sphinx'

# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#

# Theme options are theme-specific and customize the look and feel of a theme
# further.  For a list of options available for each theme, see the
# documentation.
#
# html_theme_options = {}

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = []

# Custom sidebar templates, must be a dictionary that maps document names
# to template names.
#
# The default sidebars (for documents that don't match any pattern) are
# defined by theme itself.  Builtin themes are using these templates by
# default: ``['localtoc.html', 'relations.html', 'sourcelink.html',
# 'searchbox.html']``.
#
# html_sidebars = {}
html_theme = "sphinx_book_theme"
html_logo  = '../img/el1xr_opt_logo_v6.png'
html_theme_options = {
    "repository_url": "https://github.com/EL1XR-dev/el1xr_opt",
    "use_repository_button": True,
    "use_issues_button": True,
}
# Do NOT set footer_start/footer_end/navbar_*/*extra* items that include "about.html"

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
# create docs/rst/_static/ if it doesn't exist
if not (pathlib.Path(__file__).parent / "_static").exists():
    (pathlib.Path(__file__).parent / "_static").mkdir(parents=True)

html_static_path = ['_static']

html_js_files = [
    '../custom.js',
]

mathjax3_config = {
    "tex": {
        "macros": {
            # === SETS ===
            "nB": r"\mathcal{B}",                               # All buses
            "nBEP": r"\mathcal{B}^{pcc,e}",                     # Buses at point of common coupling (PCC) with electrical connections
            "nBE": r"\mathcal{B}^{e}",                          # Buses with electrical connections
            "nBHP": r"\mathcal{B}^{pcc,h}",                     # Buses at point of common coupling (PCC) with hydrogen connections
            "nBH": r"\mathcal{B}^{h}",                          # Buses with hydrogen connections
            "nC": r"\mathcal{C}",                               # All circuits
            "nG": r"\mathcal{G}",                               # All generators
            "nGE": r"\mathcal{G}^{e}",                          # All electrical generators
            "nGER": r"\mathcal{G}^{e,res}",                     # All renewable electrical generators
            "nGVRE": r"\mathcal{G}^{e,vres}",                   # All variable renewable electrical generators
            "nGENR": r"\mathcal{G}^{e,nonres}",                 # All non-renewable electrical generators
            "nGEH": r"\mathcal{G}^{e,h}",                       # All electrical generators consuming hydrogen
            "nGH": r"\mathcal{G}^{h}",                          # All hydrogen generators
            "nGNG": r"\mathcal{G}^{h,cng}",                     # All hydrogen generators consuming natural gas
            "nGHE": r"\mathcal{G}^{h,e}",                       # All hydrogen generators consuming electricity
            "nE": r"\mathcal{E}",                               # All storage units
            "nEE": r"\mathcal{E}^{e}",                          # All electrical storage units
            "nEH": r"\mathcal{E}^{h}",                          # All hydrogen storage units
            "nEV": r"\mathcal{EV}",                             # All vehicles
            "nEVE": r"\mathcal{EV}^{e}",                        # All electrical vehicles
            "nEVH": r"\mathcal{EV}^{h}",                        # All hydrogen vehicles
            "nR": r"\mathcal{R}",                               # All retailers
            "nRE": r"\mathcal{R}^{e}",                          # All electrical retailers
            "nRH": r"\mathcal{R}^{h}",                          # All hydrogen retailers
            "nDE": r"\mathcal{D}^{e}",                          # All electrical demands
            "nDH": r"\mathcal{D}^{h}",                          # All hydrogen demands
            "nK": r"\mathcal{K}",                               # All peak demands
            "nKE": r"\mathcal{K}^{e}",                          # All electrical peak demands
            "nKH": r"\mathcal{K}^{h}",                          # All hydrogen peak demands
            "nL": r"\mathcal{L}",                               # All lines
            "nLE": r"\mathcal{L}^{e}",                          # All electrical lines
            "nLH": r"\mathcal{L}^{h}",                          # All hydrogen lines
            "nX": r"\mathcal{R}",                               # All regions
            "nZ": r"\mathcal{Z}",                               # All zones
            "nA": r"\mathcal{A}",                               # All areas
            "nT": r"\mathcal{T}",                               # All time steps
            "nV": r"\mathcal{V}",                               # All time step intervals
            "nP": r"\mathcal{P}",                               # All periods
            "nS": r"\mathcal{S}",                               # All scenarios
            "nM": r"\mathcal{M}",                               # All months in a year
            "nW": r"\mathcal{W}",                               # All weeks in a year
            "nD": r"\mathcal{D}",                               # All days in a week
            "nH": r"\mathcal{H}",                               # All hours in a day
            "nWI": r"\mathcal{H}^{w}",                          # All 20 minute intervals in an hour
            "nQ": r"\mathcal{H}^{q}",                           # All quarters in an hour
            "nTI": r"\mathcal{H}^{t}",                          # All 10 minute intervals in an hour
            "nMIN": r"\mathcal{H}^{m}",                         # All minutes in an hour


            # === INDEXES ===
            "busindex": r"nd",                                  # Index for buses
            "busindexa": r"i",                                  # Index for "from" bus of a branch
            "busindexb": r"j",                                  # Index for "to" bus of a branch
            "branchindex": r"ij",                               # Index for branch between bus i and j
            "lineindexa": r"ijc",                               # Index for "from" bus of a line
            "lineindexb": r"jic",                               # Index for "to" bus of a line
            "circuitindex": r"c",                               # Index for circuit of a line
            "elegenindex": r"g^{e}",                            # Index for electrical generator
            "elenonresgenindex": r"g^{e,nonres}",               # Index for non-renewable electrical generator
            "elenresgenindex": r"g^{e,res}",                    # Index for renewable electrical
            "elenhydgenindex": r"g^{e,h}",                      # Index for electrical generator consuming hydrogen
            "hydgenindex": r"g^{h}",                            # Index for hydrogen generator
            "hydcnggenindex": r"g^{h,cng}",                     # Index for hydrogen generator consuming natural gas
            "hydelecgenindex": r"g^{h,e}",                      # Index for hydrogen generator consuming electricity
            "genindex": r"g",                                   # Index for generator
            "eledemandindex": r"d^{e}",                         # Index for electrical load
            "hyddemandindex": r"d^{h}",                         # Index for hydrogen load
            "demandindex": r"d",                                # Index for load
            "eleconsindex": r"c^{e}",                           # Index for electrical consumer
            "hydconsindex": r"c^{h}",                           # Index for hydrogen consumer
            "consindex": r"c",                                  # Index for consumer
            "elepeakindex": r"k^{e}",                           # Index for electrical peak load
            "hydpeakindex": r"k^{h}",                           # Index for hydrogen peak load
            "peakindex": r"k",                                  # Index for peak load
            "elestorageindex": r"e^{e}",                        # Index for electrical storage
            "hydstorageindex": r"e^{h}",                        # Index for hydrogen storage
            "storageindex": r"e",                               # Index for storage
            "traderindex": r"r",                                # Index for retailer
            "eletraderindex": r"r^{e}",                         # Index for electrical retailer
            "hydtraderindex": r"r^{h}",                         # Index for hydrogen retailer
            "elevehicleindex": r"v^{e}",                        # Index for electrical vehicle
            "hydvehicleindex": r"v^{h}",                        # Index for hydrogen vehicle
            "periodindex": r"p",                                # Index for period
            "scenarioindex": r"s",                              # Index for scenario
            "weekindex": r"w",                                  # Index for week
            "monthindex": r"m",                                 # Index for month
            "dayindex": r"d",                                   # Index for day
            "hourindex": r"h",                                  # Index for hour
            "twentyminuteindex": r"w",                          # Index for 20 minute interval
            "quarterindex": r"q",                               # Index for quarter
            "tenminuteindex": r"t",                             # Index for 10 minute interval
            "minuteindex": r"m",                                # Index for minute
            "intervalindex": r"u",                              # Index for time interval
            "timeindex": r"n",                                  # Index for time  step
            "storageperiodindex": r"p'",                        # Index for storage period
            "storagetimeindex": r"n'",                          # Index for storage time step
            "storageweekindex": r"w'",                          # Index for storage week
            "storagemonthindex": r"m'",                         # Index for storage month
            "storagedayindex": r"d'",                           # Index for storage day
            "storagehourindex": r"h'",                          # Index for storage hour
            "zoneindex": r"z",                                  # Index for zone
            "areaindex": r"a",                                  # Index for area
            "regionindex": r"r",                                # Index for region

            # === FUNCTIONS ===
            # --- Market Costs ---
            "marketcost": r"C",                                 # Total market trading cost
            "elemarketcost": r"C^{trade,e}",                    # Electrical market trading cost (DA, ID, IMB)
            "elemarketcostDA": r"C^{DA,e}",                     # Electrical market trading cost DA
            "elemarketcostID": r"C^{ID,e}",                     # Electrical market trading cost ID
            "elemarketcostIMB": r"C^{IMB,e}",                   # Electrical market trading cost IMB
            "elemarketcostPPA": r"C^{PPA,e}",                   # Electrical market PPA cost
            "elemarketcostgrid": r"C^{grid,e}",                 # Electrical market grid fee cost
            "elemarketcosttax": r"C^{tax,e}",                   # Electrical market grid tax cost
            "elemarketcostVAT": r"C^{VAT,e}",                   # Electrical market VAT cost
            "hydmarketcost": r"C^{trade,h}",                    # Hydrogen market trading cost
            "hydmarketcostPPA": r"C^{PPA,h}",                   # Hydrogen market trading PPA
            "importcost": r"C^{imp}",                           # Total Import cost

            # --- Market Revenues ---
            "marketrevenue": r"R",                              # Total market trading revenue
            "elemarketrevenue": r"R^{trade,e}",                 # Electrical market trading revenue
            "elemarketrevenueDA": r"R^{DA,e}",                  # Electrical market selling profit DA
            "elemarketrevenueancillary": r"R^{f,e}",            # Electrical market selling profit ancillary services
            "elemarketrevenuePPA": r"R^{PPA,e}",                # Electrical market selling profit PPA
            "elemarketrevenuetax": r"R^{tax,e}",                # Electrical market selling profit tax
            "elemarketrevenueincentive": r"R^{inc,e}",          # Electrical market selling profit incentive
            "hydmarketrevenue": r"R^{trade,h}",                 # Hydrogen market trading cost
            "hydmarketrevenuePPA": r"R^{PPA,h}",                # Hydrogen market selling PPA
            "exportrevenue": r"R^{exp}",                        # Total Export cost

            # --- Generation Costs ---
            "generationcost": r"C^{gen}",                       # Total generation cost
            "elegenerationcost": r"C^{gen,e}",                  # Electrical generation cost
            "hydgenerationcost": r"C^{gen,h}",                  # Hydrogen generation cost
            "startcost": r"C^{start}",                          # Start-up cost
            "shutdowncost": r"C^{shut}",                        # Shut-down cost
            "noloadcost": r"C^{nl}",                            # No-load cost
            "fuelcost": r"C^{fuel}",                            # Fuel cost
            "watercost": r"C^{water}",                          # Water cost
            "maintenancecost": r"C^{maint}",                    # Maintenance cost
            "operationcost": r"C^{op}",                         # Operation cost
            "maintopercost": r"C^{0\& M}",                      # O&M cost
            "elemaintopercost": r"C^{O\& M,e}",                 # Electrical O&M cost
            "hydmaintopercost": r"C^{O\& M,h}",                 # Hydrogen O&M cost
            "degradationcost": r"C^{deg}",                      # Degradation cost
            "eledegradationcost": r"C^{deg,e}",                 # Electrical degradation cost
            "hyddegradationcost": r"C^{deg,h}",                 # Hydrogen degradation cost

            # --- Storage Costs ---
            "storagecost": r"C^{str}",                          # Storage cost

            # --- Demand & Consumption Costs ---
            "consumptioncost": r"C^{con}",                      # Total consumption cost
            "eleconsumptioncost": r"C^{con,e}",                 # Electrical consumption cost
            "hydconsumptioncost": r"C^{con,h}",                 # Hydrogen consumption cost
            "demandcost": r"C^{dem}",                           # Demand (capacity/volumetric) cost
            "eledemandcost": r"C^{dem,e}",                      # Electrical demand cost
            "hyddemandcost": r"C^{dem,h}",                      # Hydrogen demand cost
            "peakdemandcost": r"C^{peak}",                      # Peak demand cost
            "elepeakdemandcost": r"C^{peak,e}",                 # Electrical peak demand cost
            "hydpeakdemandcost": r"C^{peak,h}",                 # Hydrogen peak demand cost
            "netusecost": r"C^{netuse}",                        # Network use cost
            "elenetusecost": r"C^{netuse,e}",                   # Electrical network use cost
            "hydnetusecost": r"C^{netuse,h}",                   # Hydrogen network use cost
            "captariffcost": r"C^{cap}",                        # Capacity tariff cost
            "elecaptariffcost": r"C^{cap,e}",                   # Electrical capacity tariff cost
            "hydcaptariffcost": r"C^{cap,h}",                   # Hydrogen capacity tariff cost

            # --- Transmission & Infrastructure Costs ---
            "transmissioncost": r"C^{tr}",                      # Transmission system cost
            "linecost": r"C^{line}",                            # Line cost
            "convertercost": r"C^{conv}",                       # Converter cost
            "hvdcost": r"C^{hvdc}",                             # HVDC cost

            # --- Flexibility & Ancillary Services Costs ---
            "flexibilitycost": r"C^{flex}",                     # Flexibility provision cost
            "ancillaryservicescost": r"C^{as}",                 # Ancillary services total cost
            "spinningreservecost": r"C^{sp}",                   # Spinning reserve cost
            "nonspinningreservecost": r"C^{ns}",                # Non-spinning reserve cost
            "regulationupcost": r"C^{reg,u}",                   # Regulation-up cost
            "regulationdowncost": r"C^{reg,d}",                 # Regulation-down cost
            "reserveprocurementcost": r"C^{res,proc}",          # Reserve procurement cost
            "reserveactivationcost": r"C^{res,act}",            # Reserve activation cost

            # --- Flexibility & Ancillary Services Revenues ---
            "flexibilityrevenue": r"R^{flex}",                  # Flexibility provision revenue
            "ancillaryservicesrevenue": r"R^{as}",              # Ancillary services total revenue
            "spinningreserverevenue": r"R^{sp}",                # Spinning reserve revenue
            "nonspinningreserverevenue": r"R^{ns}",             # Non-spinning reserve revenue
            "regulationuprevenue": r"R^{reg,u}",                # Regulation-up revenue
            "regulationdownrevenue": r"R^{reg,d}",              # Regulation-down revenue
            "reserveprocurementrevenue": r"R^{res,proc}",       # Reserve procurement revenue
            "reserveactivationrevenue": r"R^{res,act}",         # Reserve activation revenue
            "freqcontnormalrevenue": r"R^{FCR-N}",              # Frequency containment reserve normal revenue
            "freqcontdisturbrevenue": r"R^{FCR-D}",             # Frequency containment reserve disturbance revenue

            # --- Curtailment & Reliability Costs ---
            "unservedenergycost": r"C^{ue}",                    # Unserved energy cost
            "eleunservedenergycost": r"C^{ens,e}",              # Electrical unserved energy cost
            "hydunservedenergycost": r"C^{ens,h}",              # Hydrogen unserved
            "demandresponsecost": r"C^{dr}",                    # Demand response cost
            "demandflexibilitycost": r"C^{df}",                 # Demand flexibility cost
            "spillcost": r"C^{spill}",                          # Spill cost
            "loadsheddingcost": r"C^{ls}",                      # Load-shedding cost
            "curtailmentcost": r"C^{curt}",                     # Curtailment cost

            # --- Environmental / Policy Costs ---
            "emissionpermitcost": r"C^{emiss,perm}",            # Emission permit (ETS) cost
            "emissiontaxcost": r"C^{emiss,tax}",                # Emission tax cost
            "carboncost": r"C^{carb}",                          # Carbon cost (if modeled separately)
            "eleemissioncost": r"C^{emiss,e}",                  # Electrical emission cost

            # --- Investment & Financial Costs ---
            "investmentcost": r"C^{inv}",                       # Investment cost
            "fixedcost": r"C^{fix}",                            # Fixed cost
            "variablecost": r"C^{var}",                         # Variable cost
            "annualcost": r"C^{ann}",                           # Annualized cost
            "totalcost": r"C^{tot}",                            # Total system cost

            # --- Total and Category Emissions ---
            "genemission": r"EM^{gen}",                         # Emissions from generation
            "conemission": r"EM^{con}",                         # Emissions from consumption
            "totemission": r"EM^{tot}",                         # Total emissions

            # --- Specific Pollutants ---
            "genCO2": r"EM^{gen,co2}",                          # CO2 from generation
            "conCO2": r"EM^{con,co2}",                          # CO2 from consumption
            "totCO2": r"EM^{tot,co2}",                          # Total CO2

            "genNOx": r"EM^{gen,nox}",                          # NOx from generation
            "conNOx": r"EM^{con,nox}",                          # NOx from consumption
            "totNOx": r"EM^{tot,nox}",                          # Total NOx

            "genSO2": r"EM^{gen,so2}",                          # SO2 from generation
            "conSO2": r"EM^{con,so2}",                          # SO2 from consumption
            "totSO2": r"EM^{tot,so2}",                          # Total SO2

            "genPM25": r"EM^{gen,pm25}",                        # PM2.5 from generation
            "conPM25": r"EM^{con,pm25}",                        # PM2.5 from consumption
            "totPM25": r"EM^{tot,pm25}",                        # Total PM2.5

            # --- Operation-related Emissions (useful decomposition) ---
            "startupemission": r"EM^{start}",                   # Start-up emissions
            "shutdownemission": r"EM^{shut}",                   # Shut-down emissions
            "noloademission": r"EM^{nl}",                       # No-load emissions
            "storageemission": r"EM^{str}",                     # Storage-induced emissions (via losses)
            "importemission": r"EM^{imp}",                      # Virtual emissions of imports
            "exportemission": r"EM^{exp}",                      # Virtual emissions of exports
            "LROcost": r"C^{LRO}",

            # === PARAMETERS ===
            "pfactorone": r"F1",                                # Generic factor 1
            "pfactortwo": r"F2",                                # Generic factor 2
            "pfactorthr": r"F3",                                # Generic factor 3
            "pfactorfou": r"F4",                                # Generic factor 4

            "pdiscountrate": r"\Gamma",                         # Discount rate
            "ptimestep": r"\nu",                                # Time step
            "ptimestepduration": r"\delta",                     # Time step duration (hours)
            "pintervalduration": r"\Delta u",                   # Time step interval duration (hours)
            "nintervals": r"N^{u}",                             # Number of time step intervals
            "nperiods": r"N^{p}",                               # Number of periods
            "nweeks": r"N^{w}",                                 # Number of weeks in a year
            "nmonths": r"N^{m}",                                # Number of months in a year
            "ndays": r"N^{d}",                                  # Number of days in a week
            "nhours": r"N^{h}",                                 # Number of hours in a day

            # --- Prices (use uppercase Pi for market prices) ---
            "pelespotprice": r"\lambda^{e}",                    # Electrical spot price
            "phydspotprice": r"\lambda^{h}",                    # Hydrogen spot price
            "pelebuyprice": r"\lambda^{b,e}",                   # Electrical buy price
            "phydbuyprice": r"\lambda^{b,h}",                   # Hydrogen buy price
            "pelesellprice": r"\lambda^{s,e}",                  # Electrical sell price
            "phydsellprice": r"\lambda^{s,h}",                  # Hydrogen sell price
            "peleimportprice": r"\lambda^{imp,e}",              # Electrical import price
            "phydimportprice": r"\lambda^{imp,h}",              # Hydrogen import price
            "peleexportprice": r"\lambda^{exp,e}",              # Electrical export price
            "phydexportprice": r"\lambda^{exp,h}",              # Hydrogen export price
            "peleppaprice": r"\lambda^{PPA,e}",                 # Electrical PPA price
            "phydppaprice": r"\lambda^{PPA,h}",                 # Hydrogen PPA price
            "pelefcrnprice": r"\Pi^{N,e}",                      # Electrical grid FCR-N price
            "pelefcrdupprice": r"\Pi^{DUP,e}",                  # Electrical grid FCR-D upwards price
            "pelefcrddwprice": r"\Pi^{DDW,e}",                  # Electrical grid FCR-D downwards price
            "pelegridaFRRUPprice": r"\Pi^{A-REG-UP,e}",         # Electrical grid aFRR upwards price
            "pelegridaFRDDWprice": r"\Pi^{A-REG-DW,e}",         # Electrical grid aFRR downwards price
            "pelegridmFRRUPprice": r"\Pi^{M-REG-UP,e}",         # Electrical grid mFRR upwards price
            "pelegridmFRDDWprice": r"\Pi^{M-REG-DW,e}",         # Electrical grid mFRR downwards price
            "pelemarketbuyingratio": r"\Upsilon^{b,e}",         # Electrical market buying ratio
            "pelemarketsellingratio": r"\Upsilon^{s,e}",        # Electrical market selling ratio
            "pelemarketcertrevenue": r"M^{cert,e}",             # Electrical certificate fee revenue
            "pelemarketpassthrough": r"M^{pass,e}",             # Electrical pass-through fee
            "pelemarketmoms": r"M^{moms,e}",                    # Electrical tax (moms)
            "pelemarketnetfee": r"M^{net,e}",                   # Electrical network fee
            "pelemarkettariff": r"M^{tariff,e}",                # Electrical tariff
            "pmaxmarketbuy": r"\overline{MB}",                  # Max market buying
            "pelemaxmarketbuy": r"\overline{MB}^{e}",           # Max electrical market buying
            "phydmaxmarketbuy": r"\overline{MB}^{h}",           # Max hydrogen market buying
            "pmaxmarketsell": r"\overline{MS}",                 # Max market selling
            "pelemaxmarketsell": r"\overline{MS}^{e}",          # Max electrical market selling
            "phydmaxmarketsell": r"\overline{MS}^{h}",          # Max hydrogen market selling
            "pminmarketbuy": r"\underline{MB}",                 # Min market buying
            "peleminmarketbuy": r"\underline{MB}^{e}",          # Min electrical market buying
            "phydminmarketbuy": r"\underline{MB}^{h}",          # Min hydrogen market buying
            "pminmarketsell": r"\underline{MS}",                # Min market selling
            "peleminmarketsell": r"\underline{MS}^{e}",         # Min electrical market selling
            "phydminmarketsell": r"\underline{MS}^{h}",         # Min hydrogen market selling

            # --- System Balancing ---
            "pbalanceerror": r"E^{bal}",                        # System balance error
            "pelebalanceerror": r"E^{bal,e}",                   # Electrical system balance error
            "phydbalanceerror": r"E^{bal,h}",                   # Hydrogen system balance balance error
            "pbalanceerrorlimit": r"\overline{E}^{bal}",        # System balance error limit
            "pelebalanceerrorlimit": r"\overline{E}^{bal,e}",   # Electrical system balance error limit
            "phydbalanceerrorlimit": r"\overline{E}^{bal,h}",   # Hydrogen system balance error limit
            # --- Point of Common Coupling (PCC) ---
            "peleppccmaximport": r"\overline{P}^{PCC,imp,e}",   # Max active power import at PCC
            "peleppccmaxexport": r"\overline{P}^{PCC,exp,e}",   # Max active power export at PCC
            "peleqpccmaximport": r"\overline{Q}^{PCC,imp,e}",   # Max reactive power import at PCC
            "peleqpccmaxexport": r"\overline{Q}^{PCC,exp,e}",   # Max reactive power export at PCC
            
            # --- Demand ---
            "peledemand": r"D^{e}",                             # Electrical demand
            "phyddemand": r"D^{h}",                             # Hydrogen demand
            "pelemaxdemand": r"\overline{D}^{e}",               # Electrical max demand
            "pelemindemand": r"\underline{D}^{e}",              # Electrical min demand
            "phydmaxdemand": r"\overline{D}^{h}",               # Hydrogen max demand
            "phydmindemand": r"\underline{D}^{h}",              # Hydrogen min demand
            "pelepeakdemand": r"D^{e,peak}",                    # Electrical peak demand
            "phydpeakdemand": r"D^{h,peak}",                    # Hydrogen peak demand
            "pdemflexible": r"D^{flex}",                        # Generic flexible demand
            "peledemflexible": r"D^{e,flex}",                   # Electrical flexible demand
            "phyddemflexible": r"D^{h,flex}",                   # Hydrogen flexible demand
            "pdemshiftedsteps": r"N^{shift}",                   # Generic demand shifted time steps
            "peledemshiftedsteps": r"N^{e,shift}",              # Electrical demand shifted time steps
            "phyddemshiftedsteps": r"N^{h,shift}",              # Hydrogen demand shifted time steps

            # --- Generation and Consumption Limits ---
            "pcontractedcapacity": r"\hat{E}",                  # Contracted capacity (generic)
            "pelecontractedcapacity": r"\hat{E}^{e}",           # Contracted electrical capacity
            "phydcontractedcapacity": r"\hat{E}^{h}",           # Contracted hydrogen capacity
            "pelemaxproduction": r"\overline{EP}",              # Max electrical generation
            "peleminproduction": r"\underline{EP}",             # Min electrical generation
            "phydmaxproduction": r"\overline{HP}",              # Max hydrogen generation
            "phydminproduction": r"\underline{HP}",             # Min hydrogen generation
            "pelemaxconsumption": r"\overline{EC}",             # Max electrical consumption
            "peleminconsumption": r"\underline{EC}",            # Min electrical consumption
            "phydmaxconsumption": r"\overline{HC}",             # Max hydrogen consumption
            "phydminconsumption": r"\underline{HC}",            # Min hydrogen consumption
            "peleconscompress": r"\overline{EC}^{compress}",    # Max electrical consumption (compression)

            # --- Technical Parameters ---
            "prampuprate": r"RU",                               # Ramp-up rate
            "prampdwrate": r"RD",                               # Ramp-down rate
            "puptime": r"TU",                                   # Minimum up time
            "pdwtime": r"TD",                                   # Minimum down time
            "peletohydfunction": r"F^{e2h}",                    # Power-to-hydrogen conversion
            "phydtoelefunction": r"F^{h2e}",                    # Hydrogen-to-power conversion

            # --- Cost Parameters ---
            "pstartupcost": r"CSU",                             # Start-up cost
            "pshutdowncost": r"CSD",                            # Shut-down cost
            "ppeakdemandtariff": r"C^{peak}",                   # Peak demand cost
            "pnoloadcost": r"C^{nl}",                           # No-load cost
            "ploadsheddingcost": r"C^{ls}",                     # Load-shedding cost
            "pspillcost": r"C^{spill}",                         # Spill cost
            "pfuelafactor": r"A",                               # Linear fuel cost coeff.
            "pfuelbfactor": r"B",                               # Quadratic fuel cost coeff.
            "pfuelcfactor": r"C",                               # Fixed fuel cost coeff.
            "pmaintenancecost": r"CM",                          # Maintenance cost
            "poperationcost": r"CO",                            # Operation cost
            "pfixedcost": r"CF",                                # Fixed cost
            "pvariablecost": r"CV",                             # Variable cost
            "pinvestmentcost": r"C^{inv}",                      # Investment cost
            "pannualcost": r"C^{ann}",                          # Annualized cost
            "pcarbonprice": r"CE",                              # Carbon price or tax

            # --- Electrical Network Parameters ---
            "peleresistantline": r"R",                          # Resistance
            "pelereactanceline": r"X",                          # Reactance
            "peleimpedanceline": r"Z",                          # Impedance magnitude
            "peleconductline": r"G",                            # Conductance
            "pelesusceptline": r"B",                            # Susceptance
            "peleshuntsuscepline": r"B^{l}",                    # Line shunt susceptance
            "pelemaxcurrent": r"\overline{I}",                  # Max current
            "pelemaxapparentpower": r"\overline{S}",            # Max apparent power
            "pelemaxrealpower": r"\overline{P}",                # Max real power
            "peleminrealpower": r"\underline{P}",               # Min real power
            "pelemaxreactivepower": r"\overline{Q}",            # Max reactive power
            "peleminreactivepower": r"\underline{Q}",           # Min reactive power
            "pelemaxvolt": r"\overline{V}",                     # Max voltage magnitude
            "peleminvolt": r"\underline{V}",                    # Min voltage magnitude
            "pelemaxvoltageangle": r"\overline{\theta}",        # Max voltage angle
            "peleminvoltageangle": r"\underline{\theta}",       # Min voltage angle
            "peleinitvolt": r"V^{0}",                           # Initial voltage
            "phydmaxflow": r"\overline{H}",                     # Max hydrogen flow
            "phydminflow": r"\underline{H}",                    # Min hydrogen flow

            # --- Storage Parameters ---
            "pelestoragecycle": r"\tau^{e}",                    # Electrical Energy storage duration
            "pelestorageinflowcycle": r"\tau^{i,e}",            # Electrical Inflow cycle duration
            "pelestorageoutflowcycle": r"\tau^{o,e}",           # Electrical Outflow cycle duration
            "pelestorageefficiency": r"\eta^{e}",               # Electrical Storage efficiency
            "pelestorchargeefficiency": r"\eta^{c,e}",          # Electrical Storage charging efficiency
            "pelestordischargeefficiency": r"\eta^{d,e}",       # Electrical Storage discharging efficiency
            "pelestoragecapacity": r"\Psi^{e}",                 # Electrical Storage capacity
            "pvarfixedavailability": r"A",                      # Variable/fixed availability ratio
            "pelemaxinflow": r"\overline{EI}^{e}",              # Max electrical inflow
            "pelemaxoutflow": r"\overline{EO}^{e}",             # Max electrical outflow
            "pelemininflow": r"\underline{EI}^{e}",             # Min electrical inflow
            "peleminoutflow": r"\underline{EO}^{e}",            # Min electrical outflow
            "pelemaxinventory": r"\overline{SI}^{e}",           # Max electrical inventory
            "pelemininventory": r"\underline{SI}^{e}",          # Min electrical inventory
            "phydstoragecycle": r"\tau^{h}",                    # Hydrogen Energy storage duration
            "phydstorageinflowcycle": r"\tau^{i,h}",            # Hydrogen Inflow cycle duration
            "phydstorageoutflowcycle": r"\tau^{o,h}",           # Hydrogen Outflow cycle duration
            "phydstorageefficiency": r"\eta^{h}",               # Hydrogen Storage efficiency
            "phydstoragecapacity": r"\Psi^{h}",                 # Hydrogen Storage capacity
            "phydmaxinflow": r"\overline{EI}^{h}",              # Max hydrogen inflow
            "phydmaxoutflow": r"\overline{EO}^{h}",             # Max hydrogen outflow
            "phydmininflow": r"\underline{EI}^{h}",             # Min hydrogen inflow
            "phydminoutflow": r"\underline{EO}^{h}",            # Min hydrogen outflow
            "phydmaxinventory": r"\overline{SI}^{h}",           # Max hydrogen inventory
            "phydmininventory": r"\underline{SI}^{h}",          # Min hydrogen inventory
            "peleminstoragestart": r"S^{e,init}",               # Initial electrical storage inventory
            "phydminstoragestart": r"S^{h,init}",               # Initial hydrogen storage inventory
            "peleminstorageend": r"S^{e,end}",                  # End electrical storage inventory
            "phydminstorageend": r"S^{h,end}",                  # End hydrogen storage inventory

            # --- Frequency Containment Parameters ---
            "pfcrnrequirement": r"FCRN",                        # Frequency containment reserve
            "pfcrduprequirement": r"FCRD^{UP}",                 # Frequency containment reserve - upwards
            "pfcrddwrequirement": r"FCRD^{DW}",                 # Frequency containment reserve - downwards
            "pafrruprequirement": r"REG^{A,UP}",                # Automatic frequency restoration reserve - upwards
            "pafrddwrequirement": r"REG^{A,DW}",                # Automatic frequency restoration reserve - downwards
            "pmfrruprequirement": r"REG^{M,UP}",                # Manual frequency restoration reserve - upwards
            "pmfrddwrequirement": r"REG^{M,DW}",                # Manual frequency restoration reserve - downwards
            "pfcrdupreqactivation": r"FCRD^{UP,act}",           # FCR-D upwards reserve activation
            "pfcrddwreqactivation": r"FCRD^{DW,act}",           # FCR-D downwards reserve activation

            # =======================================================
            #                       VARIABLES
            # =======================================================
            # symbols in lowercase to distinguish from parameters
            "velemarketbuy": r"mb^{e}",                         # Electrical market buying
            "vhydmarketbuy": r"mb^{h}",                         # Hydrogen market buying
            "vmarketbuy": r"mb",                                # Market buying
            "velemarketsell": r"ms^{e}",                        # Electrical market selling
            "vhydmarketsell": r"ms^{h}",                        # Hydrogen market selling
            "vmarketsell": r"ms",                               # Market selling
            "veleppccimport": r"imp^{e}",                       # Electrical import at PCC
            "veleppccexport": r"exp^{e}",                       # Electrical export at PCC
            "vhydimport": r"imp^{h}",                           # Hydrogen import
            "vhydexport": r"exp^{h}",                           # Hydrogen export

            "vproduction": r"p",                                # Generation
            "veleproduction": r"p^{e}",                         # Electrical generation
            "vhydproduction": r"p^{h}",                         # Hydrogen production
            "veleproductionact": r"p^{a,e}",                    # Electrical activated generation
            "vhydproductionact": r"p^{a,h}",                    # Hydrogen activated generation
            "vconsumption": r"c",                               # Consumption
            "veleconsumption": r"c^{e}",                        # Electrical consumption
            "veleconsumptionstandby": r"c^{e,standby}",         # Electrical consumption (standby)
            "veleconsumptioncompress": r"c^{e,compress}",       # Electrical consumption (compression)
            "vhydconsumption": r"c^{h}",                        # Hydrogen consumption
            "veleconsumptionact": r"c^{a,e}",                   # Electrical activated consumption
            "vhydconsumptionact": r"c^{a,h}",                   # Hydrogen activated consumption
            "vinventorystart": r"s^{init}",                     # Storage initial inventory
            "vinventory": r"si",                                # Storage inventory
            "veleinventory": r"si^{e}",                         # Electrical storage inventory
            "vhydinventory": r"si^{h}",                         # Hydrogen storage inventory
            "venergyinflow": r"ei",                             # Storage inflow
            "veleenergyinflow": r"ei^{e}",                      # Electrical storage inflow
            "vhydenergyinflow": r"ei^{h}",                      # Hydrogen storage inflow
            "venergyoutflow": r"eo",                            # Storage outflow
            "veleenergyoutflow": r"eo^{e}",                     # Electrical storage outflow
            "vhydenergyoutflow": r"eo^{h}",                     # Hydrogen storage outflow
            "vspillage": r"es",                                 # Storage spillage
            "velespillage": r"es^{e}",                          # Electrical storage spillage
            "vhydspillage": r"es^{h}",                          # Hydrogen storage spillage
            "vsecondblockproduction": r"p2b",                   # 2nd block production
            "velesecondblockproduction": r"p2b^{e}",            # Electrical 2nd block production
            "vhydsecondblockproduction": r"p2b^{h}",            # Hydrogen 2nd block production
            "vsecondblockconsumption": r"c2b",                  # 2nd block consumption
            "velesecondblockconsumption": r"c2b^{e}",           # Electrical 2nd block consumption
            "vhydsecondblockconsumption": r"c2b^{h}",           # Hydrogen 2nd block consumption
            "vmagvolt": r"v^{2}",                               # Squared voltage magnitude
            "vangvolt": r"\theta",                              # Voltage angle
            "vflow": r"f",                                      # Branch flow
            "vactflow": r"f^{P}",                               # Active power flow
            "vreactflow": r"f^{Q}",                             # Reactive power flow
            "veleflow": r"f^{e}",                               # Electrical flow
            "vhydflow": r"f^{h}",                               # Hydrogen flow
            "vdemand": r"d",                                    # Load
            "veledemand": r"d^{e}",                             # Electrical load
            "vhyddemand": r"d^{h}",                             # Hydrogen load
            "vdemflex": r"d^{flex}",                            # Flexible load
            "veledemflex": r"d^{e,flex}",                       # Electrical flexible load
            "vhyddemflex": r"d^{h,flex}",                       # Hydrogen flexible
            "vloadshed": r"ls",                                 # Load
            "veleloadshed": r"ls^{e}",                          # Electrical load-shedding
            "vhydloadshed": r"ls^{h}",                          # Hydrogen load-shedding
            "vpeakdemand": r"d^{peak}",                         # Peak demand
            "velepeakdemand": r"d^{e,peak}",                    # Electrical peak demand
            "vhydpeakdemand": r"d^{h,peak}",                    # Hydrogen peak demand

            # --- Reserves ---
            "vPupward": r"r^{p,u}",                             # Upward power reserve
            "vPdownward": r"r^{p,d}",                           # Downward power reserve
            "vCupward": r"r^{c,u}",                             # Upward consumption reserve
            "vCdownward": r"r^{c,d}",                           # Downward consumption reserve
            # power bids to FCR-N
            "velefcrnbid": r"p^{\Thetha,N}",                    # Electrical FCR-N bid
            # power bids to FCR-D UP
            "velefcrdupbid": r"p^{\Theta,DU}",                  # Electrical FCR-D UP bid
            # power bids to FCR-D DW
            "velefcrddwbid": r"p^{\Theta,DW}",                  # Electrical FCR-D DW bid
            # activated power from FCR-N UP
            "velefcrnupact": r"p^{NU}",                         # Electrical FCR-N activation
            # activated power from FCR-N DW
            "velefcrndwact": r"p^{ND}",                         # Electrical FCR-N DW activation
            # activated power from FCR-D UP
            "velefcrdupactch": r"p^{DU,ch}",                    # Electrical FCR-D UP activation when charging
            "velefcrdupactdi": r"p^{DU,di}",                    # Electrical FCR-D UP activation when discharging
            # activated power from FCR-D DW
            "velefcrddwactch": r"p^{DD,ch}",                    # Electrical FCR-D DW activation when charging
            "velefcrddwactdi": r"p^{DD,di}",                    # Electrical FCR-D DW activation when discharging
            # activation fractions for FCR-D
            "velefcrdupfraction": r"\Lambda^{DU}",              # Electrical FCR-D UP activation fraction
            "velefcrddwfraction": r"\Lambda^{DD}",              # Electrical FCR-D DW activation fraction


            # --- Binary Variables ---
            "vpeakdemandindbin": r"kd",                         # Peak demand indicator binary
            "velepeakdemandindbin": r"kd^{e}",                  # Electrical peak demand indicator binary
            "vhydpeakdemandindbin": r"kd^{h}",                  # Hydrogen peak demand indicator binary
            "vstoroperatbin": r"sf",                            # Storage operating binary
            "velestoroperatbin": r"sf^{e}",                     # Electrical storage operating binary
            "vhydstoroperatbin": r"sf^{h}",                     # Hydrogen storage operating binary
            "velestorchargebin": r"sc^{e}",                     # Electrical storage charging binary
            "velestorchargeactbin": r"sc^{a,e}",                # Electrical storage charging definite binary
            "vhydstorchargebin": r"sc^{h}",                     # Hydrogen storage charging binary
            "vhydstorchargeactbin": r"sc^{a,h}",                # Hydrogen storage charging definite binary
            "velestordischargebin": r"sd^{e}",                  # Electrical storage discharging binary
            "velestordischargeactbin": r"sd^{a,e}",             # Electrical storage discharging definite binary
            "vhydstordischargebin": r"sd^{h}",                  # Hydrogen storage discharging binary
            "vhydstordischargeactbin": r"sd^{a,h}",             # Hydrogen storage discharging definite binary
            "vmaxcommitbin": r"muc",                            # Maximum unit commitment binary
            "velemaxcommitbin": r"muc^{e}",                     # Maximum electrical unit commitment binary
            "vhydmaxcommitbin": r"muc^{h}",                     # Maximum hydrogen unit commitment binary
            "vcommitbin": r"uc",                                # Unit commitment binary
            "velecommitbin": r"uc^{e}",                         # Electrical unit commitment binary
            "vhydcommitbin": r"uc^{h}",                         # Hydrogen unit commitment binary
            "vstartupbin": r"su",                               # Start-up binary
            "velestartupbin": r"su^{e}",                        # Electrical start-up binary
            "vhydstartupbin": r"su^{h}",                        # Hydrogen start-up binary
            "vshutdownbin": r"sd",                              # Shut-down binary
            "veleshutdownbin": r"sd^{e}",                       # Electrical shut-down binary
            "vhydshutdownbin": r"sd^{h}",                       # Hydrogen shut-down binary
            "vhydcompressbin": r"hc",                           # Hydrogen compression binary
            "velefcrncommitbin": r"uc^{FCR-N,e}",               # Electrical FCR-N commitment binary
            "velefcrdupreqcommitbin": r"uc^{FCR-D,UP,e}",       # Electrical FCR-D UP commitment binary
            "velefcrddwreqcommitbin": r"uc^{FCR-D,DW,e}",       # Electrical FCR-D DW commitment binary

            # =======================================================
            #               Robustness / Uncertainty Terms
            # =======================================================
            "robustbudget": r"\Gamma",                          # Robustness budget
            "devep": r"EP^{dev}",                               # Generation deviation
            "devdem": r"D^{dev}",                               # Demand deviation
            "BigM": r"M",                                       # Big-M constant
            "robustdev": r"\omega",                             # Deviation variable
            "lightslack": r"m",                                 # Slack variable
            "dualnumber": r"\upsilon",                          # Dual variable index
            "dualperiod": r"\eta",                              # Dual variable period
            "robustindicator": r"\chi",                         # Robustness indicator
        }
    }
}

# --- Reuse MathJax macros for LaTeX/PDF (and imgmath) ---
_MJ_MACROS = mathjax3_config["tex"]["macros"]

def _latex_preamble_from_macros(macros: dict) -> str:
    lines = [r"\usepackage{amsmath,amssymb}"]  # safe, helpful
    for name, body in macros.items():
        if re.fullmatch(r"[A-Za-z]+", name):
            # normal LaTeX control word: \name
            lines.append(r"\providecommand{\%s}{%s}" % (name, body))
        else:
            # names containing digits/other chars → define with \csname ...\endcsname
            # (won't crash the build even if not directly invokable as \name)
            lines.append(
                r"\expandafter\ifx\csname %s\endcsname\relax"
                r"\expandafter\def\csname %s\endcsname{%s}"
                r"\else\relax\fi" % (name, name, body)
            )
    return "\n".join(lines)

latex_engine = "lualatex"
latex_elements = {"preamble": _latex_preamble_from_macros(_MJ_MACROS)}
imgmath_latex_preamble = latex_elements["preamble"]

latex_use_xindy = False
