import * as iam from "aws-cdk-lib/aws-iam";
import * as secrets_manager from "aws-cdk-lib/aws-secretsmanager";
import * as ssm from "aws-cdk-lib/aws-ssm";
/**
 * @internal
 */
export declare enum AuthenticationStrategy {
    /**
     * GitHub App or installation authentication
     *
     * @see https://github.com/octokit/auth-app.js/#readme
     */
    AUTH_APP = "auth-app",
    /**
     * Personal Access Token authentication
     *
     * @see https://github.com/octokit/auth-token.js#readme
     */
    AUTH_TOKEN = "auth-token",
    /**
     * unauthenticated
     *
     * @see https://github.com/octokit/auth-unauthenticated.js#readme
     */
    UNAUTHENTICATED = "unauthenticated"
}
/**
 * The auth configuration passed to the onEventHandler
 *
 * @internal
 */
export interface Auth {
    /**
     * The authentication strategy to use
     *
     * @see https://github.com/octokit/authentication-strategies.js/#authentication-strategiesjs
     */
    readonly strategy: AuthenticationStrategy;
    /**
     * Either a SecretsManager Secret Arn or an SSM ParameterStore Parameter Arn
     */
    readonly secret?: string;
}
export interface IAuthOptions {
    /**
     * @internal
     */
    readonly _auth: Auth;
    /**
     * @internal
     */
    _grantRead(grantee: iam.IGrantable): void;
}
export declare abstract class AuthOptions implements IAuthOptions {
    /**
     * GitHub App or installation authentication
     *
     * @see https://github.com/octokit/auth-app.js/#readme
     */
    static appAuth(secret: secrets_manager.ISecret): AuthOptions;
    /**
     * Personal Access Token authentication
     *
     * @see https://github.com/octokit/auth-token.js#readme
     */
    static tokenAuth(parameter: ssm.IParameter): AuthOptions;
    /**
     * unauthenticated
     *
     * @see https://github.com/octokit/auth-unauthenticated.js#readme
     */
    static unauthenticated(): AuthOptions;
    /**
     * @internal
     */
    abstract readonly _auth: Auth;
    /**
     * @internal
     */
    abstract _grantRead(grantee: iam.IGrantable): void;
}
