"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter = exports.flatten = exports.handler = void 0;
const util_arn_parser_1 = require("@aws-sdk/util-arn-parser");
const auth_app_1 = require("@octokit/auth-app");
const auth_token_1 = require("@octokit/auth-token");
const auth_unauthenticated_1 = require("@octokit/auth-unauthenticated");
const rest_1 = require("@octokit/rest");
const aws_sdk_1 = require("aws-sdk");
const execute_github_api_call_1 = require("./execute-github-api-call");
const get_secret_value_1 = require("./get-secret-value");
const auth_1 = require("../auth");
exports.handler = async (event) => {
    console.log(`Request of type ${event.RequestType} received`);
    console.log("Payload: %j", event);
    const call = parse(event.ResourceProperties[event.RequestType]);
    if (!call) {
        return;
    }
    const octokitOptions = {};
    const { strategy, secret } = event.ResourceProperties.Auth;
    switch (strategy) {
        case auth_1.AuthenticationStrategy.AUTH_APP:
            // https://github.com/octokit/authentication-strategies.js/#github-app-or-installation-authentication
            octokitOptions.authStrategy = auth_app_1.createAppAuth;
            octokitOptions.auth = JSON.parse(await get_secret_value_1.getSecretValue(secret));
            break;
        case auth_1.AuthenticationStrategy.AUTH_TOKEN:
            const parameterArn = util_arn_parser_1.parse(secret);
            const parameterName = parameterArn.resource.replace(/^parameter/, "");
            const ssm = new aws_sdk_1.SSM({ region: parameterArn.region });
            const getParameterResponse = await ssm.getParameter({ Name: parameterName, WithDecryption: true }).promise();
            // https://github.com/octokit/auth-token.js#readme
            octokitOptions.authStrategy = auth_token_1.createTokenAuth(getParameterResponse.Parameter.Value);
            break;
        case auth_1.AuthenticationStrategy.UNAUTHENTICATED:
        // https://github.com/octokit/auth-unauthenticated.js#createunauthenticatedauth-options
        default:
            octokitOptions.authStrategy = auth_unauthenticated_1.createUnauthenticatedAuth;
    }
    const octokit = new rest_1.Octokit(octokitOptions);
    try {
        const response = await execute_github_api_call_1.executeGithubApiCall(octokit, call);
        console.debug("Response: %j", response);
        const flattened = exports.flatten(response.data ?? {});
        const responsePath = call.physicalResourceId?.responsePath;
        const physicalResourceId = responsePath ? flattened[responsePath] : call.physicalResourceId?.id;
        return {
            PhysicalResourceId: physicalResourceId,
            Data: exports.filter(flattened, call.outputPaths),
        };
    }
    catch (e) {
        const err = e;
        if (!call.ignoreErrorCodesMatching || !new RegExp(call.ignoreErrorCodesMatching).test(err.message)) {
            throw e;
        }
    }
    return;
};
/**
 * Parse the GitHub API call passed in ResourceProperties OnCreate, OnUpdate and OnDelete
 *
 * @internal
 */
const parse = (stringified) => {
    return stringified ? JSON.parse(stringified) : undefined;
};
/**
 * Flattens a nested object and materializes the key paths dot separated.
 *
 * @internal
 */
exports.flatten = (object) => {
    const _flatten = (path, child) => {
        if ("object" == typeof child && null != child) {
            return Object.entries(child).reduce((result, [key, value]) => Object.assign(result, _flatten(path.concat([key]), value)), {});
        }
        return { [path.join(".")]: child };
    };
    return _flatten([], object);
};
/**
 * Filters objects keys that starts with key names provided in outputPaths.
 *
 * @internal
 */
exports.filter = (object, outputPaths) => {
    if (!outputPaths) {
        return object;
    }
    const result = {};
    for (const [key, value] of Object.entries(object)) {
        for (const outputPath of outputPaths) {
            if (key.startsWith(outputPath)) {
                result[key] = value;
            }
        }
    }
    return result;
};
//# sourceMappingURL=data:application/json;base64,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