"""Centralized model configuration for pantsonfire"""

# Default models for different tiers
# NOTE: For factoid EXTRACTION, LLM receives text content we already fetched
#       (via ExternalExtractor + Selenium). No web search needed for extraction!
# For VERIFICATION, we use search-capable models (Perplexity)

DEFAULT_FAST_MODEL = "anthropic/claude-3.5-sonnet"  # Most capable for accuracy
DEFAULT_SEARCH_MODEL = "perplexity/sonar-reasoning"  # Web search + reasoning for verification
DEFAULT_REASONING_MODEL = "anthropic/claude-3.5-sonnet:beta"  # Latest Sonnet

# Fallback models if defaults fail
FALLBACK_FAST_MODEL = "anthropic/claude-3-haiku"
FALLBACK_SEARCH_MODEL = "perplexity/sonar"
FALLBACK_REASONING_MODEL = "openai/gpt-4-turbo"

# The primary model used throughout the application
DEFAULT_MODEL = DEFAULT_FAST_MODEL

# Model compatibility settings
MODELS_WITHOUT_JSON_FORMAT = [
    "perplexity/sonar",
    "perplexity/sonar-pro",
    "perplexity/sonar-medium",
    "perplexity/sonar-small"
]


def get_default_model() -> str:
    """Get the default model for general use"""
    return DEFAULT_MODEL


def get_model_for_task(task: str) -> str:
    """
    Get the recommended model for a specific task.
    
    Args:
        task: Task type (extraction, search, reasoning)
    
    Returns:
        Model identifier
    """
    task_models = {
        "extraction": DEFAULT_FAST_MODEL,
        "factoid_extraction": DEFAULT_FAST_MODEL,
        "search": DEFAULT_SEARCH_MODEL,
        "fact_verification": DEFAULT_SEARCH_MODEL,
        "reasoning": DEFAULT_REASONING_MODEL,
        "conflict_resolution": DEFAULT_REASONING_MODEL,
    }
    
    return task_models.get(task, DEFAULT_MODEL)


def supports_json_format(model: str) -> bool:
    """
    Check if a model supports response_format={"type": "json_object"}
    
    Args:
        model: Model identifier
    
    Returns:
        True if model supports JSON format
    """
    return model not in MODELS_WITHOUT_JSON_FORMAT

