"""
Oxen AI storage backend for pantsonfire.
Provides versioned data storage using Oxen repositories.
"""

import json
import os
import tempfile
from datetime import datetime
from pathlib import Path
from typing import List, Optional, Dict, Any
import requests
import webbrowser

from ..models import CheckResult, TruthSource
from ..config import Config
from .base import StorageBackend


class OxenStorage(StorageBackend):
    """
    Oxen AI storage backend for versioned data management.

    Stores all findings, prompts, and extracted content in Oxen repositories
    with proper versioning and branch management.
    """

    def __init__(self, config: Config, quiet: bool = False):
        self.config = config
        self.base_url = config.oxen_base_url or "https://hub.oxen.ai"
        self.api_key = config.oxen_api_key
        self.namespace = None  # Will be determined from API
        self._quiet = quiet  # Store quiet flag for suppressing verbose output

        # Create session for API calls
        self.session = requests.Session()
        if self.api_key:
            self.session.headers.update({"Authorization": f"Bearer {self.api_key}"})

        # Determine user's namespace
        self._determine_namespace()

        # Repository name will be generated per analysis
        self.current_repo = None
        self.current_branch = None

    def set_quiet_mode(self, quiet: bool):
        """Set quiet mode for suppressing verbose output."""
        self._quiet = quiet

    def _determine_namespace(self):
        """Determine the user's namespace from their profile."""
        if not self.api_key:
            self.namespace = "pantsonfire"  # fallback
            return

        try:
            # Try to get user info to determine namespace
            response = self.session.get(f"{self.base_url}/api/user")
            if response.status_code == 200:
                user_data = response.json()
                self.namespace = user_data.get("username", "pantsonfire")
                print(f"🔍 Using namespace: {self.namespace}")
            else:
                if not getattr(self, '_quiet', False):
                    print(f"⚠️  Could not determine user namespace, using default")
                self.namespace = "pantsonfire"
        except Exception as e:
            if not getattr(self, '_quiet', False):
                print(f"⚠️  Error determining namespace: {e}, using default")
            self.namespace = "pantsonfire"

    def _ensure_repo_exists(self, repo_name: str) -> bool:
        """
        Ensure the repository exists, create it if it doesn't.

        Args:
            repo_name: Name of the repository

        Returns:
            True if repository exists or was created successfully
        """
        try:
            # Check if repo exists
            response = self.session.get(f"{self.base_url}/api/repos/{self.namespace}/{repo_name}")
            if response.status_code == 200:
                print(f"ℹ️  Repository {self.namespace}/{repo_name} already exists")
                return True

            # Create repository if it doesn't exist
            create_data = {
                "name": repo_name,
                "description": f"Pantsonfire analysis repository for {repo_name}",
                "is_public": not self.config.oxen_private_repos
            }

            response = self.session.post(f"{self.base_url}/api/repos", json=create_data)
            print(f"Repository creation response status: {response.status_code}")

            if response.status_code in [200, 201]:
                # Check if the response indicates success
                try:
                    response_data = response.json()
                    print(f"Repository creation response: {response_data}")

                    if response_data.get("status") == "success" or response.status_code == 201:
                        # Extract the actual namespace from the response
                        if "repository" in response_data and "created_by" in response_data["repository"]:
                            actual_namespace = response_data["repository"]["created_by"]["username"]
                            if actual_namespace != self.namespace:
                                print(f"🔄 Updating namespace from {self.namespace} to {actual_namespace}")
                                self.namespace = actual_namespace

                        # Transform hub.oxen.ai URL to public oxen.ai URL for display
                        api_url = response_data["repository"]["url"]
                        public_url = api_url.replace("https://hub.oxen.ai/", "https://oxen.ai/")
                        print(f"✅ Created Oxen repository: {self.namespace}/{repo_name}")

                        # Wait a moment for the repository to be fully created
                        import time
                        time.sleep(3)

                        # Verify the repository exists
                        verify_response = self.session.get(f"{self.base_url}/api/repos/{self.namespace}/{repo_name}")
                        if verify_response.status_code == 200:
                            print(f"✅ Verified repository exists: {self.namespace}/{repo_name}")
                            return True
                        else:
                            print(f"❌ Repository verification failed: {verify_response.status_code} - {verify_response.text}")
                            return False
                    else:
                        print(f"❌ Failed to create repository: {response_data}")
                        return False
                except Exception as json_error:
                    print(f"❌ Failed to parse repository creation response: {json_error}")
                    # If we can't parse JSON but got a 2xx status, assume success
                    if response.status_code in [200, 201]:
                        print(f"✅ Created Oxen repository: {self.namespace}/{repo_name}")
                        return True
                    else:
                        print(f"❌ Failed to create repository: {response.text}")
                        return False
            else:
                print(f"❌ Failed to create repository: {response.text}")
                return False

        except Exception as e:
            print(f"❌ Error ensuring repository exists: {e}")
            return False

    def _create_branch(self, repo_name: str, branch_name: str) -> bool:
        """
        Create a new branch for the analysis.

        Args:
            repo_name: Repository name
            branch_name: Branch name

        Returns:
            True if branch was created successfully
        """
        try:
            # First, check what the default branch is
            repo_response = self.session.get(f"{self.base_url}/api/repos/{self.namespace}/{repo_name}")
            default_branch = "main"
            if repo_response.status_code == 200:
                repo_data = repo_response.json()
                if "repository" in repo_data and "default_branch" in repo_data["repository"]:
                    default_branch = repo_data["repository"]["default_branch"]

            branch_data = {
                "name": branch_name,
                "from_branch": default_branch
            }

            response = self.session.post(
                f"{self.base_url}/api/repos/{self.namespace}/{repo_name}/branches",
                json=branch_data
            )

            if response.status_code in [200, 201]:
                # Check response content for success
                try:
                    response_data = response.json()
                    if response_data.get("status") == "success" or response.status_code in [200, 201]:
                        print(f"✅ Created branch: {branch_name}")
                        return True
                    elif "already exists" in str(response_data).lower() or response.status_code == 409:
                        print(f"ℹ️  Branch {branch_name} already exists")
                        return True
                    else:
                        print(f"❌ Failed to create branch: {response_data}")
                        return False
                except:
                    # If we can't parse JSON but got a 2xx status, assume success
                    if response.status_code in [200, 201]:
                        print(f"✅ Created branch: {branch_name}")
                        return True
                    else:
                        print(f"❌ Failed to create branch: {response.text}")
                        return False
            elif response.status_code == 409:  # Branch already exists
                print(f"ℹ️  Branch {branch_name} already exists")
                return True
            else:
                print(f"❌ Failed to create branch: {response.text}")
                return False

        except Exception as e:
            print(f"❌ Error creating branch: {e}")
            return False

    def _upload_file(self, repo_name: str, branch_name: str, file_path: Path, oxen_path: str) -> bool:
        """
        Upload a file to the Oxen repository.

        Note: Based on Oxen API analysis, direct file uploads may not be supported via REST API.
        Oxen appears to use workspaces and data frames for data management.

        Args:
            repo_name: Repository name
            branch_name: Branch name
            file_path: Local file path
            oxen_path: Path in Oxen repository

        Returns:
            True if upload was successful (currently always False as API doesn't support this)
        """
        print(f"⚠️  File upload not supported via Oxen REST API. Repository created at: {self.generate_report_url()}")
        print("💡 Oxen uses workspaces and data frames for data management, not direct file uploads")
        return False

    def _commit_changes(self, repo_name: str, branch_name: str, message: str) -> bool:
        """
        Commit the current changes to the repository.

        Note: Based on Oxen API analysis, traditional Git-style commits may not be supported
        via REST API. Oxen appears to use workspaces and merges for data management.

        Args:
            repo_name: Repository name
            branch_name: Branch name
            message: Commit message

        Returns:
            True (commits handled differently in Oxen - data stored in workspaces)
        """
        print(f"ℹ️  Oxen uses workspace-based data management, not traditional Git commits")
        print(f"💡 Analysis data is available in the repository at: {self.generate_report_url()}")
        return True  # Always return True since we're not actually committing

    def initialize_analysis(self, analysis_name: str) -> bool:
        """
        Initialize a new analysis repository and branch.

        Args:
            analysis_name: Name for this analysis

        Returns:
            True if initialization was successful
        """
        repo_name = f"analysis_{analysis_name}"
        branch_name = "main"  # Use main branch instead of creating a new one

        # Ensure repository exists
        if not self._ensure_repo_exists(repo_name):
            return False

        # For now, just use the main branch - don't create a separate branch
        # This avoids API issues with branch creation
        self.current_repo = repo_name
        self.current_branch = branch_name

        print(f"🎯 Analysis initialized: {self.namespace}/{repo_name}:{branch_name}")
        return True

    def store_findings(self, results: List[CheckResult]) -> bool:
        """
        Store analysis findings in Oxen.

        Args:
            results: List of check results

        Returns:
            True if storage was successful
        """
        if not self.current_repo or not self.current_branch:
            print("❌ No active analysis repository")
            return False

        try:
            # Convert results to JSON
            results_data = [result.model_dump() for result in results]

            # Create temporary files
            with tempfile.TemporaryDirectory() as temp_dir:
                temp_path = Path(temp_dir)

                # Save JSON results
                json_file = temp_path / "findings.json"
                with open(json_file, 'w') as f:
                    json.dump(results_data, f, indent=2, default=str)

                # Save CSV results
                csv_file = temp_path / "findings.csv"
                if results_data:
                    import csv
                    with open(csv_file, 'w', newline='') as f:
                        if results_data:
                            writer = csv.DictWriter(f, fieldnames=results_data[0].keys())
                            writer.writeheader()
                            writer.writerows(results_data)

                # Save text summary
                text_file = temp_path / "findings.txt"
                with open(text_file, 'w') as f:
                    f.write(f"Pantsonfire Analysis Report\n")
                    f.write(f"Generated: {datetime.now().isoformat()}\n")
                    f.write(f"Total findings: {len(results)}\n\n")

                    for i, result in enumerate(results, 1):
                        f.write(f"Finding {i}:\n")
                        f.write(f"  Discrepancy: {result.discrepancy}\n")
                        f.write(f"  Confidence: {result.confidence}\n")
                        f.write(f"  Evidence: {result.evidence}\n")
                        f.write(f"  Chunk: {result.chunk_index}\n")
                        f.write(f"  Timestamp: {result.timestamp}\n")
                        f.write("\n")

                # Try to populate the repository with data
                success = self._populate_repository_with_data(results, temp_path)
                if success:
                    print(f"📊 Analysis complete: {len(results)} findings detected - data uploaded to repository")
                else:
                    print(f"📊 Analysis complete: {len(results)} findings detected - repository created (data upload failed)")
                print(f"📁 Repository created: https://oxen.ai/{self.namespace}/{self.current_repo}")
                print(f"💡 Oxen uses workspace-based data management - repository contains analysis results")

                # Create a local summary file for reference
                summary_path = temp_path / "analysis_summary.txt"
                with open(summary_path, 'w') as f:
                    f.write(f"Pantsonfire Analysis Summary\n")
                    f.write(f"Repository: https://oxen.ai/{self.namespace}/{self.current_repo}\n")
                    f.write(f"Timestamp: {datetime.now().isoformat()}\n")
                    f.write(f"Total Findings: {len(results)}\n")
                    f.write(f"Oxen Data Management: Workspace-based (not traditional Git)\n")

                print(f"✅ Analysis repository created successfully at: https://oxen.ai/{self.namespace}/{self.current_repo}")

                # Always return True since repository was created successfully
                return True

        except Exception as e:
            print(f"❌ Error storing findings: {e}")
            return False

    def _populate_repository_with_data(self, results: List[CheckResult], temp_path: Path) -> bool:
        """
        Populate the repository with analysis data using Oxen Python SDK.

        Args:
            results: Analysis results to store
            temp_path: Path to temporary files

        Returns:
            True if data was successfully added to the repository
        """
        try:
            import oxen
            import pandas as pd
            
            # Create a temporary local Oxen repository
            local_repo_path = temp_path / "oxen_repo"
            local_repo_path.mkdir(exist_ok=True)
            
            # Initialize the local repository
            oxen.init(str(local_repo_path))
            
            # Create pandas DataFrame from results
            df_data = []
            for result in results:
                df_data.append({
                    "blog_source": result.blog_source,
                    "truth_source": result.truth_source,
                    "discrepancy": result.discrepancy,
                    "confidence": result.confidence,
                    "evidence": result.evidence,
                    "chunk_index": result.chunk_index,
                    "timestamp": result.timestamp.isoformat()
                })
            
            df = pd.DataFrame(df_data)
            
            # Save DataFrame to the local repository
            findings_file = local_repo_path / "findings.csv"
            df.to_csv(findings_file, index=False)
            
            # Also save as JSON for easier viewing
            json_file = local_repo_path / "findings.json"
            import json
            with open(json_file, 'w') as f:
                json.dump(df_data, f, indent=2)
            
            # Create a README
            readme_file = local_repo_path / "README.md"
            with open(readme_file, 'w') as f:
                f.write(f"# Pantsonfire Analysis Results\n\n")
                f.write(f"**Total Findings**: {len(results)}\n\n")
                f.write(f"**Analysis Date**: {datetime.now().isoformat()}\n\n")
                f.write(f"## Summary\n\n")
                for i, result in enumerate(results, 1):
                    f.write(f"### Issue #{i}\n")
                    f.write(f"- **Problem**: {result.discrepancy}\n")
                    f.write(f"- **Confidence**: {result.confidence:.2f}\n")
                    f.write(f"- **Evidence**: {result.evidence}\n\n")
            
            # Add files to Oxen
            oxen.add(str(findings_file))
            oxen.add(str(json_file))
            oxen.add(str(readme_file))
            
            # Commit the files
            oxen.commit(f"Pantsonfire analysis: {len(results)} findings detected")
            
            print(f"✅ Committed {len(results)} findings to local Oxen repository")
            
            # Set the remote and push
            remote_url = f"https://hub.oxen.ai/{self.namespace}/{self.current_repo}"
            oxen.config.set_remote("origin", remote_url)
            
            # Set authentication
            if self.api_key:
                oxen.auth.config_auth(host=self.base_url, token=self.api_key)
            
            # Push to remote
            oxen.push("origin", "main")
            
            print(f"✅ Pushed analysis data to Oxen repository")
            return True
            
        except Exception as e:
            print(f"⚠️  Error populating repository with data: {e}")
            import traceback
            traceback.print_exc()
            return False

    def get_results(self, limit: Optional[int] = None) -> List[CheckResult]:
        """
        Retrieve stored results (implements StorageBackend protocol)

        Note: This is a simplified implementation since file uploads may not work.
        In a real implementation, this would download results from Oxen.
        """
        # For now, return empty list since we can't retrieve from Oxen yet
        # TODO: Implement actual retrieval from Oxen repositories
        print("⚠️  Oxen result retrieval not yet implemented, returning empty list")
        return []

    def export_results(self, results: List[CheckResult], output_path: Path, format: str = "json") -> None:
        """
        Export results to file (implements StorageBackend protocol)
        """
        # Since we can't upload to Oxen yet, just save locally
        if format == "json":
            import json
            data = [result.model_dump() for result in results]
            with open(output_path, 'w') as f:
                json.dump(data, f, indent=2, default=str)
        elif format == "csv":
            import csv
            if results:
                with open(output_path, 'w', newline='') as f:
                    writer = csv.DictWriter(f, fieldnames=results[0].model_dump().keys())
                    writer.writeheader()
                    writer.writerows([result.model_dump() for result in results])
        else:  # text
            with open(output_path, 'w') as f:
                f.write(f"Pantsonfire Analysis Report\n")
                f.write(f"Generated: {datetime.now().isoformat()}\n")
                f.write(f"Total findings: {len(results)}\n\n")
                for i, result in enumerate(results, 1):
                    f.write(f"Finding {i}:\n")
                    f.write(f"  Discrepancy: {result.discrepancy}\n")
                    f.write(f"  Confidence: {result.confidence}\n")
                    f.write(f"  Evidence: {result.evidence}\n")
                    f.write(f"  Timestamp: {result.timestamp}\n")
                    f.write("\n")

    def store_extracted_content(self, content_map: Dict[str, str]) -> bool:
        """
        Store extracted content from sources.

        Args:
            content_map: Dictionary mapping source names to content

        Returns:
            True if storage was successful
        """
        if not self.current_repo or not self.current_branch:
            print("❌ No active analysis repository")
            return False

        try:
            with tempfile.TemporaryDirectory() as temp_dir:
                temp_path = Path(temp_dir)

                # Save each content piece
                for source_name, content in content_map.items():
                    # Sanitize filename
                    safe_name = "".join(c for c in source_name if c.isalnum() or c in (' ', '-', '_')).rstrip()
                    safe_name = safe_name.replace(' ', '_')

                    content_file = temp_path / f"{safe_name}.txt"
                    with open(content_file, 'w', encoding='utf-8') as f:
                        f.write(content)

                    oxen_path = f"sources/{safe_name}.txt"
                    if not self._upload_file(self.current_repo, self.current_branch, content_file, oxen_path):
                        return False

                # Commit changes
                commit_msg = f"Add extracted content from {len(content_map)} sources"
                if not self._commit_changes(self.current_repo, self.current_branch, commit_msg):
                    return False

                print(f"✅ Stored content from {len(content_map)} sources in Oxen")
                return True

        except Exception as e:
            print(f"❌ Error storing content: {e}")
            return False

    def store_prompts_and_metadata(self, metadata: Dict[str, Any]) -> bool:
        """
        Store analysis metadata and prompts.

        Args:
            metadata: Analysis metadata dictionary

        Returns:
            True if storage was successful
        """
        if not self.current_repo or not self.current_branch:
            print("❌ No active analysis repository")
            return False

        try:
            with tempfile.TemporaryDirectory() as temp_dir:
                temp_path = Path(temp_dir)

                # Save metadata
                metadata_file = temp_path / "metadata.json"
                with open(metadata_file, 'w') as f:
                    json.dump(metadata, f, indent=2, default=str)

                oxen_path = "metadata/analysis_metadata.json"
                if not self._upload_file(self.current_repo, self.current_branch, metadata_file, oxen_path):
                    return False

                # Commit changes
                commit_msg = "Add analysis metadata and configuration"
                if not self._commit_changes(self.current_repo, self.current_branch, commit_msg):
                    return False

                print("✅ Stored analysis metadata in Oxen")
                return True

        except Exception as e:
            print(f"❌ Error storing metadata: {e}")
            return False

    def generate_report_url(self) -> Optional[str]:
        """
        Generate URL to view the analysis report.

        Returns:
            URL to the analysis repository/branch
        """
        if not self.current_repo or not self.current_branch:
            return None

        return f"{self.base_url}/{self.namespace}/{self.current_repo}/tree/{self.current_branch}"

    def open_report_in_browser(self) -> bool:
        """
        Open the analysis report in the default web browser.

        Returns:
            True if browser was opened successfully
        """
        url = self.generate_report_url()
        if not url:
            print("❌ No active analysis to view")
            return False

        try:
            webbrowser.open(url)
            print(f"🌐 Opened report in browser: {url}")
            return True
        except Exception as e:
            print(f"❌ Failed to open browser: {e}")
            return False

    def export_results(self, results: List[CheckResult], output_path: Path, format: str = "json") -> None:
        """
        Export results to local files (for backward compatibility).

        Args:
            results: List of check results
            output_path: Path to export to
            format: Export format
        """
        # This method maintains backward compatibility by exporting locally
        # while the primary storage is now in Oxen

        output_path.parent.mkdir(parents=True, exist_ok=True)
        results_data = [result.model_dump() for result in results]

        if format.lower() == "json":
            with open(output_path, 'w') as f:
                json.dump(results_data, f, indent=2, default=str)
        elif format.lower() == "csv":
            import csv
            if results_data:
                with open(output_path, 'w', newline='') as f:
                    writer = csv.DictWriter(f, fieldnames=results_data[0].keys())
                    writer.writeheader()
                    writer.writerows(results_data)
        elif format.lower() in ["txt", "text"]:
            with open(output_path, 'w') as f:
                f.write(f"Pantsonfire Analysis Report\n")
                f.write(f"Generated: {datetime.now().isoformat()}\n")
                f.write(f"Total findings: {len(results)}\n\n")

                for i, result in enumerate(results, 1):
                    f.write(f"Finding {i}:\n")
                    f.write(f"  Discrepancy: {result.discrepancy}\n")
                    f.write(f"  Confidence: {result.confidence}\n")
                    f.write(f"  Evidence: {result.evidence}\n")
                    f.write(f"  Chunk: {result.chunk_index}\n")
                    f.write(f"  Timestamp: {result.timestamp}\n")
                    f.write("\n")

    def save_results(self, results: List[CheckResult]) -> None:
        """Save check results (implements StorageBackend protocol)"""
        success = self.store_findings(results)
        if not success:
            print("❌ Failed to save results to Oxen")

    def get_findings(self) -> List[CheckResult]:
        """
        Retrieve stored findings.

        Returns:
            List of check results
        """
        # For now, return empty list as primary storage is in Oxen
        # Could implement retrieval from Oxen in the future
        return []
