"""
Test case for Issue #4: Method name obfuscation incomplete - call sites not updated.

This test verifies that method definitions AND all their call sites
(external instance calls, internal self calls, getattr) are properly updated.
"""

import ast
import pytest

from pyobfus.config import ObfuscationConfig
from pyobfus.core.analyzer import SymbolAnalyzer
from pyobfus.core.generator import CodeGenerator
from pyobfus.transformers.name_mangler import NameMangler


class TestIssue4MethodObfuscation:
    """Test cases for Issue #4: Method call site obfuscation."""

    def test_external_method_call_obfuscation(self):
        """Test that external method calls (instance.method) are obfuscated."""
        code = """
class Calculator:
    def add(self, a, b):
        return a + b

calc = Calculator()
result = calc.add(1, 2)
"""
        # Parse
        tree = ast.parse(code)

        # Analyze
        config = ObfuscationConfig()
        analyzer = SymbolAnalyzer(config)
        analyzer.analyze(tree)

        # Transform
        mangler = NameMangler(config, analyzer)
        transformed = mangler.transform(tree)

        # Generate code
        generator = CodeGenerator()
        obfuscated_code = generator.generate(transformed)

        print("\n=== Original Code ===")
        print(code)
        print("\n=== Obfuscated Code ===")
        print(obfuscated_code)

        # Verify: should NOT contain 'add' method name anywhere
        assert "add" not in obfuscated_code, \
            "Method name 'add' should be obfuscated in all locations"

        # Execute to verify correctness
        namespace = {}
        exec(obfuscated_code, namespace)
        # Find the obfuscated 'result' variable (should be one of the I* variables with value 3)
        result_values = [v for v in namespace.values() if isinstance(v, int) and v == 3]
        assert len(result_values) > 0, "Obfuscated code should execute correctly and produce result 3"

    def test_internal_self_call_obfuscation(self):
        """Test that internal self.method calls are obfuscated."""
        code = """
class Calculator:
    def multiply(self, a, b):
        return a * b

    def square(self, x):
        return self.multiply(x, x)

calc = Calculator()
result = calc.square(5)
"""
        # Parse
        tree = ast.parse(code)

        # Analyze
        config = ObfuscationConfig()
        analyzer = SymbolAnalyzer(config)
        analyzer.analyze(tree)

        # Transform
        mangler = NameMangler(config, analyzer)
        transformed = mangler.transform(tree)

        # Generate code
        generator = CodeGenerator()
        obfuscated_code = generator.generate(transformed)

        print("\n=== Original Code ===")
        print(code)
        print("\n=== Obfuscated Code ===")
        print(obfuscated_code)

        # Verify: should NOT contain 'multiply' or 'square'
        assert "multiply" not in obfuscated_code
        assert "square" not in obfuscated_code

        # Execute to verify correctness
        namespace = {}
        exec(obfuscated_code, namespace)
        # Find the obfuscated 'result' variable (should be 25)
        result_values = [v for v in namespace.values() if isinstance(v, int) and v == 25]
        assert len(result_values) > 0, "Obfuscated code should execute correctly and produce result 25"

    def test_getattr_method_call(self):
        """Test that getattr() string references to methods are handled."""
        code = """
class Calculator:
    def divide(self, a, b):
        return a / b

calc = Calculator()
method = getattr(calc, 'divide')
result = method(10, 2)
"""
        # Parse
        tree = ast.parse(code)

        # Analyze
        config = ObfuscationConfig()
        analyzer = SymbolAnalyzer(config)
        analyzer.analyze(tree)

        # Transform
        mangler = NameMangler(config, analyzer)
        transformed = mangler.transform(tree)

        # Generate code
        generator = CodeGenerator()
        obfuscated_code = generator.generate(transformed)

        print("\n=== Original Code ===")
        print(code)
        print("\n=== Obfuscated Code ===")
        print(obfuscated_code)

        # Note: getattr with string literals is challenging to obfuscate
        # For now, we'll just verify the method definition is obfuscated
        # and note this as a known limitation

        # The string 'divide' will still exist, but method name should be obfuscated
        # This test documents the current limitation

    def test_multiple_instances_same_class(self):
        """Test obfuscation with multiple instances of the same class."""
        code = """
class Point:
    def __init__(self, x, y):
        self.x = x
        self.y = y

    def distance(self):
        return (self.x ** 2 + self.y ** 2) ** 0.5

p1 = Point(3, 4)
p2 = Point(5, 12)
d1 = p1.distance()
d2 = p2.distance()
"""
        # Parse
        tree = ast.parse(code)

        # Analyze
        config = ObfuscationConfig()
        analyzer = SymbolAnalyzer(config)
        analyzer.analyze(tree)

        # Transform
        mangler = NameMangler(config, analyzer)
        transformed = mangler.transform(tree)

        # Generate code
        generator = CodeGenerator()
        obfuscated_code = generator.generate(transformed)

        print("\n=== Original Code ===")
        print(code)
        print("\n=== Obfuscated Code ===")
        print(obfuscated_code)

        # Verify: should NOT contain 'distance'
        assert "distance" not in obfuscated_code

        # Execute to verify correctness
        namespace = {}
        exec(obfuscated_code, namespace)
        # Find the obfuscated distance values (should be 5.0 and 13.0)
        float_values = sorted([v for v in namespace.values() if isinstance(v, float)])
        assert len(float_values) >= 2
        assert abs(float_values[0] - 5.0) < 0.01
        assert abs(float_values[1] - 13.0) < 0.01

    def test_method_inheritance(self):
        """Test method obfuscation with inheritance."""
        code = """
class Animal:
    def speak(self):
        return "Some sound"

class Dog(Animal):
    def speak(self):
        return "Woof"

dog = Dog()
sound = dog.speak()
"""
        # Parse
        tree = ast.parse(code)

        # Analyze
        config = ObfuscationConfig()
        analyzer = SymbolAnalyzer(config)
        analyzer.analyze(tree)

        # Transform
        mangler = NameMangler(config, analyzer)
        transformed = mangler.transform(tree)

        # Generate code
        generator = CodeGenerator()
        obfuscated_code = generator.generate(transformed)

        print("\n=== Original Code ===")
        print(code)
        print("\n=== Obfuscated Code ===")
        print(obfuscated_code)

        # Verify: should NOT contain 'speak'
        assert "speak" not in obfuscated_code

        # Execute to verify correctness
        namespace = {}
        exec(obfuscated_code, namespace)
        # Find the obfuscated 'sound' variable (should be "Woof")
        string_values = [v for v in namespace.values() if isinstance(v, str) and v == "Woof"]
        assert len(string_values) > 0, "Obfuscated code should execute correctly and produce 'Woof'"
