from MF.V3.Settings.Camera import Camera as MF_V3_Settings_Camera_Camera
from MF.V3.Settings.Capture import Capture as MF_V3_Settings_Capture_Capture
from MF.V3.Settings.Projector import Projector as MF_V3_Settings_Projector_Projector
from MF.V3.Settings.Turntable import Turntable as MF_V3_Settings_Turntable_Turntable
from enum import Enum
from typing import List


class Scan:

    """
     Scan settings.
    """
    class Processing:

        """
         Scan processing settings.
        """
        class PhaseEdgeDetection:
            """
            Phase edge detection settings.

            Phase edge detection produces a binary mask indicating the edges of a horizontal/vertical pair of phase images.  Since flat geometries give a constant phase image gradient, we use the second derivative (Laplacian) of the phase image to detect edges rather than the gradient.

            The edge mask generated by phase edge detection is an input to both phase filtering and adaptive sampling.  If neither of these are enabled, the phase edge detection settings have no effect on the output point cloud.
            """
            def __init__(self, threshold: float, laplacianKernelRadius: int, gaussianBlurRadius: int, gaussianBlurStdDev: float, maximumWidthForProcessing: int):
                # The edge detection threshold.
                self.threshold = threshold
                # The Laplacian kernel radius.  This must be in the range [1..5].
                self.laplacianKernelRadius = laplacianKernelRadius
                """
                Gaussian blur kernel radius. (Optional)  To disable, set to 0.
                The phase images can optionally blurred before taking the Laplacian to reduce noise.
                However as a result, the detected edges are wider.
                """
                self.gaussianBlurRadius = gaussianBlurRadius
                """
                Gaussian blur kernel standard deviation.
                This parameter is ignored if `gaussianBlurSize` is zero.
                """
                self.gaussianBlurStdDev = gaussianBlurStdDev
                """
                The maximum image width for processing. (Optional) To disable, set to 0.
                If this value is greater than zero, the phase images are resized to the maximum width prior
                to computing the Laplacian and the the detected edges are then upsampled to the original size.
                This would be done to speed up processing or to detect edges on a larger scale.
                """
                self.maximumWidthForProcessing = maximumWidthForProcessing

        class PhaseFilter:

            """
             Phase filter settings.
            """
            def __init__(self, kernelRadius: int, spatialWeightStdDev: float):
                """
                The filter kernel radius.
                A neighboring value must be within this radius to be included in the filter.
                If the kernel radius is set to zero, the phase filtering is disabled.
                """
                self.kernelRadius = kernelRadius
                """
                The standard deviation of the spatial weights.
                The weight of a neighboring value is $\exp(-(r/s)^2)$ where $r$ is the distance
                to the central value and $s$ is the spatial weight standard deviation.
                If the spatial weight standard deviation is set to zero, all the spatial
                weights are uniformly set to 1.
                """
                self.spatialWeightStdDev = spatialWeightStdDev

        class AdaptiveSampling:
            """
            Adaptive sampling settings

            Adaptive sampling will downsample points in regions of low detail
            and keep points in regions of high detail.
            """
            class Type(Enum):
                NONE = "NONE"  # Do not use adaptive sampling.
                REGULAR = "REGULAR"  # Use a regular sampling mask in regions of low detail.
                RANDOM = "RANDOM"  # Use a random sampling mask in regions of low detail.

            def __init__(self, type: 'Type', rate: float):
                # Sampling type.
                self.type = type
                # The sample rate [0..1] for the regions of low detail.
                self.rate = rate

        class PointClipping:

            """
             Point clipping settings.
            """
            class Type(Enum):

                """
                 Point clipping type.
                """
                OutsideCube = "OutsideCube"  # Clip points outside a unit cube.
                OutsideCylinder = "OutsideCylinder"  # Clip points outside a unit cylinder.
                OutsideSphere = "OutsideSphere"  # Clip points outside a unit sphere.
                InsideCube = "InsideCube"  # Clip points inside a unit cube.
                InsideCylinder = "InsideCylinder"  # Clip points inside a unit cylinder.
                InsideSphere = "InsideSphere"  # Clip points inside a unit sphere.

            def __init__(self, type: 'Type', transform: List[float] = None):
                # Point clipping type.
                self.type = type
                # 4x4 transform mapping 3D points to the canonical point clipping coordinates.
                self.transform = transform

        class NormalEstimation:

            """
             Normal estimation settings.
            """
            class Method(Enum):
                NORMAL_LLS = "NORMAL_LLS"  # Linear least squares method
                NORMAL_OPEN3D = "NORMAL_OPEN3D"  # Open3D method using KD tree search for nearest neighbors

            def __init__(self, method: 'Method', maximumNeighbourCount: int, maximumNeighbourRadius: float, useMaximumNeighbourCount: bool, useMaximumNeighbourRadius: bool):
                # Normal estimation method.
                self.method = method
                """
                Maximum number of nearest neighbors used to compute the normal.
                This value is only used with the NORMAL_OPEN3D method.
                """
                self.maximumNeighbourCount = maximumNeighbourCount
                # Maximum radius for a point to be considered a neighbour.
                self.maximumNeighbourRadius = maximumNeighbourRadius
                # Use the maximum neighbour count.
                self.useMaximumNeighbourCount = useMaximumNeighbourCount
                # Use the maximum neighbour radius.
                self.useMaximumNeighbourRadius = useMaximumNeighbourRadius

        class OutlierRemoval:

            """
             Outlier removal settings.
            """
            def __init__(self, neighbourCount: int, neighbourRadius: float):
                # The minimum number of points within the radius for a point to be retained.
                self.neighbourCount = neighbourCount
                # The neighbour search radius.
                self.neighbourRadius = neighbourRadius

        def __init__(self, projectorSampleRate: float = None, imageSampleRate: float = None, edgeDetection: 'PhaseEdgeDetection' = None, phaseFilter: 'PhaseFilter' = None, adaptiveSampling: 'AdaptiveSampling' = None, pointClipping: List['PointClipping'] = None, normalEstimation: 'NormalEstimation' = None, outlierRemoval: 'OutlierRemoval' = None):
            # Projector sample rate.
            self.projectorSampleRate = projectorSampleRate
            # Image sample rate.
            self.imageSampleRate = imageSampleRate
            # Phase edge detection settings.
            self.edgeDetection = edgeDetection
            # Phase filter settings.
            self.phaseFilter = phaseFilter
            # Adaptive sampling settings.
            self.adaptiveSampling = adaptiveSampling
            # Point clipping settings.
            self.pointClipping = pointClipping
            # Normal estimation settings.
            self.normalEstimation = normalEstimation
            # Outlier removal settings.
            self.outlierRemoval = outlierRemoval

    def __init__(self, camera: MF_V3_Settings_Camera_Camera = None, projector: MF_V3_Settings_Projector_Projector = None, turntable: MF_V3_Settings_Turntable_Turntable = None, capture: MF_V3_Settings_Capture_Capture = None, processing: 'Processing' = None, alignWithScanner: bool = None, centerAtOrigin: bool = None):
        # Camera settings.
        self.camera = camera
        # Projector settings.
        self.projector = projector
        # Turntable settings.
        self.turntable = turntable
        # Capture settings.
        self.capture = capture
        # Processing settings.
        self.processing = processing
        # Align the scan with the scanner.
        self.alignWithScanner = alignWithScanner
        # Center the scan at the origin.
        self.centerAtOrigin = centerAtOrigin


