from enum import Enum
from google.protobuf import any_pb2 as _any_pb2


class TaskState(Enum):
    Empty = "None"  # The task state is not defined.
    Sent = "Sent"  # The task has been sent by the client.
    Received = "Received"  # The task has been received by the server.
    Started = "Started"  # The task started by the server.
    Completed = "Completed"  # The task is completed by the server.
    Cancelled = "Cancelled"  # The task has been cancelled by the client.
    Failed = "Failed"  # The task has failed.  A string describing the error is returned with the task.
    Dropped = "Dropped"  # The task has not been received by the server, or task IDs were sent out of sequence.
    Disconnected = "Disconnected"  # The client has been disconnected from the server before the task could finish.


class Task:
    """*
    Generic task message for the Three Scanner.

    The task message is the generic message used for requesting a task of the Three Scanner and receiving updates and results.

    Example: Apply camera settings with the "SetCameras" task.

    > Example request:

    ```json
    {
    "Task":{
    "Index":1,
    "Type":"SetCameras"
    "Input":{
    "analogGain":256,
    "digitalGain":128,
    "exposure":18000
    },
    }
    }
    ```

    > Example response:

    ```json
    {
    "Task":{
    "Index":1,
    "Type":"SetCameras"
    "Input":{
    "analogGain":256,
    "digitalGain":512,
    "exposure":18000
    },
    "Output":{
    "analogGain":{"default":512.0,"max":1024.0,"min":256.0,"value":256.0},
    "digitalGain":{"default":256,"max":65536,"min":256,"value":512},
    "exposure":{"default":27000,"max":90000,"min":9000,"value":18000},
    },
    "State":"Completed"
    }
    }
    ```
    """
    def __init__(self, Index: int, Type: str, Input: _any_pb2 = None, Output: _any_pb2 = None, State: 'TaskState' = None, Error: str = None, Progress: _any_pb2 = None):
        # A unique identifier generated by the client.  This identifier associates all incoming and outgoing task messages with a specific task requested by the client.
        self.Index = Index
        # The string identifying the task type. See task definitions for the list of valid task strings.
        self.Type = Type
        # Optional input message.  See each task definition for details.
        self.Input = Input
        # Optional output message.  See each task definition for details.
        self.Output = Output
        # The current state of the task.
        self.State = State
        # A string describing the error if the task has failed.
        self.Error = Error
        # A Progress object for tasks that give updates on long running tasks
        self.Progress = Progress


