import collections
import contextlib
from pathlib import Path
from xml.dom.minidom import Element
from xml.etree import ElementTree

import numpy as np
from packaging.version import Version
from qe_tools import CONSTANTS
from xmlschema import XMLSchema

from aiida_quantumespresso.parsers import QEOutputParsingError
from aiida_quantumespresso.utils.mapping import get_logging_container

from .exceptions import XMLParseError, XMLUnsupportedFormatError


def raise_parsing_error(message):
    raise XMLParseError(message)


def parser_assert(condition, message, log_func=raise_parsing_error):
    if not condition:
        log_func(message)


def parser_assert_equal(val1, val2, message, log_func=raise_parsing_error):
    if val1 != val2:
        msg = f'Violated assertion: {val1} == {val2}'
        if message:
            msg += ' - '
            msg += message
        log_func(msg)


def cell_volume(a1, a2, a3):
    r"""Returns the volume of the primitive cell: :math:`|\vec a_1\cdot(\vec a_2\cross \vec a_3)|`"""
    a_mid_0 = a2[1] * a3[2] - a2[2] * a3[1]
    a_mid_1 = a2[2] * a3[0] - a2[0] * a3[2]
    a_mid_2 = a2[0] * a3[1] - a2[1] * a3[0]

    return abs(float(a1[0] * a_mid_0 + a1[1] * a_mid_1 + a1[2] * a_mid_2))


def read_xml_card(dom, cardname):
    """Read an XML card from a DOM object (used for legacy XML parsing)."""
    try:
        root_node = [_ for _ in dom.childNodes if isinstance(_, Element) and _.nodeName == 'Root'][0]
        the_card = [_ for _ in root_node.childNodes if _.nodeName == cardname][0]
        return the_card
    except Exception as e:
        print(e)
        raise QEOutputParsingError(f'Error parsing tag {cardname}')


def parse_xml_child_integer(tagname, target_tags):
    """Parse an XML child tag as an integer (used for legacy XML parsing)."""
    try:
        a = [_ for _ in target_tags.childNodes if _.nodeName == tagname][0]
        b = a.childNodes[0]
        return int(b.data)
    except Exception:
        raise QEOutputParsingError(f'Error parsing tag {tagname} inside {target_tags.tagName}')


def parse_xml_child_bool(tagname, target_tags):
    """Parse an XML child tag as a boolean (used for legacy XML parsing)."""
    try:
        a = [_ for _ in target_tags.childNodes if _.nodeName == tagname][0]
        b = a.childNodes[0]
        return str2bool(b.data)
    except Exception:
        raise QEOutputParsingError(f'Error parsing tag {tagname} inside {target_tags.tagName}')


def str2bool(string):
    """Convert a string to a boolean value (used for legacy XML parsing)."""
    try:
        false_items = ['f', '0', 'false', 'no']
        true_items = ['t', '1', 'true', 'yes']
        string = str(string.lower().strip())
        if string in false_items:
            return False
        if string in true_items:
            return True
        else:
            raise QEOutputParsingError(f'Error converting string {string} to boolean value.')
    except Exception:
        raise QEOutputParsingError('Error converting string to boolean.')


def get_schema_filepath(xml):
    """Return the absolute filepath to the XML schema file that can be used to parse the given XML.

    :param xml: the pre-parsed XML object
    :return: the XSD absolute filepath
    """
    xml_schema_location_key = '{http://www.w3.org/2001/XMLSchema-instance}schemaLocation'
    # The part in curly brackets is an expanded namespace
    dir_path_schemas = Path(__file__).resolve().parent / 'schemas'

    element_root = xml.getroot()

    if element_root is None:
        raise XMLParseError('Could not find the XML root!')

    # Patch for QE v7.0: The scheme file name was not updated in the `xsi.schemaLocation` element
    with contextlib.suppress(AttributeError):
        if element_root.find('general_info').find('creator').get('VERSION') == '7.0':
            return dir_path_schemas / 'qes_211101.xsd'

    element_schema_location = element_root.get(xml_schema_location_key)
    # e.g. "http://www.quantum-espresso.org/ns/qes/qes-1.0 http://www.quantum-espresso.org/ns/qes/qes-1.0.xsd"

    if element_schema_location is None:
        raise XMLParseError('Could not find the schema location!')

    schema_location = element_schema_location.split()[1]  # e.g. "http://www.quantum-espresso.org/ns/qes/qes-1.0.xsd"
    schema_filename = schema_location.rpartition('/')[2]  # e.g. "qes-1.0.xsd"

    schema_filepath = dir_path_schemas / schema_filename

    if not schema_filepath.exists():
        raise XMLUnsupportedFormatError(
            f'Cannot find schema {schema_filepath.name} in {dir_path_schemas}.\n'
            'Make sure you are running a supported Quantum ESPRESSO version.'
        )

    return schema_filepath


def parse_xml(xml_file):
    """Parse the content of XML output file written by `pw.x` and `cp.x` with the new schema-based XML format.

    :param xml_file: file-like object containing the XML
    :returns: tuple of two dictionaries, with the parsed data and log messages, respectively
    """
    e_bohr2_to_coulomb_m2 = 57.214766  # e/a0^2 to C/m^2 (electric polarization) from Wolfram Alpha

    logs = get_logging_container()

    try:
        xml_parsed = ElementTree.parse(xml_file)
    except ElementTree.ParseError as exception:
        raise XMLParseError('error while parsing XML file') from exception

    schema_filepath = get_schema_filepath(xml_parsed)
    xsd = XMLSchema(schema_filepath)

    # Validate XML document against the schema
    # Returned dictionary has a structure where, if tag ['key'] is "simple", xml_dictionary['key'] returns its content.
    # Otherwise, the following keys are available:
    #
    #  xml_dictionary['key']['$'] returns its content
    #  xml_dictionary['key']['@attr'] returns its attribute 'attr'
    #  xml_dictionary['key']['nested_key'] goes one level deeper.

    xml_dictionary, errors = xsd.to_dict(xml_parsed, validation='lax')
    if errors:
        logs.error.append(f'{len(errors)} XML schema validation error(s) schema: {schema_filepath}:')
        for err in errors:
            logs.error.append(str(err))

    xml_version = Version(xml_dictionary['general_info']['xml_format']['@VERSION'])
    inputs = xml_dictionary.get('input', {})
    outputs = xml_dictionary['output']

    # Fix a bug of QE 6.8: the output XML is not consistent with schema, see
    # https://github.com/aiidateam/aiida-quantumespresso/pull/717
    if xml_version == '6.8' and 'timing_info' in xml_dictionary:
        timing_info = xml_dictionary['timing_info']
        partial_pwscf = timing_info.find("partial[@label='PWSCF'][@calls='0']")
        with contextlib.suppress(TypeError, ValueError):
            timing_info.remove(partial_pwscf)

    lattice_vectors = [
        [x * CONSTANTS.bohr_to_ang for x in outputs['atomic_structure']['cell']['a1']],
        [x * CONSTANTS.bohr_to_ang for x in outputs['atomic_structure']['cell']['a2']],
        [x * CONSTANTS.bohr_to_ang for x in outputs['atomic_structure']['cell']['a3']],
    ]

    has_electric_field = outputs.get('electric_field', None) is not None
    has_dipole_correction = inputs.get('electric_field', {}).get('dipole_correction', False)

    if 'occupations' in inputs.get('bands', {}):
        try:
            occupations = inputs['bands']['occupations']['$']
        except TypeError:  # "string indices must be integers" -- might have attribute 'nspin'
            occupations = inputs['bands']['occupations']
    else:
        occupations = None

    starting_magnetization = []
    magnetization_angle1 = []
    magnetization_angle2 = []

    for specie in outputs['atomic_species']['species']:
        starting_magnetization.append(specie.get('starting_magnetization', 0.0))
        magnetization_angle1.append(specie.get('magnetization_angle1', 0.0))
        magnetization_angle2.append(specie.get('magnetization_angle2', 0.0))

    constraint_mag = 0
    spin_constraints = inputs.get('spin_constraints', {}).get('spin_constraints', None)
    if spin_constraints == 'atomic':
        constraint_mag = 1
    elif spin_constraints == 'atomic direction':
        constraint_mag = 2
    elif spin_constraints == 'total':
        constraint_mag = 3
    elif spin_constraints == 'total direction':
        constraint_mag = 6

    lsda = inputs.get('spin', {}).get('lsda', False)
    spin_orbit_calculation = inputs.get('spin', {}).get('spinorbit', False)
    non_colinear_calculation = outputs.get('magnetization', {}).get('noncolin', False)
    do_magnetization = outputs.get('magnetization', {}).get('do_magnetization', False)

    # Time reversal symmetry of the system
    time_reversal = not (non_colinear_calculation and do_magnetization)

    # If no specific tags are present, the default is 1
    if non_colinear_calculation or spin_orbit_calculation:
        nspin = 4
    elif lsda:
        nspin = 2
    else:
        nspin = 1

    symmetries = []
    lattice_symmetries = []  # note: will only contain lattice symmetries that are NOT crystal symmetries
    inversion_symmetry = False

    # See also PW/src/setup.f90
    nsym = outputs.get('symmetries', {}).get('nsym', None)  # crystal symmetries
    nrot = outputs.get('symmetries', {}).get('nrot', None)  # lattice symmetries

    for symmetry in outputs.get('symmetries', {}).get('symmetry', []):
        # There are two types of symmetries, lattice and crystal. The pure inversion (-I) is always a lattice symmetry,
        # so we don't care. But if the pure inversion is also a crystal symmetry, then then the system as a whole
        # has (by definition) inversion symmetry; so we set the global property inversion_symmetry = True.
        symmetry_type = symmetry['info']['$']
        symmetry_name = symmetry['info']['@name']
        if symmetry_type == 'crystal_symmetry' and symmetry_name.lower() == 'inversion':
            inversion_symmetry = True

        sym = {
            'rotation': [
                symmetry['rotation']['$'][0:3],
                symmetry['rotation']['$'][3:6],
                symmetry['rotation']['$'][6:9],
            ],
            'name': symmetry_name,
        }

        try:
            sym['t_rev'] = '1' if symmetry['info']['@time_reversal'] else '0'
        except KeyError:
            sym['t_rev'] = '0'

        with contextlib.suppress(KeyError):
            sym['equivalent_atoms'] = symmetry['equivalent_atoms']['$']

        with contextlib.suppress(KeyError):
            sym['fractional_translation'] = symmetry['fractional_translation']

        if symmetry_type == 'crystal_symmetry':
            symmetries.append(sym)
        elif symmetry_type == 'lattice_symmetry':
            lattice_symmetries.append(sym)
        else:
            raise XMLParseError(f'Unexpected type of symmetry: {symmetry_type}')

    if (nsym != len(symmetries)) or (nrot != len(symmetries) + len(lattice_symmetries)):
        logs.warning.append(
            f'Inconsistent number of symmetries: nsym={nsym}, nrot={nrot}, len(symmetries)={len(symmetries)}, len(lattice_symmetries)={len(lattice_symmetries)}'
        )

    xml_data = {
        #'pp_check_flag': True, # Currently not printed in the new format.
        # Signals whether the XML file is complete
        # and can be used for post-processing. Everything should be in the XML now, but in
        # any case, the new XML schema should mostly protect from incomplete files.
        'lkpoint_dir': False,  # Currently not printed in the new format.
        # Signals whether kpt-data are written in sub-directories.
        # Was generally true in the old format, but now all the eigenvalues are
        # in the XML file, under output / band_structure, so this is False.
        'charge_density': './charge-density.dat',  # A file name. Not printed in the new format.
        # The filename and path are considered fixed: <outdir>/<prefix>.save/charge-density.dat
        # TODO: change to .hdf5 if output format is HDF5 (issue #222)
        'rho_cutoff_units': 'eV',
        'wfc_cutoff_units': 'eV',
        'fermi_energy_units': 'eV',
        'k_points_units': '1 / angstrom',
        'symmetries_units': 'crystal',
        'constraint_mag': constraint_mag,
        'magnetization_angle2': magnetization_angle2,
        'magnetization_angle1': magnetization_angle1,
        'starting_magnetization': starting_magnetization,
        'has_electric_field': has_electric_field,
        'has_dipole_correction': has_dipole_correction,
        'lda_plus_u_calculation': 'dftU' in outputs,
        'format_name': xml_dictionary['general_info']['xml_format']['@NAME'],
        'format_version': xml_dictionary['general_info']['xml_format']['@VERSION'],
        # TODO: check that format version: a) matches the XSD schema version; b) is updated as well
        #       See line 43 in Modules/qexsd.f90
        'creator_name': xml_dictionary['general_info']['creator']['@NAME'].lower(),
        'creator_version': xml_dictionary['general_info']['creator']['@VERSION'],
        'non_colinear_calculation': non_colinear_calculation,
        'do_magnetization': do_magnetization,
        'time_reversal_flag': time_reversal,
        'symmetries': symmetries,
        'lattice_symmetries': lattice_symmetries,
        'do_not_use_time_reversal': inputs.get('symmetry_flags', {}).get('noinv', None),
        'spin_orbit_domag': do_magnetization,
        'fft_grid': [value for _, value in sorted(outputs['basis_set']['fft_grid'].items())],
        'lsda': lsda,
        'number_of_spin_components': nspin,
        'no_time_rev_operations': inputs.get('symmetry_flags', {}).get('no_t_rev', None),
        'inversion_symmetry': inversion_symmetry,  # the old tag was INVERSION_SYMMETRY and was set to (from the code): "invsym    if true the system has inversion symmetry"
        'number_of_bravais_symmetries': nrot,  # lattice symmetries
        'number_of_symmetries': nsym,  # crystal symmetries
        'wfc_cutoff': inputs.get('basis', {}).get('ecutwfc', -1.0) * CONSTANTS.hartree_to_ev,
        'rho_cutoff': outputs['basis_set']['ecutrho'] * CONSTANTS.hartree_to_ev,  # not always printed in input->basis
        'smooth_fft_grid': [value for _, value in sorted(outputs['basis_set']['fft_smooth'].items())],
        'dft_exchange_correlation': inputs.get('dft', {}).get(
            'functional', None
        ),  # TODO: also parse optional elements of 'dft' tag
        # WARNING: this is different between old XML and new XML
        'spin_orbit_calculation': spin_orbit_calculation,
        'q_real_space': outputs['algorithmic_info']['real_space_q'],
        'energy_units': 'eV',
        'energy_accuracy_units': 'eV',
        'energy_ewald_units': 'eV',
        'energy_hartree_units': 'eV',
        'energy_one_electron_units': 'eV',
        'energy_xc_units': 'eV',
        'number_of_atoms': inputs['atomic_structure']['@nat'],
        'number_of_species': inputs['atomic_species']['@ntyp'],
    }
    xml_data['absolute_magnetization'] = outputs.get('magnetization', {}).get('absolute', 0.0)
    xml_data['total_magnetization'] = outputs.get('magnetization', {}).get('total', 0.0)

    if 'timing_info' in xml_dictionary:
        xml_data['wall_time_seconds'] = xml_dictionary['timing_info']['total']['wall']

    # alat is technically an optional attribute according to the schema,
    # but I don't know what to do if it's missing. atomic_structure is mandatory.
    output_alat_bohr = outputs['atomic_structure']['@alat']
    output_alat_angstrom = output_alat_bohr * CONSTANTS.bohr_to_ang

    # Band structure
    if 'band_structure' in outputs:
        band_structure = outputs['band_structure']

        smearing_xml = None

        if 'smearing' in outputs['band_structure']:
            smearing_xml = outputs['band_structure']['smearing']
        elif 'smearing' in inputs:
            smearing_xml = inputs['smearing']

        if smearing_xml:
            degauss = smearing_xml['@degauss']

            # Versions below 19.03.04 (Quantum ESPRESSO<=6.4.1) incorrectly print degauss in Ry instead of Hartree
            if xml_version < Version('19.03.04'):
                degauss *= CONSTANTS.ry_to_ev
            else:
                degauss *= CONSTANTS.hartree_to_ev

            xml_data['degauss'] = degauss
            xml_data['smearing_type'] = smearing_xml['$']

        num_k_points = band_structure['nks']
        num_electrons = band_structure['nelec']

        # In schema v240411 (QE v7.3.1), the `number_of_atomic_wfc` is moved to the `atomic_structure` tag as an attribute
        num_atomic_wfc = band_structure.get('num_of_atomic_wfc', None) or outputs['atomic_structure'].get(
            '@num_of_atomic_wfc', None
        )
        num_bands = band_structure.get('nbnd', None)
        num_bands_up = band_structure.get('nbnd_up', None)
        num_bands_down = band_structure.get('nbnd_dw', None)

        if 'fermi_energy' in band_structure:
            xml_data['fermi_energy'] = band_structure['fermi_energy'] * CONSTANTS.hartree_to_ev

        if 'two_fermi_energies' in band_structure:
            xml_data['fermi_energy_up'], xml_data['fermi_energy_down'] = (
                energy * CONSTANTS.hartree_to_ev for energy in band_structure['two_fermi_energies']
            )

        xml_data['number_of_atomic_wfc'] = num_atomic_wfc
        xml_data['number_of_k_points'] = num_k_points
        xml_data['number_of_electrons'] = num_electrons

        if num_bands is None and num_bands_up is None and num_bands_down is None:
            raise XMLParseError('None of `nbnd`, `nbnd_up` or `nbdn_dw` could be parsed.')

        # If both channels are `None` we are dealing with a non spin-polarized or non-collinear calculation
        if num_bands_up is None and num_bands_down is None:
            spins = False

        # If only one of the channels is `None` we raise, because that is an inconsistent result
        elif num_bands_up is None or num_bands_down is None:
            raise XMLParseError('Only one of `nbnd_up` and `nbnd_dw` could be parsed')

        # Here it is a spin-polarized calculation, where for pw.x the number of bands in each channel should be identical.
        else:
            spins = True
            if num_bands_up != num_bands_down:
                raise XMLParseError(f'different number of bands for spin channels: {num_bands_up} and {num_bands_down}')

            if num_bands is not None and num_bands != num_bands_up + num_bands_down:
                raise XMLParseError(
                    f'Inconsistent number of bands: nbnd={num_bands}, nbnd_up={num_bands_up}, nbnd_down={num_bands_down}'
                )

            if num_bands is None:
                num_bands = num_bands_up + num_bands_down  # backwards compatibility;

        if 'ks_energies' in band_structure:
            k_points = []
            k_points_weights = []

            ks_states = band_structure['ks_energies']

            for ks_state in ks_states:
                k_points.append([kp * 2 * np.pi / output_alat_angstrom for kp in ks_state['k_point']['$']])
                k_points_weights.append(ks_state['k_point']['@weight'])

            if not spins:
                band_eigenvalues = [[]]
                band_occupations = [[]]
                for ks_state in ks_states:
                    band_eigenvalues[0].append(ks_state['eigenvalues']['$'])
                    band_occupations[0].append(ks_state['occupations']['$'])
            else:
                band_eigenvalues = [[], []]
                band_occupations = [[], []]
                for ks_state in ks_states:
                    band_eigenvalues[0].append(ks_state['eigenvalues']['$'][0:num_bands_up])
                    band_eigenvalues[1].append(ks_state['eigenvalues']['$'][num_bands_up:num_bands])
                    band_occupations[0].append(ks_state['occupations']['$'][0:num_bands_up])
                    band_occupations[1].append(ks_state['occupations']['$'][num_bands_up:num_bands])

            band_eigenvalues = np.array(band_eigenvalues) * CONSTANTS.hartree_to_ev
            band_occupations = np.array(band_occupations)

            if not spins:
                parser_assert_equal(
                    band_eigenvalues.shape,
                    (1, num_k_points, num_bands),
                    'Unexpected shape of band_eigenvalues',
                )
                parser_assert_equal(
                    band_occupations.shape,
                    (1, num_k_points, num_bands),
                    'Unexpected shape of band_occupations',
                )
            else:
                parser_assert_equal(
                    band_eigenvalues.shape,
                    (2, num_k_points, num_bands_up),
                    'Unexpected shape of band_eigenvalues',
                )
                parser_assert_equal(
                    band_occupations.shape,
                    (2, num_k_points, num_bands_up),
                    'Unexpected shape of band_occupations',
                )

            bands_dict = {
                'occupations': band_occupations,
                'bands': band_eigenvalues,
                'bands_units': 'eV',
            }
            xml_data['bands'] = bands_dict
            xml_data['k_points'] = k_points
            xml_data['k_points_weights'] = k_points_weights

            if not spins:
                xml_data['number_of_bands'] = num_bands
            else:
                # For collinear spin-polarized calculations `spins=True` and `num_bands` is sum of both channels. To get the
                # actual number of bands, we divide by two using integer division
                xml_data['number_of_bands'] = num_bands // 2

            for key, value in [
                ('number_of_bands_up', num_bands_up),
                ('number_of_bands_down', num_bands_down),
            ]:
                if value is not None:
                    xml_data[key] = value

    try:
        monkhorst_pack = inputs['k_points_IBZ']['monkhorst_pack']
    except KeyError:
        pass  # not using Monkhorst pack
    else:
        xml_data['monkhorst_pack_grid'] = [monkhorst_pack[attr] for attr in ['@nk1', '@nk2', '@nk3']]
        xml_data['monkhorst_pack_offset'] = [monkhorst_pack[attr] for attr in ['@k1', '@k2', '@k3']]

    if occupations is not None:
        xml_data['occupations'] = occupations

    if 'boundary_conditions' in outputs and 'assume_isolated' in outputs['boundary_conditions']:
        xml_data['assume_isolated'] = outputs['boundary_conditions']['assume_isolated']

    # This is not printed by QE 6.3, but will be re-added before the next version
    if 'real_space_beta' in outputs['algorithmic_info']:
        xml_data['beta_real_space'] = outputs['algorithmic_info']['real_space_beta']

    conv_info = {}
    conv_info_scf = {}
    conv_info_opt = {}
    # NOTE: n_scf_steps refers to the number of SCF steps in the *last* loop only.
    # To get the total number of SCF steps in the run you should sum up the individual steps.
    # TODO: should we parse 'steps' too? Are they already added in the output trajectory?
    for key in ['convergence_achieved', 'n_scf_steps', 'scf_error']:
        with contextlib.suppress(KeyError):
            conv_info_scf[key] = outputs['convergence_info']['scf_conv'][key]
    for key in ['convergence_achieved', 'n_opt_steps', 'grad_norm']:
        with contextlib.suppress(KeyError):
            conv_info_opt[key] = outputs['convergence_info']['opt_conv'][key]
    if conv_info_scf:
        conv_info['scf_conv'] = conv_info_scf
    if conv_info_opt:
        conv_info['opt_conv'] = conv_info_opt
    if conv_info:
        xml_data['convergence_info'] = conv_info

    if 'status' in xml_dictionary:
        xml_data['exit_status'] = xml_dictionary['status']
        # 0 = convergence reached;
        # -1 = SCF convergence failed;
        # 3 = ionic convergence failed
        # These might be changed in the future. Also see PW/src/run_pwscf.f90

    if has_electric_field and 'BerryPhase' in outputs['electric_field']:
        berry_phase = outputs['electric_field']['BerryPhase']
        # This is what I would like to do, but it's not retro-compatible
        # xml_data['berry_phase'] = {}
        # xml_data['berry_phase']['total_phase']         = berry_phase['totalPhase']['$']
        # xml_data['berry_phase']['total_phase_modulus'] = berry_phase['totalPhase']['@modulus']
        # xml_data['berry_phase']['total_ionic_phase']      = berry_phase['totalPhase']['@ionic']
        # xml_data['berry_phase']['total_electronic_phase'] = berry_phase['totalPhase']['@electronic']
        # xml_data['berry_phase']['total_polarization']           = berry_phase['totalPolarization']['polarization']['$']
        # xml_data['berry_phase']['total_polarization_modulus']   = berry_phase['totalPolarization']['modulus']
        # xml_data['berry_phase']['total_polarization_units']     = berry_phase['totalPolarization']['polarization']['@Units']
        # xml_data['berry_phase']['total_polarization_direction'] = berry_phase['totalPolarization']['direction']
        # parser_assert_equal(xml_data['berry_phase']['total_phase_modulus'].lower(), '(mod 2)',
        #                    "Unexpected modulus for total phase")
        # parser_assert_equal(xml_data['berry_phase']['total_polarization_units'].lower(), 'e/bohr^2',
        #                    "Unsupported units for total polarization")
        # Retro-compatible keys:
        polarization = berry_phase['totalPolarization']['polarization']['$']
        polarization_units = berry_phase['totalPolarization']['polarization']['@Units']
        polarization_modulus = berry_phase['totalPolarization']['modulus']
        parser_assert(
            polarization_units in ['e/bohr^2', 'C/m^2'],
            f"Unsupported units '{polarization_units}' of total polarization",
        )
        if polarization_units == 'e/bohr^2':
            polarization *= e_bohr2_to_coulomb_m2
            polarization_modulus *= e_bohr2_to_coulomb_m2

        xml_data['total_phase'] = berry_phase['totalPhase']['$']
        xml_data['total_phase_units'] = '2pi'
        xml_data['ionic_phase'] = berry_phase['totalPhase']['@ionic']
        xml_data['ionic_phase_units'] = '2pi'
        xml_data['electronic_phase'] = berry_phase['totalPhase']['@electronic']
        xml_data['electronic_phase_units'] = '2pi'
        xml_data['polarization'] = polarization
        xml_data['polarization_module'] = polarization_modulus  # should be called "modulus"
        xml_data['polarization_units'] = 'C / m^2'
        xml_data['polarization_direction'] = berry_phase['totalPolarization']['direction']
        # TODO: add conversion for (e/Omega).bohr (requires to know Omega, the volume of the cell)
        # TODO (maybe): Not parsed:
        # - individual ionic phases
        # - individual electronic phases and weights

    # TODO: We should put the `non_periodic_cell_correction` string in (?)
    atomic_species_name = []
    atoms = []

    for atom in outputs['atomic_structure']['atomic_positions']['atom']:
        atomic_species_name.append(atom['@name'])
        atoms.append([atom['@name'], [coord * CONSTANTS.bohr_to_ang for coord in atom['$']]])

    species = outputs['atomic_species']['species']
    structure_data = {
        'atomic_positions_units': 'Angstrom',
        'direct_lattice_vectors_units': 'Angstrom',
        # ??? 'atoms_if_pos_list': [[1, 1, 1], [1, 1, 1]],
        'number_of_atoms': outputs['atomic_structure']['@nat'],
        'lattice_parameter': output_alat_angstrom,
        'reciprocal_lattice_vectors': [
            outputs['basis_set']['reciprocal_lattice']['b1'],
            outputs['basis_set']['reciprocal_lattice']['b2'],
            outputs['basis_set']['reciprocal_lattice']['b3'],
        ],
        'atoms': atoms,
        'cell': {
            'lattice_vectors': lattice_vectors,
            'volume': cell_volume(*lattice_vectors),
            'atoms': atoms,
        },
        'lattice_parameter_xml': output_alat_bohr,
        'number_of_species': outputs['atomic_species']['@ntyp'],
        'species': {
            'index': [i + 1 for i, specie in enumerate(species)],
            'pseudo': [specie['pseudo_file'] for specie in species],
            'mass': [specie['mass'] for specie in species],
            'type': [specie['@name'] for specie in species],
        },
    }

    xml_data['volume'] = structure_data['cell']['volume']
    xml_data['structure'] = structure_data
    xml_data['trajectory'] = collections.defaultdict(list)
    xml_data['trajectory']['atomic_species_name'] = atomic_species_name

    for frame in xml_dictionary.get('step', []):
        parse_step_to_trajectory(xml_data['trajectory'], frame)

    calculation_type = inputs.get('control_variables', {}).get('calculation', 'scf')

    # In case of an SCF calculation, there are no trajectory steps so parse from the final outputs. For a vc-relax, the
    # code performs a final SCF, the results of which are not added as a step but are part of the final outputs.
    if calculation_type in ['scf', 'vc-relax']:
        parse_step_to_trajectory(xml_data['trajectory'], outputs, skip_structure=True)

    # For some reason, the legacy trajectory structure contained a key `steps` which was a list of integers from 0 to
    # N - 1 where N is the number steps in the trajectory.
    if 'step' in xml_dictionary:
        xml_data['trajectory']['steps'] = list(range(len(xml_dictionary['step'])))

    xml_data['total_number_of_scf_iterations'] = sum(xml_data['trajectory']['scf_iterations'])

    return xml_data, logs


def parse_xml_post_6_2(xml):
    """DEPRECATED: Use parse_xml() instead.

    This function is kept for backward compatibility but will be removed in v5.0.

    :param xml: parsed XML ElementTree object
    :returns: tuple of two dictionaries, with the parsed data and log messages, respectively
    """
    import warnings
    import io
    from aiida.common.warnings import AiidaDeprecationWarning

    warnings.warn(
        'parse_xml_post_6_2() is deprecated. Use parse_xml() instead which takes a file-like object.\n'
        'This function will be removed in aiida-quantumespresso v5.0.',
        AiidaDeprecationWarning,
        stacklevel=2,
    )

    # Convert the ElementTree object back to XML string and create a file-like object
    xml_string = ElementTree.tostring(xml.getroot(), encoding='unicode')
    xml_file = io.StringIO(xml_string)

    return parse_xml(xml_file)


def parse_step_to_trajectory(trajectory, data, skip_structure=False):
    """Parse the information of a single step to the dictionary in `trajectory`."""
    if 'scf_conv' in data and 'n_scf_steps' in data['scf_conv']:
        scf_iterations = data['scf_conv']['n_scf_steps']  # Can be zero in case of initialization-only calculation
        if scf_iterations:
            trajectory['scf_iterations'].append(scf_iterations)

    if 'convergence_info' in data:
        convergence_info = data['convergence_info']
        if 'scf_conv' in convergence_info and 'n_scf_steps' in convergence_info['scf_conv']:
            trajectory['scf_iterations'].append(convergence_info['scf_conv']['n_scf_steps'])

    if 'atomic_structure' in data and not skip_structure:
        atomic_structure = data['atomic_structure']

        if 'atomic_positions' in atomic_structure:
            positions = np.array([a['$'] for a in atomic_structure['atomic_positions']['atom']])
            trajectory['positions'].append(positions * CONSTANTS.bohr_to_ang)

        if 'cell' in atomic_structure:
            cell = atomic_structure['cell']
            cell = np.array([cell['a1'], cell['a2'], cell['a3']])
            trajectory['cells'].append(cell * CONSTANTS.bohr_to_ang)

    if 'total_energy' in data:
        total_energy = data['total_energy']

        for key, key_alt in [
            ('etot', 'energy'),
            ('ehart', 'energy_hartree'),
            ('ewald', 'energy_ewald'),
            ('etxc', 'energy_xc'),
        ]:
            if key in total_energy:
                trajectory[key_alt].append(total_energy[key] * CONSTANTS.hartree_to_ev)

    if 'forces' in data and '$' in data['forces']:
        forces = np.array(data['forces']['$'])
        dimensions = data['forces']['@dims']  # Like [3, 2], should be reversed to reshape the forces array
        forces = forces * (CONSTANTS.ry_to_ev * 2) / CONSTANTS.bohr_to_ang
        trajectory['forces'].append(forces.reshape(dimensions[::-1]))

    if 'stress' in data and '$' in data['stress']:
        stress = np.array(data['stress']['$'])
        dimensions = data['stress']['@dims']  # Like [3, 3], should be reversed to reshape the stress array
        stress = stress * CONSTANTS.au_gpa
        trajectory['stress'].append(stress.reshape(dimensions[::-1]))

    if 'electric_field' in data and 'finiteElectricFieldInfo' in data['electric_field']:
        trajectory['electronic_dipole_cartesian_axes'].append(
            np.array(data['electric_field']['finiteElectricFieldInfo']['electronicDipole'])
        )
        trajectory['ionic_dipole_cartesian_axes'].append(
            np.array(data['electric_field']['finiteElectricFieldInfo']['ionicDipole'])
        )
