"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentActionGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const custom_resource_provider_1 = require("./custom-resource-provider");
const utils_1 = require("../../common/helpers/utils");
class AgentActionGroup extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        validateAgentActionGroupProps(props);
        const apiSchema = props.apiSchema?.bind(this);
        validateApiSchema(apiSchema);
        const actionGroupName = props.actionGroupName ?? (0, utils_1.generatePhysicalNameV2)(this, 'action-group', { maxLength: 100, separator: '-' });
        let actionGroupExecutor = undefined;
        if (props.actionGroupExecutor?.functionArn) {
            actionGroupExecutor = { lambda: props.actionGroupExecutor.functionArn };
            props.actionGroupExecutor.addPermission('AgentLambdaInvocationPolicy', {
                principal: new iam.ServicePrincipal('bedrock.amazonaws.com'),
                sourceArn: props.agent.agentArn,
                sourceAccount: cdk.Stack.of(this).account,
            });
        }
        const crProvider = custom_resource_provider_1.BedrockCRProvider.getProvider(this);
        const agentActionGroup = new cdk.CustomResource(this, 'ActionGroup', {
            serviceToken: crProvider.serviceToken,
            resourceType: 'Custom::Bedrock-AgentActionGroup',
            properties: {
                agentId: props.agent.agentId,
                actionGroupExecutor,
                actionGroupName,
                actionGroupState: props.actionGroupState,
                apiSchema,
                description: props.description,
                parentActionGroupSignature: props.parentActionGroupSignature,
                shouldPrepareAgent: props.shouldPrepareAgent,
            },
        });
        this.actionGroupId = agentActionGroup.getAttString('actionGroupId');
        const actionGroupCRPolicy = new iam.Policy(this, 'AgentActionGroupCRPolicy', {
            roles: [crProvider.role],
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:CreateAgentActionGroup',
                        'bedrock:DeleteAgentActionGroup',
                        'bedrock:UpdateAgentActionGroup',
                    ],
                    resources: [
                        cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'agent',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    ],
                }),
            ],
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(actionGroupCRPolicy, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'Bedrock AgentActionGroup calls have wildcards restricted to agents in the account.',
            },
        ], true);
        agentActionGroup.node.addDependency(actionGroupCRPolicy);
        props.agent._addAliasDependency(agentActionGroup.getAttString('updatedAt'));
    }
}
exports.AgentActionGroup = AgentActionGroup;
_a = JSII_RTTI_SYMBOL_1;
AgentActionGroup[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.AgentActionGroup", version: "0.1.94" };
function validateAgentActionGroupProps(props) {
    if (props.parentActionGroupSignature && props.description) {
        throw new Error('Cannot specify both description and parentActionSignature');
    }
}
function validateApiSchema(apiSchema) {
    // mutually exclusive
    if (apiSchema?.payload && apiSchema?.s3) {
        throw new Error('Cannot specify both apiSchema.payload and apiSchema.s3');
    }
}
//# sourceMappingURL=data:application/json;base64,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