/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as cdk from 'aws-cdk-lib';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as kms from 'aws-cdk-lib/aws-kms';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import { Construct } from 'constructs';
import { AgentActionGroup } from './agent-action-group';
import { AgentAlias } from './agent-alias';
import { ApiSchema } from './api-schema';
import { KnowledgeBase } from './knowledge-base';
import { BedrockFoundationModel } from './models';
/**
 * The step in the agent sequence that this prompt configuration applies to.
 */
export declare enum PromptType {
    PRE_PROCESSING = "PRE_PROCESSING",
    ORCHESTRATION = "ORCHESTRATION",
    POST_PROCESSING = "POST_PROCESSING",
    KNOWLEDGE_BASE_RESPONSE_GENERATION = "KNOWLEDGE_BASE_RESPONSE_GENERATION"
}
/**
 * Specifies whether to override the default parser Lambda function when
 * parsing the raw foundation model output in the part of the agent sequence
 * defined by the promptType. If you set the field as OVERRIDEN, the
 * overrideLambda field in the PromptOverrideConfiguration must be specified
 * with the ARN of a Lambda function.
 */
export declare enum ParserMode {
    DEFAULT = "DEFAULT",
    OVERRIDDEN = "OVERRIDDEN"
}
/**
 * Specifies whether to override the default prompt template for this
 * promptType. Set this value to OVERRIDDEN to use the prompt that you
 * provide in the basePromptTemplate. If you leave it as DEFAULT, the agent
 * uses a default prompt template.
 */
export declare enum PromptCreationMode {
    DEFAULT = "DEFAULT",
    OVERRIDDEN = "OVERRIDDEN"
}
/**
 * Specifies whether to allow the agent to carry out the step specified in the
 * promptType. If you set this value to DISABLED, the agent skips that step.
 * The default state for each promptType is as follows.
 *
 *     PRE_PROCESSING – ENABLED
 *     ORCHESTRATION – ENABLED
 *     KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
 *     POST_PROCESSING – DISABLED
 */
export declare enum PromptState {
    ENABLED = "ENABLED",
    DISABLED = "DISABLED"
}
/**
 * LLM inference configuration
 */
export interface InferenceConfiguration {
    /**
     * The likelihood of the model selecting higher-probability options while
     * generating a response. A lower value makes the model more likely to choose
     * higher-probability options, while a higher value makes the model more
     * likely to choose lower-probability options.
     *
     * Floating point
     *
     * @min 0
     * @max 1
     */
    readonly temperature: number;
    /**
     * While generating a response, the model determines the probability of the
     * following token at each point of generation. The value that you set for
     * Top P determines the number of most-likely candidates from which the model
     * chooses the next token in the sequence. For example, if you set topP to
     * 80, the model only selects the next token from the top 80% of the
     * probability distribution of next tokens.
     *
     * Floating point
     *
     * @min 0
     * @max 1
     */
    readonly topP: number;
    /**
     * While generating a response, the model determines the probability of the
     * following token at each point of generation. The value that you set for
     * topK is the number of most-likely candidates from which the model chooses
     * the next token in the sequence. For example, if you set topK to 50, the
     * model selects the next token from among the top 50 most likely choices.
     *
     * Integer
     *
     * @min 0
     * @max 500
     */
    readonly topK: number;
    /**
     * A list of stop sequences. A stop sequence is a sequence of characters that
     * causes the model to stop generating the response.
     *
     * @length 0-4
     */
    readonly stopSequences: string[];
    /**
     * The maximum number of tokens to generate in the response.
     *
     * Integer
     *
     * @min 0
     * @max 4096
     */
    readonly maximumLength: number;
}
/**
 * Contains configurations to override a prompt template in one part of an agent sequence.
 */
export interface PromptConfiguration {
    /**
     * The step in the agent sequence that this prompt configuration applies to.
     */
    readonly promptType: PromptType;
    /**
     * Contains inference parameters to use when the agent invokes a foundation
     * model in the part of the agent sequence defined by the promptType.
     */
    readonly inferenceConfiguration: InferenceConfiguration;
    /**
     * Defines the prompt template with which to replace the default prompt template.
     *
     * @length 0-100000
     */
    readonly basePromptTemplate: string;
    /**
     * Specifies whether to override the default parser Lambda function when
     * parsing the raw foundation model output in the part of the agent sequence
     * defined by the promptType. If you set the field as OVERRIDEN, the
     * overrideLambda field in the PromptOverrideConfiguration must be specified
     * with the ARN of a Lambda function.
     */
    readonly parserMode?: ParserMode;
    /**
     * Specifies whether to override the default prompt template for this
     * promptType. Set this value to OVERRIDDEN to use the prompt that you
     * provide in the basePromptTemplate. If you leave it as DEFAULT, the agent
     * uses a default prompt template.
     */
    readonly promptCreationMode: PromptCreationMode;
    /**
     * Specifies whether to allow the agent to carry out the step specified in
     * the promptType. If you set this value to DISABLED, the agent skips that
     * step. The default state for each promptType is as follows.
     *
     *     PRE_PROCESSING – ENABLED
     *     ORCHESTRATION – ENABLED
     *     KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
     *     POST_PROCESSING – DISABLED
     */
    readonly promptState: PromptState;
}
/**
 * Contains configurations to override prompts in different parts of an agent sequence.
 */
export interface PromptOverrideConfiguration {
    /**
     * Contains configurations to override a prompt template in one part of an agent sequence.
     */
    readonly promptConfigurations: PromptConfiguration[];
    /**
     * The ARN of the Lambda function to use when parsing the raw foundation
     * model output in parts of the agent sequence. If you specify this field,
     * at least one of the promptConfigurations must contain a parserMode value
     * that is set to OVERRIDDEN.
     */
    readonly overrideLambda?: string;
}
/**
 * Properties for a Bedrock Agent.
 */
export interface AgentProps {
    /**
     * The Bedrock text foundation model for the agent to use.
     */
    readonly foundationModel: BedrockFoundationModel;
    /**
     * The name of the agent.
     *
     * @default - A name is automatically generated.
     */
    readonly name?: string;
    /**
     * A narrative instruction to provide the agent as context.
     */
    readonly instruction: string;
    /**
     * A description of the agent.
     *
     * @default - No description is provided.
     */
    readonly description?: string;
    /**
     * Knowledge Bases to make available to the agent.
     *
     * @default - No knowledge base is used.
     */
    readonly knowledgeBases?: KnowledgeBase[];
    /**
     * How long sessions should be kept open for the agent.
     *
     * @default - 1 hour
     */
    readonly idleSessionTTL?: cdk.Duration;
    /**
     * KMS encryption key to use for the agent.
     *
     * @default - An AWS managed key is used.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Overrides for the agent.
     *
     * @default - No overrides are provided.
     */
    readonly promptOverrideConfiguration?: PromptOverrideConfiguration;
    /**
     * Name of the alias for the agent.
     *
     * @default - No alias is created.
     */
    readonly aliasName?: string;
    /**
     * Whether to prepare the agent for use.
     *
     * @default - false
     */
    readonly shouldPrepareAgent?: boolean;
}
/**
 * Properties to add an Alias to an Agent
 */
export interface AddAgentAliasProps {
    /**
     * The name for the agent alias.
     */
    readonly aliasName: string;
    /**
     * The version of the agent to associate with the agent alias.
     *
     * @default - Creates a new version of the agent.
     */
    readonly agentVersion?: string;
}
/**
 * Properties to add an ActionGroup to an Agent
 */
export interface AddAgentActionGroupProps {
    /**
     * The Lambda function containing the business logic that is carried out upon invoking the action.
     */
    readonly actionGroupExecutor?: lambda.IFunction;
    /**
     * The name of the action group.
     *
     * @default - a name is generated by CloudFormation.
     */
    readonly actionGroupName?: string;
    /**
     * Specifies whether the action group is available for the agent to invoke or not when sending an InvokeAgent request.
     */
    readonly actionGroupState?: 'ENABLED' | 'DISABLED';
    /**
     * Contains details about the S3 object containing the OpenAPI schema for the action group. For more information, see
     * [Action group OpenAPI schemas](https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html).
     */
    readonly apiSchema?: ApiSchema;
    /**
     * A description of the action group.
     *
     * @note This object is a Union. Only one member of this object can be specified or returned.
     */
    readonly description?: string;
    /**
     * If you specify this value as AMAZON.UserInput, the agent will prompt additional information from the user when it
     * doesn't have enough information to respond to an utterance. Leave this field blank if you don't want the agent to
     * prompt additional information.
     */
    readonly parentActionGroupSignature?: 'AMAZON.UserInput';
}
/**
 * Deploy a Bedrock Agent.
 */
export declare class Agent extends Construct implements cdk.ITaggableV2 {
    /**
     * The name of the agent.
     */
    readonly name: string;
    /**
     * The IAM role for the agent.
     */
    readonly role: iam.Role;
    /**
     * The unique identifier of the agent.
     */
    readonly agentId: string;
    /**
     * The ARN of the agent.
     */
    readonly agentArn: string;
    /**
     * The unique identifier of the agent alias.
     */
    readonly aliasId?: string;
    /**
     * The ARN of the agent alias.
     */
    readonly aliasArn?: string;
    /**
     * The name for the agent alias.
     */
    readonly aliasName?: string;
    /**
     * TagManager facilitates a common implementation of tagging for Constructs
     */
    readonly cdkTagManager: cdk.TagManager;
    /**
     * A list of values to indicate if PrepareAgent or an Alias needs to be updated.
     * @private
     */
    private resourceUpdates;
    /**
     * If prepare agent should be called on resource updates.
     * @private
     */
    private readonly shouldPrepareAgent;
    constructor(scope: Construct, id: string, props: AgentProps);
    /**
     * Add an alias to the agent.
     */
    addAlias(props: AddAgentAliasProps): AgentAlias;
    /**
     * Add an action group to the agent.
     */
    addActionGroup(props: AddAgentActionGroupProps): AgentActionGroup;
    /**
     * Register a dependency for aliases.
     *
     * @param updatedAt - The updatedAt of the resource that will be registered as a dependency.
     *
     * @internal This is an internal core function and should not be called directly.
     */
    _addAliasDependency(updatedAt: string): void;
}
/**
 * Validate the inferenceConfiguration of a prompt override.
 *
 * @internal This is an internal core function and should not be called directly.
 */
export declare function validateInferenceConfiguration(inferenceConfiguration: InferenceConfiguration): void;
/**
 * Validate the promptOverrideConfiguration.
 *
 * @internal This is an internal core function and should not be called directly.
 */
export declare function validatePromptOverrideConfiguration(promptOverrideConfiguration: PromptOverrideConfiguration | undefined): void;
