"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.validatePromptOverrideConfiguration = exports.validateInferenceConfiguration = exports.Agent = exports.PromptState = exports.PromptCreationMode = exports.ParserMode = exports.PromptType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const agent_action_group_1 = require("./agent-action-group");
const agent_alias_1 = require("./agent-alias");
const custom_resource_provider_1 = require("./custom-resource-provider");
const utils_1 = require("../../common/helpers/utils");
/**
 * The step in the agent sequence that this prompt configuration applies to.
 */
var PromptType;
(function (PromptType) {
    PromptType["PRE_PROCESSING"] = "PRE_PROCESSING";
    PromptType["ORCHESTRATION"] = "ORCHESTRATION";
    PromptType["POST_PROCESSING"] = "POST_PROCESSING";
    PromptType["KNOWLEDGE_BASE_RESPONSE_GENERATION"] = "KNOWLEDGE_BASE_RESPONSE_GENERATION";
})(PromptType || (exports.PromptType = PromptType = {}));
/**
 * Specifies whether to override the default parser Lambda function when
 * parsing the raw foundation model output in the part of the agent sequence
 * defined by the promptType. If you set the field as OVERRIDEN, the
 * overrideLambda field in the PromptOverrideConfiguration must be specified
 * with the ARN of a Lambda function.
 */
var ParserMode;
(function (ParserMode) {
    ParserMode["DEFAULT"] = "DEFAULT";
    ParserMode["OVERRIDDEN"] = "OVERRIDDEN";
})(ParserMode || (exports.ParserMode = ParserMode = {}));
/**
 * Specifies whether to override the default prompt template for this
 * promptType. Set this value to OVERRIDDEN to use the prompt that you
 * provide in the basePromptTemplate. If you leave it as DEFAULT, the agent
 * uses a default prompt template.
 */
var PromptCreationMode;
(function (PromptCreationMode) {
    PromptCreationMode["DEFAULT"] = "DEFAULT";
    PromptCreationMode["OVERRIDDEN"] = "OVERRIDDEN";
})(PromptCreationMode || (exports.PromptCreationMode = PromptCreationMode = {}));
/**
 * Specifies whether to allow the agent to carry out the step specified in the
 * promptType. If you set this value to DISABLED, the agent skips that step.
 * The default state for each promptType is as follows.
 *
 *     PRE_PROCESSING – ENABLED
 *     ORCHESTRATION – ENABLED
 *     KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
 *     POST_PROCESSING – DISABLED
 */
var PromptState;
(function (PromptState) {
    PromptState["ENABLED"] = "ENABLED";
    PromptState["DISABLED"] = "DISABLED";
})(PromptState || (exports.PromptState = PromptState = {}));
/**
 * Deploy a Bedrock Agent.
 */
class Agent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * TagManager facilitates a common implementation of tagging for Constructs
         */
        this.cdkTagManager = new cdk.TagManager(cdk.TagType.MAP, 'Custom::Bedrock-Agent');
        /**
         * A list of values to indicate if PrepareAgent or an Alias needs to be updated.
         * @private
         */
        this.resourceUpdates = [];
        validatePromptOverrideConfiguration(props.promptOverrideConfiguration);
        this.shouldPrepareAgent = props.shouldPrepareAgent ?? false;
        validateModel(props.foundationModel);
        this.name = props.name ?? (0, utils_1.generatePhysicalNameV2)(this, 'bedrock-agent', { maxLength: 32, lower: true, separator: '-' });
        this.role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('bedrock.amazonaws.com'),
            roleName: (0, utils_1.generatePhysicalNameV2)(this, 'AmazonBedrockExecutionRoleForAgents_', { maxLength: 64, lower: false }),
        });
        this.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            principals: [new iam.ServicePrincipal('bedrock.amazonaws.com')],
            conditions: {
                StringEquals: {
                    'aws:SourceAccount': cdk.Stack.of(this).account,
                },
                ArnLike: {
                    'aws:SourceArn': cdk.Stack.of(this).formatArn({
                        service: 'bedrock',
                        resource: 'agent',
                        resourceName: '*',
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                    }),
                },
            },
        }));
        new iam.Policy(this, 'AgentFMPolicy', {
            roles: [this.role],
            statements: [
                new iam.PolicyStatement({
                    actions: ['bedrock:InvokeModel'],
                    resources: [props.foundationModel.asArn(this)],
                }),
            ],
        });
        if (props.knowledgeBases && props.knowledgeBases.length > 0) {
            new iam.Policy(this, 'AgentKBPolicy', {
                roles: [this.role],
                statements: [
                    new iam.PolicyStatement({
                        actions: [
                            'bedrock:UpdateKnowledgeBase',
                            'bedrock:Retrieve',
                        ],
                        resources: props.knowledgeBases.map(kb => kb.knowledgeBaseArn),
                    }),
                ],
            });
        }
        const crProvider = custom_resource_provider_1.BedrockCRProvider.getProvider(this);
        const agent = new cdk.CustomResource(this, 'Agent', {
            serviceToken: crProvider.serviceToken,
            resourceType: 'Custom::Bedrock-Agent',
            properties: {
                agentName: this.name,
                foundationModel: String(props.foundationModel),
                instruction: props.instruction,
                description: props.description,
                idleSessionTTLInSeconds: props.idleSessionTTL?.toSeconds(),
                agentResourceRoleArn: this.role.roleArn,
                customerEncryptionKeyArn: props.encryptionKey?.keyArn,
                tags: this.cdkTagManager.renderedTags,
                promptOverrideConfiguration: props.promptOverrideConfiguration,
                shouldPrepareAgent: this.shouldPrepareAgent,
            },
        });
        this.agentId = agent.getAttString('agentId');
        this.agentArn = agent.getAttString('agentArn');
        const agentCRPolicy = new iam.Policy(this, 'AgentCRPolicy', {
            roles: [crProvider.role],
            statements: [
                new iam.PolicyStatement({
                    actions: ['iam:PassRole'],
                    resources: [this.role.roleArn],
                }),
                new iam.PolicyStatement({
                    actions: ['bedrock:CreateAgent'],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:DeleteAgent',
                        'bedrock:UpdateAgent',
                        'bedrock:TagResource',
                        'bedrock:GetAgent',
                        'bedrock:PrepareAgent',
                    ],
                    resources: [
                        cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'agent',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    ],
                }),
            ],
        });
        agent.node.addDependency(agentCRPolicy);
        agent.node.addDependency(crProvider);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(agentCRPolicy, [
            {
                id: 'AwsSolutions-IAM5',
                reason: "Bedrock CreateAgent can't be restricted by resource.",
            },
        ], true);
        this._addAliasDependency(agent.getAttString('updatedAt'));
        if (props.knowledgeBases && props.knowledgeBases.length > 0) {
            const kbAssocCRPolicy = new iam.Policy(this, 'KBAssocCRPolicy', {
                roles: [crProvider.role],
                statements: [
                    new iam.PolicyStatement({
                        actions: [
                            'bedrock:AssociateAgentKnowledgeBase',
                            'bedrock:UpdateAgentKnowledgeBase',
                            'bedrock:DisassociateAgentKnowledgeBase',
                        ],
                        resources: [
                            cdk.Stack.of(this).formatArn({
                                service: 'bedrock',
                                resource: 'agent',
                                resourceName: '*',
                                arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                            }),
                            cdk.Stack.of(this).formatArn({
                                service: 'bedrock',
                                resource: 'knowledge-base',
                                resourceName: '*',
                                arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                            }),
                        ],
                    }),
                ],
            });
            cdk_nag_1.NagSuppressions.addResourceSuppressions(kbAssocCRPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'Bedrock Agent/KB associations have wildcards restricted to agents and kbs in the account.',
                },
            ], true);
            for (let kb of props.knowledgeBases ?? []) {
                if (!kb.instruction) {
                    throw new Error('Agent Knowledge Bases require instructions.');
                }
                const kbAssoc = new cdk.CustomResource(this, `KBAssoc-${kb.name}`, {
                    serviceToken: crProvider.serviceToken,
                    resourceType: 'Custom::Bedrock-AgentKnowledgeBase',
                    properties: {
                        agentId: this.agentId,
                        knowledgeBaseId: kb.knowledgeBaseId,
                        description: kb.instruction,
                        shouldPrepareAgent: this.shouldPrepareAgent,
                    },
                });
                kbAssoc.node.addDependency(kbAssocCRPolicy);
                kbAssoc.node.addDependency(crProvider);
                this._addAliasDependency(kbAssoc.getAttString('updatedAt'));
            }
        }
        if (props.aliasName) {
            const alias = this.addAlias({
                aliasName: props.aliasName,
            });
            this.aliasId = alias.aliasId;
            this.aliasArn = alias.aliasArn;
            this.aliasName = alias.aliasName;
        }
    }
    /**
     * Add an alias to the agent.
     */
    addAlias(props) {
        const alias = new agent_alias_1.AgentAlias(this, `AgentAlias-${props.aliasName}`, {
            agentId: this.agentId,
            agentVersion: props.agentVersion,
            resourceUpdates: cdk.Lazy.list({ produce: () => this.resourceUpdates }),
            aliasName: props.aliasName,
        });
        return alias;
    }
    /**
     * Add an action group to the agent.
     */
    addActionGroup(props) {
        const actionGroupName = props.actionGroupName ?? (0, utils_1.generatePhysicalNameV2)(this, 'action-group', { maxLength: 100, separator: '-' });
        return new agent_action_group_1.AgentActionGroup(this, `AgentActionGroup-${actionGroupName}`, {
            agent: this,
            actionGroupName,
            description: props.description,
            parentActionGroupSignature: props.parentActionGroupSignature,
            actionGroupExecutor: props.actionGroupExecutor,
            actionGroupState: props.actionGroupState,
            apiSchema: props.apiSchema,
            shouldPrepareAgent: this.shouldPrepareAgent,
        });
    }
    /**
     * Register a dependency for aliases.
     *
     * @param updatedAt - The updatedAt of the resource that will be registered as a dependency.
     *
     * @internal This is an internal core function and should not be called directly.
     */
    _addAliasDependency(updatedAt) {
        if (updatedAt) {
            this.resourceUpdates.push(updatedAt);
        }
    }
}
exports.Agent = Agent;
_a = JSII_RTTI_SYMBOL_1;
Agent[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.Agent", version: "0.1.94" };
/**
 * Validate that Bedrock Agents can use the selected model.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateModel(foundationModel) {
    if (!foundationModel.supportsAgents) {
        throw new Error(`The model ${foundationModel} is not supported by Bedrock Agents.`);
    }
}
/**
 * Validate the inferenceConfiguration of a prompt override.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateInferenceConfiguration(inferenceConfiguration) {
    if (inferenceConfiguration.topK < 0 || inferenceConfiguration.topK > 500) {
        throw new Error('topK must be between 0 and 500');
    }
    if (!Number.isInteger(inferenceConfiguration.topK)) {
        throw new Error('topK must be an integer');
    }
    if (inferenceConfiguration.stopSequences.length > 4) {
        throw new Error('stopSequences cannot contain more than 4 elements');
    }
    if (inferenceConfiguration.maximumLength < 0 || inferenceConfiguration.maximumLength > 4096) {
        throw new Error('maximumLength must be between 0 and 4096');
    }
    if (!Number.isInteger(inferenceConfiguration.maximumLength)) {
        throw new Error('maximumLength must be an integer');
    }
    if (inferenceConfiguration.topP < 0 || inferenceConfiguration.topP > 1) {
        throw new Error('topP must be between 0 and 1');
    }
    if (inferenceConfiguration.temperature < 0 || inferenceConfiguration.temperature > 1) {
        throw new Error('temperature must be between 0 and 1');
    }
}
exports.validateInferenceConfiguration = validateInferenceConfiguration;
/**
 * Validate the promptOverrideConfiguration.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validatePromptOverrideConfiguration(promptOverrideConfiguration) {
    if (!promptOverrideConfiguration) {
        return;
    }
    if (promptOverrideConfiguration.overrideLambda &&
        promptOverrideConfiguration.promptConfigurations.some(pc => pc.parserMode !== ParserMode.OVERRIDDEN)) {
        throw new Error('overrideLambda can only be used if all promptConfigurations have a parserMode value of OVERRIDDEN');
    }
    if (!promptOverrideConfiguration.overrideLambda &&
        promptOverrideConfiguration.promptConfigurations.some(pc => pc.parserMode === ParserMode.OVERRIDDEN)) {
        throw new Error('At least one promptConfiguration has a parserMode value of OVERRIDDEN, but no overrideLambda is specified');
    }
    // check inferenceConfiguration number types
    Object.values(promptOverrideConfiguration.promptConfigurations).forEach(pc => {
        validateInferenceConfiguration(pc.inferenceConfiguration);
    });
    return;
}
exports.validatePromptOverrideConfiguration = validatePromptOverrideConfiguration;
//# sourceMappingURL=data:application/json;base64,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