/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as cdk from 'aws-cdk-lib';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { BedrockFoundationModel } from './models';
import { VectorIndex } from '../opensearch-vectorindex';
import { VectorCollection } from '../opensearchserverless';
/**
 * Properties for a knowledge base
 */
export interface KnowledgeBaseProps {
    /**
     * The embeddings model for the knowledge base
     */
    readonly embeddingsModel: BedrockFoundationModel;
    /**
     * The description of the knowledge base.
     *
     * @default - No description provided.
     */
    readonly description?: string;
    /**
     * A narrative description of the knowledge base.
     *
     * A Bedrock Agent can use this instruction to determine if it should
     * query this Knowledge Base.
     *
     * @default - No description provided.
     */
    readonly instruction?: string;
    /**
     * The name of the vector index.
     *
     * @default - 'bedrock-knowledge-base-default-index'
     */
    readonly indexName?: string;
    /**
     * The name of the field in the vector index.
     *
     * @default - 'bedrock-knowledge-base-default-vector'
     */
    readonly vectorField?: string;
    /**
     * The vector store for the knowledge base.
     *
     * @default - A new OpenSearch Serverless vector collection is created.
     */
    readonly vectorStore?: VectorCollection;
    /**
     * The vector index for the knowledge base.
     *
     * @default - A new vector index is created on the Vector Collection
     */
    readonly vectorIndex?: VectorIndex;
}
/**
 * Deploys a Bedrock Knowledge Base and configures a backend vector store.
 *
 * At the moment, only OpenSearch Serverless is supported as a vector store.
 * This construct creates the collection and index.
 */
export declare class KnowledgeBase extends Construct implements cdk.ITaggableV2 {
    /**
     * The name of the knowledge base.
     */
    readonly name: string;
    /**
     * The role the Knowledge Base uses to access the vector store and data source.
     */
    readonly role: iam.Role;
    /**
     * The vector store for the knowledge base.
     */
    readonly vectorStore: VectorCollection;
    /**
     * A narrative instruction of the knowledge base.
     */
    readonly instruction?: string;
    /**
     * The ARN of the knowledge base.
     */
    readonly knowledgeBaseArn: string;
    /**
     * The ID of the knowledge base.
     */
    readonly knowledgeBaseId: string;
    /**
     * TagManager facilitates a common implementation of tagging for Constructs
     */
    readonly cdkTagManager: cdk.TagManager;
    /**
     * The OpenSearch vector index for the knowledge base.
     * @private
     */
    private vectorIndex?;
    constructor(scope: Construct, id: string, props: KnowledgeBaseProps);
}
